/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef _DDATACONTROL_H
#define _DDATACONTROL_H

#include <qdatastream.h>
#include <qmap.h>
#include <qobject.h>

#include "settings.h"
#include "aux.h"
#include "DBase.h"
#include "DItem.h"

class DissertParser;
class DGenerator;
class QTextStream;
class KURL;
class DDataItem;
class QDir;
class KTempDir;
class DSpell;
class DGuiView;
/**
 * This class handles the necessary code for registering
 * and handling the mindmap items (data side)
 *
 * @short Main handler for mindmap item data
 * @author Thomas Nagy <tnagy2^8@yahoo.fr>
 * @version 0.3.1
 */
class DDataControl : public QObject, public DBase
{
	Q_OBJECT

	friend class DissertParser;
	friend class FFParser;
	friend class DSpell;
	public:
		enum flags
		{
			e_warning,
			e_good,
			e_idea,
			e_work,
			e_clarify,
			e_question,
			e_trash,
			e_meeting,
			e_trouble
		};

		/// default constructor
		DDataControl(QObject* parent = 0, const char * name = 0);

		/// default destructor
		~DDataControl();

		// create a new item (makes a unique id, and checks for bugs)
		int createItem(int id=DItem::NOITEM);

		/*
		 * removes a mindmap item of particular id. 
		 * does nothing if the item does not exist
		 */
		void removeItem( int );

		/// sets a mindmap item's parent
		void linkItems( int parent, int child );

		/// unlinks two mindmap items
		void unlinkItems( int parent, int child );

		/*
		 *  remove the parent of a mindmap item 
		 *  this is a shortcut for unlinkItem(NOITEM, child)
		 */
		void setOrphan(int child);

		/*
		 * this unlinks all children from a particular mindmap item
		 */
		void killChildren(int id);

		/*
		 * this unlinks all children from a particular mindmap item recursively
		 */
		void killFamily(int id);

		/*
		 * this disconnects an item completely from
		 * its parents and children
		 */
		void disconnectItem(int);

		/// notify that a mindmap item has changed
		void notifyChildChange(int id);

		/// notify item children that something has changed
		void notifyChildren(int id);

		/// save the mindmap data to a file
		bool saveToFile(const KURL &);

		/// load a mindmap from a file
		bool loadFromFile(const KURL &);

		/// used to dump the data to a file
		void printTree(QTextStream&);

		/// returns to total amount of items
		int countItems();

		/// returns the root of the biggest mindmap tree
		int rootID();

		/*
		 *  tells if the mindmap is valid in particular
		 *  when trying to use the document generators
		 */
		bool canGenerate();

		/*
		 * debug a mindmap item, dumping its connections
		 */
		void debugItem(int id);

		/*
		 * This checks the consistency of the mindmap trees
		 * and tries to fix them when an error is detected
		 */
		void checkConsistency();

		/// return the temporary directory for the cached documents (pictures)
		QDir * getTmpDir();

		/*
		 * Compare two items on the position in the hierarchy
		 */
		int compare(int id1, int id2) const;

		// notify that one item is selected in a view
		void setItemSelected(int, DGuiView*);

		// emits itemChanged
		void notifyItemModified(int);

		/// per-document settings

		/// author
		QString m_fname;
		QString m_sname;

		/// pic preview size
		int m_pixSize;
		QColor m_col_background;
		QColor m_col_link;
		QColor m_col_ref;

		QFont m_canvasFont;
		bool m_picturesOnCanvas;
		bool m_reverseVideo;

		bool m_showDirection;
		int m_showDirectionSize;

		/// link style
		int m_linkstyle;


		/*
		 * Use this to ask to add or remove a reference
		 */
		bool requestRefChange(int orig, int dest, bool add = true, bool check=true);

	signals:
		/*
		 *  Emitted whenever a particular item changes
		 *  - to refresh the views
		 */
		void itemChanged(int);

		/*
		 * Emitted to refresh the views when an item is created
		 */
		void itemCreated(int);

		/*
		 * Emitted to refresh the views when an item is removed
		 */
		void itemRemoved(int);

		/*
		 * Emitted when a reference is added or removed
		 */
		void refChanged(int, int, bool add);

		/*
		 * Emitted when one item is selected in a view
		 */
		void itemSelected(int, DGuiView*);

		/*
		 * This signal is emitted to set the document state :
		 * true means that the document will need to be saved
		 */
		void documentChanged(bool);


		void settingsChanged();

	public slots:
		/// this method clears the current document
		void clearDocument();

		/// this method is used to set the mindmap state to "changed"
		void docChanged();

		/// this method is used to set the mindmap state to "not changed"
		void docNotChanged();

		void slotFileSpell();

		/// 
		void updateSettings();

	protected:
		/*
		 * this method is used when loading a document, to create a mindmap
		 * item with a particular id
		 */
		DDataItem* createItemWithId(int id);

	private:
		/// last id encountered, mandatory for generating new items (need unique id)
		int m_lastid;

		/// find a dataitem given its id
		DDataItem* dataItem(int id) const;

		/// temporary directory for caching documents (pictures especially)
		KTempDir *m_fileholder;
		DSpell *m_spell;
};

#endif // _DDATACONTROL_H

