/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <pwd.h>
#include <sys/types.h>
#include <unistd.h>

#include <qdir.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qtextstream.h>
#include <qfile.h>
#include <qregexp.h>

#include <karchive.h>
#include <ktar.h>
#include <klocale.h>
#include <ktempfile.h>
#include <ktempdir.h>
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include <kurl.h>
#include <ktempdir.h>
#include <kdebug.h>

#include "settings.h"
#include "DBase.h"
#include "DDataItem.h"
#include "DGuiView.h"
#include "DDataControl.h"
#include "DSpell.h"
#include "FFParser.h"
#include "DissertParser.h"

static const int document_version = 5;

DDataControl::DDataControl(QObject * parent, const char * name) : QObject(parent, name)
{
	m_lastid = 0;
	m_spell = NULL;

	/// author
	m_fname = Settings::fname();
	m_sname = Settings::sname();

	if (!m_fname.length() && !m_sname.length())
	{
		struct passwd *pw = ::getpwuid(getuid());
		if (pw)
		{
			m_sname = QString::fromLocal8Bit( pw->pw_gecos );
			Settings::setSname( m_sname );
		}
	}

	/// pic preview size
	m_pixSize = Settings::pixSize();
	m_col_background = Settings::col_background();
	m_col_link = Settings::col_link();
	m_col_ref = Settings::col_ref();

	m_canvasFont = Settings::canvasFont();
	m_picturesOnCanvas = Settings::picturesOnCanvas();
	m_reverseVideo = Settings::reverseVideo();

	m_showDirectionSize = Settings::showDirectionSize();

	m_linkstyle = Settings::linkStyle();

	//internal connections
	connect(this, SIGNAL(itemChanged(int)), this, SLOT(docChanged()));
	connect(this, SIGNAL(itemCreated(int)), this, SLOT(docChanged()));
	connect(this, SIGNAL(itemRemoved(int)), this, SLOT(docChanged()));

	m_fileholder = new KTempDir();
}

DDataControl::~DDataControl()
{
	m_fileholder->unlink();

	delete m_spell;
	delete m_fileholder;
}

void DDataControl::updateSettings()
{
	emit settingsChanged();
}

DDataItem* DDataControl::createItemWithId(int id)
{
	DDataItem* item = NULL;

	if (isRegistered(id))
	{
		kdWarning()<<"attempted to create an item that already exists!"<<endl;
		return NULL;
	}
	item = new DDataItem(this, id);
	registerItem(item);

	return item;
}

int DDataControl::createItem(int parent)
{
	DDataItem* item = NULL;

	while ( isRegistered(m_lastid) )
		m_lastid++;

	item = createItemWithId(m_lastid);
	if (item)
	{
		emit itemCreated( item->Id() );
	}

	if (parent != DItem::NOITEM)
	{
		linkItems(parent, item->Id());
	}

	setItemSelected(item->Id(), NULL);
	docChanged();

	return item->Id();
}

void DDataControl::linkItems(int parent, int child)
{
	// kdWarning()<<"call to DDataControl::linkItems"<<endl;

	// link only valid items
	if (parent == DItem::NOITEM || child == DItem::NOITEM)
	{
		kdWarning()<<"BUG in DDataControl::linkItems : "<<parent<<" "<<child<<endl;
		return;
	}

	if (parent == child)
		return;

	// check against cycles
	int iter = parent;
	while (iter != DItem::NOITEM)
	{
		if (iter == child)
		{
			//kdWarning()<<"DDataControl::linkItems : attempted to make a cycle (avoided)"<<endl;
			return;
		}

		iter = dataItem(iter)->Parent();
	}

	// for linking, the child must have no parent
	if ( dataItem(child)->Parent() == DItem::NOITEM )
	{
		dataItem(parent)->addChild(child);
		dataItem(child)->setParent(parent);

		emit itemChanged(parent);
		emit itemChanged(child);
	}
}

void DDataControl::unlinkItems(int id1, int id2)
{
	//kdWarning()<<"unlink "<<id1<<" "<<id2<<endl;

	if (id1 == DItem::NOITEM || id2 == DItem::NOITEM)
		return;

	if (id1 == id2)
		return;

	bool changed = false;

	if (!dataItem(id1))
	{
		dataItem(id2)->removeChild(id1);
	}
	else if (!dataItem(id2))
	{
		dataItem(id1)->removeChild(id2);
	}
	else if (dataItem(id1)->Parent() == id2)
	{
		changed = true;
		dataItem(id1)->setParent(DItem::NOITEM);
		dataItem(id2)->removeChild(id1);
	}
	else if (dataItem(id2)->Parent() == id1)
	{
		changed = true;
		dataItem(id2)->setParent(DItem::NOITEM);
		dataItem(id1)->removeChild(id2);
	}

	if (changed)
	{
		emit itemChanged(id1);
		emit itemChanged(id2);
	}
}

void DDataControl::removeItem(int id)
{
	//kdWarning()<<"removeItem "<<id<<endl;
	if (!isRegistered(id) )
		return;

	disconnectItem(id);
	unregisterItem(id);
	emit itemRemoved( id );
	emit itemSelected( DItem::NOITEM, NULL );

	docChanged();
}

void DDataControl::disconnectItem(int id)
{
	if (!isRegistered(id))
		return;

	killChildren(id);
	setOrphan(id);

	docChanged();
}

void DDataControl::setOrphan(int child)
{
	if (!isRegistered(child))
		return;

	if (!child == DItem::NOITEM)
		return;
	int parent = dataItem(child)->Parent();

	if (parent == DItem::NOITEM)
		return;

	if (child == parent)
		return;

	unlinkItems(child, parent);
}

void DDataControl::killChildren(int id)
{
	if (!isRegistered(id))
		return;

	DDataItem* item = dataItem(id);
	while (item->countChildren()>0)
	{
		unlinkItems(id, item->childNum(0));
	}
}

void DDataControl::killFamily(int id)
{
	while (dataItem(id)->countChildren() > 0)
	{
		killFamily( dataItem(id)->childNum(0) );
	}
	setOrphan(id);
}

DDataItem* DDataControl::dataItem(int id) const
{
	return (DDataItem*) DBase::Item(id);
}

void DDataControl::clearDocument()
{
	// just to make sure :)
	checkConsistency();

	// delete all items
	QValueList<int>::iterator it;
	QValueList<int> allitems = m_map.keys();
	for ( it = allitems.begin(); it != allitems.end(); ++it )
	{
		removeItem(*it);
	}

	// clear the documents added
	m_fileholder->unlink();
	delete m_fileholder;
	m_fileholder = new KTempDir();
}

int DDataControl::compare(int id1, int id2) const
{
	if (id1 == id2)
		return 0;

	DDataItem *item1 = dataItem(id1);
	DDataItem *item2 = dataItem(id2);
	if (!item1 || !item2)
		return 0;

	if (item1->Parent() != item2->Parent())
		return 0;

	if (item1->Parent() == DItem::NOITEM && item2->Parent() == DItem::NOITEM)
	{
		if (item1->countChildren() >= item2->countChildren())
		{
			if (item1->countChildren() > item2->countChildren())
				return -1;
			return 0;
		}
		return 1;
	}

	DDataItem *dataitem_parent = dataItem(item1->Parent());
	if (dataitem_parent->childIdx(id1) > dataitem_parent->childIdx(id2))
		return -1;
	else
		return 1;
}

void DDataControl::notifyChildren(int id)
{
	DDataItem * item = dataItem(id);
	if (item)
	{
		// update the children positions
		for (int i = 0; i<item->countChildren(); i++)
		{
			emit itemChanged(item->childNum(i));
		}
	}
}

void DDataControl::printTree(QTextStream & s)
{
	s.setEncoding( QTextStream::UnicodeUTF8 );

	s<<QString("<?xml version=\"1.0\" encoding=\"utf8\"?>\n");
	s<<"<kdissertdoc>\n";
	s<<"<version>"<<document_version<<"</version>\n";

	s<<"<meta ";
	s<<"fname=\""<<m_fname<<"\" ";
	s<<"sname=\""<<m_sname<<"\" ";

	s<<"pixsize=\""<<m_pixSize<<"\" ";
	s<<"colbackground=\""<<m_col_background.name()<<"\" ";
	s<<"collink=\""<<m_col_link.name()<<"\" ";
	s<<"colref=\""<<m_col_ref.name()<<"\" ";

	s<<"canvasfont=\""<<m_canvasFont.toString()<<"\" ";
	s<<"picturesoncanvas=\""<<(int) m_picturesOnCanvas<<"\" ";
	s<<"reversevideo=\""<<(int) m_reverseVideo<<"\" ";

	s<<"showdirection=\""<<(int) m_showDirection<<"\" ";
	s<<"showdirectionsize=\""<<m_showDirectionSize<<"\" ";
	s<<"linkstyle=\""<<m_linkstyle<<"\" />\n";


	QMapIterator<int, DItem*> it;
	for (it = m_map.begin(); it != m_map.end(); ++it)
	{
		DDataItem* item = (DDataItem*) (*it);
		if (item)
			item->printSelf(s);
		else
			kdWarning()<<"BUG : in DDataControl::printTree"<<endl;
	}

	s<<"</kdissertdoc>\n";
}

bool DDataControl::loadFromFile(const KURL & url)
{
	clearDocument();

	QString filename = url.url();
	bool safe = false;
	if (filename.endsWith(".kdi", false)) safe=true;
	if (!safe) if (filename.endsWith(".kno", false)) safe=true;
	if (!safe) if (filename.endsWith(".mm", false)) safe=true;

	if (!safe)
	{
		kdDebug()<<"open kdissert documents ?"<<endl;
		return false;
	}

	QString tmpFile;
	if (! KIO::NetAccess::download(url, tmpFile, NULL))
	{
		QString text;

		if ( url.isLocalFile() )
			text = i18n( "Unable to find the kdissert file %1 !" );
		else
			text = i18n( "Unable to download the kdissert file ! "
					"Please check that the address %1 is correct." );

		KMessageBox::sorry( NULL, text.arg( url.prettyURL() )  );

		docNotChanged();

		return false;
	}


	if (tmpFile.endsWith(".kdi", false))
	{
		KTar arch(tmpFile);
		if (! arch.open(IO_ReadOnly) )
		{
			QString text = i18n("Unable to open the kdissert file %1 !");
			KMessageBox::sorry( NULL, text.arg( url.prettyURL() )  );
			KIO::NetAccess::removeTempFile( tmpFile );

			docNotChanged();

			return false;
		}

		const KArchiveDirectory *archiveDir = arch.directory();
		QStringList lst = archiveDir->entries();

		// Find the main document or return false
		bool cond = false;
		for (unsigned int i=0; i<lst.count(); i++)
		{
			if (lst[i] == "maindoc.xml")
				cond = true;
		}

		if (!cond)
		{
			QString text = i18n("Unable to parse the kdissert file %1 !");
			KMessageBox::sorry( NULL, text.arg( url.prettyURL() )  );
			KIO::NetAccess::removeTempFile( tmpFile );

			docNotChanged();

			return false;
		}

		const KArchiveEntry* entry = archiveDir->entry("maindoc.xml");
		if (!entry->isFile())
		{
			QString text = i18n("The kdissert file %1 does not seem to be a valid kdissert file !");
			KMessageBox::sorry( NULL, text.arg( url.prettyURL() )  );
			KIO::NetAccess::removeTempFile( tmpFile );

			docNotChanged();

			return false;
		}

		KArchiveFile* maindocentryfile = (KArchiveFile*) entry;
		KTempDir tmpdir;
		maindocentryfile->copyTo(tmpdir.name());

		// Parse the maindoc file now
		QString tmpdocname = tmpdir.name() + maindocentryfile->name();
		QFile maindocfile(tmpdocname);
		DissertParser handler(this);
		QXmlInputSource source( &maindocfile );
		QXmlSimpleReader reader;
		reader.setContentHandler( &handler );

		bool result = reader.parse( &source , true );
		if (!result)
		{
			QString text = i18n("The kdissert file %1 does not seem to be a valid kdissert file !");
			KMessageBox::sorry( NULL, text.arg( url.prettyURL() )  );
			clearDocument();
			KIO::NetAccess::removeTempFile( tmpFile );
			tmpdir.unlink();

			docNotChanged();

			return false;
		}

		// Load the cached documents 
		// TODO: pictures for the moment, can be other things in the future

		QRegExp rx("^pic.\\d+....$");
		for (unsigned int i=0; i<lst.count(); i++)
		{
			if (rx.search(lst[i]) == 0)
			{
				QString val = lst[i];
				val.remove(0, 4);
				val = val.left(lst[i].length()-8);
				int id = val.toInt();

				//kdWarning()<<"file is "<<lst[i]<<" and id is : "<<id<<endl;

				DDataItem* item = dataItem(id);
				if (item)
				{
					entry = archiveDir->entry(lst[i]);
					if (entry->isFile())
					{
						KArchiveFile* picfile = (KArchiveFile*) entry;
						picfile->copyTo(tmpdir.name());
						item->loadPix(tmpdir.name() + lst[i]);
					}
				}
				else
				{
					kdWarning() <<"BUG: a pix item is missing"<<endl;
				}
			}
		}

		tmpdir.unlink();
	}
	else if (tmpFile.endsWith(".mm", false))
	{
		QFile maindocfile(tmpFile);
		FFParser handler(this);
		QXmlInputSource source( &maindocfile );
		QXmlSimpleReader reader;
		reader.setContentHandler( &handler );

		bool result = reader.parse( &source , true );
		if (!result)
		{
			QString text = i18n("The mindmap file %1 does not seem to be a valid freemind file !");
			KMessageBox::sorry( NULL, text.arg( url.prettyURL() )  );
			clearDocument();
			KIO::NetAccess::removeTempFile( tmpFile );

			docNotChanged();

			return false;
		}
		docChanged();
	}
	else if (tmpFile.endsWith("kno", false))
	{
		QFile maindocfile(tmpFile);

		if (!maindocfile.open(IO_ReadOnly)) 
		{
			KMessageBox::error(0, i18n("Unable to open the knowit file %1 !").arg(maindocfile.name()));
			return false;
		}

		QTextStream docstream(&maindocfile);
		docstream.setEncoding( QTextStream::UnicodeUTF8 );

		QValueList<int> ancestors;
		int lastdepth = 0;
		int lastitem = DItem::NOITEM;
		int id = 0;
		URLObject obj;

		QString line = docstream.readLine();
		while (! line.isNull())
		{
			if (line.left(9) == "\\NewEntry" || line.left(13) == "\\CurrentEntry")
			{
				QString name;
				uint depth;

				QTextIStream itemline(&line);
				itemline >> name >> depth;

				name = itemline.readLine();
				name = name.stripWhiteSpace();

				// request a new item
				createItemWithId( id ); 
				lastitem = id;
				id++;


				DDataItem *item = dataItem( lastitem );
				item->setXY( 1500+random()%1000, 1500+random()%1000);
				item->m_summary = name;

				if (depth+1 > ancestors.count())
				{
					ancestors += lastitem;
				}
				ancestors[depth] = lastitem;

				if (depth> 0)
				{
					dataItem( ancestors[depth-1] )->addChild( lastitem );
					item->setParent( ancestors[depth-1] );
				}
				else
				{
					item->setParent( DItem::NOITEM );
				}

				lastdepth++;
			}
			else if (line.left(5) == "\\Link" )
			{
				obj.m_url = line.right( line.length()- 6 );
			}
			else if (line.left(6) == "\\Descr" )
			{
				obj.m_caption = line.right( line.length()- 7 );
				DDataItem* currentitem = dataItem( lastitem );
				if (currentitem)
					currentitem->m_urllist.append( obj );
			}
			else
			{
				DDataItem* currentitem = dataItem( lastitem );

				if (currentitem)
					currentitem->m_text += line;
			}
			line = docstream.readLine();
		}
	}

	KIO::NetAccess::removeTempFile( tmpFile );

	// Notify the views attached
	QValueList<int>::iterator it;
	QValueList<int> allitems = m_map.keys();
	for ( it = allitems.begin(); it != allitems.end(); ++it )
	{
		emit itemCreated(*it);
	}

	// make sure everything is correct (perhaps a bit paranoid ..)
	for ( it = allitems.begin(); it != allitems.end(); ++it )
	{
		DDataItem *item = dataItem(*it);
		//kdWarning()<<"item is "<<item->Id()<<endl;
		for (int i=0; i<item->ref_count(); i++)
		{
			DRef *ref = item->ref_idx(i);
			//kdWarning()<<"updating ref "<<item->Id()<<" "<<ref->m_ref<<endl;
			emit refChanged(item->Id(), ref->m_ref, true);
		}
		emit itemChanged(*it);
	}	

	emit settingsChanged();
	return true;
}

bool DDataControl::saveToFile(const KURL & url)
{
	// check the name of the file to write to
	QString fname = url.url();
	if (! fname.endsWith(".kdi", false))
		fname.append(".kdi");

	// create a tremporary file in which the data will be written
	KTempFile archfile;
	QString tmpfilename = archfile.name();

	// create also a temporary file for writing the xml document
	KTempFile docfile;

	if (archfile.status() != 0 || docfile.status() != 0)
	{
		KMessageBox::information(NULL, i18n("Cannot create a temporary file '%1' for output.").arg(tmpfilename), 
				i18n("Save Document") );
		return false;
	}

	KTar arch(tmpfilename, "application/x-gzip");
	if (! arch.open(IO_WriteOnly) )
	{
		KMessageBox::information(NULL, i18n("Cannot open file '%1' for output.").arg(tmpfilename), 
				i18n("Save Document") );
		archfile.unlink();
		docfile.unlink();
		return false;
	}

	// add the doc file
	printTree(*docfile.textStream());
	docfile.close();
	arch.addLocalFile(docfile.name(), "maindoc.xml");

	// add the cached documents
	arch.addLocalDirectory( getTmpDir()->absPath() , "." );

	// close the archive and save it
	arch.close();

	bool issaved = KIO::NetAccess::upload(tmpfilename, KURL(fname), NULL);
	emit documentChanged( !issaved );

	archfile.unlink();
	docfile.unlink();
	return issaved;
}

void DDataControl::docChanged()
{
	emit documentChanged( true );
}

void DDataControl::docNotChanged()
{
	emit documentChanged( false );
}

int DDataControl::countItems()
{
	return m_map.count();
}

bool DDataControl::canGenerate()
{
	DDataItem *data = dataItem( rootID() );

	if (!data)
		return false;

	if (data->countChildren() > 0)
	{
		return true;
	}

	return false;
}

int DDataControl::rootID()
{
	int cnt = 0, rootid = DItem::NOITEM;

	QValueList<int>::iterator it;
	QValueList<int> allitems = m_map.keys();
	QValueList<int>::iterator end = allitems.end();

	for ( it = allitems.begin(); it != end; ++it )
	{
		if ( dataItem( *it )->Parent() == DItem::NOITEM )
		{
			if ( dataItem( *it )->countFamily() > cnt )
			{
				cnt = dataItem( *it )->countFamily();
				rootid = *it;
			}
		}
	}
	return rootid;
}

void DDataControl::debugItem(int id)
{
	DDataItem *item = dataItem( id );
	if (!item)
	{
		kdWarning()<<"DDataControl::debugItem : invalid item of id : "<<id<<endl;
		return;
	}
	kdWarning()<<"***"<<endl;

	kdWarning()<<"item of id : "<<id<<endl;

	for (int i=0; i<item->countChildren(); i++)
	{
		kdWarning()<<"   - child "<<dataItem(item->childNum(i))->Id()<<endl;
	}

	kdWarning()<<"***"<<endl;
}

void DDataControl::notifyChildChange(int id)
{
	emit itemChanged(id);
}

QDir * DDataControl::getTmpDir()
{
	return m_fileholder->qDir();
}

void DDataControl::checkConsistency()
{
	QValueList<int>::iterator it;
	QValueList<int> allitems = m_map.keys();
	QValueList<int>::iterator end = allitems.end();
	for ( it = allitems.begin(); it != end; ++it )
	{
		bool changed = false;

		// check the parent
		DDataItem* item = (DDataItem*) dataItem(*it);
		if (item->Parent() != DItem::NOITEM)
		{
			if (!dataItem(item->Parent()))
			{
				item->setParent(DItem::NOITEM);
				changed = true;	
			}
		}

		// check the children
		int count = 0;
		while (count < item->countChildren())
		{
			if (!dataItem(item->childNum(count)))
			{
				item->removeChild( item->childNum(count) );
				count = 0;
				changed = true;
			}
			else
			{
				count++;
			}
		}

		// notify if something changed
		if (changed)
		{
			kdWarning()<<"inconsistency detected - item : "<<item->Id()<<endl;	
			emit itemChanged(item->Id());
		}
	}
}

void DDataControl::slotFileSpell()
{
	if (!m_spell) m_spell = new DSpell( this );
	m_spell->spell();
}

void DDataControl::setItemSelected(int val, DGuiView *view)
{
	emit itemSelected(val, view);
}

void DDataControl::notifyItemModified(int val)
{
	emit itemChanged(val);
}

bool DDataControl::requestRefChange(int orig, int dest, bool add, bool check)
{
	DDataItem *item = dataItem(orig);
	if (check)
	{
		if (!item)
		{
			//kdWarning()<<"item orig does not exist! "<<orig<<endl;
			return false;
		}
		if (!dataItem(dest))
		{
			//kdWarning()<<"item dest does not exist! "<<dest<<endl;
			return false;
		}
	}

	if (add)
	{
		if ( item->isRef(dest) ) return false;
		//kdWarning()<<"adding "<<dest<<" for orig "<<orig<<endl;
		item->addRef(dest);
	}
	else
	{
		if ( !item->isRef(dest) ) return false;
		item->rmRef(dest);
	}
	if (check) emit refChanged(orig, dest, add);
	return true;
}

#include "DDataControl.moc"
