/***************************************
  $Header: /cvsroot/petscgraphics/geomview.c,v 1.6 2003/04/30 19:34:55 hazelsct Exp $

  This file has the Geomview interface, including the PETSc vector gather
  operations to bring everything to CPU 0.
***************************************/


#include <stdio.h>
#include <petscvec.h>
#include "config.h" /* esp. for inline */
#include "illuminator.h" /* Just to make sure the interface is "right" */

/* Build with -DDEBUG for debugging output */
#undef DPRINTF
#ifdef DEBUG
#define DPRINTF(fmt, args...) PetscPrintf (PETSC_COMM_WORLD, "%s: " fmt, __FUNCT__, args)
#else
#define DPRINTF(fmt, args...)
#endif

/*+ Stream holding the pipe to the Geomview process.  It is initialized to
  NULL, then set by GeomviewBegin, and cleared by GeomviewEnd.  This can be
  used as a flag to determine whether the Geomview display is open. +*/
static FILE *pfd = NULL;

/*+ Declared in illuminator.c, this gives the current number of triangles on
  this node. +*/
extern int num_triangles;

/*+ Declared in illuminator.c, this array stores the corner coordinates and
  color information for each triangle. +*/
extern PetscScalar vertices[];


#undef __FUNCT__
#define __FUNCT__ "GeomviewBegin"

/*++++++++++++++++++++++++++++++++++++++
  Spawn a new geomview process.  Most of this was shamelessly ripped from Ken
  Brakke's Surface Evolver.

  int GeomviewBegin Returns 0 or an error code.

  MPI_Comm comm MPI communicator for rank information, if NULL it uses
  PETSC_COMM_WORLD.
  ++++++++++++++++++++++++++++++++++++++*/

int GeomviewBegin(MPI_Comm comm)
{
  int rank, ierr, gv_pid,gv_pipe[2], to_gv_pipe[2];
  char gv_version[100];

  if (!comm)
    comm = PETSC_COMM_WORLD;
  ierr = MPI_Comm_rank (comm,&rank); CHKERRQ(ierr);
  if (!rank)
    {
      if (pfd)
	SETERRQ (PETSC_ERR_ARG_WRONGSTATE,
		 "Multiple geomview displays not allowed");

      pipe (gv_pipe); /* from geomview stdout */
      pipe (to_gv_pipe); /* to geomview stdin */
      gv_pid = fork ();

      if(gv_pid==0) /* child */
	{
	  close (0);
	  dup (to_gv_pipe[0]);
	  close (to_gv_pipe[0]);
	  close (to_gv_pipe[1]);
	  close (1);
	  dup (gv_pipe[1]);
	  close (gv_pipe[0]);
	  close (gv_pipe[1]);
	  /* signal(SIGINT,SIG_IGN); */
	  execlp (GEOMVIEW,GEOMVIEW,"-c","(interest (pick world))","-",NULL);
	  perror (GEOMVIEW); /* only error gets here */
	  SETERRQ (PETSC_ERR_ARG_WRONGSTATE,"Geomview invocation failed.\n");
	}

      /* PETSc program execution resumes here */
      close (gv_pipe[1]);
      close (to_gv_pipe[0]);
      pfd = fdopen (to_gv_pipe[1], "w"); /* hooked to stdin of geomview */
      fprintf (pfd, "(echo (geomview-version) \"\n\")\n");
      fflush (pfd);
      read (gv_pipe[0], gv_version, sizeof (gv_version));
    }
  return 0;
}


#undef __FUNCT__
#define __FUNCT__ "GeomviewEnd"

/*++++++++++++++++++++++++++++++++++++++
  Exit the current running Geomview process and close its pipe.  Based in part
  on Ken Brakke's Surface Evolver.

  int GeomviewEnd Returns 0 or an error code.

  MPI_Comm comm MPI communicator for rank information, if NULL it uses
  PETSC_COMM_WORLD.
  ++++++++++++++++++++++++++++++++++++++*/

int GeomviewEnd (MPI_Comm comm)
{
  int rank, ierr;

  if (!comm)
    comm = PETSC_COMM_WORLD;
  ierr = MPI_Comm_rank (comm, &rank); CHKERRQ (ierr);
  if (!rank)
    {
      if (pfd)
	{
	  fprintf (pfd, "(exit)");
	  fclose (pfd);
	  pfd = NULL;
	}
    }
  return 0;
}


#undef __FUNCT__
#define __FUNCT__ "GeomviewDisplayTriangulation"

/*++++++++++++++++++++++++++++++++++++++
  Pipe the current triangulation to Geomview for display.  Much of this is
  based on Ken Brakke's Surface Evolver.

  int GeomviewDisplayTriangulation Returns 0 or an error code.

  MPI_Comm comm MPI communicator for rank information, if NULL it uses
  PETSC_COMM_WORLD.

  PetscScalar *minmax Position of block corners: xmin, xmax, ymin, ymax, zmin, zmax.

  char *name Name to give the Geomview OOGL object which we create.

  PetscTruth transparent Geomview transparency flag.
  ++++++++++++++++++++++++++++++++++++++*/

int GeomviewDisplayTriangulation
(MPI_Comm comm, PetscScalar *minmax, char *name, PetscTruth transparent)
{
  int ierr, i, total_entries, start, end, rank;
  PetscScalar *localvals;
  Vec globalverts, localverts;
  VecScatter vecscat;
  IS islocal;

  /* Simple "assertion" check */
  if (!comm)
    comm = PETSC_COMM_WORLD;
  ierr = MPI_Comm_rank (comm, &rank); CHKERRQ (ierr);
  if (!rank && !pfd)
    SETERRQ (PETSC_ERR_ARG_WRONGSTATE,"Geomview display is not open!");

  /*+ First, this creates global and local vectors for all of the triangle
    vertices. +*/
  ierr = VecCreateMPIWithArray (comm, 13*num_triangles, PETSC_DETERMINE,
				vertices, &globalverts); CHKERRQ (ierr);
  ierr = VecGetSize (globalverts, &total_entries); CHKERRQ (ierr);
  ierr = VecCreateMPI (comm, (rank == 0) ? total_entries:0, total_entries,
		       &localverts); CHKERRQ (ierr);
  DPRINTF ("Total triangles: %d\n", total_entries/13);

  /* Diagnostics which may be useful to somebody sometime */
  /* ierr = VecGetOwnershipRange (globalverts, &start, &end); CHKERRQ (ierr);
  ierr = PetscSynchronizedPrintf
    (comm, "[%d] Global vector local size: %d, ownership from %d to %d\n",
     rank, end-start, start, end); CHKERRQ (ierr);
  ierr = PetscSynchronizedFlush (comm); CHKERRQ (ierr);

  ierr = VecGetOwnershipRange (localverts, &start, &end); CHKERRQ (ierr);
  ierr = PetscSynchronizedPrintf
    (comm, "[%d] Local vector local size: %d, ownership from %d to %d\n", rank,
     end-start, start, end); CHKERRQ (ierr);
  ierr = PetscSynchronizedFlush (comm); CHKERRQ (ierr);
  ierr = PetscPrintf (comm, "Global vector size: %d\n", total_entries);
  CHKERRQ (ierr); */

  /*+ It then gathers (``scatters'') all vertex data to processor zero, +*/
  ierr = ISCreateStride (comm, total_entries, 0, 1, &islocal); CHKERRQ (ierr);
  ierr = VecScatterCreate (globalverts, PETSC_NULL, localverts, islocal,
			   &vecscat); CHKERRQ (ierr);
  DPRINTF ("Starting triangle scatter to head node\n",0);
  ierr = VecScatterBegin (globalverts, localverts, INSERT_VALUES,
			  SCATTER_FORWARD, vecscat); CHKERRQ (ierr);
  DPRINTF ("Triangle scatter to head node in progress...\n",0);
  ierr = VecScatterEnd (globalverts, localverts, INSERT_VALUES,
			SCATTER_FORWARD, vecscat); CHKERRQ (ierr);
  DPRINTF ("Triangle scatter to head node complete\n",0);
  ierr = VecScatterDestroy (vecscat); CHKERRQ (ierr);
  ierr = ISDestroy (islocal); CHKERRQ (ierr);

  /*+ and puts them in an array. +*/
  ierr = VecGetArray (localverts, &localvals); CHKERRQ (ierr);

  /*+ Finally, it sends everything to Geomview, +*/
  if (!rank) {
    fprintf (pfd, "(geometry \"%s\" { : bor })", name);
    fprintf (pfd, "(read geometry { define bor \n");
    fprintf (pfd, "appearance {%s}\nOFF\n", transparent ? "+transparent" : "");
    fprintf (pfd, "%d %d 0\n", 3*(total_entries/13) + 2, total_entries/13);
    for (i=0; i<total_entries; i+=13)
      fprintf (pfd, "%g %g %g\n%g %g %g\n%g %g %g\n", localvals[i],
	       localvals[i+1], localvals[i+2], localvals[i+3], localvals[i+4],
	       localvals[i+5], localvals[i+6], localvals[i+7], localvals[i+8]);
    fprintf (pfd, "%g %g %g\n%g %g %g\n", minmax[0], minmax[2], minmax[4],
	     minmax[1], minmax[3], minmax[5]);
    for (i=0; i<total_entries/13; i++)
      fprintf (pfd, "3 %d %d %d %g %g %g %g\n", 3*i,3*i+1,3*i+2,
	       localvals[13*i+9], localvals[13*i+10], localvals[13*i+11],
	       localvals[13*i+12]);
    fprintf (pfd, "})\n");
    fflush (pfd);
  }

  /*+ and cleans up the mess, resetting the number of triangles to zero. +*/
  ierr = VecRestoreArray (localverts, &localvals); CHKERRQ (ierr);
  ierr = VecDestroy (localverts); CHKERRQ (ierr);
  ierr = VecDestroy (globalverts); CHKERRQ (ierr);
  num_triangles = 0;

  return 0;
}
