/* vim:tabstop=4:expandtab:shiftwidth=4
 * 
 * Idesk -- XCaption.cpp
 *
 * Copyright (c) 2002, Chris (nikon) (nikon@sc.rr.com)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *      Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *      
 *      Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *      
 *      Neither the name of the <ORGANIZATION> nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 * (See the included file COPYING / BSD )
 */

#include "XCaption.h"
#include "XIconWithShadow.h"

XCaption::XCaption(AbstractContainer * c, AbstractIcon * iParent,
                   AbstractConfig * con, AbstractIconConfig * iConfig) 
           : AbstractCaption(c, iParent, con, iConfig), FONTSHIFT(4)
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);
    imlibData = xContainer->getImlibData();
    spareRoot = xContainer->getSpareRoot();
    visual = DefaultVisual(xContainer->getDisplay(), 
                           DefaultScreen( xContainer->getDisplay() ) );
    cmap = DefaultColormap( xContainer->getDisplay(),
                            DefaultScreen( xContainer->getDisplay() ) );

    configure();
    createFont();
}

XCaption::~XCaption()
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    XftColorFree( xContainer->getDisplay(), visual, cmap, &fontColor );

    if (shadowOn)
    {
    XftColorFree( xContainer->getDisplay(), visual, cmap, &shadowFontColor );
    }

    XFreePixmap(xContainer->getDisplay(), buffer);
    XFreeGC(xContainer->getDisplay(), gc);
    XFreeGC(xContainer->getDisplay(), bufferGc);
}

void XCaption::configure()
{
    DesktopIconConfig * dIconConfig =
        dynamic_cast<DesktopIconConfig *>(iconConfig);

    //grab the text
    text = iconConfig->getCaption();
    
    //set boldness
    if (dIconConfig->getBoldness())
        boldVal = XFT_WEIGHT_BOLD;
    else
        boldVal = XFT_WEIGHT_MEDIUM;

    //set shadow options
    shadowOn = dIconConfig->isShadow();
    shadowX = dIconConfig->getShadowX();
    shadowY = dIconConfig->getShadowY();
    shadowColor = dIconConfig->getShadowColor();

    //captionOnHover
    captionOnHover = dIconConfig->getCaptionOnHover();
}

void XCaption::createFont()
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);
    DesktopIconConfig * dIconConfig =
        dynamic_cast<DesktopIconConfig *>(iconConfig);

    font = XftFontOpen( xContainer->getDisplay(),
                        DefaultScreen( xContainer->getDisplay()), 
                        XFT_FAMILY,
                        XftTypeString,
                        (char*)dIconConfig->getFont().c_str(),
                        XFT_SIZE,
                        XftTypeDouble,
                        (double)dIconConfig->getFontSize(), 
                        XFT_WEIGHT, XftTypeInteger, boldVal,
                        NULL );
    
    XftColorAllocName(xContainer->getDisplay(), visual, cmap,
                      (char*)dIconConfig->getFontColor().c_str(), &fontColor );

    if (shadowOn)
        XftColorAllocName(xContainer->getDisplay(), visual, cmap,
                          (char*)shadowColor.c_str(), &shadowFontColor );
}


Window * XCaption::getWindow()
{
    return &window;
}

void XCaption::createWindow()
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    XSetWindowAttributes attr;

    attr.override_redirect = True;
    attr.event_mask = SubstructureRedirectMask |
                      SubstructureNotifyMask   |
                      ButtonPressMask          |
                      ButtonReleaseMask        |
                      PointerMotionMask        |
                      EnterWindowMask          |
                      LeaveWindowMask          |
                      PropertyChangeMask       |
                      ExposureMask;

    XftTextExtents8( xContainer->getDisplay(), font, (XftChar8*)text.c_str(), text.length(), &fontInfo );

    // fix window extents so shadow text is not cut
    // TODO: fix window extents if shadow is negative:
    // Would have to edit x and y of the window
    width = fontInfo.width + FONTSHIFT;
    height = font->height;
    if (shadowOn)
    {
        width += shadowX > 0 ? shadowX : 0;
        height += shadowY > 0 ? shadowY : 0;
    }
    window = XCreateWindow( xContainer->getDisplay(),
                            xContainer->getRootWindow(),
                            0,
                            0,
                            fontInfo.width + FONTSHIFT,
                            font->height,
                            0,
                            CopyFromParent,
                            CopyFromParent,
                            CopyFromParent,
                            CWOverrideRedirect|CWEventMask,
                            &attr );

    buffer = XCreatePixmap(xContainer->getDisplay(), window,
                           fontInfo.width + FONTSHIFT, font->height,
                           DefaultDepth(xContainer->getDisplay(), 
                                    DefaultScreen(xContainer->getDisplay())));
    gc = XCreateGC(xContainer->getDisplay(), window, 0, 0);
    bufferGc = XCreateGC(xContainer->getDisplay(), buffer, 0, 0);
    XSetForeground(xContainer->getDisplay(), bufferGc, 0xffffff);
    XSetLineAttributes(xContainer->getDisplay(), bufferGc, 2, LineSolid,
                       CapRound, JoinRound);
    getXftDrawHandle();
}

void XCaption::getXftDrawHandle()
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    fontDrawHandle = XftDrawCreate(xContainer->getDisplay(), buffer, visual,
                                   cmap);
}

void XCaption::lowerWindow()
{   
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    XLowerWindow( xContainer->getDisplay(), window );
}

void XCaption::draw()
{   
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);
    XIconWithShadow * sIcon = dynamic_cast<XIconWithShadow *>(iconParent);
    
    ImlibImage *textBackground;
    int fX, fY;

    // using the spareRoot image (the background image created at startup) 
    // instead of dynamically geting the window to speed things up.
    
    fX = iconParent->getX() + ( iconParent->getWidth() - fontInfo.width )/2;
    fY = iconParent->getY() + iconParent->getHeight() + FONTSHIFT;

    textBackground = Imlib_crop_and_clone_image(imlibData, spareRoot, fX, fY,
                                                fontInfo.width + FONTSHIFT,
                                                font->height);

    Imlib_paste_image(imlibData, textBackground, buffer, 0, 0,
                      fontInfo.width + FONTSHIFT, font->height);
    Imlib_kill_image(imlibData, textBackground);

    // if the icon shadow is enabled and the shadow is visible underneath the
    // text window
    if (sIcon &&
        sIcon->getShadowX() > fX - sIcon->getWidth() &&
        sIcon->getShadowY() > fY - sIcon->getHeight() &&
        sIcon->getShadowX() < fX + sIcon->getWidth() &&
        sIcon->getShadowY() < fY + sIcon->getHeight() )
            sIcon->renderShadowToImage(buffer, fX, fY);
    
    
    updateText();

    XMoveWindow( xContainer->getDisplay(), window, fX, fY );
    
    //apply the buffer to the window
    XSetFillStyle(xContainer->getDisplay(), gc, FillTiled);
    XSetTile(xContainer->getDisplay(), gc, buffer);
    XFillRectangle(xContainer->getDisplay(), window, gc, 0, 0,
            fontInfo.width + FONTSHIFT, font->height);
}

void XCaption::updateText()
{   
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);
    
    // unnecessary with the double buffering
    //XClearWindow( xContainer->getDisplay(), buffer );
    
    XSetForeground(xContainer->getDisplay(), gc, 0xffffff);
    XSetLineAttributes(xContainer->getDisplay(), gc, 2, LineSolid, CapRound,
                       JoinRound);
    XDrawArc(xContainer->getDisplay(), window, gc, 0, 3, width, height,
            0, 0);
    
    if (shadowOn)
        XftDrawString8( fontDrawHandle, &shadowFontColor, font, FONTSHIFT/2 + shadowX, font->ascent + shadowY, (XftChar8*)text.c_str(), text.length() );

    XftDrawString8( fontDrawHandle, &fontColor, font, FONTSHIFT/2, font->ascent, (XftChar8*)text.c_str(), text.length() );
}

int XCaption::getFontWidth()
{
    return fontInfo.width;
}

int XCaption::getFontHeight()
{
    return font->height;
}
void XCaption::mapWindow()
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    XMapWindow( xContainer->getDisplay(), window );
}

void XCaption::initMapWindow()
{
    if (!captionOnHover)
        mapWindow();
}

void XCaption::unmapWindow()
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    XUnmapWindow( xContainer->getDisplay(), window );
}

void XCaption::moveWindow(int xC, int yC)
{
    XDesktopContainer * xContainer =
        dynamic_cast<XDesktopContainer *>(container);

    XMoveWindow( xContainer->getDisplay(), window, xC, yC );
}

void XCaption::mouseOverEffect()
{
    if (captionOnHover)
        mapWindow();
}

void XCaption::mouseOffEffect()
{
    if (captionOnHover)
        unmapWindow();
}
