/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// textfile.c
//
// Text file functions
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "htp.h"

BOOL OpenFile(const char *filename, const char *openFlags, TEXTFILE *textFile)
{
    char *os_filename;
    assert(filename != NULL);
    assert(openFlags != NULL);
    assert(textFile != NULL);

    /* convert the filename to the native machine's format */
    /* (MS-DOS to UNIX and vice-versa) */
    /* this returns an allocated copy of the string, so it must be freed */
    /* when the structure is destroyed */
    if((os_filename = ConvertDirDelimiter(filename)) == NULL)
    {
        DEBUG_PRINT(("unable to convert dir delimiter"));
        return FALSE;
    }

    /* initialize the rest of the structure */
    textFile->lineNumber = 1;
    textFile->bytesReadThisLine = 0;
    textFile->bytesWrittenThisLine = 0;
    textFile->pendingSpaces = 0;
    textFile->flags = TEXTFILE_FLAG_NONE;
    textFile->file = fopen(os_filename, openFlags);
    FreeMemory(os_filename);

    if(textFile->file == NULL)
    {
        DEBUG_PRINT(("unable to open file %s", filename));
        return FALSE;
    }

    return TRUE;
}   


BOOL OpenFD(int filedes, const char *openFlags, TEXTFILE *textFile)
{
    assert(openFlags != NULL);
    assert(textFile != NULL);

    /* initialize the rest of the structure */
    textFile->lineNumber = 1;
    textFile->bytesReadThisLine = 0;
    textFile->bytesWrittenThisLine = 0;
    textFile->pendingSpaces = 0;
    textFile->flags = TEXTFILE_FLAG_NONE;
    textFile->file = fdopen(filedes, openFlags);

    if(textFile->file == NULL)
    {
        DEBUG_PRINT(("unable to open fd %d", filedes));
        return FALSE;
    }

    return TRUE;
}   

void CreateNullFile(TEXTFILE *textFile)
{
    assert(textFile != NULL);

    memset(textFile, 0, sizeof(TEXTFILE));
    textFile->flags = TEXTFILE_FLAG_NULL_FILE;
}

void CloseFile(TEXTFILE *textFile)
{
    assert(textFile != NULL);

    if(textFile->flags & TEXTFILE_FLAG_NULL_FILE)
    {
        return;
    }

    assert(textFile->file != NULL);
    fclose(textFile->file);

    textFile->file = NULL;
}   

BOOL GetFileChar(TEXTFILE *textFile, char *ch)
{
    int getCh;

    assert(textFile != NULL);

    if(textFile->flags & TEXTFILE_FLAG_NULL_FILE)
    {
        return FALSE;
    }

    assert(ch != NULL);
    assert(textFile->file != NULL);

    do
    {
        if((getCh = fgetc(textFile->file)) == EOF)
        {
            return FALSE;
        }
    } while(getCh == '\r');

    if((*ch = (char) getCh) != '\n')
    {
        textFile->bytesReadThisLine++;
    }
    else
    {
        textFile->lineNumber++;
        textFile->bytesReadThisLine = 0;
        textFile->bytesWrittenThisLine = 0;
    }

    return TRUE;
}   

BOOL PutFileChar(TEXTFILE *textFile, char ch)
{
    uint bytesWrittenThisLine;

    assert(textFile != NULL);

    if(textFile->flags & TEXTFILE_FLAG_NULL_FILE)
    {
        return TRUE;
    }

    if(ch == '\r')
    {
        return TRUE;
    }
    else if(ch == '\n')
    {
        bytesWrittenThisLine = textFile->bytesWrittenThisLine;

        textFile->lineNumber++;
        textFile->bytesReadThisLine = 0;
        textFile->bytesWrittenThisLine = 0;

        /* suppress linefeeds? */
        if(textFile->flags & TEXTFILE_FLAG_NO_CR)
        {
            if(bytesWrittenThisLine != 0) {
                /* emit a SPACE to avoid a run-on word from the C/R being */
                /* dropped between words */
                textFile->pendingSpaces++;
            }
            return TRUE;
        }
        else if(textFile->flags & TEXTFILE_FLAG_FEW_CR) {
            if(bytesWrittenThisLine == 0) {
                textFile->pendingSpaces = 0;
                return TRUE;
            }
        }
    }
    else if (ch == ' ')
    {
        textFile->pendingSpaces++;
        return TRUE;
    }
    else
    {
        textFile->bytesWrittenThisLine++;
    }

    assert(textFile->file != NULL);

    if(textFile->flags & TEXTFILE_FLAG_NO_CR)
    {
        if (textFile->pendingSpaces > 0) {
            textFile->bytesWrittenThisLine++;
            fputc(' ', textFile->file);
        }
        textFile->pendingSpaces = 0;
    }
    else
    {
        textFile->bytesWrittenThisLine += textFile->pendingSpaces;
        while (textFile->pendingSpaces > 0) {
            fputc(' ', textFile->file);
            textFile->pendingSpaces--;
        }
    }
    fputc(ch, textFile->file);

    return TRUE;
}   
