/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
//
// macro.c
//
// specialized markup processors
//
// Copyright (c) 1995-96 Jim Nelson.  Permission to distribute
// granted by the author.  No warranties are made on the fitness of this
// source code.
//
*/

#include "macro.h"

#include "defs.h"
#include "def-proc.h"

/*
// specialized markup processors
*/

#include "option.h"
#include "image-proc.h"
#include "file-proc.h"
#include "def-proc.h"
#include "use-proc.h"
#include "set-proc.h"
#include "bool-proc.h"
#include "misc-proc.h"
#include "while-proc.h"

static MARKUP_PROCESSORS markupProcessor[] =
{
    { "IMG", MARKUP_TYPE_HTML, ImageProcessor },
    { "OPT", MARKUP_TYPE_HTP, OptionProcessor },
    { "FILE", MARKUP_TYPE_HTP, FileProcessor },
    { "INC", MARKUP_TYPE_HTP, IncProcessor },
    { "SET", MARKUP_TYPE_HTP, SetProcessor },
    { "BLOCK", MARKUP_TYPE_HTP, BlockProcessor },
    { "BLOCKDEF", MARKUP_TYPE_HTP, BlockProcessor },
    { "USE", MARKUP_TYPE_HTP, UseProcessor },
    { "IF", MARKUP_TYPE_HTP, BooleanProcessor },
    { "/IF", MARKUP_TYPE_HTP, BooleanProcessor },
    { "IFNOT", MARKUP_TYPE_HTP, ConditionalWarning },
    { "ELSE", MARKUP_TYPE_HTP, BooleanProcessor },
    { "/BODY", MARKUP_TYPE_HTML, CommentProcessor },
    { "UNSET", MARKUP_TYPE_HTP, UnsetProcessor },
    { "PRE", MARKUP_TYPE_HTML, PreProcessor },
    { "/PRE", MARKUP_TYPE_HTML, PreProcessor },
    { "ALTTEXT", MARKUP_TYPE_HTP, AltTextProcessor },
    { "DEF", MARKUP_TYPE_HTP, BlockProcessor },
    { "OUTPUT", MARKUP_TYPE_HTP, OutputProcessor },
    { "UNDEF", MARKUP_TYPE_HTP, UndefProcessor },
    { "WHILE", MARKUP_TYPE_HTP, WhileProcessor },
    { "/WHILE", MARKUP_TYPE_HTP, WhileProcessor },
    { "!---", MARKUP_TYPE_HTP, HtpCommentProcessor },
    { "QUOTE", MARKUP_TYPE_HTP, QuoteProcessor }
};
#define MARKUP_PROCESSOR_COUNT (sizeof(markupProcessor)/sizeof(markupProcessor[0]))

/*
// HTML processing
*/
static BOOL FindRecursiveMacroName(TASK *task, const char **text, 
                                   char *name, int namelen)
{
    const char *textPtr;
    const char *textStart;
    BOOL quoted, changed;
    int braceLevel;
    textPtr = *text;

    DEBUG_PRINT(("FindRecursiveMacroName on %s", textPtr));
    if (*textPtr != '{') {
        HtpMsg(MSG_ERROR, task->infile, "starting brace not found in macro name");
        return FALSE;
    }
    
    textPtr++;
    textStart = textPtr;
    braceLevel = 0;
    while (TRUE) {
        switch(*textPtr) {
        case NUL:
            HtpMsg(MSG_ERROR, task->infile, "ending brace not found in macro name");
            return FALSE;

        case '}':
            if (!braceLevel) {
                char submacro[MAX_VARVALUE_LEN];
                int len = textPtr - textStart;

                if (len + 1 >= MAX_VARVALUE_LEN) {
                    HtpMsg(MSG_ERROR, task->infile, "macro name too long");
                    return FALSE;
                }
                
                memcpy(submacro, textStart, len);
                submacro[len] = 0;
                if (!ExpandMacrosInString(task, submacro, name, namelen,
                                          &quoted, &changed))
                    return FALSE;
                if (!changed) {
                    StringCopy(name, submacro, namelen);
                }
                *text = textPtr;
                DEBUG_PRINT(("name is %s", name));
                return TRUE;
            }
            braceLevel--;
            break;

        case '{':
            braceLevel++;
            break;

        }
        textPtr++;
    }
}
       
BOOL ExpandMacrosInString(TASK *task, const char *text, char *newText,
    uint newTextSize, BOOL *quoted, BOOL *changed)
{
    const char *expansion;
    char macro[MAX_VARVALUE_LEN];
    const char *textPtr;
    uint expansionLength;
    uint skipped;
    uint textLength;
    int singletonMacro;
    BOOL dummy;

    assert(task != NULL);
    assert(text != NULL);
    assert(newText != NULL);
    assert(newTextSize > 0);
    assert(quoted != NULL);
    assert(changed != NULL);

    *changed = FALSE;

    /* optimization: if no '$' in text, no need to go futher */
    if(strchr(text, '$') == NULL)
    {
#if DEBUG
        expandSkipped++;
#endif
        return TRUE;
    }

#if DEBUG
    expandPerformed++;
#endif

    textLength = strlen(text);

    /* Allan Todd fix: only check buffer size when its known to be needed */
    if (textLength + 1 >= newTextSize) {
        HtpMsg(MSG_ERROR, task->infile, 
               "ExpandMacroToString: text \"%s\" too long.", macro);
        return FALSE;
    }

    /* assume singletonMacro until we find some other text */
    singletonMacro = 1;

    /* loop repeatedly to evaluate the string until no more macros are found */
    for (;;)
    {
        /* search the value for a $-preceded macro name */
        textPtr = strchr(text, '$');

        /* if nothing found, exit */
        if(textPtr == NULL)
        {
            break;
        }

        /* Copy skipped text to newText */
        if (textPtr > text) {
            int len = textPtr - text;
            memcpy(newText, text, len);
            newText += len;
            newTextSize -= len;
            text += len;
            textLength -= len;
            singletonMacro = 0;
        }

        /* skip the '$' */
        text++;
        textLength--;

        if (*text == '$')
        {
            /* convert $$ to $ */
            *newText++ = '$';
            newTextSize--;
            text++;
            textLength--;
            *changed = TRUE;
            continue;
        }

        macro[0] = NUL;
        /* this is used later more than once, but could change each iteration */
        /* process the macro */

        /* copy macro name into macro[] array and set up the text pointer */
        /* macro specified with braces? */
        if(*text == '{')
        {
            textPtr = text;
            if (! FindRecursiveMacroName(task, &textPtr, 
                                         macro, MAX_VARVALUE_LEN))
                return FALSE;

            /* set end insertion point */
            skipped = textPtr - text + 1;
        }
        else
        {
            /* start of macro, no braces */
            /* copy macro name until EOS */
            StringCopy(macro, text, MAX_VARVALUE_LEN);

            /* set end insertion point */
            skipped = textLength;
        }
        
        text += skipped;
        textLength -= skipped;

        /* if no macro name found, stop */
        if(macro[0] == NUL)
        {
            break;
        }

        /* make sure variable exists in store */
        if(VariableExists(task->varstore, macro) != TRUE)
        {
            /* only a warning ... stop processing to prevent infinite */
            /* loop */
            HtpMsg(MSG_WARNING, task->infile, 
                   "unrecognized macro name \"%s\"", macro);
            break;
        }

        /* block macros cannot be expanded inside of markup tags */
        if(GetVariableType(task->varstore, macro) == VAR_TYPE_BLOCK_MACRO)
        {
            STREAM *blockFile = 
                (STREAM*) GetVariableValue(task->varstore, macro);
            assert(blockFile->sflags == STREAM_FLAG_BUFFER);

            expansion = blockFile->u.buffer.buffer;
            expansionLength = blockFile->u.buffer.offset;
        }
        else
        {
            /* get the macros value and replace the attribute value */
            expansion = GetVariableValue(task->varstore, macro);
            if (expansion == NULL)
                expansion = "";
            expansionLength = strlen(expansion);
        }

        HtpMsg(MSG_INFO, task->infile, "expanding macro \"%s\" to \"%s\"",
               macro, expansion);

        if (expansionLength + textLength + 1 >= newTextSize) {
            HtpMsg(MSG_ERROR, task->infile, "Macro \"%s\" too long (%d + %d >= %d).",
                   macro, expansionLength, textLength, newTextSize);
            return FALSE;
        }

        /* Recursively expand macro in case it contains macros again. */
        if (!ExpandMacrosInString(task, expansion, newText, newTextSize,
                                  &dummy, changed))
            return FALSE;
        /* Macro did not change, copy it directly */
        if (!*changed) {
            memcpy(newText, expansion, expansionLength);
            newText += expansionLength;
            newTextSize -= expansionLength;
            *newText = 0;
        } else {
            int len = strlen(newText);
            newText += len;
            newTextSize -= len;
        }

        /* increment the change count */
        *changed = TRUE;

        if (singletonMacro && !textLength) {
            /* since the macro is the entire value, no harm (and */
            /* more robust) to surround it by quotes */
            *quoted = TRUE;
            return TRUE;
        }

        /* need to go back and re-evaluate the value for more macros */
    }
    StringCopy(newText, text, newTextSize);
    return TRUE;
}

BOOL ExpandMacros(TASK *task, HTML_MARKUP *htmlMarkup)
{
    uint ctr;
    HTML_ATTRIBUTE *attrib;
    union
    {
        char newTag[MAX_VARNAME_LEN];
        char newName[MAX_VARNAME_LEN];
        char newValue[MAX_VARVALUE_LEN];
    } newText;
    BOOL quoted;
    BOOL changed;

    assert(task != NULL);
    assert(htmlMarkup != NULL);

    /* expand any macros in the tags */
    if(htmlMarkup->tag != NULL)
    {
        if(ExpandMacrosInString(task, htmlMarkup->tag, newText.newTag,
            MAX_VARNAME_LEN, &quoted, &changed) != TRUE)
        {
            return FALSE;
        }

        if(changed)
        {
            ChangeMarkupTag(htmlMarkup, newText.newTag);
        }
    }
    
    if (stricmp(htmlMarkup->tag, "WHILE") == 0
        || stricmp(htmlMarkup->tag, "!---") == 0) {
        return TRUE;
    }

    /* do the same for all attributes, both name and value */
    for(ctr = 0; ctr < htmlMarkup->attribCount; ctr++)
    {
        attrib = &htmlMarkup->attrib[ctr];

        if(attrib->name != NULL)
        {
            memset(newText.newName, 0, MAX_VARNAME_LEN);
            if (!ExpandMacrosInString(task, attrib->name, newText.newName,
                                      MAX_VARNAME_LEN, &quoted, &changed))
            {
                return FALSE;
            }
        
            if(changed)
            {
                ChangeAttributeName(attrib, newText.newName);
            }
        }


        if(attrib->value != NULL)
        {
            quoted = attrib->quoted;
            memset(newText.newValue, 0, MAX_VARVALUE_LEN);
            if (!ExpandMacrosInString(task, attrib->value, newText.newValue,
                                      MAX_VARVALUE_LEN, &quoted, &changed))
            {
                return FALSE;
            }
        
            if(changed)
            {
                ChangeAttributeValue(attrib, newText.newValue, quoted);
            }
        }
    } 
    
    return TRUE;
}   


uint ExpandMetatag(TASK *task, HTML_MARKUP *htmlMarkup)
{
    const char *options;
    char *optionCopy;
    FIND_TOKEN findToken;
    char *optionPtr;
    uint optionCount;
    uint ctr;
    uint optionCtr;
    const char *optionArray[METATAG_MAX_OPTIONS];
    int  wildcardOptions[METATAG_MAX_OPTIONS];
    int  wildcardNum;
    int  wildcardLength;
    const HTML_ATTRIBUTE *attrib;
    BOOL found;
    VARSTORE defVarstore;
    uint flag;
    const char *defName;
    STREAM *blockStream;
    STREAM *defStream;
    STREAM defFile;
    TASK newTask;
    BOOL result;
    BOOL hasWildcard;
    uint macroType;

    /* first things first: find the tag in the metatag store */
    if(VariableExists(task->varstore, htmlMarkup->tag) == FALSE)
    {
        /* don't change a thing */
        return MARKUP_OKAY;
    }

    /* verify the macro in the store is a metatag definition */
    macroType = GetVariableType(task->varstore, htmlMarkup->tag);

    if(macroType != VAR_TYPE_DEF_MACRO
       && macroType != VAR_TYPE_BLOCKDEF_MACRO)
    {
        return MARKUP_OKAY;
    }

    /* get a pointer to the name */
    defName = htmlMarkup->tag;

    /* get the filename the DEF macro is held in */
    if ((defStream = (STREAM *) GetVariableValue(task->varstore, defName))
        == NULL)
    {
        /* this shouldnt be */
        HtpMsg(MSG_ERROR, task->infile, "DEF macro \"%s\" was not store properly",
            defName);
        return MARKUP_ERROR;
    }

    /* get options to compare against markups paramater list */
    options = GetVariableParam(task->varstore, defName);

    /* initialize a local variable store, even if its not used */
    InitializeVariableStore(&defVarstore);

    /* if NULL, then no options allowed */
    if(options == NULL)
    {
        if(htmlMarkup->attribCount > 0)
        {
            HtpMsg(MSG_ERROR, task->infile, "DEF macro \"%s\" does not specify any options",
                defName);
            DestroyVariableStore(&defVarstore);
            return MARKUP_ERROR;
        }
    }
    else
    {
        /* options should be space-delimited, use StringToken() */
        if((optionCopy = DuplicateString(options)) == NULL)
        {
            HtpMsg(MSG_ERROR, task->infile, "Unable to duplicate option macro (out of memory?)");
            DestroyVariableStore(&defVarstore);
            return MARKUP_ERROR;
        }

        /* build array of pointers to null-terminated option */
        optionCount = 0;
        optionPtr = StringFirstToken(&findToken, optionCopy, " ");
        hasWildcard = FALSE;
        while((optionPtr != NULL) && (optionCount < METATAG_MAX_OPTIONS))
        {
            /* ignore multiple spaces */
            if(*optionPtr != ' ')
            {
                /* save a pointer to the token */
                optionArray[optionCount++] = optionPtr;
            }

            optionPtr = StringNextToken(&findToken);

            if (strcmp(optionArray[optionCount-1], "*") == 0)
            {
                hasWildcard = TRUE;
            }
        }

        wildcardLength = 0;
        wildcardNum = 0;

        /* now, see if every paramater in the markup is also in the option list */
        /* (the reverse is not required) */
        for(ctr = 0; (attrib = MarkupAttribute(htmlMarkup, ctr)) != NULL; ctr++)
        {
            found = FALSE;
            for(optionCtr = 0; optionCtr < optionCount; optionCtr++)
            {
                if(stricmp(optionArray[optionCtr], attrib->name) == 0)
                {
                    found = TRUE;
                    break;
                }
            }

            if (found)
            {
                /* since this is a good attribute, add it to the local store */
                flag = (attrib->quoted == TRUE) ? VAR_FLAG_QUOTED : VAR_FLAG_NONE;
                /* No need to duplicate values, as varstore is
                 * destroyed before this function returns.  
                 */
                if(StoreVariable(&defVarstore, attrib->name, attrib->value,
                                 VAR_TYPE_SET_MACRO, flag, NULL, NULL) == FALSE)
                {
                    HtpMsg(MSG_ERROR, task->infile,
                           "Unable to store local macro for metatag");
                    DestroyVariableStore(&defVarstore);
                    FreeMemory(optionCopy);
                    return MARKUP_ERROR;
                }
            }
            else if (hasWildcard)
            {
                wildcardOptions[wildcardNum++] = ctr;
                wildcardLength += 
                    strlen(attrib->name) + strlen(attrib->value) + 2;
                if (attrib->quoted == TRUE)
                    wildcardLength += 2;
            }
            else
            {
                HtpMsg(MSG_ERROR, task->infile,
                       "DEF macro \"%s\" does not accept a parameter named \"%s\"",
                       defName, attrib->name);
                FreeMemory(optionCopy);
                DestroyVariableStore(&defVarstore);
                return MARKUP_ERROR;
            }
        }

        /* looks good, this is no longer needed */
        FreeMemory(optionCopy);

        if (wildcardLength > 0) {
            /* Build the wildcard string. */
            char *ptr, *wildcards;
            wildcards = AllocMemory(wildcardLength);
            ptr = wildcards;
            for (ctr = 0; ctr < wildcardNum; ctr++) {
                attrib = MarkupAttribute(htmlMarkup, wildcardOptions[ctr]);
                strcpy(ptr, attrib->name);
                ptr += strlen(attrib->name);
                *ptr++ = '=';
                if (attrib->quoted)
                    *ptr++ = '"';
                strcpy(ptr, attrib->value);
                ptr += strlen(attrib->value);
                if (attrib->quoted)
                    *ptr++ = '"';
                *ptr++ = ' ';
            }
            /* Replace last space with NUL to terminate wildcards */
            *(ptr-1) = 0;
            if(StoreVariable(&defVarstore, "*", wildcards,
                             VAR_TYPE_SET_MACRO, VAR_FLAG_DEALLOC_VALUE,
                             NULL, NULL) == FALSE)
            {
                HtpMsg(MSG_ERROR, task->infile,
                       "Unable to store local macro for metatag");
                DestroyVariableStore(&defVarstore);
                FreeMemory(optionCopy);
                FreeMemory(wildcards);
                return MARKUP_ERROR;
            }
        } else {
            if(StoreVariable(&defVarstore, "*", "",
                             VAR_TYPE_SET_MACRO, VAR_FLAG_NONE,
                             NULL, NULL) == FALSE)
            {
                HtpMsg(MSG_ERROR, task->infile,
                       "Unable to store local macro for metatag");
                DestroyVariableStore(&defVarstore);
                FreeMemory(optionCopy);
                return MARKUP_ERROR;
            }
        }
    }

    /* If this is a BLOCKDEF macro expand the trailing block into */
    /* the macro named block */
    if (macroType == VAR_TYPE_BLOCKDEF_MACRO) 
    {
        blockStream = (STREAM*) AllocMemory(sizeof(STREAM));
        if (!ReadinBlock(task, htmlMarkup, blockStream)) 
        {
            DestroyVariableStore(&defVarstore);
            return MARKUP_ERROR;
        }

        /* store the block file name and the block macro name as a variable */
        if(StoreVariable(&defVarstore, "BLOCK", blockStream, 
                         VAR_TYPE_BLOCK_MACRO, VAR_FLAG_NONE,
                         NULL, BlockDestructor) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, 
                   "unable to store macro information (out of memory?)");
            DestroyVariableStore(&defVarstore);
            return MARKUP_ERROR;
        }
    }

    /* expand the DEF macro like a block macro ... */

    /* open the file the macro is held in */
    if(CreateBufferReader(&defFile, defStream) == FALSE)
    {
        HtpMsg(MSG_ERROR, task->infile,
            "unable to open block for DEF macro \"%s\"",
            defName);
        DestroyVariableStore(&defVarstore);
        return MARKUP_ERROR;
    }

    HtpMsg(MSG_INFO, task->infile, "dereferencing %s macro \"%s\"", 
           macroType == VAR_TYPE_BLOCKDEF_MACRO ? "blockdef" : "def",
           defName);

    /* build a new task structure */
    newTask.infile = &defFile;
    newTask.outfile = task->outfile;
    newTask.sourceFilename = task->sourceFilename;

    if (options != NULL || macroType == VAR_TYPE_BLOCKDEF_MACRO) {
        /* Put in the defVarstore */
        PushVariableStoreContext(task->varstore, &defVarstore);
        newTask.varstore = &defVarstore;
    } else {
        newTask.varstore = task->varstore;
    }


    /* process the new input file */
    result = ProcessTask(&newTask);

    /* remove the new context if necessary */
    if(newTask.varstore == &defVarstore)
    {
        assert(PeekVariableStoreContext(&defVarstore) == &defVarstore);
        PopVariableStoreContext(&defVarstore);
    }

    /* no matter what, destroy the local store */
    DestroyVariableStore(&defVarstore);

    CloseStream(&defFile);

    if (!result) {
        /* Error message was already spitted out.  However, we should
         * give a hint where the meta-tag was called.
         */
        HtpMsg(MSG_ERROR, task->infile,
               "... in metatag \"%s\"", defName);
        return MARKUP_ERROR;
    }

    return DISCARD_MARKUP;
}


uint ExpandAll(TASK *task, HTML_MARKUP *htmlMarkup, 
               char** newPlaintextPtr, uint markupType)
{
    uint ctr;
    uint markupResult;

    if(ExpandMacros(task, htmlMarkup) == FALSE)
    {
        /* problem encountered trying to expand macros */
        return MARKUP_ERROR;
    }

    /* give the metatag processor a chance to expand metatags */
    /* this is a little strange, but if MARKUP_OKAY it means the the */
    /* metatag processor didnt recognize the tag, and therefore should */
    /* be handled by the other processors */
    if((markupResult = ExpandMetatag(task, htmlMarkup)) == MARKUP_OKAY)
    {
        /* find the first processor that wants to do something with the */
        /* markup tag */
        for(ctr = 0; ctr < MARKUP_PROCESSOR_COUNT; ctr++)
        {
            if(markupProcessor[ctr].markupType & markupType)
            {
                if(IsMarkupTag(htmlMarkup, markupProcessor[ctr].tag))
                {
                    assert(markupProcessor[ctr].markupFunc != NULL);
                    
                    markupResult = markupProcessor[ctr]
                        .markupFunc(task, htmlMarkup, newPlaintextPtr);
                    
                    break;
                }
            }
        }
    }
    
    /* unless the function requested to use its new markup string, */
    /* take the HTML_MARKUP structure and build a new markup */
    if((markupResult != NEW_MARKUP) && (markupResult != DISCARD_MARKUP))
    {
        if(MarkupToPlaintext(htmlMarkup, newPlaintextPtr) == FALSE)
        {
            HtpMsg(MSG_ERROR, task->infile, "unable to build plain text from markup (out of memory?)");
            return MARKUP_ERROR;
        }
    }

    return markupResult;
}
