<?php
/**
 * IMAP_Thread provides functions for working with imap_thread() output.
 *
 * $Horde: framework/IMAP/IMAP/Thread.php,v 1.4.10.6 2005/03/16 02:43:42 slusarz Exp $
 *
 * Copyright 2004-2005 Michael Slusarz <slusarz@curecanti.org>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Michael Slusarz <slusarz@curecanti.org>
 * @version $Revision: 1.4.10.6 $
 * @since   Horde 3.0
 * @package Horde_IMAP
 */
class IMAP_Thread {

    /**
     * imap_thread() output.
     *
     * @var array $_thread
     */
    var $_threadarray;

    /**
     * The list of message IDs to ignore.
     *
     * @var array $_ignore
     */
    var $_ignore = array();

    /**
     * Internal thread data structure.
     *
     * @var array $_thread
     */
    var $_thread = array();

    /**
     * Array index to Message index lookup array.
     *
     * @var array $_lookup
     */
    var $_lookup = array();

    /**
     * Constructor.
     *
     * @access public
     *
     * @param array $ob               Output from imap_thread().
     * @param optional array $ignore  Ignore list.
     */
    function IMAP_Thread($ob, $ignore = array())
    {
        $this->_threadarray = $ob;
        $this->_ignore = $ignore;
        $this->_processStructure();
    }

   /**
     * Gets the indention level for an IMAP message index.
     *
     * @access public
     *
     * @param integer $index  The IMAP message index.
     *
     * @return mixed  Returns the thread indent level if $index found.
     *                Returns false on failure.
     */
    function getThreadIndent($index)
    {
        $key = $this->_getKey($index);
        if (!is_null($key)) {
            if (isset($this->_thread[$key]['level'])) {
                return $this->_thread[$key]['level'];
            }
        }

         return false;
    }

    /**
     * Gets the base thread index for an IMAP message index.
     *
     * @access public
     *
     * @param integer $index  The IMAP message index.
     *
     * @return mixed  Returns the base IMAP index if $index is part of a
     *                thread.
     *                Returns false on failure.
     */
    function getThreadBase($index)
    {
        $key = $this->_getKey($index);
        if (!is_null($key)) {
            if (!empty($this->_thread[$key]['base'])) {
                return $this->_thread[$key]['base'];
            }
        }

        return false;
    }

    /**
     * Is this index the last in the current level?
     *
     * @access public
     *
     * @param integer $index  The IMAP message index.
     *
     * @return boolean  Returns true if $index is the last element in the
     *                  current thread level.
     *                  Returns false if not, or on failure.
     */
    function lastInLevel($index)
    {
        $key = $this->_getKey($index);
        if (!is_null($key)) {
            if (!empty($this->_thread[$key]['last'])) {
                return $this->_thread[$key]['last'];
            }
        }

        return false;
    }

    /**
     * Do the message index -> array index lookup.
     *
     * @access private
     *
     * @param integer $index  The IMAP message index.
     *
     * @return mixed  The array index value or null if no index.
     */
    function _getKey($index)
    {
        return (isset($this->_lookup[$index])) ? $this->_lookup[$index] : null;
    }

    /**
     * Return the sorted list of messages indices.
     *
     * @access public
     *
     * @param boolean $new  True for newest first, false for oldest first.
     *
     * @return array  The sorted list of messages.
     */
    function messageList($new)
    {
        return ($new) ? array_reverse(array_keys($this->_lookup)) : array_keys($this->_lookup);
    }

    /**
     * Returns the list of messages in the current thread.
     *
     * @access public
     *
     * @param integer $index  The IMAP index of the current message.
     *
     * @return array  A list of IMAP message indices.
     */
    function getThread($index)
    {
        /* Find the beginning of the thread. */
        $begin = $this->getThreadBase($index);
        $key = $this->_getKey($begin);
        if (is_null($key) || empty($begin)) {
            return array($index);
        }

        /* Work forward from the first thread element to find the end of the
         * thread. */
        $thread_list = array($this->_thread[$key]['index']);
        while (++$key) {
            if (!isset($this->_thread[$key])) {
                break;
            }
            $curr = $this->_thread[$key];
            if ($curr['base'] == $begin) {
                $thread_list[] = $this->_thread[$key]['index'];
            } else {
                break;
            }
        }

        return $thread_list;
    }

    /**
     * Process the output from imap_thread() into an internal data structure.
     *
     * @access private
     */
    function _processStructure()
    {
        $container = $container_base = $last_index = $thread_base = $thread_base_idx = null;
        $indices = array();
        $i = $last_i = $level = 0;

        foreach ($this->_threadarray as $key => $val) {
            $pos = strpos($key, '.');
            $index = substr($key, 0, $pos);
            $type = substr($key, $pos + 1);

            switch ($type) {
            case 'num':
                if ($val === 0) {
                    $container = $index;
                } else {
                    $i++;
                    if (is_null($container) && empty($level)) {
                        $thread_base = $val;
                        $thread_base_idx = $index;
                    }
                    $this->_lookup[$val] = $index;
                    $this->_thread[$index] = array();
                    $this->_thread[$index]['index'] = $val;
                }
                break;

            case 'next':
                if (!is_null($container) && ($container === $index)) {
                    $container_base = $this->_threadarray[$val . '.num'];
                } else {
                    $i++;
                    if (!is_null($container)) {
                        $this->_thread[$index]['base'] = $container_base;
                    } else {
                        $this->_thread[$index]['base'] = (!empty($level) || ($val != 0)) ? $thread_base : null;
                    }
                    $level++;
                }
                break;

            case 'branch':
                if ($container === $index) {
                    $container = $container_base = null;
                    $this->_thread[$last_index]['last'] = true;
                } else {
                    $this->_thread[$index]['level'] = $level--;
                    $this->_thread[$index]['last'] = !(!is_null($container) && empty($level));
                    if ($index === $thread_base_idx) {
                        $index = null;
                    } elseif (!empty($level) &&
                              !is_null($last_index) &&
                              isset($this->_thread[$last_index])) {
                        $this->_thread[$last_index]['last'] = ($last_i == ($i - 1));
                    }
                }
                $last_index = $index;
                $last_i = $i++;
                break;
            }
        }
    }

}
