#include "error.h"
#ifndef __htmlh__
  #include "html.h"
#endif

#include <iostream>
#include <cstdio>

static string StdError(ErrNum num){
  switch(num){
    case e_CloseNoDef: return "</@DEF> without <@DEF>";
    case e_CloseNoIf: return "</@IF> without <@IF>";
    case e_ElseNoClose: return "<@ELSE> without </@IF>";
    case e_ElseNoIf: return "<@ELSE> without <@IF>";
    case e_ElseIfNoClose: return "<@ELSEIF> without </@IF>";
    case e_ElseIfNoIf: return "<@ELSEIF> without <@IF>";
    case e_DefNoClose: return "<@DEF> without </@DEF>";
    case e_IfNoClose: return "<@IF> without </@IF>";
    case e_CommentNoClose: return "<@COMMENT> without </@COMMENT>";
    case e_CloseNoComment: return "</@COMMENT> without <@COMMENT>";

    case e_BadCompare: return "COMPARE must be EQ, NE, LT, GT, LE, GE, EQINT, NEINT, LTINT, GTINT, LEINT, GEINT, IN or NI";
    case e_BadScope: return "SCOPE must be LOCAL, PARENT or GLOBAL";
    case e_BadVarName: return "Illegal variable name";
    case e_BadProcName: return "Illegal procedure name";
    case e_BadFormat: return "FORMAT must be one of FOLDED, OPEN or FLAT";
    case e_BadFormat2: return "FORMAT must be one of HITOP, SAFE or VERBATIM";

    case e_NoName: return "NAME missing";
    case e_NoSrc: return "SRC missing";
    case e_NoEntry: return "ENTRY missing";
    case e_NoValue: return "VALUE missing";
    case e_NoDataSrc: return "<@NAV> must have either a DATA or SRC";
    case e_NoFrom: return "FROM missing";
    case e_NoTo: return "TO missing";

    case e_UndefVar: return "Undefined variable";
    case e_UndefProc: return "Undefined procedure";
    case e_UndefEntry: return "Undefined ENTRY procedure";
    case e_UndefPregroup: return "Undefined PREGROUP procedure";
    case e_UndefPostgroup: return "Undefined POSTGROUP procedure";
    case e_UndefStringProc: return "Undefined string process";

    case e_FileNotOpen: return "Couldn't open file";
    case e_TooDeep: return "Maximum recursion level exceeded";
    case e_RunOnVar: return "Unterminated inline variable";
    case e_MissingColon: return "Missing colon";
    case e_ExpectBrackets: return "Expecting brackets";
    case e_ExpectCloseBracket: return "Expecting close bracket";
    case e_UntermQuotes: return "Unterminated quotes";
    case e_DataBadFormat: return "Error in DATA format";
    case e_NavBadFormat: return "Error in format of the navigation file";
    case e_StepIsZero: return "STEP cannot be zero";
    case e_EndedInQuotes: return "The file ended while still inside quotes. (Possible run-on string?)";
    case e_EndedInSingles: return "The file ended while still inside single quotes. (Possible run-on string?)";
    case e_EndedInElement: return "The file ended while still inside angle brackets. (Possible run-on string?)";

    case e_ParamWrong: return " has the wrong number of parameters";
    case e_ParamNone: return " shouldn't have any parameters";
    case e_ParamOne: return " should have one parameter";
  }
  return "";
}

extern void doError(const string& message);
extern void doWarn(const string& message);
extern void doDebug(const string& message);

void Warn(const HTML& cur,const string& mess){
  doWarn(cur.Origin()+": "+mess+" in \""+cur.Print()+"\"");
}

void Warn(const string& mess){
  doWarn(mess);
}

void Warn(ErrNum mess){
  Warn(StdError(mess));
}

void Debug(const HTML& cur,const string& mess){
  doDebug(cur.Origin()+": "+mess);
}

void Error(const HTML& cur,const string& mess){
  doError(cur.Origin()+": "+mess+" in \""+cur.Print()+"\"");
}

void Error(const HTML& cur,ErrNum mess){
  Error(cur,StdError(mess));
}

void Error(const HTML& cur,ErrNum mess,const string& data){
  Error(cur,StdError(mess)+" '"+data+"'");
}

void Error(const HTML& cur,const string& data,ErrNum mess){
  Error(cur,data+" "+StdError(mess));
}

void Error(const string& mess){
  doError(": "+mess);
}

void Error(ErrNum mess){
  Error(StdError(mess));
}

