//$Id: form-editor-canvas.cc,v 1.11 2004/01/27 18:56:13 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "form-editor/form-editor-canvas.h"
#include "form-editor/form-editor.h"
#include "colorpalette.h"

#include <gnome--/canvas.h>
#include <gnome--/canvas-group.h>
#include <gnome--/canvas-text.h>
#include <gnome--/canvas-polygon.h>

#include <gdk-pixbuf/gnome-canvas-pixbuf.h>

#include "bitmapfamily-res.h"

#include <list>

using namespace Guikachu::GUI;
using Guikachu::GUI::FormEditor::Font;
using Guikachu::GUI::FormEditor::Glyph;

namespace
{
    GdkPixbuf * clip_pos_pixbuf (GdkPixbuf     *pixbuf,
				 bool           text,
				 int           &x,
				 int           &y,
				 int            clip_width,
				 int            clip_height,
				 GtkAnchorType  anchor);
            
    typedef std::list<const Glyph*> typeset_t;
    
    GdkPixbuf * render_text_impl (const Font &font,
				  const std::string &text,
				  const Gdk_Color   &color);
    
    GdkPixbuf * render_text_multi_line_impl (const Font        &font,
					     const std::string &text,
					     const Gdk_Color   &color);

    void render_raw_line (const Font      &font,
			  const typeset_t &typeset,
			  const Gdk_Color &color,
			  guint8          *target,
			  int              target_rowstride);
}

Gnome::CanvasItem* FormEditor::draw_text (Gnome::CanvasGroup &group,
                                          const std::string  &text,
                                          const Font         &font,
                                          const Gdk_Color    &color,
                                          int                 x,
                                          int                 y,
                                          int                 clip_width,
                                          int                 clip_height,
                                          GtkAnchorType       anchor,
                                          bool                multi_line)
{
    Gnome::CanvasGroup *text_group = new Gnome::CanvasGroup (group, 0, 0);
    
    GdkPixbuf *pixbuf = multi_line ?
	render_text_multi_line_impl (font, text, color) : render_text_impl (font, text, color);

    if (!pixbuf)
	return 0;

    GdkPixbuf *clipped_pixbuf;
    clipped_pixbuf = clip_pos_pixbuf (pixbuf, true, x, y, clip_width, clip_height, anchor);
    gdk_pixbuf_unref (pixbuf);
    
    int width = gdk_pixbuf_get_width (clipped_pixbuf);
    int height = gdk_pixbuf_get_height (clipped_pixbuf);

    // Draw the text
    draw_pixbuf (*text_group, clipped_pixbuf, x, y);
    gdk_pixbuf_unref (clipped_pixbuf);

    // Draw a transparent background block (to get mouse events for
    // the whole area)
    FormEditor::draw_rectangle (*text_group,
				x, y,
				x + width, y + height);
    
    return text_group;
}

Gnome::CanvasItem* FormEditor::draw_pixbuf (Gnome::CanvasGroup &group,
					    GdkPixbuf          *pixbuf,
					    int                 x,
					    int                 y)
{
    return Gtk::wrap (gnome_canvas_item_new (group.gtkobj (),
					     gnome_canvas_pixbuf_get_type (),
					     "pixbuf", pixbuf,
					     "x", (double) x,
					     "y", (double) y,
					     0));
}

namespace
{

// Thar Be Dragons Here
GdkPixbuf * clip_pos_pixbuf (GdkPixbuf     *pixbuf,
			     bool           text,
			     int           &x,
			     int           &y,
			     int            clip_width,
			     int            clip_height,
			     GtkAnchorType  anchor)
{
    int clip_x = x;
    int clip_y = y;
    
    int width = gdk_pixbuf_get_width (pixbuf);
    int height = gdk_pixbuf_get_height (pixbuf);

    // Make sure (x,y) is at the upper-left corner
    
    // Horizontal offset
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_SOUTH_WEST:
	break;
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_SOUTH:
	x += (clip_width - width + 1) / 2;
	break;
    case GTK_ANCHOR_NORTH_EAST:
    case GTK_ANCHOR_EAST:
    case GTK_ANCHOR_SOUTH_EAST:
	x += (clip_width - width);
    }

    // Vertical offset
    int text_dy;
    switch (anchor)
    {
    case GTK_ANCHOR_NORTH_WEST:
    case GTK_ANCHOR_NORTH:
    case GTK_ANCHOR_NORTH_EAST:
	break;
    case GTK_ANCHOR_WEST:
    case GTK_ANCHOR_CENTER:
    case GTK_ANCHOR_EAST:
        // Ugly code, but this is how PalmOS works -- don't ask me why
        text_dy = clip_height - height;
        if (text && text_dy < 0 && (clip_height > (height + 1) / 2))
            --text_dy;
	if (!text)
	    ++text_dy;
	    
        y += text_dy / 2;
	break;
    case GTK_ANCHOR_SOUTH_WEST:
    case GTK_ANCHOR_SOUTH:
    case GTK_ANCHOR_SOUTH_EAST:
	y += (clip_height - height);
	break;
    }    

    int clip_offset_x = std::max (0, clip_x - x);
    int clip_offset_y = std::max (0, clip_y - y);

    int clip_size_x = width - clip_offset_x;
    if (clip_width)
        clip_size_x -= std::max (0, (x + width - (clip_x + clip_width)));

    int clip_size_y = height - clip_offset_y;
    if (clip_height)
        clip_size_y -= std::max (0, (y + height - (clip_y + clip_height)));

    GdkPixbuf *dest = gdk_pixbuf_new (GDK_COLORSPACE_RGB, true, 8, clip_size_x, clip_size_y);

    x += clip_offset_x;
    y += clip_offset_y;
    
    gdk_pixbuf_copy_area (pixbuf,
                          clip_offset_x, clip_offset_y,
			  clip_size_x, clip_size_y,
			  dest, 0, 0);
    
    return dest;
}

// Get next character from a string, with escape codes resolved
static char next_char (const std::string &text,
		       std::string::const_iterator &i)
{
    // Not an escape sequence
    if (*i != '\\')
	return *(i++);

    // Trailing '\'
    if (++i == text.end ())
	return '\\';

    // Check escape type
    char escape_type = *i;

    // Simple escape sequences
    switch (escape_type)
    {
    case 'a':
	i++;
	return '\x07';
    case 'b':
	i++;
	return '\x08';
    case 'e':
	i++;
	// FIXME: Figure out the value of '\e'
	return 'e';
    case 'f':
	i++;
	return '\x0c';
    case 'n':
	i++;
	return '\x0a';
    case 'r':
	i++;
	return '\n';
    case 't':
	i++;
	return '\x09';
    case 'v':
	i++;
	return '\x0b';
    case 'z':
	i++;
	// FIXME: Figure out the value of '\z'
	return 'z';
    }

#define IS_OCTAL(c)  ((c) >= '0' && (c) <= '7')
#define GET_OCTAL(c) ((c) - '0')

#define IS_HEX(c)  (((c) >= '0' && (c) <= '9') ||	\
		    ((c) >= 'a' && (c) <= 'f') ||	\
		    ((c) >= 'A' && (c) <= 'F'))

#define GET_HEX(c) (((c) >= '0' && (c) <= '9') ?	\
		     (c) - '0' :			\
		     ((c) >= 'a' && (c) <= 'f') ?	\
		      (c) - 'a' + 10 : (c) - 'A' + 10)
    
    
    // Octal numbers
    if (IS_OCTAL (escape_type))
    {
	char octal_value = 0;
	
	// Get next at most three octal numbers
	for (int j = 0; j < 3 && i != text.end (); i++, j++)
	{
	    if (!IS_OCTAL (*i))
		// Not an octal number
		break;

	    octal_value = octal_value * 8 + GET_OCTAL (*i);
	}

	return octal_value;
    }

    // Hexadecimal numbers
    if (escape_type == 'x')
    {
	char hex_value = 0;

	// Skip leading 0's
	while (*(++i) == '0' && i != text.end ());
	
	// Read at most two hexadecimal characters
	for (int j = 0; j < 2 && i != text.end (); i++, j++)
	{
	    if (!IS_HEX (*i))
		break;

	    hex_value = hex_value * 16 + GET_HEX (*i);
	}

	return hex_value;
    }

    // Unknown escape sequence: return escaped character
    return *(i++);
}

void set_pixel (guint8          *pixel,
		const Gdk_Color &color)
{
    pixel[0] = color.red   >> 8;
    pixel[1] = color.green >> 8;
    pixel[2] = color.blue  >> 8;
    pixel[3] = 255;
}

void free_pixdata_gfree (guint8 *pixels, gpointer data)
{
    g_free (pixels);
}

void free_pixdata_delete (guint8 *pixels, gpointer data)
{
    delete[] pixels;
}

void render_raw_line (const Font      &font,
		      const typeset_t &typeset,
		      const Gdk_Color &color,
		      guint8          *target,
		      int              target_rowstride)
{
    int line_height = font.get_line_height ();
    
    for (typeset_t::const_iterator i = typeset.begin (); i != typeset.end (); i++)
    {
	const Glyph *glyph = *i;
	
	bool **curr_bitmap = glyph->bitmap;
	guint8 *curr_row = target;
	
	for (int y = 0; y < line_height; y++)
	{
	    guint8 *curr_pixel = curr_row;
	    
	    for (int x = 0; x < glyph->width; x++)
		{
		    if (curr_bitmap[y][x])
			set_pixel (curr_pixel, color);
		    
		    curr_pixel += 4;
		}
		
		curr_row += target_rowstride;
	    }
	    
	    target += glyph->width * 4;
	}
    }
    
GdkPixbuf* render_text_impl (const Font        &font,
			     const std::string &text,
			     const Gdk_Color   &color)
{
    if (text == "")
	return 0;

    int width = 0;
    int height = font.get_line_height ();
    typeset_t typeset;
    
    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	const Glyph &g = font.get_glyph (c);
	
	typeset.push_back (&g);
	width += g.width;
    }

    guint8 *pixbuf_data = g_new0 (guint8, width * height * 4);
    render_raw_line (font, typeset, color, pixbuf_data, width * 4);
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_gfree, 0);
}

GdkPixbuf* render_text_multi_line_impl (const Font        &font,
					const std::string &text,
					const Gdk_Color   &color)
{
    if (text == "")
	return 0;

    std::list <typeset_t> typesets;
    int width = 0;
    int height = 0;

    int current_width = 0;
    std::list <const Glyph*> current_typeset;
    int line_height = font.get_line_height ();
    
    std::string::const_iterator i = text.begin ();
    while (i != text.end ())
    {
	char c = next_char (text, i);
	if (c != '\n')
	{
	    const Glyph &g = font.get_glyph (c);
	    
	    current_typeset.push_back (&g);
	    current_width += g.width;
	} else {
	    typesets.push_back (current_typeset);
	    current_typeset.clear ();

	    width = std::max<int> (width, current_width);
	    current_width = 0;
	    height += line_height;
	}
    }
    
    // Add last line
    typesets.push_back (current_typeset);
    width = std::max<int> (width, current_width);
    height += font.get_line_height ();
    
    guint8 *pixbuf_data = g_new0 (guint8, width * height * 4);

    int offset = 0;
    for (std::list <typeset_t>::const_iterator i = typesets.begin ();
	 i != typesets.end (); i++)
    {
	render_raw_line (font, *i, color, pixbuf_data + offset, width * 4);

	offset += (line_height * width) * 4;
    }
    
    return gdk_pixbuf_new_from_data (pixbuf_data,
				     GDK_COLORSPACE_RGB,
				     true,
				     8,
				     width, height,
				     width * 4,
				     free_pixdata_gfree, 0);
}

} // anonymous namespace
    
Gnome::CanvasItem* FormEditor::draw_rectangle (Gnome::CanvasGroup &group,
					       int x1, int y1,
					       int x2, int y2)
{
    Gnome::CanvasPoints points;
    using Gnome::Art::Point;

    points.push_back (Point (x1, y1));
    points.push_back (Point (x2, y1));
    points.push_back (Point (x2, y2));
    points.push_back (Point (x1, y2));
    
    return new Gnome::CanvasPolygon (group, points);
}

GdkPixbuf* FormEditor::render_bitmap (const Magick::Image &image, Resources::Bitmap::BitmapType bitmap_type)
{
    if (!image.isValid ())
        return 0;
    
    unsigned int width = image.columns ();
    unsigned int height = image.rows ();
    guint8 *pixels = new guint8[width * height * 4];
    
    // Magick::Image::write is not a const method
    const_cast<Magick::Image&>(image).write (0, 0, image.columns (), image.rows (),
                                            "RGB", Magick::CharPixel, pixels);
    
    dither_pixels (bitmap_type, pixels, width, height);
    
    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_data (pixels, GDK_COLORSPACE_RGB, FALSE, 8, width, height,
                                                  width * 3, free_pixdata_delete, 0);

    GdkPixbuf *retval = gdk_pixbuf_add_alpha (pixbuf, TRUE, 0xFF, 0xFF, 0xFF);
    gdk_pixbuf_unref (pixbuf);

    return retval;
}

Gnome::CanvasItem * FormEditor::draw_bitmap (Gnome::CanvasGroup            &group,
                                             const Magick::Image           &image,
                                             Resources::Bitmap::BitmapType  bitmap_type,
                                             const std::string             &background_color,
                                             int                            x,
                                             int                            y,
                                             int                            clip_width,
                                             int                            clip_height,
                                             GtkAnchorType                  anchor)
{
    GdkPixbuf *pixbuf;
    GdkPixbuf *pixbuf_img = render_bitmap (image, bitmap_type);
    if (!pixbuf_img)
        return 0;

    // Clip pixbuf to desired size
    pixbuf = clip_pos_pixbuf (pixbuf_img, false, x, y, clip_width, clip_height, anchor);
    gdk_pixbuf_unref (pixbuf_img);

    // Create parent group to store background + pixbuf
    Gnome::CanvasGroup *ret_val = new Gnome::CanvasGroup (group, x, y);

    // Create clickable background
    using namespace Gnome::CanvasHelpers;
    *(draw_rectangle (*ret_val, 0, 0, gdk_pixbuf_get_width (pixbuf), gdk_pixbuf_get_height (pixbuf)))
        << fill_color (background_color);

    // Create pixbuf item
    draw_pixbuf (*ret_val, pixbuf);
    gdk_pixbuf_unref (pixbuf);

    return ret_val;
}

namespace
{
    typedef std::pair<Magick::Image, Guikachu::Resources::Bitmap::BitmapType> image_pair_t;
    
#define TRY_IMAGETYPE(t)                                                \
    image = res->get_image (Guikachu::Resources::Bitmap::TYPE_##t);     \
    if (image.isValid ())                                               \
        return image_pair_t (image, Guikachu::Resources::Bitmap::TYPE_##t);

    image_pair_t get_grey_image (Guikachu::Resources::BitmapFamily *res)
    {
        Magick::Image image;

        TRY_IMAGETYPE(GREY_16);
        TRY_IMAGETYPE(GREY_4);
        TRY_IMAGETYPE(MONO);

        return image_pair_t (image, Guikachu::Resources::Bitmap::TYPE_MONO);
    }
                                   
    image_pair_t get_color_image (Guikachu::Resources::BitmapFamily *res)
    {
        Magick::Image image;

        TRY_IMAGETYPE(COLOR_16K);
        TRY_IMAGETYPE(COLOR_256);
        TRY_IMAGETYPE(COLOR_16);

        return get_grey_image (res);
    }

} // anonymous namespace

Gnome::CanvasItem * FormEditor::draw_bitmapref (Gnome::CanvasGroup            &group,
                                                const Properties::ResourceRef &ref_prop,
                                                bool                           color,
                                                const std::string             &background_color,
                                                int                            x,
                                                int                            y,
                                                int                            clip_width,
                                                int                            clip_height,
                                                GtkAnchorType                  anchor)
{
    Resource *res = ref_prop.resolve ();
    Magick::Image image;
    Resources::Bitmap::BitmapType bitmap_type;
    
    if (Resources::Bitmap *bitmap = dynamic_cast<Resources::Bitmap*> (res))
    {
        bitmap_type = bitmap->bitmap_type;
        if (!color)
            bitmap_type = std::min (bitmap_type, Resources::Bitmap::TYPE_GREY_16);
        image = bitmap->get_image ();
    } else if (Resources::BitmapFamily *bitmapfamily = dynamic_cast<Resources::BitmapFamily*> (res)) {
        image_pair_t image_pair = color ?
            get_color_image (bitmapfamily) : get_grey_image (bitmapfamily);
        image = image_pair.first;
        bitmap_type = image_pair.second;
    }
    
    if (!image.isValid ())
        return 0;
    
    return draw_bitmap (group, image, bitmap_type, background_color,
                        x, y, clip_width, clip_height, anchor);
}
