/* This is for emacs: -*-Mode: C++;-*- */
/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#if !defined(__INC_PRINT_H)
#define __INC_PRINT_H

#include <stdio.h>

#include <string>

#include "gql++/result-set.h"

namespace GQLShell
{

class TablePrinter
{
  public:
    enum PrintFormat
    {
      PRINT_NOTHING = 0,
      PRINT_UNALIGNED,
      PRINT_ALIGNED,
      PRINT_HTML,
      PRINT_LATEX
    };
    
    /** Constructor.
     * \param fmt Output format.
     * \param expanded Expanded/vertical output (if supported
     *   by output format) 
     * \param pager If \c true, use a pager (if to stdout and stdout 
     *   is a tty).
     * \param tuples_only If \c true, don't output headers row counts, etc. 
     * 
     */
    TablePrinter(PrintFormat fmt = PRINT_ALIGNED, 
                 bool expanded = false, 
                 bool tuples_only = false,
                 bool pager = true,
                 unsigned short border = 1,
                 const std::string& field_sep = std::string("|"), 
                 const std::string& record_sep = std::string("\n"),
                 const std::string& table_attr = std::string());

    bool expanded() const { return expanded_; }
  
    /*
     * Use this to print just any table in the supported formats.
     * - title is just any string (NULL is fine)
     * - headers is the column headings (NULL ptr terminated). 
     *   It must be given and complete since the column count is generated 
     *   from this.
     * - cells are the data cells to be printed. Now you know why the correct
     *   column count is important
     * - footers are lines to be printed below the table
     * - align is an 'l' or an 'r' for every column, if the output 
     *   format needs it. (You must specify this long enough. Otherwise 
     *   anything could happen.)
     */
    void print(const char *title, const char *const *headers,
               const char *const *cells, const char *const *footers,
               const char *align,
               FILE *fout);
  private:
    PrintFormat format_;
    unsigned short border_;
    std::string field_sep_;
    std::string record_sep_;
    std::string table_attr_;
    unsigned expanded_ : 1;
    unsigned tuples_only_ : 1;
    unsigned pager_;
};

class ResultSetPrinter
{
  public:
    ResultSetPrinter(TablePrinter& tbl_printer, 
                     const std::string& title = std::string(),
                     const std::string& null_print = std::string(), 
                     bool quote = false, 
                     const char *const *footers = NULL,
                     bool default_footer = true);
    
    const TablePrinter& table_printer() const { return *tbl_printer_; }
    TablePrinter& table_printer() { return *tbl_printer_; }
    
    void print(GQL::ResultSet *rs, FILE *fout);
  private:
    TablePrinter *tbl_printer_;
    std::string title_;
    std::string null_print_;
    const char *const *footers_;
    unsigned default_footer_ : 1;
    unsigned quote_ : 1;
};

}

#endif
