#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# GFEntry.py
#
# DESCRIPTION:
"""
The primary data entry widget in forms
"""
# NOTES:
#

from gnue.forms import GFDisplayHandler
from GFTabStop import GFTabStop
import string

############################################################
# GFEntry
#
# Matches the Entry form tag in the gfd
#
# It send events to its parent GFBlock
#
class GFEntry(GFTabStop):
  def __init__(self, parent=None, value=None):
    GFTabStop.__init__(self, parent, 'GFEntry')

    # Default attributes (these may be replaced by parser)
    self.Char__height = int(gConfigForms('widgetHeight'))
    self.Char__width = int(gConfigForms('widgetWidth'))
    self.style = "default"
    self.label = ""
    self._queryable = True

    self._inits = [self.initialize]

    self._rows = 1
    self._gap = 0

    #
    # Trigger exposure
    #
    self._validTriggers ={ 'PRE-FOCUSOUT':   'Pre-FocusOut',
                           'POST-FOCUSOUT':  'Post-FocusOut',
                           'PRE-FOCUSIN':    'Pre-FocusIn',
                           'POST-FOCUSIN':   'Post-FocusIn',
                           'ON-NEXT-ENTRY':  'On-Next-Entry',
                           'ON-PREVIOUS-ENTRY':  'On-Previous-Entry',
                         }
    self._triggerFunctions={'set':{'function':self.triggerSetValue},
                            'get':{'function':self.getValue}
                             }
    self._triggerSet = self.triggerSetValue
    self._triggerGet = self.getValue

    self._triggerProperties = {
          'rows':     { 'get': self._getRows},
          'value':    { 'set':self.triggerSetValue,
                        'get':self.getValue },
          'hidden':   { 'set':self.triggerSetHidden,
                        'get':self.triggerGetHidden },
          'navigable':{ 'set':self.triggerSetNavigable,
                        'get':self.triggerGetNavigable },
      }

  def _buildObject(self):
    return GFTabStop._buildObject(self)

  def initialize(self):

    #
    # Find parents/relatives
    #

    # ...Block
    try:
      self._block = block = self.findParentOfType('GFForm')._logic._blockMap[self.block]
      block._entryList.append(self)
    except KeyError:
      raise gException, u_("Entry references non-existent block '%s'") \
                        % self.block

    # ...Field
    try:
      self._field = field = block._fieldMap[self.field]
      field._entryList.append(self)
    except KeyError:
      raise gException, u_("Entry references non-existent field '%s'") \
                        % self.field

    # ...Page
    self._page = self.findParentOfType('GFPage')
    self._page._entryList.append(self)

    self._form = self._block._form


    # Navigable? or no?
    self._navigable = self.navigable
    if self.style == 'label' or self.hidden:
      self._navigable = False
      self._queryable = False

    self._formatmask = ""
    self._inputmask = hasattr(self,'inputmask') and self.inputmask or ""
    self._displaymask = hasattr(self,'displaymask') and self.displaymask or ""

    # TODO: Conversion: I'm torn... does this go here or in GFField??
    # Create an appropriate display handler
    if self.style == 'checkbox':
      self._displayHandler = GFDisplayHandler.CheckboxDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler)
    elif self.style == 'dropdown':
      self._displayHandler = GFDisplayHandler.DropdownDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler)
    elif self.style == 'listbox':
      self._displayHandler = GFDisplayHandler.ListboxDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler)
    elif self.style == 'password':
      self._displayHandler = GFDisplayHandler.PasswordDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler,
               self._displaymask, self._inputmask)
    elif self._field.typecast == 'number':
      self._displayHandler = GFDisplayHandler.NumberDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler,
               self._displaymask, self._inputmask)
    elif self._field.typecast == 'date':
      self._displayHandler = GFDisplayHandler.DateDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler,
               self._displaymask, self._inputmask)
    else:
      self._displayHandler = GFDisplayHandler.TextDisplayHandler(self,
               self._form._instance.eventController, self.subEventHandler,
               self._displaymask, self._inputmask)

    # Row settings
    if hasattr(self,'rows'):
      self._rows = self.rows
    else:
      self._rows = self._field._rows

    if hasattr(self,'rowSpacer'):
      self._gap = self.rowSpacer
    else:
      self._gap = self._field._gap


  def _getRows(self):
    return self._rows

  def isNavigable(self, mode):
    # TODO: Check for read-only, etc
    return self._navigable and self._block.navigable and not self.hidden

  def isEditable(self, mode):
    return self._field.isEditable(mode)

  ####################################################################
  #
  # Trigger functions
  #
  def getValue(self, *args, **parms):
    return self._field.getValue(*args, **parms)

  def triggerSetValue(self, *args, **parms):
    return self._field.triggerSetValue(*args, **parms)

  def triggerSetHidden(self, value):
    self.hidden = not not value # Force boolean

  def triggerGetHidden(self):
    return self.hidden

  def triggerSetNavigable(self, value):
    self.navigable = not not value # Force boolean

  def triggerGetNavigable(self):
    return self.navigable

