#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# DESCRIPTION:
# Driver to provide access to data via SQLite's Python Driver.
# Requires PySQLite (http://pysqlite.sf.net/)
#
# NOTES:
#
#   Supported attributes (via connections.conf or <database> tag)
#
#     dbname = This is the SQLite database to use (required)
#
# $Id: Connection.py 7007 2005-02-11 16:18:59Z reinhard $

__all__ = ['Connection']

####                                   ####
#### IF YOU MODIFY ANY CONNECTION      ####
#### ATTRIBUTES, PLEASE UPDATE info.py ####
####                                   ####

from gnue.common.apps import errors
from gnue.common.datasources import Exceptions, GConnections
from gnue.common.datasources.drivers import DBSIG2

from DataObject import *

try:
  import sqlite as SIG2api
  
  # This is a workaround for a bug in PySQLite. All the following mebers are
  # not imported from main.py in __init__.py 
  if not hasattr (SIG2api, 'Timestamp') and SIG2api.main.have_datetime:
    SIG2api.Date               = SIG2api.main.Date
    SIG2api.Time               = SIG2api.main.Time
    SIG2api.Timestamp          = SIG2api.main.Timestamp
    SIG2api.DateFromTicks      = SIG2api.main.DateFromTicks
    SIG2api.TimeFromTicks      = SIG2api.main.TimeFromTicks
    SIG2api.TimestampFromTicks = SIG2api.main.TimestampFromTicks

    SIG2api.DateTimeType       = SIG2api.main.DateTimeType
    SIG2api.DateTimeDeltaType  = SIG2api.main.DateTimeDeltaType

except ImportError:
  raise GConnections.DependencyError, ('SQLitedbapi', None)

from gnue.common.datasources.drivers.sqlite.Schema.Discovery.Introspection \
    import Introspection
from gnue.common.datasources.drivers.sqlite.Schema.Creation.Creation \
    import Creation


# =============================================================================
# GConnection object for SQLite driver
# =============================================================================

class Connection (DBSIG2.Connection):

  _driver         = SIG2api
  _DatabaseError  = SIG2api.DatabaseError
  defaultBehavior = Introspection
  defaultCreator  = Creation
  _boolean_true   = 1
  _boolean_false  = 0
  _numbers_as_string = False

  supportedDataObjects = {
    'object': DataObject_Object,
    'sql':    DataObject_SQL
  }


  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, name, parameters):
    """
    This constructor extends the inherited object by the 'noTransactions' flag
    """
    DBSIG2.Connection.__init__ (self, connections, name, parameters)
    self.noTransactions = False



  # ---------------------------------------------------------------------------
  # Connect to a given database according to the connectData dictionary
  # ---------------------------------------------------------------------------

  def connect (self, connectData):
    """
    This function creates and returns a new database connection using
    parameters from the given connectData dictionary.

    @param connectData: dictionary with parameters used for the connection
    """

    gDebug (9, "SQLite database driver initializing <%s>")

    if not hasattr (self, '_DatabaseError'):
      try:
        self._DatabaseError = self._driver.Error
      except:
        self._DatabaseError = self._driver.DatabaseError

    try:
      self.noTransactions = connectData.get ('appserver', False)
      self.native = SIG2api.connect (db         = connectData['dbname'],
                                     encoding   = self._encoding,
                                     autocommit = self.noTransactions)

    except self._DatabaseError, value:
      gDebug (1, "Exception %s" % value)
      raise Exceptions.LoginError, \
           u_("The SQLite driver returned the following error:\n\t%s") \
           % errors.getException () [2]


  # ---------------------------------------------------------------------------
  # Commit a pending transactiona pending transaction
  # ---------------------------------------------------------------------------

  def commit (self):
    """
    This function performs a commit depending on the current transaction-flag.
    """
    if not self.noTransactions:
      DBSIG2.Connection.commit (self)



  # ---------------------------------------------------------------------------
  # Return a sequence of required login fields
  # ---------------------------------------------------------------------------

  def getLoginFields (self):
    """
    This function returns an empty sequence since SQLite doesn't use any user
    authentication.
    """
    return []
