# GNU Enterprise Common - SQLite DB Driver - Schema Introspection
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Introspection.py 6851 2005-01-03 20:59:28Z jcater $

__all__ = ['Introspection']

import string
import re

from gnue.common.datasources import GIntrospection

# =============================================================================
# This class implements schema introspection for SQLite backends
# =============================================================================

class Introspection (GIntrospection.Introspection):

  # list of the types of Schema objects this driver provides
  types      = [ ('view', _('Views'), True), ('table', _('Tables'), True) ]

  _REPCOMMAS = re.compile ('\(\s*(\d+)\s*,\s*(\d+)\s*\)')
  _ALIGN     = re.compile ('\s*\(\s*(.*?)\s*\)')
  _LEN_SCALE = re.compile ('^\s*(\w+)\s*\((\d+)[;]{0,1}(\d*)\)\s*')
  _TEXTTYPE  = re.compile ('.*(BLOB|CHAR|CLOB|TEXT){1}.*')
  _BLANKS    = re.compile ('\s+')
  _NOTNULL   = re.compile ('(.*)(NOT NULL)(.*)', re.IGNORECASE)
  _CONSTRAINTS = re.compile ('.*?((UNIQUE|CHECK|PRIMARY KEY)\s*\(.*?\)).*', \
                             re.IGNORECASE)
  _PRIMARYKEY  = re.compile ('.*?PRIMARY KEY\s*\((.*?)\).*', re.IGNORECASE)
  _INDEX = re.compile ('CREATE\s*(\w+){0,1}\s*INDEX\s*(\w+)\s*ON\s*\w+\s*' \
                       '\((.*?)\).*', re.IGNORECASE)

  # ---------------------------------------------------------------------------
  # Get a schema object matching name and type
  # ---------------------------------------------------------------------------

  def find (self, name = None, type = None):
    """
    """

    result    = []
    condition = ""

    if name is None:
      if type is not None:
        condition = "WHERE type = '%s'" % type.lower ()
    else:
      condition = "WHERE name = '%s'" % name

    cmd = u"SELECT type, name, sql FROM sqlite_master %s" % condition
    cursor = self._connection.makecursor (cmd)

    try:
      for rs in cursor.fetchall ():
        if not rs [0] in [tp [0] for tp in self.types]:
          continue

        attrs = {'id'  : rs [1],
                 'name': rs [1],
                 'type': rs [0],
                 'sql' : rs [2]}
        
        attrs ['primarykey'] = self.__getPrimaryKey (rs [2])
        attrs ['indices']    = self.__getIndices (rs [1])

        result.append ( \
          GIntrospection.Schema (attrs, getChildSchema = self._getChildSchema))

    finally:
      cursor.close ()

    return len (result) and result or None


  # ---------------------------------------------------------------------------
  # Get all fields from a table / view
  # ---------------------------------------------------------------------------

  def _getChildSchema (self, parent):
    result = []

    cmd = "SELECT sql FROM sqlite_master WHERE type = '%s' AND name = '%s'" \
        % (parent.type, parent.name)

    cursor = self._connection.makecursor (cmd)


    try:
      for rs in cursor.fetchall ():
        code = string.join (rs [0].splitlines (), ' ')
        code = code [string.find (code, '(') + 1:string.rfind (code, ')')]

        code = self._BLANKS.sub (' ', code)
        code = self._REPCOMMAS.sub (r'(\1;\2)', code)
        code = self._ALIGN.sub (r' (\1)', code)

        # we currently skip all constraints (primary key, unique, check)
        cma = self._CONSTRAINTS.match (code)
        while cma is not None:
          constraint = cma.groups () [0]
          code = string.replace (code, constraint, '')
          cma = self._CONSTRAINTS.match (code)

        for item in [i.strip () for i in code.split (',')]:
          if not len (item):
            continue

          parts = item.split ()
          attrs = {'id'  : "%s.%s" % (parent.name, parts [0]),
                   'name': parts [0],
                   'type': 'field'}

          datatype = string.join (parts [1:], ' ')
          lsmatch = self._LEN_SCALE.match (datatype)
          if lsmatch is not None:
            (typename, length, scale) = lsmatch.groups ()
          else:
            typename = parts [1]
            length   = 0
            scale    = 0

          nativetype = typename
          add = []
          if length: add.append (length)
          if scale : add.append (scale)
          if len (add):
            nativetype += "(%s)" % string.join (add, ",")

          attrs ['length']     = length
          attrs ['precision']  = scale or 0
          attrs ['nativetype'] = nativetype
          attrs ['required']   = self._NOTNULL.match (item) is not None

          if self._TEXTTYPE.match (typename.upper ()):
            attrs ['datatype'] = 'text'
          elif typename.lower () in ['date', 'datetime', 'time']:
            attrs ['datatype'] = 'date'
          else:
            attrs ['datatype'] = 'number'

          result.append (GIntrospection.Schema (attrs))

    finally:
      cursor.close ()

    return result


  # ---------------------------------------------------------------------------
  # Try to extract a primary key definition from SQL code
  # ---------------------------------------------------------------------------

  def __getPrimaryKey (self, sql):
    """
    This function extracts a primary key from the given SQL code.

    @param sql: SQL code to extract primary key from
    @return: sequence of fields building the primary key or None
    """

    result = None
    pk     = self._PRIMARYKEY.match (sql)

    if pk is not None:
      result = [f.strip () for f in pk.groups () [0].split (',')]
    
    return result


  # ---------------------------------------------------------------------------
  # Try to extract all index definitions for a given table
  # ---------------------------------------------------------------------------

  def __getIndices (self, table):
    """
    This function fetches all indices for a given table and returns them as a
    dictionary or None if no indices are available. NOTE: the primary key is
    *NOT* listed as index.

    @param table: name of the table to fetch indices from
    @return: dictionary with index information or None
    """

    result = {}

    cmd = u"SELECT name, sql FROM sqlite_master " \
           "WHERE type = 'index' AND tbl_name = '%s' AND sql IS NOT NULL" \
          % table

    cursor = self._connection.makecursor (cmd)

    try:
      for rs in cursor.fetchall ():
        ixm = self._INDEX.match (rs [1])
        if ixm is not None:
          (unique, name, fields) = ixm.groups ()
          result [name] = { \
              'unique' : unique is not None and unique.lower () == 'unique',
              'primary': False,
              'fields' : [f.strip () for f in fields.split (',')]}

    finally:
      cursor.close ()

    return len (result.keys ()) and result or None
