# GNU Enterprise Common - Database Drivers - Base Record Set
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: RecordSet.py 7113 2005-03-08 15:19:22Z reinhard $

__all__ = ['RecordSet']

import string
from gnue.common.datasources import Exceptions

# =============================================================================
# This class implements the basic record set
# =============================================================================

class RecordSet:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, parent, initialData = {}, dbIdentifier = None,
                defaultData = {}):
    self._detailObjects = []
    self._dbIdentifier  = dbIdentifier
    self._deleteFlag    = False
    self._updateFlag    = False
    self._parent        = parent
    self._fieldOrder    = {}
    self._modifiedFlags = {}      # If field name is present as a key,
                                  # then field has been modified

    self._cachedDetailResultSets = {}

    self._initialData = initialData

    if self._initialData and len (self._initialData):
      self._insertFlag = False
      self._emptyFlag  = False
      self._fields     = {}
      self._fields.update (initialData)
    else:
      self._insertFlag = True
      self._emptyFlag  = True
      self._fields     = {}
      self._fields.update (defaultData)

    gDebug (8, "Initial Data: %s" % self._fields)


  # ---------------------------------------------------------------------------
  # Dictionary emulation
  # ---------------------------------------------------------------------------

  def __setitem__(self, attr, val):
    self.setField (attr, val)

  def __getitem__ (self, attr):
    return self.getField (attr)
  
  def keys (self):
    return self._fields.keys ()
  
  def values (self):
    return self._fields.values ()

  def items (self):
    return self._fields.items ()
    
  # ---------------------------------------------------------------------------
  # Status of this record
  # ---------------------------------------------------------------------------

  def isEmpty (self):
    """
    Returns True if the record is empty, which means that it has been newly
    inserted, but neither has any field been changed nor has a detail for this
    record been inserted with a status other than empty.
    """
    if self._emptyFlag:
      result = True
      for child in self._cachedDetailResultSets.values ():
        if child.isPending ():
          result = False
          break
    else:
      result = False
    return result

  def isInserted (self):
    """
    Returns True if the record has been newly inserted and has either changes
    or a detail has been inserted. Records with this status will be inserted
    into the database on post.
    """
    return self._insertFlag and not self._deleteFlag and not self.isEmpty ()

  def isModified (self):
    """
    Returns True if the record is an existing record with local changes.
    Records with this status will be updated in the database on post.
    """
    return self._updateFlag and not self._insertFlag and not self._deleteFlag

  def isDeleted (self):
    """
    Returns True if the record is an existing record that has been deleted.
    Records with this status will be deleted in the database on post.
    """
    return self._deleteFlag and not self._insertFlag

  def isPending (self):
    """
    Returns True if the record has any local changes that make it necessary to
    post it to the database. Equal to isInserted or isModified or isDeleted.
    """
    return self.isInserted () or self.isModified () or self.isDeleted ()


  # ---------------------------------------------------------------------------
  # Field access
  # ---------------------------------------------------------------------------

  # Returns current value of "field"
  def getField(self, field):
    try:
      return self._fields[field]
    except KeyError:
      try:

        # TODO: When we're confident that
        # TODO: all field names are lowercase,
        # TODO: then this can be removed.

        return self._fields[string.lower(field)]
      except KeyError:
        # If a field value has yet to be set
        # (either from a query or via a setField),
        # then _fields will not contain a key
        # for the requested field even though
        # the field name may still be valid.
        return None


  # Sets current value of "field"
  # If trackMod is set to 0 then the modification flag isn't raised
  def setField(self, field, value, trackMod = 1):
    do = self._parent._dataObject
    gDebug (8, "setField: %s to %s" % (field, value))
    # If this field is bound to a datasource and the datasource is read only,
    # generate an error.
    if self._parent.isFieldBound(field) and self._parent.isReadOnly():
      # Provide better feedback??
      tmsg = u_("Attempted to modify read only field '%s'") % field
      raise Exceptions.ReadOnlyError, tmsg
    else:
      fn = string.lower (field)
      self._fields [fn] = value
      if trackMod:
        if self._parent.isFieldBound(field):
          self._emptyFlag = False
          self._updateFlag = True
          self._modifiedFlags [fn] = True
          # self._modifiedFlags[field] = True

          try:
            do._dataSource._onModification(self)
          except AttributeError:
            pass
    return value

  # Batch mode of above setField method
  # If trackMod is set to 0 then the modification flag isn't raised
  def setFields(self, updateDict, trackMod = 1):
    # If this field is bound to a datasource and the datasource is read only,
    # generate an error.
    gDebug (8, "calling setField from updateDict: %s" % updateDict)
    for field in updateDict.keys():
      self.setField(field, updateDict[field], trackMod)


  def getFieldsAsDict(self):
    """
    Returns the record set as a dictionary.

    @return: A python dictionary of field name/value pairs.
    """

    results = {}
    for field in self._fields.keys():
      results[field] = self.getField(field)
    return results

    
    
  # Returns 1=Field has been modified
  def isFieldModified(self, fieldName):
    #TODO: the string.lower() line should never be called but is left here
    #TODO: until the code is clean
    return self._modifiedFlags.has_key(fieldName) or \
           self._modifiedFlags.has_key (string.lower(fieldName))

  # ---------------------------------------------------------------------------
  # Mark record as deleted
  # ---------------------------------------------------------------------------

  def delete(self):
    if self._parent.isReadOnly():
      # Provide better feedback??
      tmsg = _("Attempted to delete from a read only datasource")
      raise Exceptions.ReadOnlyError, tmsg
    else:
      self._deleteFlag = True

  # ---------------------------------------------------------------------------
  # Post changes to database
  # ---------------------------------------------------------------------------

  def post (self, recordNumber = None):
    # Should a post() to a read only datasource cause a ReadOnlyError?
    # It does no harm to attempt to post since nothing will be posted,
    # But does this allow sloppy programming?

    gDebug (8, 'Preparing to post datasource %s' \
               % self._parent._dataObject.name)

    # Save the initial status so we know if any triggers changed us
    status = (self._insertFlag, self._deleteFlag, self._updateFlag)

    # Call the hooks for commit-level hooks
    if not self.isEmpty() and hasattr(self._parent._dataObject,'_dataSource'):

      if self._insertFlag and not self._deleteFlag:
        self._parent._dataObject._dataSource._beforeCommitInsert(self)
      elif self._deleteFlag and not self._insertFlag:
        self._parent._dataObject._dataSource._beforeCommitDelete(self)
      elif self._updateFlag:
        self._parent._dataObject._dataSource._beforeCommitUpdate(self)

    #
    # If the record status changed while we were doing the triggers,
    # start from the beginning and run the triggers again.
    #
    if status != (self._insertFlag, self._deleteFlag, self._updateFlag):
      self.post(recordNumber)
      return


    if self.isPending():
      gDebug (8, 'Posting datasource %s' % self._parent._dataObject.name)
      self._postChanges (recordNumber)


    # Post all detail records
    for child in (self._cachedDetailResultSets.keys()):
      c = self._cachedDetailResultSets[child]._dataObject
      # Set the primary key for any new child records
      fk = {}
      for i in range(len(c._masterfields)):
        fk[c._detailfields[i]] = self.getField(c._masterfields[i])

      self._cachedDetailResultSets[child].post(foreign_keys=fk)


  # ---------------------------------------------------------------------------
  # Sets the ResultSet associated with this master record
  # ---------------------------------------------------------------------------

  def addDetailResultSet(self, resultSet):
    self._cachedDetailResultSets[resultSet._dataObject] = resultSet


  # ---------------------------------------------------------------------------
  # Nice string representation
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    do = self._parent._dataObject
    if hasattr (do, 'table'):
      return "<RecordSet for %s>" % do.table
    else:
      return "<NIL-Table RecordSet>"

  # ---------------------------------------------------------------------------
  # Virtual methods to be overwritten by the drivers
  # ---------------------------------------------------------------------------

  def _postChanges (self, recordNumber = None):
    """
    Post any changes (deletes, inserts, and updates) to the database.
    Descendants can either override this function, or the three functions
    _postDelete, _postInsert, and _postUpdate.

    _postChanges is guaranteed to be only called for records that have pending
    changes. Implementations of this function can check the kind of pending
    change by querying the _deleteFlag, _insertFlag, and _updateFlag instance
    variables.
    """
    if self._deleteFlag:
      self._postDelete ()

    elif self._insertFlag or self._updateFlag:
      modifiedFields = {}
      for field in self._modifiedFlags.keys ():
        modifiedFields [field] = self._fields [field]

      if self._insertFlag:
        self._postInsert (modifiedFields)
      else:
        self._postUpdate (modifiedFields)

      self._modifiedFlags = {}

    self._deleteFlag = False
    self._insertFlag = False
    self._updateFlag = False

  def _postDelete (self):
    """
    Post a deletion to the backend. Descendants should override this function
    (or the general _postChanges function).

    _postDelete is guaranteed to be only called for records that pend a
    deletion.
    """
    pass

  def _postInsert (self, fields):
    """
    Post an insert to the backend. Descendants should override this function
    (or the general _postChanges function).

    _postInsert is guaranteed to be only called for records that pend an
    insert (i.e. that were newly created).

    @param fields: a dictionary with field names as keys and field values as
        values.
    """
    pass

  def _postUpdate (self, fields):
    """
    Post an update to the backend. Descendants should override this function
    (or the general _postChanges function).

    _postUpdate is guaranteed to be only called for records that pend an
    update (i.e. for existing records that were modified).

    @param fields: a dictionary with field names as keys and field values as
        values.
    """
    pass
