/**
 ** mp3_control.c - control and UI functions of the mp3 player
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include "gnomp3.h"
#include "mpg123_play.h"
#include "mp3_control.h"
#include "rule_list.h"
#include "playlist.h"
#include "mp3list.h"
#include "xmms_play.h"

struct mp3_control mp3_control_state;

void mp3_control_play(MP3 *mp3, float pos)
{
    mp3_control_state.playing = TRUE;
    mp3_control_state.paused  = FALSE;
    mp3_control_state.update_seek = TRUE;
    mp3->playing = TRUE;
    mp3->played  = TRUE;
    mp3->no_plays++;
    playlist_update_display();

    mp3_control_state.mp3 = mp3;

    if(!gnomp3.use_xmms)
	mpg123_play_song( mp3, pos );
    else
	xmms_play_song(mp3, pos);
}

/* halt playback */
void mp3_control_stop()
{
    GtkAdjustment *adj;
    
    if(mp3_control_state.mp3)
	mp3_control_state.mp3->playing = FALSE;
    mp3_control_state.playing = FALSE;
    if(!gnomp3.use_xmms)
	mpg123_play_stop();
    else
	xmms_play_stop();

    gtk_label_set_text( GTK_LABEL(gnomp3.mp3_control_label), "Stopped");
    adj = gtk_range_get_adjustment( GTK_RANGE (gnomp3.mp3_hscale) );
    adj->value = 0.0;
    gtk_signal_emit_by_name (GTK_OBJECT (adj), "changed");

}

void mp3_control_update_display()
{
    char text[128];
    GtkAdjustment *adj;

    if( mp3_control_state.update_seek ){
	adj = gtk_range_get_adjustment( GTK_RANGE (gnomp3.mp3_hscale) );
	adj->upper = (double)100.0;
	adj->value = mp3_control_state.progress;
	gtk_signal_emit_by_name (GTK_OBJECT (adj), "changed");
    }
    g_snprintf( text, 128, "Time: %s -- %dkbps %dkHz", mp3_control_state.time,
		mp3_control_state.bitrate, mp3_control_state.freq/1000);
    gtk_label_set_text( GTK_LABEL(gnomp3.mp3_control_label), text);
    if(mp3_control_state.mp3)
	gtk_label_set_text( GTK_LABEL(gnomp3.mp3_control_song_name), 
			    mp3_control_state.mp3->display_name);
}

/* 
 * Searchs the playlist for the next unplayed song. 
 * RETURNS the song in the mp3list 
 */
GList *mp3_control_find_next_unplayed()
{
    GList *ptr;

    if(!mp3_control_state.next_playlist_song)
	goto second_loop;

    /* find the next song from here to the end of the list */
    ptr = mp3_control_state.next_playlist_song->next;
    while( ptr ){
	if( ! MP3_NODE(ptr)->played )
	    return ptr;
	ptr = ptr->next;
    }
    
  second_loop:
    /* if not found yet start looking at the begining of the list */
    ptr = playlist;
    while( ptr ){
	if( ! MP3_NODE(ptr)->played )
	    return ptr;
	ptr = ptr->next;
    }
    
    /* not found */
    return NULL;
}

/* 
 * finds the next song to be played and begins playing it. If the random button
 * is pressed then the next song is found randomly, otherwise the next song in
 * the playlist sequence is used. If the 'repeat' button is not depressed then
 * playback halts once all songs in the playlist have been played 
 */ 
void mp3_control_play_next()
{
    int i, count = 1;
    
    mp3_control_stop();

    if (GTK_TOGGLE_BUTTON (gnomp3.random_button)->active){
	count = 1 + (int)
	    ((double)(GTK_CLIST(gnomp3.play_clist)->rows-1) * rand()
	     /(RAND_MAX+1.0));
    }

    for( i = 0; i < count; i++ ){
	mp3_control_state.next_playlist_song = mp3_control_find_next_unplayed();
	if( !mp3_control_state.next_playlist_song ){
	    if( GTK_TOGGLE_BUTTON (gnomp3.repeat_button)->active){
		playlist_set_all_unplayed();
		mp3_control_state.next_playlist_song = mp3_control_find_next_unplayed();
	    }else{
		mp3_control_stop();
		playlist_update_display();
		return;
	    }
	}
    }
    mp3_control_play(mp3_control_state.next_playlist_song->data, 0);
}

void mp3_control_play_prev()
{
    if( !mp3_control_state.next_playlist_song )
	return;

    mp3_control_state.next_playlist_song = mp3_control_state.next_playlist_song->prev;
    mp3_control_play_cb();
}

/* play the song with the given position in the play_clist */
void mp3_control_play_nth(int num)
{
    MP3 *mp3;

    mp3 = gtk_clist_get_row_data(GTK_CLIST(gnomp3.play_clist), num);
    mp3_control_state.next_playlist_song = g_list_find(playlist, mp3);
    mp3_control_play(mp3_control_state.next_playlist_song->data, 0);
}

/* start play back of the surrently selected song in the play_clist */
void mp3_control_play_selected()
{
    if(!GTK_CLIST(gnomp3.play_clist)->selection)
	return;

    mp3_control_play_nth((int)GTK_CLIST(gnomp3.play_clist)->selection->data);
}

/* plays the selected songs from lists other than the play_clist */
void mp3_control_play_selected_song(GtkWidget *w, gpointer data)
{
    GtkCListRow *clist_row;
    void *row_data;

    mp3_control_stop();

    if(!GTK_CLIST(data)->selection)
	return;
        
    if( GTK_IS_CTREE(data)){
	clist_row = GTK_CLIST(data)->selection->data;
	row_data = GTK_CTREE_ROW(clist_row)->row.data;
    }else if( GTK_IS_CLIST(data) ){
	clist_row = g_list_nth( GTK_CLIST(data)->row_list, (int)GTK_CLIST(data)->selection->data )->data;
	row_data = clist_row->data;
    }
  
    if(!row_data)return;

    mp3_control_play(row_data, 0);
}
/*****************************************************************************/
/*
 * Button callbacks.
 */
void mp3_control_play_cb()
{
    MP3 *mp3;
    if( !mp3_control_state.next_playlist_song ){
	mp3_control_state.next_playlist_song = playlist;
    }
    mp3 = mp3_control_state.next_playlist_song->data;
    
    mp3_control_play(mp3, 0);
}

void mp3_control_next_cb()
{
    mp3_control_play_next();
}

void mp3_control_seek_press(GtkWidget *w, gpointer data)
{
    mp3_control_state.update_seek = FALSE;
}

void mp3_control_seek_release(GtkWidget *w, gpointer data)
{
    GtkAdjustment *adj;
    
    mp3_control_state.update_seek = TRUE;

    if( !mp3_control_state.playing)
	return;

    adj = gtk_range_get_adjustment( GTK_RANGE (gnomp3.mp3_hscale) );

    mp3_control_play( mp3_control_state.mp3, adj->value );

    /* FIXME should not resort to current_playing_song as may not be valid */
}

void mp3_control_pause_cb()
{
    if(gnomp3.use_xmms)
	xmms_play_pause();
    else
	if(mp3_control_state.paused){
	    mp3_control_state.paused = FALSE;
//	mp3play_song( MP3_NODE(current_playing_song), 100);
	    mp3_control_seek_release(NULL, NULL);
	}else{
	    mp3_control_state.paused = TRUE; 
	    mpg123_play_stop();
	}
}



