/* jukebox.c
   Functions related to the communication with jukebox hardware
   using the libnjb library.
   Copyright (C) 2001-2003 Linus Walleij

This file is part of the GNOMAD package.

GNOMAD is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

You should have received a copy of the GNU General Public License
along with GNOMAD; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. 

*/

#include <time.h>
#include "common.h"
#include "jukebox.h"
#include "xfer.h"
#include "prefs.h"
#include "filesystem.h"
#include "filenaming.h"
#include "util.h"

/* External global used by libnjb */
extern int njb_error;

/* Local variables */
static njb_t njbs[NJB_MAX_DEVICES];
static njb_t *njb = NULL;
static njbid_t *njbid = NULL;
static gboolean njb_connected = FALSE;
/* Jukebox properties variables */
static gchar *jukebox_ownerstring = NULL;
static gchar *jukebox_timestring = NULL;
static gchar *jukebox_firmware = NULL;
static u_int64_t jukebox_totalbytes = 0;
static u_int64_t jukebox_freebytes = 0;
static u_int64_t jukebox_usedbytes = 0;
static guint jukebox_songs = 0;
static guint jukebox_playlists = 0;
static guint jukebox_datafiles = 0;
/* Variables used for playing tracks */
static GList *playlist;
static GList *playlistitr;
static GList *playlistlast;
static GtkWidget *songnamelabel;
static gboolean created_play_mutex = FALSE;
static GMutex *play_thread_mutex = NULL;
static gboolean passed_first_zero = FALSE;
/* After initial scan, this hash table always contains
 * all the songs arranged by song ID */
static GHashTable *songhash = NULL;
/* This is used temporarily for storing a list of
 * playlist entries (gnomadplaylist_entry_t), which 
 * in turn stores the playlists as GSLists with track
 * IDs.  
 *  
 * playlistlist -> list-> list -> list -> ... -> NULL
 *                 |      |       |
 *                 entry  entry   entry
 *                 |
 *                 +-> name
 *                 +-> plid
 *                 +-> tracklist
 *                     |
 *                     +-> track -> track -> ... -> NULL
 */
static GSList *playlistlist = NULL;


/***********************************************************************************/
/* Public and private functions                                                    */
/***********************************************************************************/

static void refresh_id(void)
{
  if (njb != NULL) {
    if (njbid != NULL) {
      g_free(njbid);
      njbid = NULL;
    }
    njbid = NJB_Ping(njb);
  }
}

/* Returns the unique jukebox ID */
gchar *jukebox_get_idstring(void)
{
  if (njb != NULL) {
    return (gchar *) njb->idstring;
  } else {
    return NULL;
  }
}

/* Returns a string representing the firmware revision */
gchar *jukebox_get_firmware(void)
{
  if (jukebox_firmware != NULL) {
    g_free(jukebox_firmware);
    jukebox_firmware = NULL;
  }
  refresh_id();
  if (njbid != NULL) {
    /* if (njb->device_type == NJB_DEVICE_NJB1) */
    jukebox_firmware = 
      g_strdup_printf("%u.%u", njbid->fwMajor, njbid->fwMinor);
    /* else g_strdup_printf("%u.%u.%u", njbid->fwMajor, njbid->fwMinor, njbid->fwRel); */
    return jukebox_firmware;
  } else {
    return NULL;
  }
}

/* Returns the product name */
gchar *jukebox_get_prodname(void)
{
  refresh_id();
  if (njbid != NULL) {
    return (gchar *) njbid->productName;
  } else {
    return NULL;
  }
}

/* Returns data about if the power cord is connected */
gboolean jukebox_get_power(void)
{
  refresh_id();
  if (njbid != NULL) {
    return (njbid->power > 0) ? TRUE : FALSE;
  } else {
    return FALSE;
  }
}

gchar *jukebox_get_time(void)
{
  njb_time_t *time;
  
  if (njb != NULL) {
    time = NJB_Get_Time(njb);
    if (time != NULL) {
      if (jukebox_timestring != NULL) {
	g_free(jukebox_timestring);
	jukebox_timestring = NULL;
      }
      jukebox_timestring = g_strdup_printf("%u-%.2u-%.2u %.2u:%.2u:%.2u", 
					   time->year, time->month, time->day,
					   time->hours, time->minutes, 
					   time->seconds);
      free(time);
    }
  } else {
    if (jukebox_timestring != NULL) {
      g_free(jukebox_timestring);
    }
    jukebox_timestring = NULL;
  }
  return jukebox_timestring;
}


/* Functions for manipulating the song hash */
static void destroy_hash (gpointer key,
			  gpointer value,
			  gpointer user_data)
{
  metadata_t *meta = (metadata_t *) value;
  destroy_metadata_t(meta);
}


static void destroy_hashes()
{
  /* Destroy the song hash */
  if (songhash != NULL) {
    g_hash_table_foreach(songhash,
			 (GHFunc) destroy_hash,
			 NULL);
    g_hash_table_destroy(songhash);
    songhash = NULL;
  }
}


static void add_row_from_songhash (gpointer key,
				 gpointer value,
				 gpointer user_data)
{
  metadata_t *meta = (metadata_t *) value;
  add_metadata_to_model(meta, JB_LIST);
  //g_print("Added %s\n", meta->title);
}


static void jblist_from_songhash(gboolean threaded)
{
  if (threaded)
    gdk_threads_enter();

  /* Little trick I learned from Sven Neumann, recreate the store each time,
   * because otherwise you cannot remove the sorting attribute from the store,
   * and that in turn will result in O(n^2) addition times for this store.
   * So we add rows in O(n) time to a fresh store and then connect it to the
   * view and sort it afterwards.
   */
  recreate_list_store(JB_LIST);
  /* This does not need to be threaded! */
  g_hash_table_foreach(songhash,
		       (GHFunc) add_row_from_songhash,
		       NULL);
  /* Then sort everything */
  view_and_sort_list_store(JB_LIST);
  if (threaded)
    gdk_threads_leave();
}

/* This callback handle all "cancel" buttons during
 * jukebox operations */
void cancel_jukebox_operation_click ( GtkButton *button,
				      gpointer data )
{
  cancel_jukebox_operation = TRUE;
}


/*
 * Discover devices on the USB bus and return an array of 
 * descriptions that may be used to select the jukebox with
 * index from 0 upwards.
 */
gchar **jukebox_discover(void) {
  int jukeboxcount;
  gchar **retarray;

  /*
   * Intercept USB communications - invoke gnomad2
   * with gnomad2 -D7 to activate a lot of USB debugging
   * information 
   */
  if (gnomad_debug != 0) {
    NJB_Set_Debug(gnomad_debug);
  }

  /* Select UTF8 because we are running Gnome 2 */
  /* Also test with NJB_UC_8859 */
  NJB_Set_Unicode(NJB_UC_UTF8);

  if ( NJB_Discover(njbs, 0, &jukeboxcount) == -1 ) {
    create_error_dialog(_("Could not try to locate jukeboxes\nUSB error?"));
    njb_error_dump(stderr);
    return NULL;
  }
  
  if ( jukeboxcount == 0 ) {
    create_error_dialog(_("No jukeboxes found on USB bus"));
    return NULL;
  }

  // Add a nice selection dialog here njb->device_type tells the type,
  // and owner string is useful too.
  if ( jukeboxcount >= 1) {
    int i;

    // Space for jukebox strings.
    retarray = g_malloc((jukeboxcount+1) * sizeof(gchar *));
    for (i = 0; i < jukeboxcount; i++) {
      njb = &njbs[i];
      switch (njb->device_type)
	{
	/* Put in hack to display D.A.P. Jukebox in europe? */
	case NJB_DEVICE_NJB1:
	  retarray[i] = g_strdup_printf("Nomad Jukebox 1 (D.A.P.)");
	  break;
	case NJB_DEVICE_NJB2:
	  retarray[i] = g_strdup_printf("Nomad Jukebox 2");
	  break;
	case NJB_DEVICE_NJB3:
	  retarray[i] = g_strdup_printf("Nomad Jukebox 3");
	  break;
	case NJB_DEVICE_NJBZEN:
	  retarray[i] = g_strdup_printf("Nomad Jukebox Zen");
	  break;
	case NJB_DEVICE_NJBZEN2:
	  retarray[i] = g_strdup_printf("Nomad Jukebox Zen USB 2.0");
	  break;
	case NJB_DEVICE_NJBZENNX:
	  retarray[i] = g_strdup_printf("Nomad Jukebox Zen NX");
	  break;
	case NJB_DEVICE_NJBZENXTRA:
	  retarray[i] = g_strdup_printf("Nomad Jukebox Zen Xtra");
	  break;
	case NJB_DEVICE_DELLDJ:
	  retarray[i] = g_strdup_printf("Dell Digital Jukebox");
	  break;
	case NJB_DEVICE_NJBZENTOUCH:
	  retarray[i] = g_strdup_printf("Nomad Jukebox Zen Touch");
	  break;
	default:
	  retarray[i] = g_strdup_printf(_("Unknown jukebox type"));
	}
    }
    retarray[jukeboxcount] = NULL;
    njb = NULL;
    return retarray;
  }

  return NULL;
}

// This selects a certain jukebox for use and sets the
// global njb variable to point to thus jukebox.
gboolean jukebox_select(gint i) {
  njb = &njbs[i];
  
  if (NJB_Open(njb) == -1) {
    create_error_dialog(_("Could not open jukebox"));
    njb_error_dump(stderr);
    return FALSE;
  }

  /*
   * Causing problems. Removing for now. 2004-04-12
  if ((njbid = NJB_Ping(njb)) == NULL) {
    create_error_dialog(_("Could not ping jukebox"));
    njb_error_dump(stderr);
    NJB_Close(njb);
    return FALSE;
  }
  */

  if (NJB_Capture(njb) == -1) {
    create_error_dialog(_("Could not capture jukebox"));
    njb_error_dump(stderr);
    NJB_Close(njb);
    return FALSE;
  }
  njb_connected = TRUE;
  return TRUE;
}

// Check if a jukebox has been selected.
gboolean jukebox_selected(void)
{
  if (!njb) {
    // g_print("njb was null.\n");
    return FALSE;
  }
  if (!njb_connected) {
    // g_print("njb was not connected.\n");
    return FALSE;
  }
  return TRUE;
}

/* Called at end of session to release jukebox */
void jukebox_release(void)
{
  if (njb_connected) {
    njbid = NJB_Ping(njb);
    NJB_Release(njb);
    NJB_Close(njb);
    njb_connected = FALSE;
  }
}

/* Builds the list in memory that the dialog
 * drop down is then built from */
static void build_playlist_list()
{
  playlist_t *playlist;

  if (jukebox_playlist != NULL) {
    GList *tmplist = jukebox_playlist;
    
    while(tmplist) {
      if (tmplist->data != NULL)
	g_free(tmplist->data);
      tmplist = tmplist->next;
      tmplist = tmplist->next;
    }
    g_list_free(jukebox_playlist);
    jukebox_playlist = NULL;
  }
  NJB_Reset_Get_Playlist(njb);
  while (playlist = NJB_Get_Playlist(njb)) {
    jukebox_playlists++;
    jukebox_playlist = g_list_append(jukebox_playlist, g_strdup(playlist->name));
    jukebox_playlist = g_list_append(jukebox_playlist, GUINT_TO_POINTER(playlist->plid));
    // Dangerous?
    playlist_destroy(playlist);
  }
}


static void create_tree_structure()
{
  playlist_t *playlist;

  /* This is for the actual playlist tree scan */
  NJB_Reset_Get_Playlist(njb);
  while (playlist = NJB_Get_Playlist(njb)) {
    playlist_track_t *track;
    gnomadplaylist_entry_t *entry;
    gint i,j;
    gchar *tmp;

    entry = (gnomadplaylist_entry_t *) g_malloc(sizeof(gnomadplaylist_entry_t));
    entry->name = NULL;
    entry->plid = 0;
    entry->tracklist = NULL;
    entry->name = g_strdup(playlist->name);
    tmp = g_strdup_printf("%lu", playlist->plid);
    entry->plid = tmp;
    playlist_reset_gettrack(playlist);
    while (track = playlist_gettrack(playlist)) {
      entry->tracklist = g_slist_append(entry->tracklist,
					GUINT_TO_POINTER(track->trackid));
    }
    playlist_destroy(playlist);
    playlistlist = g_slist_append(playlistlist, (gpointer) entry);
    if (cancel_jukebox_operation)
      break;
  }
  if ( njb_error != EO_EOM ) njb_error_dump(stderr);
}


static void build_tree_widget(GtkTreeStore *pltreestore, gboolean threaded)
{
  GSList *tmplist;

  if (threaded)
    gdk_threads_enter();
  clear_list_store(PL_TREE);
  tmplist = playlistlist;
  // Draw the tree...
  while (tmplist != NULL) {
    gnomadplaylist_entry_t *entry;
    GSList *tmplist2;
    gint i;
    GtkTreeIter treeiter;
    
    entry = (gnomadplaylist_entry_t *) tmplist->data;
    // g_print("Playlist: %s, ID %lu\n", entry->name, entry->plid);
    gtk_tree_store_append (GTK_TREE_STORE(pltreestore), &treeiter, NULL);
    gtk_tree_store_set(GTK_TREE_STORE(pltreestore), &treeiter,
		       PLIST_PLAYLISTNAME_COLUMN, entry->name,
		       PLIST_PLID_COLUMN, entry->plid,
		       PLIST_SONGID_COLUMN, "0",
		       -1);

    tmplist2 = entry->tracklist;
    while (tmplist2 != NULL) {
      metadata_t *meta;
      GtkTreeIter treeiter2;

      // g_print("Track: %lu\n", GPOINTER_TO_UINT(tmplist2->data));
      meta = (metadata_t *) g_hash_table_lookup(songhash, tmplist2->data);
      /*
       * This might have caused lots of trouble to people with bad tracks 
       * in their playlists... Fixed 2004-03-09 Linus Walleij
       */
      if (meta != NULL) {
	gtk_tree_store_append (GTK_TREE_STORE(pltreestore), &treeiter2, &treeiter);
	gtk_tree_store_set(GTK_TREE_STORE(pltreestore), &treeiter2,
			   PLIST_ARTIST_COLUMN, meta->artist,
			   PLIST_TITLE_COLUMN, meta->title,
			   PLIST_LENGTH_COLUMN, meta->length,
			   PLIST_PLID_COLUMN, entry->plid,
			   PLIST_SONGID_COLUMN, meta->path,
			   -1);
      } else {
	g_print("Bad track detected in playlist, track ID: %lu\n", GPOINTER_TO_UINT(tmplist2->data));
      }
      tmplist2 = tmplist2->next;
    }
    tmplist = tmplist->next;
  }
  if (threaded)
    gdk_threads_leave();  
  
}

static void destroy_tree_structure()
{
  /* This destroys the playlists in memory again */
  GSList *tmplist;

  tmplist = playlistlist;
  while (tmplist != NULL) {
    gnomadplaylist_entry_t *entry;
    
    entry = (gnomadplaylist_entry_t *) tmplist->data;
    g_free(entry->name);
    g_free(entry->plid);
    g_slist_free(entry->tracklist);
    tmplist = tmplist->next;
  }
  g_slist_free(playlistlist);
}


static void build_playlist_tree(GtkTreeStore *pltreestore,
				gboolean threaded)
{
  /* First build the structure for dialog boxes */
  build_playlist_list();
  /* Set this too NULL to be certain */
  playlistlist = NULL;
  /* Next get the playlists and their respective tracks
   * from the jukebox */
  create_tree_structure();
  /* Now we are finished with the jukebox communication and we have
   * all playlists in memory, so let's enter the GTK thread and
   * redraw the tree. */
  build_tree_widget(pltreestore, threaded);
  /* Destroy the memory allocated by temporary playlist
   * tree structure */
  destroy_tree_structure();
}

static void add_to_playlist(guint playlistid, GList *metalist, 
			    GtkTreeStore *pltreestore, gboolean threaded)
{
  playlist_track_t *pl_track;
  playlist_t *playlist;
  GList *tmplist = metalist;
  gboolean found = FALSE;

  if (!metalist)
    return;
  NJB_Reset_Get_Playlist(njb);
  while (playlist = NJB_Get_Playlist(njb)) {
    /* g_print("Is it %lu?\n", playlist->plid); */
    if (playlist->plid == playlistid) {
      /* g_print("YES!\n"); */
      found = TRUE;
      break;
    }
    /* g_print("NO.\n"); */
    playlist_destroy(playlist);
  }
  if (!found) {
    if (threaded)
      gdk_threads_enter();
    create_error_dialog(_("Could not find playlist"));
    if (threaded)
      gdk_threads_leave();
  } else if (tmplist) {
    while(tmplist) {
      metadata_t *meta;
      u_int32_t id;

      meta = (metadata_t *) tmplist->data;
      id = string_to_guint(meta->path);
      /* g_print("Adding track %lu to playlist %lu\n", id, playlistid); */
      pl_track = playlist_track_new(id);
      playlist_addtrack(playlist, pl_track, NJB_PL_END);
      tmplist = g_list_next(tmplist);
    }
    if (NJB_Update_Playlist(njb, playlist) == -1)
      njb_error_dump(stderr);
    playlist_destroy(playlist);
  }
  /* Rebuild the tree afterwards */
  build_playlist_tree(pltreestore, threaded);
}

/* Update the usage information */
static void flush_usage()
{
  u_int64_t totalbytes;
  u_int64_t freebytes;
  
  if (NJB_Get_Disk_Usage (njb, &totalbytes, &freebytes) == -1)
    njb_error_dump(stderr);

  jukebox_totalbytes = totalbytes;
  jukebox_freebytes = freebytes;
  jukebox_usedbytes = jukebox_totalbytes - jukebox_freebytes;
}

/***********************************************************************************/
/* Scanning thread - reads in the Jukebox track directory, the datafile directory  */
/* and all playlists, along with some general information about the jukebox.       */
/***********************************************************************************/

gpointer scan_thread(gpointer thread_args)
{
  songid_t *songtag;
  datafile_t *datatag;
  gchar *scan_songs = _("Scanning songs...");
  gchar *scan_playlists = _("Scanning playlists...");
  gchar *scan_datafiles = _("Scanning datafiles...");
  scan_thread_arg_t *args = (scan_thread_arg_t *) thread_args;

  /* Destroy old song hashes */
  destroy_hashes();
  jukebox_songs = 0;
  jukebox_datafiles = 0;
  jukebox_playlists = 0;

  /* Retrieve general information about the jukebox */
  gdk_threads_enter();
  gtk_label_set_text(GTK_LABEL(args->label), scan_songs);
  gdk_threads_leave();
  /* If it has already been read, free the old string */
  if (jukebox_ownerstring != NULL)
    g_free(jukebox_ownerstring);
  jukebox_ownerstring = g_strdup(NJB_Get_Owner_String (njb));
  flush_usage();
  
  /* Next retrieve the track listing */
  songhash = g_hash_table_new(NULL, NULL);
  /* Activate extended metadata mode if desired */
  // FIXME: Add preset. Modify metadata editor to reflect this.
  if (get_prefs_extended_metadata()) {
    NJB_Get_Extended_Tags(njb, 1);
  } else {
    NJB_Get_Extended_Tags(njb, 0);
  }
  NJB_Reset_Get_Track_Tag(njb);
  /* FIXME/IMPROVEMENT: If jukebox_songs != 0 start displaying progress bar? */

  while (songtag = NJB_Get_Track_Tag(njb)) {
    gchar *label;
    gchar *data;
    guint numdata;
    gint i;
    metadata_t *meta;
    songid_frame_t *frame;
    gchar *scanstring;

    jukebox_songs++;
    /* Number of songs scanned during scan. */
    scanstring = g_strdup_printf(_("%u songs scanned"), jukebox_songs);
    gdk_threads_enter();
    gtk_label_set_text(GTK_LABEL(args->label), scanstring);
    gdk_threads_leave();
    g_free(scanstring);

    songid_reset_getframe(songtag);
    /* Create a structure to hold the data in the columns */
    meta = new_metadata_t();
    meta->path = g_strdup_printf("%lu", songtag->trid);

    /* Loop through the song tags */
    while(frame = songid_getframe(songtag)){
      // FIXME: add progress bar for scanning?
      label = (gchar *) g_malloc (frame->labelsz + 1);
      memcpy(label, frame->label, frame->labelsz);
      /* Terminate string */
      label[frame->labelsz] = '\0';
      
      if (frame->type == ID_DATA_BIN) {
	/* If it is not ASCII data */
	if (!strcmp(label, FR_LENGTH)) {
	  data = seconds_to_mmss(songid_frame_data32(frame));
	}
	else {
	  if (frame->datasz > 4) {
	    /* 
	     * There is no way to handle this right now, though 
	     * it could easily be added to the libnjb if needed
	     */
	    g_print("Error: Found 64 bit data in songid tag frame. Ignoring.\n");
	  } else
	    numdata = songid_frame_data32(frame);
	}
      } else if (frame->type == ID_DATA_ASCII) {
	/* Strings that are actually integers are converted */
	if (!strcmp(label, FR_YEAR) ||
	    !strcmp(label, FR_TRACK)) {
	  numdata = string_to_guint(frame->data);
	} else {
	  /* If it is ASCII data */
	  if (frame->datasz) {
	    data = g_strdup(frame->data);
	  } else
	    data = g_strdup("<Unknown>");
	}
      }

      if (!strcmp(label, FR_ARTIST))
	meta->artist = data;
      else if (!strcmp(label, FR_TITLE))
	meta->title = data;
      else if (!strcmp(label, FR_ALBUM))
	meta->album = data;
      else if (!strcmp(label, FR_GENRE))
	meta->genre = data;
      else if (!strcmp(label, FR_LENGTH))
	meta->length = data;
      else if (!strcmp(label, FR_SIZE))
	meta->size = numdata;
      else if (!strcmp(label, FR_CODEC))
	meta->codec = data;
      else if (!strcmp(label, FR_TRACK))
	meta->trackno = numdata;
      else if (!strcmp(label, FR_YEAR))
	meta->year = numdata;
      else if (!strcmp(label, FR_FNAME))
	meta->filename = data;
      else if (!strcmp(label, FR_PROTECTED)) {
	meta->protect = TRUE;
	g_free(data);
      }
      else {
	if (data) {
	  g_print("Discarded: %s -> %s\n", label, data);
	  g_free(data);
	}
      }
      g_free(label);
    }
    /* Compensate for missing tag information */
    if (!meta->artist)
      meta->artist = g_strdup("<Unknown>");
    if (!meta->title)
      meta->title = g_strdup("<Unknown>");
    if (!meta->album)
      meta->album = g_strdup("<Unknown>");
    if (!meta->genre)
      meta->genre = g_strdup("<Unknown>");
    if (!meta->length)
      meta->length = g_strdup("0:00");
    /* Add to hash */
    g_hash_table_insert(songhash,
			GUINT_TO_POINTER(string_to_guint(meta->path)),
			(gpointer) meta);
    songid_destroy (songtag);
    if (cancel_jukebox_operation)
      break;
  }
  /* Build songlist from hash */
  jblist_from_songhash(TRUE);

  /* Fill in the list of data files */
  NJB_Reset_Get_Datafile_Tag(njb);
  gdk_threads_enter();
  gtk_label_set_text(GTK_LABEL(args->label), scan_datafiles);

  /* Hack to get the datafile list store built in O(n) instead of O(n^2) */
  recreate_list_store(JBDATA_LIST);

  gdk_threads_leave();
  while (datatag = NJB_Get_Datafile_Tag (njb)) {
    metadata_t *meta;
    u_int64_t filesize;
    gint i;
    gchar *scanstring;

    jukebox_datafiles++;
    scanstring = g_strdup_printf(_("%u data files scanned"), jukebox_datafiles);
    gdk_threads_enter();
    gtk_label_set_text(GTK_LABEL(args->label), scanstring);
    gdk_threads_leave();
    g_free(scanstring);

    meta = new_metadata_t();
    meta->filename = g_strdup(datatag->filename);
    /* g_print("Scan: filename %s, size %d\n", datatag->filename, strlen(datatag->filename)); */
    /* Convert filesize from 64 bit unsigned integer value */
    filesize = (u_int64_t) datatag->msdw;
    filesize = filesize << 32;
    filesize = filesize | (u_int64_t) datatag->lsdw;
    /* data = g_strdup_printf("%Lu", filesize); */
    meta->size = (guint) filesize; /* FIXME: hows it right? Also for tracks? */
    /* File ID */
    meta->path = g_strdup_printf("%lu", datatag->dfid);
    gdk_threads_enter();
    add_metadata_to_model(meta, JBDATA_LIST);
    gdk_threads_leave();
    destroy_metadata_t(meta);
    datafile_destroy (datatag);
    if (cancel_jukebox_operation)
      break;
  }

  gdk_threads_enter();
  /* Then sort everything */
  view_and_sort_list_store(JBDATA_LIST);
  gtk_label_set_text(GTK_LABEL(args->label), scan_playlists);
  gdk_threads_leave();
  build_playlist_tree(args->pltreestore, TRUE);
  
  gdk_threads_enter();
  gtk_widget_destroy(args->dialog);
  gdk_threads_leave();
  jukebox_locked = FALSE;
}

/*
 * Progression callback for transfer both back and forth to the
 * jukebox
 */
static int progress (u_int64_t sent, u_int64_t total, const char* buf, unsigned len, void *data)
{
  gdouble fraction;

  fraction = ((gdouble) sent / (gdouble) total);
  gdk_threads_enter();
  // gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), percentage);
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progress_bar), fraction);
  gdk_threads_leave();
  /* This cancels any jukebox operation */
  if (cancel_jukebox_operation)
    return -1;
  return 0;
}


/***********************************************************************************/
/* Transferring music from jukebox thread                                          */
/***********************************************************************************/

gpointer jb2hd_thread(gpointer thread_args)
{
  jb2hd_thread_arg_t *args = (jb2hd_thread_arg_t *) thread_args;
  GList *tmplist = args->metalist;

  while (tmplist &&
	 !cancel_jukebox_operation)
    {
      metadata_t *jbmeta;
      metadata_t *meta;
      u_int32_t id;
      gchar *filename;
      gchar *tmpfname;
      
      /* Get title from filelist */
      jbmeta = (metadata_t *) tmplist->data;
      filename = compose_filename(jbmeta);
      tmpfname = filename_fromutf8(filename);
      id = string_to_guint(jbmeta->path);
      gdk_threads_enter();
      gtk_label_set_text(GTK_LABEL(args->label), filename);
      gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), (gfloat) 0);
      gdk_threads_leave();
      /* g_print("Transferring %s...\n", file); */
      /* Transfer track */
      if ( NJB_Get_Track(njb, id, jbmeta->size, tmpfname, progress, NULL) == -1 ) {
	njb_error_dump(stderr);
	g_free(filename);
	goto clean_up_and_return;
      }
      /* Copy all metadata, but replace jukebox ID with path */
      meta = clone_metadata_t(jbmeta);
      g_free(meta->path);
      meta->path = g_strdup(filename);
      g_free(filename);
      /* Tag with ID3 unless already present */
      if (get_prefs_useid3()) {
	if (meta->codec != NULL &&
	    !strcmp(meta->codec, "MP3")) {
	  set_tag_for_mp3file (meta, tmpfname, get_prefs_id3override());
	}
      }
      
      /* Add to disk listbox and re-sort */
      g_free(tmpfname);
      gdk_threads_enter();
      add_metadata_to_model(meta, HD_LIST);
      gdk_flush(); // Time consuming?
      // FIXME: trig resort gtk_clist_sort(GTK_CLIST(args->harddiskclist));
      gdk_threads_leave();
      /* Destroy metadata */
      destroy_metadata_t(meta);
      tmplist = tmplist->next;
    }
  
 clean_up_and_return:

  jukebox_locked = FALSE;
  gdk_threads_enter();
  gtk_widget_destroy(args->dialog);
  fill_in_dir(HD_LIST, get_current_dir());
  gdk_threads_leave();
  /* Free the memory used by the list */
  destroy_metalist(args->metalist);
}


/***********************************************************************************/
/* Transferring data files from jukebox thread                                     */
/***********************************************************************************/

gpointer jb2hd_data_thread(gpointer thread_args)
{
  jb2hd_data_thread_arg_t *args = (jb2hd_data_thread_arg_t *) thread_args;
  GList *metalist = args->metalist;

  while (metalist &&
	 !cancel_jukebox_operation)
    {
      metadata_t *jbmeta;
      u_int32_t id;
      gchar *tmpfname;
      
      jbmeta = (metadata_t *) metalist->data;
      id = string_to_guint(jbmeta->path);
      gdk_threads_enter();
      gtk_label_set_text(GTK_LABEL(args->label), jbmeta->filename);
      gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), (gfloat) 0);
      gdk_threads_leave();
      /* g_print("Transfering %s\n", jbmeta->filename); */
      /* Transfer file */
      tmpfname = filename_fromutf8(jbmeta->filename);
      if (NJB_Get_File(njb, id, jbmeta->size, tmpfname, progress, NULL) == -1) {
	g_free(tmpfname);
	njb_error_dump(stderr);
	goto data_error;
      }
      g_free(tmpfname);
      /* Add to disk listbox and re-sort */
      gdk_threads_enter();
      add_metadata_to_model(jbmeta, HDDATA_LIST);
      gdk_flush(); // Time consuming?
      gdk_threads_leave();
      metalist = g_list_next(metalist);
    }
  
 data_error:

  destroy_metalist(args->metalist);
  jukebox_locked = FALSE;
  gdk_threads_enter();
  gtk_widget_destroy(args->dialog);
  fill_in_dir(HDDATA_LIST, get_current_dir());
  gdk_threads_leave();
}


/***********************************************************************************/
/* Transferring music to jukebox thread                                            */
/***********************************************************************************/

gpointer hd2jb_thread(gpointer thread_args)
{
  hd2jb_thread_arg_t *args = (hd2jb_thread_arg_t *) thread_args;
  GList *templist = args->metalist;
  GList *new_metalist = NULL;
  gchar *tmpdirname;

  // Create a temporary directory for tag-stripped files.
  if (get_prefs_id3remove()) {
    tmpdirname = tmpnam(NULL);
    // Create it.
    if (!create_directory(tmpdirname)) {
      goto hd2jb_cleanup;
    }
  }
  
  while (templist &&
	 !cancel_jukebox_operation)
    {
      metadata_t *hdmeta;
      metadata_t *jbmeta;
      gchar *tmpyear;
      gint protectint;
      gchar *tmpfname;
      gchar *tmp;
      u_int32_t id, length;
      
      hdmeta = (metadata_t *) templist->data;
      // g_print("Storing %s on jukebox...\n", hdmeta->filename);
      gdk_threads_enter();
      tmp = g_strdup_printf("%s - %s", hdmeta->artist, hdmeta->title);
      gtk_label_set_text(GTK_LABEL(args->label), tmp);
      g_free(tmp);
      gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), (gfloat) 0);
      gdk_threads_leave();
      length = mmss_to_seconds(hdmeta->length);
      tmpyear = g_strdup_printf("%lu", hdmeta->year);
      protectint = hdmeta->protect ? 1 : 0;
      /* Optionally remove ID3 tag */
      if (get_prefs_id3remove() && !strcmp(hdmeta->codec, "MP3")) {
	gchar *tmpfname8;
	gchar *tmpfname;

	tmpfname8 = g_build_filename(tmpdirname, hdmeta->filename, NULL);
	tmpfname = filename_fromutf8(tmpfname8);
	/* Call the filesystem to clone the file and remove the tag */
	if (clone_and_strip_id3(hdmeta->path, tmpfname8)) {
	  // g_print("Transferring track.\n");
	  if (NJB_Send_Track (njb, 
			      tmpfname, 
			      hdmeta->codec, 
			      hdmeta->title, 
			      hdmeta->album, 
			      hdmeta->genre,
			      hdmeta->artist, 
			      length, 
			      hdmeta->trackno, 
			      tmpyear, 
			      protectint, 
			      progress, 
			      NULL, 
			      &id) == -1) {
	    njb_error_dump(stderr);
	    goto hd2jb_cleanup;
	  }
	  /* Afterwards delete the tempfile */
	  unlink(tmpfname);
	  g_free(tmpfname8);
	  g_free(tmpfname);
	  jukebox_songs++;
	}
      } else {
	tmpfname = filename_fromutf8(hdmeta->path);
	if (NJB_Send_Track (njb, 
			    tmpfname, 
			    hdmeta->codec, 
			    hdmeta->title, 
			    hdmeta->album, 
			    hdmeta->genre,
			    hdmeta->artist, 
			    length, 
			    hdmeta->trackno, 
			    tmpyear, 
			    protectint, 
			    progress, 
			    NULL, 
			    &id) == -1) {
	  g_free(tmpfname);
	  njb_error_dump(stderr);
	  goto hd2jb_cleanup;
	}
	/* g_print("%s stored on jukebox with ID: %lu\n", file, id); */
	g_free(tmpfname);
	jukebox_songs++;
      }
      g_free(tmpyear);
      /* Add to disk listbox and re-sort */
      /* Add correct trackid in col 7, add the row */
      jbmeta = clone_metadata_t(hdmeta);
      g_free(jbmeta->path);
      jbmeta->path = g_strdup_printf("%lu", id);
      gdk_threads_enter();
      add_metadata_to_model(jbmeta, JB_LIST);
      gdk_flush();
      gdk_threads_leave();
      /* Then add the song to the global hash table */
      g_hash_table_insert(songhash,
			  GUINT_TO_POINTER(id),
			  (gpointer) jbmeta);
      new_metalist = g_list_append (new_metalist, jbmeta);
      templist = g_list_next(templist);
    }
  // Remove temporary directory for tag-stripped files.
  if (get_prefs_id3remove()) {
    rmdir(tmpdirname);
    g_free(tmpdirname);
  }
  /* At last add the tracks to default playlists if
   * any such are selected. */
  if (args->playlists != NULL) {
    add_tracks_to_playlists(args->playlists, new_metalist, args->pltreestore, TRUE);
  }
  g_list_free(new_metalist);
  flush_usage();
  
 hd2jb_cleanup:

  destroy_metalist(args->metalist);
  g_list_free(args->playlists);
  gdk_threads_enter();
  gtk_widget_destroy(args->dialog);
  jukebox_locked = FALSE;
  gdk_threads_leave();
  g_free(args);
}


/***********************************************************************************/
/* Transferring data file to jukebox thread                                        */
/***********************************************************************************/

gpointer hd2jb_data_thread(gpointer thread_args)
{
  hd2jb_data_thread_arg_t *args = (hd2jb_data_thread_arg_t *) thread_args;
  GList *metalist = args->metalist;

  while (metalist &&
	 !cancel_jukebox_operation)
    {
      gchar *tmpfname;
      u_int32_t id;
      metadata_t *hdmeta;
      metadata_t *jbmeta;
      
      hdmeta = (metadata_t *) metalist->data;
      /* g_print("Storing %s on jukebox as %s...\n", hdmeta->path, hdmeta->filename); */
      gdk_threads_enter();
      gtk_label_set_text(GTK_LABEL(args->label), hdmeta->filename);
      gtk_progress_bar_update(GTK_PROGRESS_BAR(progress_bar), (gfloat) 0);
      gdk_threads_leave();
      tmpfname = filename_fromutf8(hdmeta->path);
      if (NJB_Send_File (njb, tmpfname, hdmeta->filename, progress, NULL, &id) == -1) {
	g_free(tmpfname);
	njb_error_dump(stderr);
	goto hd2jb_data_cleanup;
      }
      g_free(tmpfname);
      /* g_print("%s stored on jukebox with ID: %lu\n", hdmeta->filename, id); */
      jukebox_datafiles++;
      /* Add to disk listbox and re-sort */
      /* Add correct trackid in col 7, add the row
       * then restore the old contents so they will 
       * be freed correctly */
      jbmeta = clone_metadata_t(hdmeta);
      g_free(jbmeta->path);
      jbmeta->path = g_strdup_printf("%lu", id);
      gdk_threads_enter();
      add_metadata_to_model(jbmeta, JBDATA_LIST);
      gdk_threads_leave();
      destroy_metadata_t(jbmeta);
      metalist = g_list_next(metalist);
    }
  flush_usage();

 hd2jb_data_cleanup:

  destroy_metalist(args->metalist);
  gdk_threads_enter();
  gtk_widget_destroy(args->dialog);
  jukebox_locked = FALSE;
  gdk_threads_leave();
}


/* Called after file deletion to remove the deleted
 * tracks from any playlists they may occur in. UNTHREADED */
static void remove_tracks_from_playlists(GList *metalist,
					 GtkTreeStore *pltreestore)
{
  GList *tmplist;
  playlist_t *playlist;

  if (!metalist)
    return;
  /* g_print("Called remove_tracks_from_playlists()\n"); */
  NJB_Reset_Get_Playlist(njb);
  while (playlist = NJB_Get_Playlist(njb)) {
    playlist_track_t *track;
    
    playlist_reset_gettrack(playlist);
    while (track = playlist_gettrack(playlist)) {
      tmplist = metalist;
      while (tmplist) {
	metadata_t *meta;
	u_int32_t id;

	meta = (metadata_t *) tmplist->data;
	id = string_to_guint(meta->path);
	if (id == track->trackid) {
	  /* When the track is located in a playlist, remove it */
	  /* g_print("Removing track: %lu from playlist %lu\n", 
	     track->trackid, playlist->plid); */
	  if (track->prev != NULL)
	    track->prev->next = track->next;
	  else
	    playlist->first = track->next;
	  if (track->next != NULL)
	    track->next->prev = track->prev;
	  playlist_track_destroy(track);
	  playlist->ntracks--;
	  playlist->_state= NJB_PL_CHTRACKS;
	}
	tmplist = g_list_next(tmplist);
      }
    }
    /* If the playlist changed, update it */
    if (playlist->_state==NJB_PL_CHTRACKS) {
      if (NJB_Update_Playlist(njb, playlist) == -1)
	njb_error_dump(stderr);
    }
    playlist_destroy(playlist);
  }
  if (njb_error != EO_EOM) {
    njb_error_dump(stderr);
  }
  build_playlist_tree(pltreestore, FALSE);
}

/* Delete a list of files from the jukebox */
void jukebox_delete_files(GList *metalist)
{
  GList *tmplist = metalist;

  if (!metalist)
    return;
  jukebox_locked = TRUE;
  while (tmplist) {
    u_int32_t id;
    char *dummy;
    metadata_t *meta;
    
    meta = (metadata_t *) tmplist->data;
    id = string_to_guint(meta->path);
    /* g_print("Deleting: %u from jukebox library\n", id); */
    if (id) {
      if ( NJB_Delete_Datafile(njb, id) == -1 ) {
	njb_error_dump(stderr);
      }
      if (jukebox_datafiles)
	jukebox_datafiles--;
    }
    tmplist = g_list_next(tmplist);
  }
  flush_usage();
  jukebox_locked = FALSE;
}

/* Delete a list of files from the jukebox */
void jukebox_delete_tracks(GList *metalist,
			   GtkTreeStore *pltreestore)
{
  GList *tmplist = metalist;

  if (!metalist)
    return;
  jukebox_locked = TRUE;
  /* First remove tracks from any playlists they are in */
  remove_tracks_from_playlists(metalist, pltreestore);
  while (tmplist) {
    metadata_t *meta;
    u_int32_t id;
    
    meta = (metadata_t *) tmplist->data;
    id = string_to_guint(meta->path);
    /* g_print("Deleting: %lu from jukebox library\n", id); */
    if (id) {
      metadata_t *tmpmeta;
      if ( NJB_Delete_Track(njb, id) == -1 ) {
	njb_error_dump(stderr);
      }
      /* Remove song from hash */
      tmpmeta = (metadata_t *) g_hash_table_lookup(songhash, GUINT_TO_POINTER(id));
      // Sometimes tracks are removed that are not part of any playlist. Weird but happens.
      if (tmpmeta != NULL) {
	destroy_hash(NULL, tmpmeta, NULL);
	g_hash_table_remove(songhash,
			    GUINT_TO_POINTER(id));
      }
      if (jukebox_songs)
	jukebox_songs--;
    }
    tmplist = tmplist->next;
  }
  flush_usage();
  jukebox_locked = FALSE;
}

gboolean jukebox_begin_metadata_set(void)
{
  /* Begin metadata transaction, return TRUE if transaction
   * is OK to begin */
  jukebox_locked = TRUE;
  return TRUE;
}

void jukebox_set_metadata (metadata_t *meta)
{
  /* Set metadata on the file with ID id */
  u_int32_t id, length;
  metadata_t *tmpmeta;
  gchar *tmpyear;
  gint protectint;

  id = string_to_guint(meta->path);
  length = mmss_to_seconds(meta->length);
  tmpyear = g_strdup_printf("%lu", meta->year);
  protectint = meta->protect ? 1 : 0;

  if (NJB_Replace_Track_Tag (njb,
			     id,
			     meta->codec,
			     meta->title,
			     meta->album, 
			     meta->genre,
   			     meta->artist,
			     length,
			     meta->trackno,
			     meta->size, 
			     meta->filename,
			     tmpyear, 
			     protectint) == -1) {
    njb_error_dump(stderr);
  }
  g_free(tmpyear);
  /* Replace the data in the hash table */
  tmpmeta = (metadata_t *) g_hash_table_lookup(songhash,
					       GUINT_TO_POINTER(id));
  // Make sure we actually find it! Sometimes it is non-existant...
  if (tmpmeta != NULL) {
    destroy_hash(NULL, tmpmeta, NULL);
    g_hash_table_remove(songhash,
			GUINT_TO_POINTER(id));
  }
  /* Clone metadata */
  tmpmeta = clone_metadata_t(meta);
  /* Insert the new row */
  g_hash_table_insert(songhash,
		      GUINT_TO_POINTER(id),
		      (gpointer) tmpmeta);
}

void jukebox_end_metadata_set(void)
{
  /* End metadata transaction */
  jukebox_locked = FALSE;
  /* Update the window with the new metadata */
  // jblist_from_songhash(FALSE);
}

/* Creates a new playlist and returns its playlist ID UNTHREADED */
guint jukebox_create_playlist(gchar *plname, GtkTreeStore *pltreestore)
{
  playlist_t *playlist;
  guint plid = 0;

  /* g_print("Called create playlist\n"); */
  /* Create the new playlist in memory */
  playlist=playlist_new();
  if (playlist == NULL) {
    return 0;
  }
  if (playlist_set_name(playlist, plname) == -1) {
    njb_error_dump(stderr);
    return 0;
  }
  jukebox_locked = TRUE;
  /* g_print("Calling NJB_Update_Playlist\n"); */
  if (NJB_Update_Playlist(njb, playlist) == -1) {
    gdk_threads_enter();
    create_error_dialog(_("Could not create playlist"));
    gdk_threads_leave();
    njb_error_dump(stderr);
  }
  jukebox_playlists++;
  plid = playlist->plid;
  playlist_destroy(playlist);
  build_playlist_tree(pltreestore, FALSE);
  jukebox_locked = FALSE;
  /* g_print("Created playlist\n"); */
  return plid;
}

/* Delete a playlist with playlist ID plid UNTHREADED */
void jukebox_delete_playlist(guint plid)
{
  jukebox_locked = TRUE;
  if (NJB_Delete_Playlist(njb, plid) == -1) {
    gdk_threads_enter();
    create_error_dialog(_("Could not delete playlist"));
    gdk_threads_leave();
    njb_error_dump(stderr);
  }
  jukebox_playlists--;
  build_playlist_list();
  jukebox_locked = FALSE;
}

/* Rename the playlist with ID plid UNTHREADED */
void jukebox_rename_playlist(guint plid, gchar *name, GtkTreeStore *pltreestore, gboolean threaded)
{
  playlist_t *playlist;
  gboolean found = FALSE;
  
  jukebox_locked = TRUE;
  NJB_Reset_Get_Playlist(njb);
  while (playlist = NJB_Get_Playlist(njb)) {
    /* g_print("Playlist %u == Playlist %u ... ",playlist->plid, plid); */
    if (playlist->plid == plid) {
      /* g_print("Yes.\n"); */
      found = TRUE;
      break;
    }
    /* g_print("No.\n"); */
    playlist_destroy(playlist);
  }
  if (found) {
    if (playlist_set_name(playlist, name) == -1) {
      njb_error_dump(stderr);
    } else {
      if (NJB_Update_Playlist(njb, playlist) == -1) {
	if (threaded)
	  gdk_threads_enter();
	create_error_dialog(_("Could not rename playlist"));
	if (threaded)
	  gdk_threads_leave();
	njb_error_dump(stderr);
      }
    }
    playlist_destroy(playlist);
    build_playlist_tree(pltreestore, FALSE);
  } else {
    if (threaded)
      gdk_threads_enter();
    create_error_dialog(_("Could not locate playlist to rename!"));
    if (threaded)
      gdk_threads_leave();
  }
  jukebox_locked = FALSE;
}

/* This is a public caller func for the above UNTHREADED */
void add_tracks_to_playlists(GList *playlists, GList *metalist, GtkTreeStore *pltreestore, gboolean threaded)
{
  GList *templist = g_list_first(playlists);

  /* g_print("Called add_tracks_to_playlist()\n"); */
  jukebox_locked = TRUE;
  /* For thread message in the unthreaded code. Kludgy. */
  cancel_jukebox_operation = FALSE;
  while (templist != NULL) {
    guint plid = GPOINTER_TO_UINT(templist->data);
    add_to_playlist(plid, metalist, pltreestore, threaded);
    templist = g_list_next(templist);
  }
  jukebox_locked = FALSE;
}


/* Delete file from playlist UNTHREADED */
guint jukebox_delete_track_from_playlist(guint trackid, guint plist)
{
  guint new_plid = plist;

  if (trackid != 0 && plist != 0) {
    playlist_t *playlist;
    jukebox_locked = TRUE;
    gboolean found = FALSE;

    g_print("Removing track: %lu from playlist %lu\n", 
	    trackid, plist);
    NJB_Reset_Get_Playlist(njb);
    while (playlist = NJB_Get_Playlist(njb)) {
      g_print("Is it in %lu?\n", playlist->plid);
      if (playlist->plid == plist) {
	g_print("YES!");
	found = TRUE;
	break;
      }
      g_print("NO.");
      playlist_destroy(playlist);
    }
    /* If the playlist was found, remove the track */
    if (found) {
      playlist_track_t *track;
      
      g_print("Found playlist %lu\n", playlist->plid);
      while (track = playlist_gettrack(playlist)) {
	if (trackid == track->trackid) {
	  /* When the track is located in a playlist, remove it */
	  g_print("Removing track: %lu from playlist %lu\n", 
		  track->trackid, playlist->plid);
	  
	  if (track->prev != NULL)
	    track->prev->next = track->next;
	  else
	    playlist->first = track->next;
	  if (track->next != NULL)
	    track->next->prev = track->prev;
	  playlist_track_destroy(track);
	  playlist->ntracks--;
	  playlist->_state = NJB_PL_CHTRACKS;
	  break;
	}
      }
      if (playlist->_state==NJB_PL_CHTRACKS) {
	/* g_print("Called NJB_Update_Playlist...\n"); */
	if (NJB_Update_Playlist(njb, playlist) == -1)
	  njb_error_dump(stderr);
	new_plid = playlist->plid;
      }
      playlist_destroy(playlist);
    } else {
      create_error_dialog(_("Could not find the track in the playlist"));
    }
  }
  jukebox_locked = FALSE;
  // build_playlist_tree(playlisttree, FALSE);
  g_print("Old playlist ID: %lu, new playlist ID: %lu\n", plist, new_plid);
  return new_plid;
}

/* Retrieve ownerstring */
gchar *jukebox_get_ownerstring(void)
{
  return jukebox_ownerstring;
}

/* Set the ownerstring */
void jukebox_set_ownerstring(gchar *owner)
{
  jukebox_locked = TRUE;
  NJB_Set_Owner_String (njb, owner);
  if (jukebox_ownerstring != NULL)
    g_free(jukebox_ownerstring);
  jukebox_ownerstring = g_strdup(owner);
  /* The ownerstring max length is 64 bytes
   * including the null-terminator according
   * to the protocol specification */
  if (strlen(jukebox_ownerstring) >= OWNER_STRING_LENGTH)
    jukebox_ownerstring[OWNER_STRING_LENGTH-1] = '\0';
  jukebox_locked = FALSE;
}

/* Return disk usage */
#ifdef G_HAVE_GINT64
void jukebox_getusage(guint64 *total, guint64 *free, guint64 *used, 
		      guint *songs, guint *playlists, guint *datafiles)
{
  *total = jukebox_totalbytes;
  *free = jukebox_freebytes;
  *used = jukebox_usedbytes;
  *songs = jukebox_songs;
  *playlists = jukebox_playlists;
  *datafiles = jukebox_datafiles;
}
#endif

/* Returns the hh:mm:ss representation from seconds */
static void hhmmss (u_int16_t seconds, u_int16_t *hh, u_int16_t *mm, u_int16_t *ss)
{
  if ( seconds >= 3600 ) {
    *hh= seconds/3600;
    seconds-= 3600*(*hh);
  } else
    *hh = 0;
  if ( seconds >= 60 ) {
    *mm= seconds/60;
    seconds-= 60*(*mm);
  } else
    *mm = 0;
  *ss= seconds;
}

/* Reset EAX retrieval routine */
void jukebox_reset_get_eax(void)
{
  jukebox_locked = TRUE;
  NJB_Reset_Get_EAX_Type(njb);
}


/* Gets an EAX setting struct, returns NULL if
 * something fails. */
njb_eax_t *jukebox_get_eax(void)
{
  njb_eax_t *eax = NJB_Get_EAX_Type(njb);

  if (eax == NULL)
    jukebox_locked = FALSE;
  return eax;
}

/* Free the memory used by the EAX structure */
void jukebox_destroy_eax(njb_eax_t *eax)
{
  NJB_Destroy_EAX_Type(eax);
}

void jukebox_adjust_eax(guint16 effect, guint16 patch, gint16 value)
{
  if (created_play_mutex) {
    g_mutex_lock(play_thread_mutex);
    NJB_Adjust_EAX(njb, effect, patch, value);
    g_mutex_unlock(play_thread_mutex);
  }
}

/* Returns the playlist in a format suiting for
 * the player thread below */
GList *jukebox_get_playlist_for_play(guint plid)
{
  GList *retlist = NULL;
  playlist_t *playlist;
  gboolean found = FALSE;

  jukebox_locked = TRUE;
  NJB_Reset_Get_Playlist(njb);
  while (playlist = NJB_Get_Playlist(njb)) {
    if (playlist->plid == plid) {
      found = TRUE;
      break;
    }
    // Dangerous?
    playlist_destroy(playlist);
  }
  /* Get the tracks from the playlist */
  if (found) {
    playlist_track_t *track;
    metadata_t *meta;
    metadata_t *newmeta;
    gchar *tmp;

    while (track = playlist_gettrack(playlist)) {
      meta = (metadata_t *) g_hash_table_lookup(songhash,
						GUINT_TO_POINTER(track->trackid));
      // Sometimes it's non-existant
      if (meta != NULL) {
	newmeta = clone_metadata_t(meta);
	retlist = g_list_append(retlist, (gpointer) newmeta);
      }
    }
  }
  jukebox_locked = FALSE;  
  return retlist;
}

/* Transfer playlist to jukebox */
static void send_playlist(GList *list)
{
  gboolean first = TRUE;
  GList *tmplist = list;
  
  if (tmplist != NULL)
    NJB_Stop_Play(njb);
  while (tmplist) {
    metadata_t *meta;
    u_int32_t id;

    meta = (metadata_t *) tmplist->data;
    id = string_to_guint(meta->path);
    if (id) {
      if (first) {
	/* g_print("Playing: %lu\n", id); */
	first = FALSE;
	if (NJB_Play_Track(njb, id) == -1)
	  njb_error_dump(stderr);
      } else {
	/* g_print("Queueing: %lu\n", id); */
	if (NJB_Queue_Track(njb, id) == -1)
	  njb_error_dump(stderr);
      }
    }
    tmplist = tmplist->next;
  }
}

/* Sets the songname in the dialog - you have to
 * lock mutexes and gdk thread (if needed) before 
 * calling this function */
static void set_songname_label(gchar *artist, gchar *title, gchar *seconds)
{
  gchar *tmp;

  tmp = g_strdup_printf("%s - %s, %s", artist, title, seconds);
  gtk_label_set_text(GTK_LABEL(songnamelabel), tmp);
  g_free(tmp);
}

void jukebox_previous(void)
{
  if (created_play_mutex) {
    g_mutex_lock(play_thread_mutex);
    /* Twice, because each read advances three steps */
    if (playlistitr->prev != NULL) {
      metadata_t *meta;

      playlistitr = g_list_previous(playlistitr);
      meta = (metadata_t *) playlistitr->data;
      set_songname_label(meta->artist, meta->title, meta->length);
    }
    send_playlist(playlistitr);
    passed_first_zero = FALSE;
    g_mutex_unlock(play_thread_mutex);
  }
}

/* The parameter tells if the next song is already
 * playing, so that the list does not need to be
 * resent. */
void jukebox_next(gboolean already_playing)
{
  if (created_play_mutex) {
    g_mutex_lock(play_thread_mutex);
    if (playlistitr != playlistlast) {
      playlistitr = g_list_next(playlistitr);
      if (playlistitr != NULL) {
	metadata_t *meta;

	meta = (metadata_t *) playlistitr->data;
	set_songname_label(meta->artist, meta->title, meta->length);
	if (!already_playing) {
	  send_playlist(playlistitr);
	  passed_first_zero = FALSE;
	}
      }
    }
    g_mutex_unlock(play_thread_mutex);
  }
}

static void jukebox_current(void)
{
  GList *tmplist;

  /* g_print("Called jukebox_current...\n"); */
  if (playlistitr != NULL) {
    metadata_t *meta;
    
    meta = (metadata_t *) playlistitr->data;
    set_songname_label(meta->artist, meta->title, meta->length);
    send_playlist(playlistitr);
  }
}

/* Start playing a bunch of selected files */
gpointer play_thread(gpointer thread_args)
{
  play_thread_arg_t *args = (play_thread_arg_t *) thread_args;
  u_int16_t sec, hh, mm, ss;
  gint change = 0;
  gboolean repeat = TRUE;
  gfloat position;

  /* This mutex is to avoid collisions with EAX changes */
  if (!created_play_mutex) {
    g_assert (play_thread_mutex == NULL);
    play_thread_mutex = g_mutex_new();
    created_play_mutex = TRUE;
  }
  jukebox_locked = TRUE;

  /* Lock playing mutex and initialize */
  g_mutex_lock(play_thread_mutex);
  playlist = args->metalist;
  songnamelabel = args->songlabel;
  passed_first_zero = FALSE;
  if (!playlist) {
    g_mutex_unlock(play_thread_mutex);
    return;
  }
  /* The iterator is used for walking the playlist */
  playlistitr = playlist;
  /* Set playlistlast to the pointer of the last song */
  playlistlast = g_list_last(playlist);
  /* Setup for the first round */
  gdk_threads_enter();
  jukebox_current();
  gdk_threads_leave();
  g_mutex_unlock(play_thread_mutex);

  while (!cancel_jukebox_operation &&
	 repeat) {
    if (change) {
      if (playlistitr == NULL) {
	/* It would be nice if the jukebox sent a signal
	 * like this at the end of playing all tracks,
	 * but unfortunately it doesn't so this clause
	 * is not executed on any device I've seen. */
	repeat = FALSE;
      } else {
	gdk_threads_enter();
	jukebox_next(TRUE);
	gdk_threads_leave();
      }
      change = 0;
    } else {
      gchar tmp[10];
      metadata_t *meta;
      guint seconds;

      g_mutex_lock(play_thread_mutex);
      NJB_Elapsed_Time(njb, &sec, &change);
      g_mutex_unlock(play_thread_mutex);
      hhmmss(sec, &hh, &mm, &ss);
      sprintf(tmp, "%02u:%02u:%02u", hh, mm, ss);
      meta = (metadata_t *) playlistitr->data;
      //dump_metadata_t(meta);
      seconds = mmss_to_seconds(meta->length);
      position = (gfloat) 100 * ((gfloat) sec / (gfloat) seconds);
      /* This is a guard against things that may happen if
       * you have MP3s with variable bitrate -- the second
       * tag may be incorrect */
      if (position < 0)
	position = 100.0;
      gdk_threads_enter();
      gtk_label_set_text(GTK_LABEL(args->timelabel), tmp);
      gtk_adjustment_set_value(GTK_ADJUSTMENT(args->adj), position);
      gdk_threads_leave();
    }
    /* The Jukebox reports zero seconds of time at the
     * beginning of play, and after the song ends. This
     * detects it and uses the situation after the last
     * song ends to kill the player window */
    g_mutex_lock(play_thread_mutex);
    if (playlistitr == playlistlast &&
	sec > 0)
      passed_first_zero = TRUE;
    if (passed_first_zero &&
	sec == 0 &&
	playlistitr == playlistlast) {
      /* g_print("End of last track\n"); */
      g_mutex_unlock(play_thread_mutex);
      break;
    }
    g_mutex_unlock(play_thread_mutex);
    /* Sleep for a second */
    sleep(1);
  }

  /* Free the argument list */
  gdk_threads_enter();
  gtk_widget_destroy(args->dialog);
  gdk_threads_leave();
  g_mutex_lock(play_thread_mutex);
  destroy_metalist(playlist);
  playlist = NULL;
  NJB_Stop_Play(njb);
  g_mutex_unlock(play_thread_mutex);
  jukebox_locked = FALSE;
}

/* This routine synchronize the time on the
 * jukebox to that of the host system
 */
void jukebox_synchronize_time(void)
{
  njb_time_t *jukeboxtime;
  GTimeVal currenttimeval;
  GTime currenttime;
  struct tm *tm;
  
  // Get the time from the jukebox, then modify it.
  jukeboxtime = NJB_Get_Time(njb);
  g_get_current_time(&currenttimeval);
  currenttime = (GTime) currenttimeval.tv_sec;

  // Uses UNIX library calls, might need to be portablized
  tm = localtime((time_t *) &currenttime);
  /*
    g_print("Setting: %d-%d-%d (wkd: %d) %d:%d:%d\n", 1900+tm->tm_year,
    1+tm->tm_mon, tm->tm_mday, tm->tm_wday-1,
    tm->tm_hour, tm->tm_min, tm->tm_sec);
  */
  jukeboxtime->year = 1900+tm->tm_year;
  jukeboxtime->month = 1+tm->tm_mon;
  jukeboxtime->day = tm->tm_mday;
  jukeboxtime->weekday = tm->tm_wday-1;
  jukeboxtime->hours = tm->tm_hour;
  jukeboxtime->minutes = tm->tm_min;
  jukeboxtime->seconds = tm->tm_sec;

  NJB_Set_Time(njb, jukeboxtime);
  NJB_Destroy_Time(jukeboxtime);
}
