/* gnobog_app.c
 *
 * Copyright (C) 2000 Frdric LESPEZ  & Renaud CHAILLAT
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <gnome.h>
#include <glade/glade.h>
#include <sys/stat.h> /* mkdir */
#include "gnobog_save_file_dialog.h"
#include "gnobog_arborescence.h"
#include "gnobog_bookmarks.h"
#include "gnobog_pixmaps.h"
#include "gnobog_popup_menu.h"
#include "gnobog_app.h"
#include "gnobog_menus.h"

#define LOCK_FILENAME  "lock"
#define LOCK_ERROR "Lock file %s exist !\n\
Either another Gnobog is running\n\
Or Gnobog let a stale lock file.\n\
Please investigate :\n\
If another Gnobog is running, use it :-)\n\
If Gnobog is *not* running, please remove lock file\n\
and restart Gnobog\n\
Gnobog will close now."
#define LOCK_CREATING_ERROR "Unable to create lock file %s !\n\
Please investigate :\n\
Be sure Gnobog have permissions\n\
to create lock file. Check this and rerun.\n\
Gnobog will close now."

#define DROP_ZONE_FILENAME  "storing_zone.html"
#define DROP_ZONE_TITLE     "Storing Zone"

#define DROP_ZONE_LOADING_ERROR "Unable to load Storing Zone located here :\n\
%s\n\
Please investigate :\n\
First, be sure Gnobog have permissions\n\
to access Storing Zone file. Check this and rerun.\n\
If Gnobog have enough permission, then Storing Zone\n\
file exists but is not valid (Corrupted ?).\n\
Move away the file (backup it) and rerun Gnobog.\n\
Gnobog will close now."

struct _Document {
  GnobogBookmarks*  bookmarks; /* can uniquely identify a document */
  GList*            windows_list;
};
typedef struct _Document Document;


/* Column titles */
static gchar*            Titles[] = {N_("Name"), N_("Location")};
static gchar*            DocumentsTitles[] = {N_("Name"), N_("Views")};

/* to store the current app pointer */
static GtkWidget*        gnobog_app = NULL;

/* to store the active arborescence */
static GtkWidget*        active_view = NULL;

/* Keep a list of all open documents and their windows */
static GList*            gnobog_app_documents_list = NULL;
static GtkWidget*        documents_view            = NULL;

/* The clipboards : internal bookmarks objects */
static GnobogBookmarks*  manual_clipboard    = NULL;
static GnobogBookmarks*  drop_zone_bookmarks = NULL;
gchar*                   gnobog_dir          = NULL;



/* utility functions */
static GtkWidget* gnobog_app_window_create 
                            (GnobogBookmarks* bookmarks);
static Document*  gnobog_app_document_create 
                            (GnobogBookmarks* bookmarks,
			     GtkWidget* document_window);
static void       gnobog_app_real_copy_to_clipboard 
                            (GnobogBookmarks* clipboard, 
			     GnobogBookmarks* source_bookmarks, 
			     GList* list);
static void       gnobog_app_real_paste_from_clipboard 
                            (GnobogBookmarks* clipboard, 
			     GnobogBookmarks* target_bookmarks, 
			     GnobogBookmarksNode anchor_node, 
			     GnobogBookmarksInsertMode insert_mode);
static void       gnobog_app_real_clear_clipboard 
                            (GnobogBookmarks* clipboard);
static gboolean   gnobog_app_real_is_clipboard_empty 
                            (GnobogBookmarks* clipboard);
static Document*  gnobog_app_find_document_from_bookmarks 
                            (GnobogBookmarks* bookmarks);

/* callbacks to 'delete-event' signal */
static gint       gnobog_app_delete_event_cb 
                            (GtkWidget* w, 
			     GdkEventAny* e, 
			     gpointer data);
static gint       gnobog_app_document_window_delete_event_cb 
                            (GtkWidget* w, 
			     GdkEventAny* e, 
			     gpointer data);

/* Callbacks to arborescence signals */
/* TODO : gnobog_app...              */
static void       gnobog_url_launch 
                            (GnobogArborescence *arborescence, 
			     gint row, 
			     gpointer data);
static void       gnobog_app_popup_menu 
                            (GnobogArborescence *arborescence, 
			     GdkEventButton *event, 
			     gpointer data);

static gboolean   gnobog_app_window_focus_cb 
                            (GtkWidget* widget,
			     GdkEventFocus *event,
			     gpointer user_data);

static void       gnobog_app_tree_select_cb 
                            (GtkCTree *ctree,
			     GList *node,
			     gint column,
			     gpointer user_data);

static gboolean   gnobog_app_special_keys_cb 
                            (GtkWidget* widget,
			     GdkEventKey* event,
			     gpointer user_data);

GtkWidget*
gnobog_app_new (void)
{
  GtkWidget*    status;
  GtkWidget*    main_view;
  GtkWidget*    documents_list;
  GtkWidget*    documents_clist;
  GtkWidget*    documents_scrolled_window;
  GtkWidget*    drop_zone;
  GtkWidget*    drop_zone_arborescence;
  GtkWidget*    drop_zone_scrolled_window;
  GnomeDialog*  dialog;
  gchar*        lock_filename;
  gchar*        lock_error;
  gchar*        lock_creating_error;
  gchar*        drop_zone_filename;
  gchar*        drop_zone_loading_error;
  GList*        list;
  gchar*        i18n_titles[2];
  FILE*         fd;

  /* Get user's home and look for gnobog directory */
  home_dir   = g_get_home_dir ();
  g_message ("User Home Directory : %s", home_dir); 
  gnobog_dir = g_strconcat (home_dir, "/.gnobog/", NULL);
  g_message ("Gnobog Working Directory : %s", gnobog_dir); 
  if ( !g_file_exists (gnobog_dir) ) {
    /* Create the directory */
    mkdir (gnobog_dir, 0777);
  }
        
  lock_filename = g_strconcat (gnobog_dir, LOCK_FILENAME, NULL);
  if ( g_file_exists (lock_filename) ) {
    /* Lock exist ! Either another Gnobog is running or Gnobog crashed */
    /* Anyway, for now, just let the users, handles this */
    /* So inform him and exit */
    lock_error = g_strdup_printf (_(LOCK_ERROR), lock_filename);
    dialog = GNOME_DIALOG (gnome_error_dialog (lock_error));
    gnome_dialog_run_and_close (dialog);
    g_free (lock_error);
    /* Gnobog must close now */
    g_free (gnobog_dir);
    g_free (lock_filename);
    return NULL;
  } else {
    /* Create lock file */
    fd = fopen (lock_filename, "w");
    if (fd == NULL) {
      /* Ok, we've got a big problem with lock file */
      /* Tell the user */
      lock_creating_error = g_strdup_printf (_(LOCK_CREATING_ERROR), lock_filename);
      dialog = GNOME_DIALOG (gnome_error_dialog (lock_creating_error));
      gnome_dialog_run_and_close (dialog);
      g_free (lock_creating_error);
      /* Gnobog must close now */
      g_free (gnobog_dir);
      g_free (lock_filename);
      return NULL;
    }
    fclose (fd);
  }
  g_free (lock_filename);

  /* Default storing  zone for quickly dropping bookmarks */
  /* if drop zone file exists load it */
  drop_zone_filename = g_strconcat (gnobog_dir, DROP_ZONE_FILENAME, NULL);
  if ( g_file_exists (drop_zone_filename) ) {
    /* Create bookmarks from file */
    //    drop_zone_bookmarks = GNOBOG_BOOKMARKS (gnobog_bookmarks_new_from_file (drop_zone_filename));
    drop_zone_bookmarks = gnobog_bookmarks_new_from_file (drop_zone_filename);
    if (drop_zone_bookmarks == NULL) {
      /* Ok, we've got a big problem with drop_zone */
      /* Tell the user */
      drop_zone_loading_error = g_strdup_printf (_(DROP_ZONE_LOADING_ERROR), drop_zone_filename);
      dialog = GNOME_DIALOG (gnome_error_dialog (drop_zone_loading_error));
      gnome_dialog_run_and_close (dialog);
      g_free (drop_zone_loading_error);
      /* Gnobog must close now */
      g_free (gnobog_dir);
      g_free (drop_zone_filename);
      return NULL;
    }
    /* Force Storing Zone title (to stay compatible with 0.4.1 and earlier version) */
    g_free (gnobog_bookmarks_get_title (drop_zone_bookmarks));
    gnobog_bookmarks_set_title (drop_zone_bookmarks, _(DROP_ZONE_TITLE));
  } else {
    //    drop_zone_bookmarks = GNOBOG_BOOKMARKS (gnobog_bookmarks_new_with_title ("Storing Zone", ""));
    drop_zone_bookmarks = gnobog_bookmarks_new_with_title (_(DROP_ZONE_TITLE), "");
    gnobog_bookmarks_set_filename (drop_zone_bookmarks, drop_zone_filename);
  }
  g_free (drop_zone_filename);

  /* Create the clipboard */
  manual_clipboard = gnobog_bookmarks_new_with_title (_("Clipboard"), "");

  /*** gnobog-widgets */
  /* main window */
  gnobog_app = gnome_app_new (PACKAGE, _("Gnobog"));

  gtk_window_set_policy (GTK_WINDOW (gnobog_app), FALSE, TRUE, FALSE);
  gtk_window_set_default_size (GTK_WINDOW (gnobog_app), 190, 500);
  gtk_window_set_wmclass (GTK_WINDOW (gnobog_app), "gnobog", "gnobog");

  /* Pixmaps */
  gnobog_pixmaps_create ();

  /* documents list to perform tasks on them */
  documents_list = gtk_frame_new (_("Documents list"));
  gtk_frame_set_shadow_type (GTK_FRAME (documents_list), GTK_SHADOW_IN);
  documents_scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (documents_scrolled_window), 
                                  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  /* i18n stuff */
  i18n_titles[0] = g_strdup_printf ("%s", _(DocumentsTitles[0]));
  i18n_titles[1] = g_strdup_printf ("%s", _(DocumentsTitles[1]));
  documents_clist = gtk_clist_new_with_titles (2, i18n_titles);
  g_free (i18n_titles[0]);
  g_free (i18n_titles[1]);
  gtk_clist_set_column_auto_resize (GTK_CLIST (documents_clist), 0, TRUE);
  documents_view = documents_clist;
  gtk_container_add (GTK_CONTAINER (documents_scrolled_window), 
		     documents_clist);
  gtk_container_add (GTK_CONTAINER (documents_list), 
		     documents_scrolled_window );
        
 /* Build main UI */
  drop_zone = gtk_frame_new (gnobog_bookmarks_get_title (drop_zone_bookmarks));
  gtk_frame_set_shadow_type (GTK_FRAME (drop_zone), GTK_SHADOW_IN);
  drop_zone_arborescence = gnobog_arborescence_new_with_titles (drop_zone_bookmarks,
								2, 
                                                                Titles);
  gnobog_arborescence_set_column_visibility (GNOBOG_ARBORESCENCE (drop_zone_arborescence), 
					     2,
					     FALSE);
  gnobog_arborescence_set_selection_mode (GNOBOG_ARBORESCENCE (drop_zone_arborescence), 
                                          WINDOWS_MODE);        
  gtk_signal_connect (GTK_OBJECT (drop_zone_arborescence), 
		      "popup_menu", 
                      (GtkSignalFunc) gnobog_app_popup_menu, 
		      NULL);
  gtk_signal_connect (GTK_OBJECT (drop_zone_arborescence), 
		      "open_row", 
                      (GtkSignalFunc) gnobog_url_launch, 
		      NULL);
  drop_zone_scrolled_window = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (drop_zone_scrolled_window), 
                                  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (drop_zone_scrolled_window), 
		     drop_zone_arborescence);
  gtk_container_add (GTK_CONTAINER (drop_zone), 
		     drop_zone_scrolled_window);

  /* Ask Drop Zone its contents and insert it into the empty arbo */
  /* FIXME : What's happening if a bookmarks signal appears meanwhile ? (low probability) */
  if ( (list = gnobog_bookmarks_get_list (drop_zone_bookmarks)) != NULL ) {
    gnobog_arborescence_list_insert (GNOBOG_ARBORESCENCE (drop_zone_arborescence),
                                     NULL, 
				     INSERT_INTO, 
				     list, 
				     NULL );
  }
        
  /* Since we have just loaded, a list insert signal is received and all is selected */
  gnobog_arborescence_unselect_all (GNOBOG_ARBORESCENCE (drop_zone_arborescence));

  /* assemble in the main window */
  main_view = gtk_vpaned_new ();
  gtk_paned_add1 (GTK_PANED (main_view), documents_list);
  gtk_paned_add2 (GTK_PANED (main_view), drop_zone);
                
  gnome_app_set_contents (GNOME_APP (gnobog_app), main_view);

  /* Status bar */
  status = gnome_appbar_new (FALSE, TRUE, GNOME_PREFERENCES_NEVER);
  gnome_app_set_statusbar (GNOME_APP(gnobog_app), status);

  /* Menus */
  gnobog_menus_install_menus_and_toolbar (gnobog_app);
  gnobog_popup_menu_set (gnobog_app);

  /* gnobog-widgets ***/

  /*** gnobog-signals */
  gtk_signal_connect (GTK_OBJECT (gnobog_app),
                      "delete_event",
                      GTK_SIGNAL_FUNC (gnobog_app_delete_event_cb),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (gnobog_app),
                      "destroy_event",
                      GTK_SIGNAL_FUNC (gnobog_app_delete_event_cb),
                      NULL);

  /* store a pointer to the view in the window object,
   * and connect focus event to have a means to know the active view
   */
  gtk_object_set_data (GTK_OBJECT (gnobog_app), 
		       "view",
		       drop_zone_arborescence);
  gtk_signal_connect (GTK_OBJECT (gnobog_app), 
		      "focus-in-event", 
                      (GtkSignalFunc) gnobog_app_window_focus_cb, 
		      NULL);
  gtk_signal_connect (GTK_OBJECT (drop_zone_arborescence), 
		      "tree-select-row", 
                      (GtkSignalFunc) gnobog_app_tree_select_cb, 
		      NULL);
  /* gnobog-signals ***/

  return gnobog_app;
}

GtkWidget*  
gnobog_app_get_app (void)
{
  return gnobog_app;
}

GtkWidget*  
gnobog_app_get_active_view (void)
{
  return active_view;
}


void
gnobog_app_document_load (gchar* filename)
{
  GnobogBookmarks*  bookmarks;
  GtkWidget*        document_window;
  Document*         new_document;
  GnomeDialog*      dialog;

  g_return_if_fail (filename != NULL);

  /* Create new bookmarks from filename */
  bookmarks = gnobog_bookmarks_new_from_file (filename);

  /* If load fails, destroy bookmark and inform user */
  if (bookmarks == NULL) {
    dialog = GNOME_DIALOG (gnome_error_dialog (_("Unable to load file")));
    gnome_dialog_run_and_close (dialog);
    return;
  }

  /* Create new Document */
  new_document = g_new (Document, 1);

  /* Create view */
  document_window = gnobog_app_window_create (bookmarks);

  /* Create document */
  new_document    = gnobog_app_document_create (bookmarks, document_window);

  /* connect to close signal */
  gtk_signal_connect (GTK_OBJECT (document_window),
                      "delete_event",
                      GTK_SIGNAL_FUNC (gnobog_app_document_window_delete_event_cb),
                      new_document); /* document associated to know when all views are closed */
  gtk_signal_connect (GTK_OBJECT (document_window),
                      "destroy_event",
                      GTK_SIGNAL_FUNC (gnobog_app_document_window_delete_event_cb),
                      new_document); /* document associated to know when all views are closed */

  gtk_widget_show_all (document_window);
                        
  return;
}

void
gnobog_app_document_new (void)
{
  GnobogBookmarks*  bookmarks;
  GtkWidget*        document_window;
  Document*         new_document;

  bookmarks       =  gnobog_bookmarks_new ();

  document_window = gnobog_app_window_create (bookmarks);

  new_document    = gnobog_app_document_create (bookmarks, document_window);

  /* connect to close signal */
  gtk_signal_connect (GTK_OBJECT (document_window),
                      "delete_event",
                      GTK_SIGNAL_FUNC (gnobog_app_document_window_delete_event_cb),
                      new_document); /* document associated to know when all views are closed */


  /* Display the document */
  gtk_widget_show_all (document_window);

  return;
}

void
gnobog_app_document_new_from_selection (void)
{
  GnobogBookmarks*  bookmarks;
  GtkWidget*        document_window;
  Document*         new_document;

  bookmarks       =  gnobog_bookmarks_new ();

  gnobog_app_paste_from_clipboard (bookmarks,
                                   NULL, /* root */
                                   INSERT_DEFAULT_MODE);

  document_window = gnobog_app_window_create (bookmarks);

  new_document    = gnobog_app_document_create (bookmarks, document_window);

  /* connect to close signal */
  gtk_signal_connect (GTK_OBJECT (document_window),
                      "delete_event",
                      GTK_SIGNAL_FUNC (gnobog_app_document_window_delete_event_cb),
                      new_document); /* document associated to know when all views are closed */


  /* Display the document */
  gtk_widget_show_all (document_window);

  return;
}

void
gnobog_app_document_window_new (GnobogBookmarks* bookmarks)
{
  GtkWidget*  document_window;
  gchar*       document_info;
  gint        row;
  Document*   document;

  g_return_if_fail (bookmarks != NULL);

  document_window = gnobog_app_window_create (bookmarks);
        
  /* find the document pointer */
  /* but NOT for the drop_zone */
  /* TODO : drop_zone should *not* be a special bookmarks object, 
   *        but a bookmarks object with certain flags values
   */
  if ( bookmarks != drop_zone_bookmarks ) {
    document = gnobog_app_find_document_from_bookmarks (bookmarks);
        
    /* Update documents list and display */
    document->windows_list = g_list_prepend (document->windows_list, document_window);
    document_info = g_strdup_printf ("%d", g_list_length (document->windows_list));
    row = gtk_clist_find_row_from_data (GTK_CLIST (documents_view), document);
    gtk_clist_set_text (GTK_CLIST (documents_view), row, 1, document_info);
    g_free (document_info);

    /* connect signals */
    gtk_signal_connect(GTK_OBJECT (document_window),
                       "delete_event",
                       GTK_SIGNAL_FUNC (gnobog_app_document_window_delete_event_cb),
                       document);
  }
        
  gtk_widget_show_all (document_window);
                        
  return;
}

static Document*
gnobog_app_document_create (GnobogBookmarks* bookmarks,
			    GtkWidget* document_window)
{
  Document*  new_document;
  gchar*     document_info[2];
  gint       row;

  new_document = g_new (Document, 1);

  /* Store and display document information */
  new_document->bookmarks    = bookmarks;
  new_document->windows_list = g_list_prepend (NULL, document_window);
  gnobog_app_documents_list  = g_list_append (gnobog_app_documents_list, 
					     new_document);
  /* LEAK :  document_info[0] = g_strdup (gnobog_bookmarks_file_get_name (bookmarks));
   * Actually it's a leak if we don't free data after it's copied into the clist.
   * But copying the string locally here is not really needed.
   */
  document_info[0] = gnobog_bookmarks_get_filename (bookmarks);
  /* If document is new, put something instead of the filename */
  if (document_info[0] == NULL) {
    document_info[0] = _("Not yet attached to a file");
  }
  document_info[1] = "1";
  row = gtk_clist_append (GTK_CLIST (documents_view), document_info);
  gtk_clist_set_row_data (GTK_CLIST (documents_view), row, new_document);

  return new_document;
}

static GtkWidget*
gnobog_app_window_create (GnobogBookmarks* bookmarks)
{
  GtkWidget*  document_arborescence;
  GtkWidget*  document_window;
  GtkWidget*  scrolled_window;
  GtkWidget*  frame;
  GtkWidget*  popup;
  gchar*      window_title;
  gchar*      filename;
  GList*      list;

  g_return_val_if_fail (bookmarks != NULL, NULL);

  /* Create a new arborescence pointing to the same bookmarks */
  document_arborescence = gnobog_arborescence_new_with_titles (bookmarks, 2, Titles);
  gnobog_arborescence_set_column_visibility (GNOBOG_ARBORESCENCE (document_arborescence), 
					     2,
					     FALSE);
  gnobog_arborescence_set_selection_mode (GNOBOG_ARBORESCENCE (document_arborescence), 
                                          WINDOWS_MODE);        
        
  /* ask bookmarks its contents and insert it into the empty arbo */
  /* FIXME : What's happening if a bookmarks signal appears meanwhile ? (low probability) */
  if ( (list = gnobog_bookmarks_get_list (bookmarks)) != NULL ) {
    gnobog_arborescence_list_insert (GNOBOG_ARBORESCENCE (document_arborescence),
                                     NULL, 
				     INSERT_INTO, 
				     list, 
				     NULL );
  }
        
  filename = gnobog_bookmarks_get_filename (bookmarks);
  if (filename == NULL) {
    window_title = g_strdup_printf (_("Gnobog: -(Not yet attached to a file)-"));
  } else {
    //    frame = gtk_frame_new (g_basename (filename)); 
    window_title = g_strdup_printf ("Gnobog: %s", filename);
  }

  /* Create window and decoration */
  document_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  //  gtk_window_set_wmclass (GTK_WINDOW (gnobog_app), "gnobog", "gnobog");
  gtk_window_set_policy (GTK_WINDOW (document_window), FALSE, TRUE, FALSE);
  gtk_window_set_default_size (GTK_WINDOW (document_window), 190, 500);
  gtk_window_set_title (GTK_WINDOW (document_window), window_title);
  g_free (window_title);
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window), 
                                  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);

  //    frame = gtk_frame_new (g_basename (filename)); 
  frame = gtk_frame_new (gnobog_bookmarks_get_title (bookmarks));

  /* TODO: We should remember document state... */
  gtk_ctree_expand_to_depth (GTK_CTREE (document_arborescence), NULL, 3);
  gnobog_arborescence_unselect_all (GNOBOG_ARBORESCENCE (document_arborescence));
        
  /* Connect arborescence signals and pass the document as data to popup menu */
  gtk_signal_connect (GTK_OBJECT (document_arborescence), 
		      "popup_menu", 
                      (GtkSignalFunc) gnobog_app_popup_menu, 
		      NULL);
  gtk_signal_connect (GTK_OBJECT (document_arborescence), 
		      "open_row", 
                      (GtkSignalFunc) gnobog_url_launch, 
		      NULL);
  gtk_signal_connect (GTK_OBJECT (document_window), 
		      "focus-in-event", 
                      (GtkSignalFunc) gnobog_app_window_focus_cb, 
		      NULL);
  gtk_signal_connect (GTK_OBJECT (document_arborescence), 
		      "tree-select-row", 
                      (GtkSignalFunc) gnobog_app_tree_select_cb, 
		      NULL);
  gtk_signal_connect (GTK_OBJECT (document_arborescence), 
		      "key_press_event",
		      GTK_SIGNAL_FUNC (gnobog_app_special_keys_cb),
		      NULL);

  /* store a pointer to the view in the window */
  gtk_object_set_data (GTK_OBJECT (document_window), 
		       "view",
		       document_arborescence);

  /* Add popup menu keyboard accelerators */
  popup = gnobog_popup_menu_get_popup ();
  gtk_window_add_accel_group (GTK_WINDOW (document_window),
    			      gnome_popup_menu_get_accel_group (GTK_MENU (popup)));
  //  gtk_accel_group_attach (GNOME_APP (gnobog_app)->accel_group, 
  //			  GTK_OBJECT (document_window));
  //  gtk_window_add_accel_group (GTK_WINDOW (document_window), 
  //		      GNOME_APP (gnobog_app)->accel_group);

  /* Assemble everything together */
  gtk_container_add (GTK_CONTAINER (scrolled_window), document_arborescence);
  gtk_container_add (GTK_CONTAINER (frame), scrolled_window);
  gtk_container_add (GTK_CONTAINER (document_window), frame);

  return document_window;
}

void
gnobog_app_document_save (GnobogBookmarks* bookmarks)
{
  GnomeDialog*  dialog;

  if (gnobog_bookmarks_get_filename (bookmarks) != NULL) {
    /* If save fails, inform user */
    if (gnobog_bookmarks_save_file (bookmarks) == FALSE) {
      dialog = GNOME_DIALOG (gnome_error_dialog (_("Unable to save file")));
      gnome_dialog_run_and_close (dialog);
      /* Go in "Save as" mode */
      gnobog_app_document_save_as (bookmarks);
    }
  } else {
    gnobog_app_document_save_as (bookmarks);
  }
}

void
gnobog_app_document_save_as (GnobogBookmarks* bookmarks)
{
  gchar*        selected_filename;
  GList*        iterator;
  Document*     document;
  gchar*        window_title;
  GtkFrame*     frame;
  GtkWindow*    window;
  GnomeDialog*  dialog;
  gboolean      success;
  gint          row;

  g_return_if_fail (bookmarks != NULL);

  /* This flag is set to TRUE when save is successful */
  success = FALSE;

  do {
    /* Ask user the filename via a nice UI */
    selected_filename = gnobog_ui_save_file (gnobog_bookmarks_get_filename (bookmarks));
        
    /* If filename is NULL, user chose to "Cancel" the save */
    if ( selected_filename == NULL ) {
      g_message ("Save Cancel");
      return;
    }

    /* Save with that filename */

    /* Blah blah... */
    g_message ("Selected file : %s", selected_filename);
        
    /* If save fails, inform user */
    success = gnobog_bookmarks_save_as_file
      (bookmarks, selected_filename,
       gnobog_bookmarks_get_file_format (bookmarks));
    if (success == FALSE) {
      dialog = GNOME_DIALOG (gnome_error_dialog (_("Unable to save file")));
      gnome_dialog_run_and_close (dialog);
      g_free (selected_filename);
      /* Reask a filename to user */
      continue;
    }
     
    /* We must change the window title and frame label ! */
    /* In all views of the document */
    /* sanity check, we should have an api for that */
    g_message ("Chosen name : %s, actual name: %s",
               selected_filename,
               gnobog_bookmarks_get_filename (bookmarks));

    if ( bookmarks != drop_zone_bookmarks ) {
      document = gnobog_app_find_document_from_bookmarks (bookmarks);
      if ( document != NULL ) {   /* i.e. if we're not called during a close */
        window_title = g_strdup_printf ("Gnobog: %s", selected_filename);
        iterator = document->windows_list;
        while ( iterator != NULL ) {  /* for each window */
          window = iterator->data;
          gtk_window_set_title (window, window_title);
          frame = GTK_FRAME ((gtk_container_children (GTK_CONTAINER (window)))->data);
          gtk_frame_set_label (frame, g_basename (selected_filename));
          iterator = g_list_next (iterator);              
        } 
        g_free (window_title);
        /* Update filename in document list */
        row = gtk_clist_find_row_from_data (GTK_CLIST (documents_view), document);
        gtk_clist_set_text (GTK_CLIST (documents_view), row, 0, selected_filename);
      }
    }
    g_free (selected_filename);
  } while (success == FALSE);
}


void
gnobog_app_document_save_copy_as (GnobogBookmarks* bookmarks)
{
  gchar*    selected_filename;
  gboolean  success;
  GnomeDialog*  dialog;

  g_return_if_fail (bookmarks != NULL);

  /* This flag is set to TRUE when save is successful */
  success = FALSE;

  do {
    /* Ask user the filename via a nice UI */
    selected_filename = gnobog_ui_save_file (gnobog_bookmarks_get_filename (bookmarks));
        
    /* If filename is NULL, user chose to "Cancel" the save */
    if ( selected_filename == NULL ) {
      g_message ("Save Cancel");
      return;
    }

    /* Blah blah... */
    g_message ("Selected file : %s", selected_filename);
        
    /* Save with that filename */
    /* If save fails, inform user */
    success = gnobog_bookmarks_save_copy_as_file
                    (bookmarks,
                     selected_filename,
                     gnobog_bookmarks_get_file_format (bookmarks));
    if (success == FALSE) {
      dialog = GNOME_DIALOG (gnome_error_dialog (_("Unable to save file")));
      gnome_dialog_run_and_close (dialog);
      continue;
    }    
  } while (success == FALSE);
 
  /* We must *not* change the window title and frame label ! */
}

/* Close one window of a document, if it's the last window
 * then close the document
 */
void
gnobog_app_document_close_window (GtkWidget* window, 
                                  gpointer data /* has been set with connect */ )
{
  Document*  document;
  gchar*     document_info;
  gint       row;
        
  g_return_if_fail (window != NULL);
  g_return_if_fail (data != NULL);

  document = (Document *) data;
        
  /* Remove window in windows list and in documents list display */
  document->windows_list = g_list_remove (document->windows_list, window);
  document_info = g_strdup_printf ("%d", g_list_length (document->windows_list));
  row = gtk_clist_find_row_from_data (GTK_CLIST (documents_view), document);
  gtk_clist_set_text (GTK_CLIST (documents_view), row, 1, document_info);
  g_free (document_info);
        
  /* Is there any window left ? */
  if ( document->windows_list == NULL ) {
    gnobog_app_document_close (document);
  }
        
  gtk_widget_destroy (window);
}

/* Close a document and all its windows
 * Is used by the popup on the documents list 
 */
void
gnobog_app_document_close (gpointer data /* data associated with the row */)
{
  Document*  document;
  GList*     windows_list;
  gint       row;
        
  g_return_if_fail (data != NULL);

  document = (Document *) data;
  windows_list = document->windows_list;

  /* Close all windows if any */
  while ( windows_list != NULL ) {
    gnobog_app_document_close_window (windows_list->data, document);
    windows_list = document->windows_list;
  }
        
  /* Remove the document from document list if necessary */
  gnobog_app_documents_list = g_list_remove (gnobog_app_documents_list, document);
  row = gtk_clist_find_row_from_data (GTK_CLIST (documents_view), document);
  if ( row >= 0 ) {
    gtk_clist_remove (GTK_CLIST (documents_view), row);

    /* If document was modified, ask for saving it */
    if ( gnobog_bookmarks_is_modified (document->bookmarks) ) {
      gnobog_app_document_save_as (document->bookmarks);
    }
    gnobog_bookmarks_destroy (GTK_OBJECT (document->bookmarks));
  }
  //  g_free (document);
}

/* Close all documents and their windows */
void
gnobog_app_document_close_all (void)
{
  GList*  iterator;
        
  iterator = gnobog_app_documents_list;
  while ( iterator != NULL ) {
    gnobog_app_document_close (iterator->data);
    iterator = gnobog_app_documents_list;               
  }     
        
  g_list_free (gnobog_app_documents_list);

}

void
gnobog_app_quit (GtkWidget* gnobog_app /* the main window */)
{
  GnomeDialog*  dialog;
  gchar*        selected_filename;
  gboolean      success;
  gchar*        lock_filename;
        
  g_return_if_fail (gnobog_app != NULL);
  g_return_if_fail (GNOME_IS_APP (gnobog_app));

  /* free clipboards */
  gnobog_bookmarks_destroy (GTK_OBJECT (manual_clipboard));
        
  /* ask for closing all documents, save them and free memory */
  gnobog_app_document_close_all ();
  /* TODO: check if we have freed all infos used by the document list */
      
           
  /* SAVE the drop zone and free memory */
  success = gnobog_bookmarks_save_file (drop_zone_bookmarks);
  if (success == FALSE) {
    dialog = GNOME_DIALOG (gnome_error_dialog (_("Unable to save Storing Zone")));
    gnome_dialog_run_and_close (dialog);
  }    
  while (success == FALSE) {
    /* Ask user the filename via a nice UI */
    selected_filename = gnobog_ui_save_file
                              (gnobog_bookmarks_get_filename (drop_zone_bookmarks));
        
    /* If filename is NULL, user chose to "Cancel" the save */
    if ( selected_filename == NULL ) {
      g_message ("Save Cancel");
      break;
    }

    /* Blah blah... */
    g_message ("Selected file : %s", selected_filename);
        
    /* Save with that filename */
    /* If save fails, inform user */
    success = gnobog_bookmarks_save_as_file
                    (drop_zone_bookmarks,
                     selected_filename,
                     gnobog_bookmarks_get_file_format (drop_zone_bookmarks));
    if (success == FALSE) {
      dialog = GNOME_DIALOG (gnome_error_dialog (_("Unable to save file")));
      gnome_dialog_run_and_close (dialog);
      continue;
    }    

  }
  gnobog_bookmarks_destroy (GTK_OBJECT (drop_zone_bookmarks));

  /* Remove lock file */
  lock_filename = g_strconcat (gnobog_dir, LOCK_FILENAME, NULL);
  (void) remove (lock_filename);
  g_free (lock_filename);


  /* Free other things ! */
  g_free (gnobog_dir);
  g_free (home_dir);
                
  /* Destroy main widget */
  gtk_widget_destroy (gnobog_app);

  /* Free Pixmaps */
  gnobog_pixmaps_delete ();
        
  gtk_main_quit ();
}


/* could use a hash table...*/
static Document*
gnobog_app_find_document_from_bookmarks (GnobogBookmarks* bookmarks)
{
  GList*     iterator;
  Document*  document;

  g_return_val_if_fail (bookmarks != NULL, NULL);
        
  iterator = gnobog_app_documents_list;
  while ( (iterator != NULL) &&
          ((Document*)(iterator->data))->bookmarks != bookmarks ) {
    iterator = g_list_next (gnobog_app_documents_list);
  }
  if ( iterator != NULL ) {
    document = (Document *)(iterator->data);
  } else {
    document = NULL;
  }

  return document;
}


/* TODO: it's a mess, use abstract stuctures/objects and allow 
 *       a much more flexible behaviour
 */
void    
gnobog_app_toggle_split_view (GnobogArborescence* arborescence)
{
  GtkWidget*  view_container; /* currently a frame */
  GList*      view_children; /* a pane or directly a scrolled window */
  GtkWidget*  view_child;
  GtkWidget*  hpaned;
  GtkWidget*  current_scroll_win;
  GtkWidget*  other_scroll_win;
  GtkWidget*  other_arbo;
  GList*      list;
        
  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (arborescence));

  /* get current scrolled window pointer */
  current_scroll_win = gtk_widget_get_ancestor (GTK_WIDGET (arborescence), 
                                                GTK_TYPE_SCROLLED_WINDOW);
  /* get the view container of this arborescence */
  view_container = gtk_widget_get_ancestor (GTK_WIDGET (arborescence), 
                                            GTK_TYPE_FRAME);
  /* look for its child */
  view_children = gtk_container_children (GTK_CONTAINER (view_container));
  view_child = GTK_WIDGET (view_children->data);

  if ( GTK_IS_HPANED (view_child) ) {

    /* remove current scroll win from hpaned */
    gtk_widget_ref (current_scroll_win);
    gtk_container_remove (GTK_CONTAINER (view_child), current_scroll_win);

    /* remove and destroy paned view */
    gtk_widget_ref (view_child);
    gtk_container_remove (GTK_CONTAINER (view_container), view_child);
    gtk_widget_destroy (view_child);
    gtk_widget_unref (view_child);

    /* add current scroll win to frame */
    gtk_container_add (GTK_CONTAINER (view_container), current_scroll_win);
    gtk_widget_unref (current_scroll_win);

    /* quick hack to allow the popup to be in sync */
    /* We could use this kind of data to know if we are in split or not,
     * but what we really need is to rewrite the whole thing, see TODO.
     */
    gtk_object_set_data (GTK_OBJECT (arborescence), 
			 "splitted window",
			 GINT_TO_POINTER(FALSE));
  } else {
    /* create paned view */
    hpaned = gtk_hpaned_new ();

    /* create and add other scroll win and arbo */
    other_scroll_win = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (other_scroll_win),
                                   GTK_POLICY_AUTOMATIC, 
				   GTK_POLICY_AUTOMATIC);
    other_arbo = gnobog_arborescence_new_with_titles (arborescence->bookmarks_watched, 
                                                      2, 
						      Titles);
    gnobog_arborescence_set_column_visibility (GNOBOG_ARBORESCENCE (other_arbo), 2, FALSE);
    gnobog_arborescence_set_selection_mode 
                       (GNOBOG_ARBORESCENCE (other_arbo),
			gnobog_arborescence_get_selection_mode (arborescence));  
    gtk_clist_set_column_width(GTK_CLIST (other_arbo), 0, 200);
    gtk_clist_set_column_width(GTK_CLIST (other_arbo), 1, 100);

    /* ask bookmarks its contents and insert it into the empty arbo */
    if ( (list = gnobog_bookmarks_get_list (arborescence->bookmarks_watched)) != NULL ) {
      gnobog_arborescence_list_insert (GNOBOG_ARBORESCENCE (other_arbo),
                                       NULL, 
				       INSERT_INTO, 
				       list, 
				       NULL);
      gnobog_arborescence_unselect_all (GNOBOG_ARBORESCENCE (other_arbo));
    }

    /* connect signals */
    gtk_signal_connect (GTK_OBJECT (other_arbo), 
			"popup_menu", 
                        (GtkSignalFunc) gnobog_app_popup_menu, 
			NULL);
    gtk_signal_connect (GTK_OBJECT (other_arbo), 
			"open_row", 
                        (GtkSignalFunc) gnobog_url_launch, 
			NULL);
    gtk_signal_connect (GTK_OBJECT (other_arbo), 
			"tree-select-row", 
			(GtkSignalFunc) gnobog_app_tree_select_cb, 
			NULL);

    /* quick hack to allow the popup to be in sync */
    gtk_object_set_data (GTK_OBJECT (arborescence), 
			 "splitted window",
			 GINT_TO_POINTER(TRUE));
    gtk_object_set_data (GTK_OBJECT (other_arbo), 
			 "splitted window",
			 GINT_TO_POINTER(TRUE));


    /* add arbo to scroll win */
    gtk_container_add (GTK_CONTAINER (other_scroll_win), other_arbo);

    /* assemble all together */
    gtk_widget_ref (current_scroll_win);
    gtk_container_remove (GTK_CONTAINER (view_container), current_scroll_win);
    gtk_paned_pack1 (GTK_PANED (hpaned), current_scroll_win, TRUE, FALSE);
    gtk_widget_unref (current_scroll_win);
    gtk_paned_pack2 (GTK_PANED (hpaned), other_scroll_win, TRUE, FALSE);
    gtk_container_add (GTK_CONTAINER (view_container), hpaned);
    gtk_widget_show_all (hpaned);
  }

}


void    
gnobog_app_toggle_selection_all (void)
{

}










/**************************************************************************/
/****  Clipboards management                                             ***/
/**************************************************************************/

static void
gnobog_app_real_copy_to_clipboard (GnobogBookmarks* clipboard, 
                                   GnobogBookmarks* source_bookmarks,
                                   GList* bookmarks_list)
{
  //  g_return_if_fail (bookmarks_list != NULL);
        
  gnobog_bookmarks_clear (clipboard);
        
  /* Duplicate nodes from bookmarks to clipboard */
  gnobog_bookmarks_list_copy_from (clipboard, 
				   source_bookmarks, 
                                   NULL, 
				   INSERT_INTO, 
				   bookmarks_list);
                
}

static void     
gnobog_app_real_paste_from_clipboard (GnobogBookmarks* clipboard,
                                      GnobogBookmarks* target_bookmarks,
                                      GnobogBookmarksNode anchor_node,
                                      GnobogBookmarksInsertMode insert_mode)
{
  gnobog_bookmarks_list_copy_from (target_bookmarks, 
				   clipboard, 
                                   anchor_node, 
				   insert_mode,
                                   gnobog_bookmarks_get_list (clipboard));
}

static void
gnobog_app_real_clear_clipboard (GnobogBookmarks* clipboard)
{
  gnobog_bookmarks_clear (clipboard);
}

static gboolean
gnobog_app_real_is_clipboard_empty (GnobogBookmarks* clipboard)
{
  return ( NULL == gnobog_bookmarks_get_list (clipboard) );
}


void    
gnobog_app_copy_to_clipboard (GnobogBookmarks* source_bookmarks, 
			      GList* bookmarks_list)
{
  gnobog_app_real_copy_to_clipboard (manual_clipboard, 
				     source_bookmarks, 
				     bookmarks_list);
}

void    
gnobog_app_paste_from_clipboard (GnobogBookmarks* target_bookmarks,
                                 GnobogBookmarksNode anchor_node,
                                 GnobogBookmarksInsertMode insert_mode)
{
  gnobog_app_real_paste_from_clipboard (manual_clipboard, 
					target_bookmarks, 
                                        anchor_node, 
					insert_mode);
}

void
gnobog_app_clear_clipboard (void)
{
  gnobog_app_real_clear_clipboard (manual_clipboard);
}

gboolean
gnobog_app_is_clipboard_empty (void)
{
  return gnobog_app_real_is_clipboard_empty (manual_clipboard);
}








/**************************************************************************/
/****  Callbacks                                                        ***/
/**************************************************************************/

static gint
gnobog_app_document_window_delete_event_cb (GtkWidget* window, 
					    GdkEventAny* e, 
					    gpointer data)
{
  gnobog_app_document_close_window (window, data);
        
  /* we have already destroyed the window */
  return TRUE;
}


static gint
gnobog_app_delete_event_cb (GtkWidget* window, 
			    GdkEventAny* e, 
			    gpointer data)
{
  gnobog_app_quit (window);

  /* Prevent the window's destruction, since we destroyed it
   * ourselves with gnobog_app_quit()
   */
  return TRUE;
}


static gboolean    
gnobog_app_window_focus_cb (GtkWidget* widget,
			    GdkEventFocus *event,
			    gpointer user_data)
{
  GtkWidget*  new_active_view = NULL;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_WINDOW (widget), FALSE);

  //  g_message ("FOCUS");
  new_active_view = gtk_object_get_data (GTK_OBJECT (widget),
					 "view");

  if ( NULL != new_active_view ) {
    active_view = new_active_view;
  } else {
    g_message ("!! No view associated with this window !!");
  }

  return FALSE;
}

static void        
gnobog_app_tree_select_cb (GtkCTree* ctree,
			   GList* node,
			   gint column,
			   gpointer user_data)
{
  g_return_if_fail (NULL != ctree);
  g_return_if_fail (GNOBOG_IS_ARBORESCENCE (ctree));

  active_view = GTK_WIDGET (ctree);

  //  g_message ("TREE SELECT");
}

static gboolean        
gnobog_app_special_keys_cb (GtkWidget* widget,
			    GdkEventKey* event,
			    gpointer user_data)
{
  //gchar*      gdk_keyval_name                 (guint keyval);
  //guint       gdk_keyval_from_name            (const gchar *keyval_name);
  
  if ( event->keyval == gdk_keyval_from_name ("Delete") &&
       event->type == GDK_KEY_PRESS ) {
    if ( GNOBOG_IS_ARBORESCENCE (widget) ) {
      g_message ("Delete");
      gnobog_popup_cb_cut (NULL, NULL);
    }
  } 

  return FALSE; /* let other handlers do their job (default handlers will anyway) */
}



/**************************************************************************/
/****  Test Handlers                                                    ***/
/**************************************************************************/

static void
gnobog_url_launch (GnobogArborescence* arborescence, 
		   gint row, 
		   gpointer data)
{
  GnobogBookmarksNode   bookmarks_node;

  g_return_if_fail (arborescence != NULL);
  g_return_if_fail (row >= 0);

  bookmarks_node = gnobog_arborescence_get_bookmarks_node_from_row (arborescence, row);
  /* TODO: should not be in bookmarks, it's an higher functionnality */
  /* TODO: Default Browser should be configurable */

  gnobog_bookmarks_node_view_in_browser (arborescence->bookmarks_watched,
					 bookmarks_node,
					 DEFAULT_BROWSER,
					 FALSE);
}

static void
gnobog_app_popup_menu (GnobogArborescence* arborescence, 
		       GdkEventButton* event, 
		       gpointer data)
{
  g_return_if_fail (arborescence != NULL);

  /* TODO: use flags for some bookmarks properties (what's allowed, forbidden...) */
  gnobog_popup_menu_do_popup (GTK_WIDGET(arborescence), 
			      event, 
			      drop_zone_bookmarks /* bookmarks where save as is not allowed */);
}


