/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "ui_parent.h"
#include "ui_child.h"

/**/

#define CLOSE_BUTTON_SIZE (16)

enum
{
	PROP_0,
	
	PROP_TEXT,
	PROP_STOCK,
	PROP_HIGHLIGHT,
	PROP_CLOSE
};

static gpointer parent_class = NULL;

static void giftui_child_init (GiftuiChild *child);
static void giftui_child_class_init (GiftuiChildClass *class);
static void giftui_child_get_property (GObject *object, guint param_id,
				       GValue *value, GParamSpec *pspec);
static void giftui_child_set_property (GObject *object, guint param_id,
				       const GValue *value, GParamSpec *pspec);
static void giftui_child_close_clicked (GtkWidget *child);
static void giftui_child_dispose (GObject *object);

/**/

GType
giftui_child_get_type (void)
{
	static GType child_type = 0;
	
	if (!child_type)
	{
		static const GTypeInfo child_type_info =
			{
				sizeof (GiftuiChildClass),
				NULL,		/* base_init */
				NULL,		/* base_finalize */
				(GClassInitFunc) giftui_child_class_init,
				NULL,		/* class_finalize */
				NULL,		/* class_data */
				sizeof (GiftuiChild),
				0,		/* n_preallocs */
				(GInstanceInitFunc) giftui_child_init,
				NULL
			};
		
		child_type = g_type_register_static (GTK_TYPE_VBOX, "GiftuiChild",
						     &child_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	
	return child_type;
}

static void
giftui_child_class_init (GiftuiChildClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	
	parent_class = g_type_class_peek_parent (class);
	
	object_class->get_property = giftui_child_get_property;
	object_class->set_property = giftui_child_set_property;
	
	object_class->dispose = giftui_child_dispose;
	
	g_object_class_install_property (object_class,
					 PROP_TEXT,
					 g_param_spec_string ("label",
							      "Label",
							      "Label of the tab",
							      NULL,
							      G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class,
					 PROP_STOCK,
					 g_param_spec_string ("stock",
							      "Stock id",
							      "Stock icon of the tab",
							      NULL,
							      G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class,
					 PROP_HIGHLIGHT,
					 g_param_spec_boolean ("highlight",
							       "Highlight",
							       "Highlight the tab",
							       FALSE,
							       G_PARAM_READWRITE));
	
	g_object_class_install_property (object_class,
					 PROP_CLOSE,
					 g_param_spec_boolean ("close",
							       "Close",
							       "Tab can be closed",
							       TRUE,
							       G_PARAM_READWRITE));
	
	return;
}

static void
giftui_child_init (GiftuiChild *child)
{
	GtkRequisition size;
	GtkWidget *image;
	GtkRcStyle *rcstyle;
	
	child->parent = NULL;
	
	child->highlight = FALSE;
	child->close = TRUE;
	
	/* Tab label etc.. */
	child->hbox = gtk_hbox_new (FALSE, 1);
	g_object_ref (G_OBJECT (child->hbox));
	gtk_object_sink (GTK_OBJECT (child->hbox));
	
	child->icon = gtk_image_new_from_stock (NULL, GTK_ICON_SIZE_MENU);
	gtk_box_pack_start (GTK_BOX (child->hbox), child->icon, FALSE, FALSE, 5);
	
	child->label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (child->hbox), child->label, FALSE, FALSE, 5);
	
	child->close_button = gtk_button_new ();
	gtk_button_set_relief (GTK_BUTTON (child->close_button), GTK_RELIEF_NONE);
	
	rcstyle = gtk_rc_style_new ();
	rcstyle->xthickness = rcstyle->ythickness = 0;
	gtk_widget_modify_style (child->close_button, rcstyle);
	gtk_rc_style_unref (rcstyle),
	
	image = gtk_image_new_from_stock (GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);
	gtk_widget_size_request (image, &size);
	gtk_widget_size_request (image, &size);
	
	gtk_widget_set_size_request (child->close_button, CLOSE_BUTTON_SIZE, CLOSE_BUTTON_SIZE);
	gtk_container_add (GTK_CONTAINER (child->close_button), image);
	gtk_box_pack_start (GTK_BOX (child->hbox), child->close_button, FALSE, FALSE, 0);
	g_signal_connect_swapped (child->close_button, "clicked",
				  G_CALLBACK (giftui_child_close_clicked), GTK_WIDGET (child));
	
	/* Menu label */
	child->menu_label = gtk_label_new (NULL);
	
	g_object_ref (G_OBJECT (child->menu_label));
	gtk_object_sink (GTK_OBJECT (child->menu_label));
	
	gtk_widget_show_all (child->hbox);
	
	return;
}

static void
giftui_child_close_clicked (GtkWidget *child)
{
	giftui_parent_remove_child (GIFTUI_PARENT (GIFTUI_CHILD (child)->parent), child);
	gtk_widget_destroy (child);
	
	return;
}

static void
giftui_child_dispose (GObject *object)
{
	GiftuiChild *child = GIFTUI_CHILD (object);
	
	/* Parent => destroy signal from parent */
	if (child->hbox)
	{
		g_object_unref (G_OBJECT (child->hbox));
		child->hbox = NULL;
	}
	
	if (child->menu_label)
	{
		g_object_unref (G_OBJECT (child->menu_label));
		child->menu_label = NULL;
	}
	
	G_OBJECT_CLASS (parent_class)->dispose (object);
	
	return;
}

static void
giftui_child_get_property (GObject *object, guint param_id,
			   GValue *value, GParamSpec *pspec)
{
	GiftuiChild *child = GIFTUI_CHILD (object);
	
	switch (param_id)
	{
	case PROP_TEXT:
		g_object_get_property (G_OBJECT (child->label), "label", value);
		break;
		
	case PROP_STOCK:
		g_object_get_property (G_OBJECT (child->icon), "stock", value);
		break;
		
	case PROP_HIGHLIGHT:
		g_value_set_boolean (value, child->highlight);
		break;
		
	case PROP_CLOSE:
		g_value_set_boolean (value, child->close);
		break;
		
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
	
	return;
}

void
giftui_child_set_text (GiftuiChild *child, const gchar *text)
{
	g_return_if_fail (GIFTUI_IS_CHILD (child));
	
	gtk_label_set_text (GTK_LABEL (child->label), text);
	gtk_label_set_text (GTK_LABEL (child->menu_label), text);
	
	return;
}

void
giftui_child_set_image_from_stock (GiftuiChild *child, const gchar *stock_id)
{
	g_return_if_fail (GIFTUI_IS_CHILD (child));
	
	gtk_image_set_from_stock (GTK_IMAGE (child->icon), stock_id, GTK_ICON_SIZE_MENU);
	
	return;
}

void
giftui_child_set_highlight (GiftuiChild *child, gboolean highlight)
{
	GdkColor c;
	GiftuiChild *current;
	
	g_return_if_fail (GIFTUI_IS_CHILD (child));
	
	/* Nothing to do. */
	if (highlight == child->highlight)
		return;
	
	/* No highlight for the current tab. */
	if (child->parent)
		current = GIFTUI_CHILD (giftui_parent_current_child (GIFTUI_PARENT (child->parent)));
	else
		current = NULL;
	
	if (current == child)
		return;
	
	c.red = 65535;
	c.green = 0;
	c.blue = 0;
	
	child->highlight = highlight;
	
	if (highlight)
		gtk_widget_modify_fg (child->label, GTK_STATE_ACTIVE, &c);
	else
		gtk_widget_modify_fg (child->label, GTK_STATE_ACTIVE, NULL);
	
	return;
}

void
giftui_child_set_close (GiftuiChild *child, gboolean close)
{
	g_return_if_fail (GIFTUI_IS_CHILD (child));
	
	child->close = close;
	
	if (close)
		gtk_widget_show (child->close_button);
	else
		gtk_widget_hide (child->close_button);
	
	return;
}

static void
giftui_child_set_property (GObject *object, guint param_id,
			   const GValue *value, GParamSpec *pspec)
{
	GiftuiChild *child = GIFTUI_CHILD (object);
	
	switch (param_id)
	{
	case PROP_TEXT:
		g_object_set_property (G_OBJECT (child->label), "label", value);
		g_object_set_property (G_OBJECT (child->menu_label), "label", value);
		break;
		
	case PROP_STOCK:
		g_object_set_property (G_OBJECT (child->icon), "stock", value);
		break;
		
	case PROP_HIGHLIGHT:
		giftui_child_set_highlight (child, g_value_get_boolean (value));
		break;
		
	case PROP_CLOSE:
		giftui_child_set_close (child, g_value_get_boolean (value));
		break;
		
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
	
	return;
}
