// UTILITY.H : some utility functions and classes...

// Copyright (C) 1998 Tommi Hassinen, Jarno Huuskonen, Geoff Hutchison.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef UTILITY_H
#define UTILITY_H

class element;
class bondtype;

/*################################################################################################*/

#include "v3d.h"
#include "prefs.h"

/*################################################################################################*/

/**	This function can be used to modify a 3D-vector object using a 4x4 transformation 
	matrix. Will first make a [1*4]x[4x4]-type matrix product and after that bring the 
	result back to 3D.
	
	For more complete documentation search for "##homogeneous coordinates"...
*/

void TransformVector(v3d<fGL> &, const fGL *);

/*################################################################################################*/

#define ELEMENT_SYMBOLS 110

/// An element class.

class element
{
	private:
	
	i32s atomic_number;
	
	static const char string[ELEMENT_SYMBOLS][32];
	static const char symbol[ELEMENT_SYMBOLS][4];
	
	static const fGL color[ELEMENT_SYMBOLS][3];
	
	static const fGL mass[ELEMENT_SYMBOLS];
	static const fGL vdwr[ELEMENT_SYMBOLS];

	public:
	
	/// This is the current element that user has selected.
	
	static element current_element;
	
	public:
	
	element(void);
	element(char *);	// symbols
	element(i32s);		// atomic numbers
	~element(void);
	
	const i32s GetAtomicNumber(void) const;
	const char * GetString(void) const;
	const char * GetSymbol(void) const;	
	const fGL * GetColor(void) const;
	fGL GetAtomicMass(void) const;
	fGL GetVDWRadius(void) const;

	// If you have a prefs object, we can check that
	// for a user-defined radius or color
	fGL GetVDWRadius(prefs *) const;
	const fGL * GetColor(prefs *) const;

	void operator++(void);
	void operator--(void);
};

/*################################################################################################*/

#define BONDTYPE_CNJGTD 0
#define BONDTYPE_SINGLE 1
#define BONDTYPE_DOUBLE 2
#define BONDTYPE_TRIPLE 3
#define BONDTYPE_QUADRP 4

/// A bondtype class.

class bondtype
{
	private:
	
	i32s type;
	
	static const char * string[5];
	
	static const char symbol1[5];
	static const char symbol2[5];
	
	public:
	
	/// This is the current element that user has selected.
	
	static bondtype current_bondtype;
	
	public:
	
	bondtype(void);
	bondtype(char);		// symbols
	bondtype(i32s);		// number codes
	~bondtype(void);
	
	const char * GetString(void) const;
	
	i32s GetValue(void) const;
	char GetSymbol1(void) const;
	char GetSymbol2(void) const;
	
	void operator++(void);
	void operator--(void);
	
	friend ostream & operator<<(ostream &, bondtype &);
};

/*################################################################################################*/

/**	An utility class (template) to release memory when program closes.
	
	If you have, say a table that you have to create when program starts, 
	and to destroy it when program closes. For example
	
	        int * table = new int[100];
	
	The memory will be automatically released if you attach an "cleaner" for it:
	
	        int * table = new int[100];
	        singleton_cleaner<int> table_cleaner(table);
*/

template<class TYPE1> class singleton_cleaner
{
	private:
	
	TYPE1 * instance;
	
	public:
	
	singleton_cleaner(TYPE1 * p1 = NULL) { instance = p1; }
	~singleton_cleaner(void) { if (instance) delete instance; }
	
	bool SetInstance(TYPE1 * p1) { if (instance) return false; instance = p1; return true; }
};

/*################################################################################################*/

#endif	// UTILITY_H

// eof
