/*  $Id: PixmapArray.h,v 1.3 2002/04/03 06:11:28 sarrazip Exp $
    PixmapArray.h - Object containing an array of Pixmaps.

    gengameng - Generic 2D Game Engine library
    Copyright (C) 1999, 2000, 2001 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#ifndef _H_PixmapArray
#define _H_PixmapArray

#include <gengameng/Couple.h>

#include <SDL.h>
#include <SDL_image.h>

#include <vector>

using namespace std;


class PixmapArray
/*  Object containing an array of Pixmaps.
*/
{
public:

    typedef SDL_Surface *Pixmap;
    /*  The Pixmap type is local to this class.
	This helps avoid global namespace conflicts.
	It was defined in the global name space in version 2.0 by
	the X11 #includes.
    */


    PixmapArray(size_t numImages);
    /*  Creates a pixmap array capable of containing 'numImages' images.
	'numImages' must be positive.
	This object is the owner of the pixmaps, and the destructor will
	take care of freeing them.
    */

    static void freePixmap(Pixmap pm);
    /*  Frees the designated pixmap.
	Does nothing if 'pm' is null.
	Portability wrapper for XFreePixmap().
    */

    void freePixmaps();
    /*  Frees the pixmaps that are stored in this object.
	'display' must not be null.
	Frees the arrays that hold the pixmaps.
	Assumes that the arrays passed to the constructor of this object
	were allocated with new Pixmap[], or that the PixmapArray(size_t)
	constructor was used.
    */

    ~PixmapArray();
    /*  Calls freePixmaps().
    */

    Pixmap getImage(size_t i) const;
    size_t getNumImages() const;
    /*  Returns the Pixmap of the image at index 'i' of the
	arrays given to the constructor of this object.
	'i' must be lower than the value returned by getNumImages().
	This method must not be called if freePixmaps() has been called
	on this object.
    */

    void setArrayElement(size_t i, Pixmap image);
    /*  'i' must be a valid index in the array.
	'image' must not be null.
    */

    void setImageSize(Couple size);
    Couple getImageSize() const;
    /*  Sets or gets the size in pixels of the images in the pixmap array.
	All images in the array are assumed to be of the same size.
	Neither size.x nor size.y are allowed to be zero.
    */

private:
    vector<Pixmap> images;
    Couple imageSize;  // size in pixels of the images; all assumed same size

    /*	Forbidden operations:
    */
    PixmapArray(const PixmapArray &x);
    PixmapArray &operator = (const PixmapArray &x);
};


/*  INLINE METHODS
*/

inline PixmapArray::Pixmap
PixmapArray::getImage(size_t i) const { return images[i]; }
inline size_t
PixmapArray::getNumImages() const { return images.size(); }
inline Couple
PixmapArray::getImageSize() const { return imageSize; }


#endif  /* _H_PixmapArray */
