/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <math.h>
#include <GL/gl.h>

#include "gdis.h"
#include "coords.h"
#include "interface.h"
#include "gtkshorts.h"
#include "matrix.h"
#include "spatial.h"
#include "numeric.h"
#include "morph.h"
#include "opengl.h"
#include "select.h"
#include "zone.h"

/* data structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/***********************************************/
/* construct a list of bonded atoms for a core */
/***********************************************/
GSList *con_core_bond_list(struct core_pak *core)
{
GSList *blist, *clist=NULL;
struct bond_pak *bond;

for (blist=core->bonds ; blist ; blist=g_slist_next(blist))
  {
  bond = (struct bond_pak *) blist->data;

  if (bond->atom1 == core)
    clist = g_slist_prepend(clist, bond->atom2);
  else
    clist = g_slist_prepend(clist, bond->atom1);
  }
return(clist);
}

/*************************************/
/* determine if two cores are bonded */
/*************************************/
gint con_cores_bonded(struct core_pak *core1, struct core_pak *core2)
{
GSList *list;
struct bond_pak *bond;

for (list=core1->bonds ; list ; list=g_slist_next(list))
  {
  bond = (struct bond_pak *) list->data;

  if (bond->atom1 == core2 || bond->atom2 == core2)
    return(TRUE);
  }
return(FALSE);
}

/****************************/
/* display hydrogen bonding */
/****************************/
void build_hbonds(struct model_pak *model)
{
gint status;
GSList *list;
struct bond_pak *bond;

if (model->build_hydrogen)
  status = NORMAL;
else
  status = HIDDEN;

for (list=model->bonds ; list ; list=g_slist_next(list))
  {
  bond = (struct bond_pak *) list->data;

  if (bond->type == BOND_HBOND)
    bond->status = status;
  }
}

/************************************/
/* build zeolite style connectivity */
/************************************/
void build_zeolite(struct model_pak *data)
{
GSList *list, *list1, *list2;
struct bond_pak *bond;
struct core_pak *core, *core1, *core2;

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

/* search for O */
  if (core->atom_code != 8)
    continue;

  for (list1=core->bonds ; list1 ; list1=g_slist_next(list1))
    {
    core1 = (struct core_pak *) list1->data;

/* search for Al or Si */
    if (core1->atom_code == 13 || core1->atom_code == 14)
      {
/* search for Al or Si */
      list2 = g_slist_next(list1);
      while (list2)
        {
        core2 = (struct core_pak *) list2->data;

        if (core2->atom_code == 13 || core2->atom_code == 14)
          {
/* hide the oxygen */
          core->status |= ZEOL_HIDDEN;
/* replace with one zeolite connection */
          bond = g_malloc(sizeof(struct bond_pak)); 
          bond->type = ZEOLITE;
          bond->atom1 = core1;
          bond->atom2 = core2;
          data->bonds = g_slist_prepend(data->bonds, bond);
          }
        list2 = g_slist_next(list2);
        }
      }
    }
  }
}

/***************************************/
/* build polyhedral style connectivity */
/***************************************/
#define DEBUG_BUILD_POLYHEDRAL 0
void create_polyhedra(struct model_pak *data)
{
gint a, b, c, k, n, flag;
gdouble angle, vec1[3], vec2[3], vec3[3], vec4[3];
GSList *list1, *list2, *clist, *slist, *mlist;
struct spatial_pak *spatial;
struct vec_pak *p1, *p2, *p3, *v1, *v2, *v3;
struct core_pak *core1;
struct bond_pak *bond;

/* if selection, apply only to those cores, else all cores */
if (data->selection)
  mlist = data->selection;
else
  mlist = data->cores;

/* iterate over candidate center atoms */
for (list1=mlist ; list1 ; list1=g_slist_next(list1))
  {
  core1 = (struct core_pak *) list1->data;
  ARR3SET(vec1, core1->x);

/* search for bonds */
  n=0;
  clist=NULL;
  for (list2=core1->bonds ; list2 ; list2=g_slist_next(list2))
    {
    bond = (struct bond_pak *) list2->data;

/* NEW - ignore bonds between same atom types (both candidate center atoms) */
    if ((bond->atom1)->atom_code == (bond->atom2)->atom_code)
      continue;

/* find the candidate polyhedral vertex */
    if (core1 == bond->atom2)
      {
      p1 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(p1->x, (bond->atom1)->x);

      if (bond->periodic == BOND_SPLIT)
        {
        ARR3SUB(p1->x, bond->pic);
        }

      clist = g_slist_prepend(clist, p1);
      n++;
      }
    else
      {
g_assert(core1 == bond->atom1);

      p1 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(p1->x, (bond->atom2)->x);

      if (bond->periodic == BOND_SPLIT)
        {
        ARR3ADD(p1->x, bond->pic);
        }

      clist = g_slist_prepend(clist, p1);
      n++;
      }
    }

#if DEBUG_BUILD_POLYHEDRAL
printf("This [%s] has %d candidates.\n", core1->label, n);
#endif

/* minimum of 4 points for a closed polyhedron */
  if (n > 3)
    {
    k=0;
    slist = NULL;
/* replace all triples with a triangle */
    for (a=0 ; a<n-2 ; a++)
    for (b=a+1 ; b<n-1 ; b++)
    for (c=b+1 ; c<n ; c++)
      {
      p1 = (struct vec_pak *) g_slist_nth_data(clist, a);
      p2 = (struct vec_pak *) g_slist_nth_data(clist, b);
      p3 = (struct vec_pak *) g_slist_nth_data(clist, c);

/* spatial */
      spatial = g_malloc(sizeof(struct spatial_pak));
      spatial->type = SPATIAL_POLYHEDRA;
      spatial->method = GL_POLYGON;
/* central atom's colour */
      ARR3SET(p1->colour, core1->colour);
      VEC3MUL(p1->colour, 1.0/65535.0);
      ARR3SET(p2->colour, core1->colour);
      VEC3MUL(p2->colour, 1.0/65535.0);
      ARR3SET(p3->colour, core1->colour);
      VEC3MUL(p3->colour, 1.0/65535.0);

      spatial->data = NULL;
/* triangle, defined by 3 surrounding oxygens */
/* compute midpoints */
      ARR3SET(vec2, p1->x);
      ARR3ADD(vec2, p2->x);
      VEC3MUL(vec2, 0.5);

      ARR3SET(vec3, p1->x);
      ARR3ADD(vec3, p3->x);
      VEC3MUL(vec3, 0.5);

      ARR3SET(vec4, p2->x);
      ARR3ADD(vec4, p3->x);
      VEC3MUL(vec4, 0.5);

/* compare with central atom */
      ARR3SUB(vec2, vec1);
      ARR3SUB(vec3, vec1);
      ARR3SUB(vec4, vec1);
      vecmat(data->latmat, vec2);
      vecmat(data->latmat, vec3);
      vecmat(data->latmat, vec4);

/* if one matches, then this is an internal facet, and will be discarded */
      flag=0;
/*
      if (VEC3MAGSQ(vec2) < POSITION_TOLERANCE)
        flag++;
      if (VEC3MAGSQ(vec3) < POSITION_TOLERANCE)
        flag++;
      if (VEC3MAGSQ(vec4) < POSITION_TOLERANCE)
        flag++;
*/
      if (VEC3MAGSQ(vec2) < 0.5)
        flag++;
      if (VEC3MAGSQ(vec3) < 0.5)
        flag++;
      if (VEC3MAGSQ(vec4) < 0.5)
        flag++;

      if (flag)
        {
        g_free(spatial);
        continue;
        }
      else
        {
        slist = g_slist_prepend(slist, spatial);
        k++;
        }

/* TODO - a general routine, given a list of points, and a */
/* reference -> reorder (if nec.) so this is true */
/* compute the clockwise normal */
      ARR3SET(vec2, p2->x);
      ARR3SUB(vec2, p1->x);
      ARR3SET(vec3, p3->x);
      ARR3SUB(vec3, p1->x);
      crossprod(vec4, vec2, vec3);
/* CURRENT */
/* keep the normal */
      normalize(vec4, 3);
      ARR3SET(p1->n, vec4);
      ARR3SET(p2->n, vec4);
      ARR3SET(p3->n, vec4);
/* get a vector from poly center to face */
      ARR3SUB(vec2, p1->x);
      VEC3MUL(vec2, -1.0);
      angle = via(vec2, vec4, 3);

/* copy the vertices (since they may be free'd individually later) */
/* TODO - implement copy/duplicate etc. vector method */
      v1 = g_malloc(sizeof(struct vec_pak));
      v2 = g_malloc(sizeof(struct vec_pak));
      v3 = g_malloc(sizeof(struct vec_pak));
      ARR3SET(v1->x, p1->x);
      ARR3SET(v2->x, p2->x);
      ARR3SET(v3->x, p3->x);
      ARR3SET(v1->n, p1->n);
      ARR3SET(v2->n, p2->n);
      ARR3SET(v3->n, p3->n);
      ARR3SET(v1->colour, p1->colour);
      ARR3SET(v2->colour, p2->colour);
      ARR3SET(v3->colour, p3->colour);

/* enforce an outwards pointing clockwise normal */
      if (angle < PI/2.0)
        {
        spatial->data = g_slist_append(spatial->data, v1);
        spatial->data = g_slist_append(spatial->data, v2);
        spatial->data = g_slist_append(spatial->data, v3);
        }
      else
        {
        spatial->data = g_slist_append(spatial->data, v1);
        spatial->data = g_slist_append(spatial->data, v3);
        spatial->data = g_slist_append(spatial->data, v2);
        VEC3MUL(v1->n, -1.0);
        VEC3MUL(v2->n, -1.0);
        VEC3MUL(v3->n, -1.0);
        }
      }

/* only add if more than 3 facets found */
/* ie min for a closed polyhedron - not infallible tho... */
    if (k > 3)
      data->spatial = g_slist_concat(data->spatial, slist);
    else
      free_slist(slist);
    }
/* free all vertices */
  g_slist_free(clist);
  }
}

/*******************************/
/* debugging routines for bonds */
/*******************************/
void dump_atom_bonds(struct model_pak *data)
{
GSList *list1, *list2;
struct core_pak *core;
struct bond_pak *bond;

for (list1=data->cores ; list1 ; list1=g_slist_next(list1))
  {
  core = (struct core_pak *) list1->data;
printf("core [%s][%p], bonds: ", core->label, core);
  for (list2=core->bonds ; list2 ; list2=g_slist_next(list2))
    {
    bond = (struct bond_pak *) list2->data;

    printf("(");

    switch (bond->periodic)
      {
      case BOND_SPLIT:
        printf("s ");
        break;
      case BOND_MERGED:
        printf("m ");
        break;
      default:
        printf("n ");
        break;
      }

    switch (bond->type)
      {
      case BOND_HBOND:
        printf("h ");
        break;
      }

    printf(")");

    }
printf("\n");
  }
}

/*******************************/
/* debugging routines for bonds */
/*******************************/
void dump_bonds(struct model_pak *data)
{
gint h,n,m,p;
GSList *item;
struct bond_pak *bond;

h=n=m=p=0;
for (item=data->bonds ; item ; item=g_slist_next(item))
  {
  bond = (struct bond_pak *) item->data;

if (bond->type == BOND_HBOND)
  h++;
else
  switch (bond->periodic)
    {
    case BOND_SPLIT:
      printf("  split bond [%p][%4s-%4s][%p-%p] : pic [%d %d %d]\n",
              bond,
              (bond->atom1)->label, (bond->atom2)->label, 
              bond->atom1, bond->atom2,
              bond->pic[0], bond->pic[1], bond->pic[2]);
      p++;
      break;

    case BOND_MERGED:
      printf("  merged bond [%p][%4s-%4s][%p-%p] : pic [%d %d %d]\n",
              bond,
              (bond->atom1)->label, (bond->atom2)->label, 
              bond->atom1, bond->atom2,
              bond->pic[0], bond->pic[1], bond->pic[2]);
      m++;
      break;

    default:
      printf("  normal bond [%p][%4s-%4s][%p-%p] : pic [%d %d %d]\n",
              bond,
              (bond->atom1)->label, (bond->atom2)->label, 
              bond->atom1, bond->atom2,
              bond->pic[0], bond->pic[1], bond->pic[2]);
      n++;
    }
  }
printf("bond total=%d : normal=%d, periodic=%d, merged=%d, hbond=%d\n",
                          g_slist_length(data->bonds),n,p,m,h);
}

/*******************************/
/* debugging routines for mols */
/*******************************/
void dump_mols(struct model_pak *model)
{
GSList *list, *list2;
struct mol_pak *mol;
struct core_pak *core;

printf("Found %d molecules.\n", g_slist_length(model->moles));

for (list=model->moles ; list ; list=g_slist_next(list))
  {
  mol = (struct mol_pak *) list->data;

  printf("mol [%p] has %d cores: ", mol, g_slist_length(mol->cores));

  for (list2=mol->cores ; list2 ; list2=g_slist_next(list2))
    {
    core = (struct core_pak *) list2->data;

    printf("%s ", core->label);
    }
  printf("\n");
  }
}

/********************/
/* duplicate a bond */
/********************/
struct bond_pak *dup_bond(struct bond_pak *bond)
{
struct bond_pak *copy;

copy = g_malloc(sizeof(struct bond_pak));

memcpy(copy, bond, sizeof(struct bond_pak));

return(copy);
}

/********************************/
/* update bond list for an atom */
/********************************/
/* NEW - using the new zone scheme, the cell translation bit is */
/* encapsulated in zone lookup */
#define DEBUG_ATOM_BONDS 0
void calc_atom_bonds(struct core_pak *core, struct model_pak *data)
{
guint i;
gint a, b, c, zi;
gint flag, status, type;
gint g[3], start[3], stop[3];
gint m1[3], m2[3];
gdouble ri, r2, r2cut;
gdouble x1[3], x2[3], pic[3], xlat[3];
GSList *list;
struct bond_pak *bond;
struct core_pak *comp;
/*
struct zone_pak *zone, *zone2;
*/
gpointer zone, zone2;

/* allow connectivity to be turned off */
if (!data->build_molecules)
  return;

/* setup for compare */
ri = core->bond_cutoff;

/* get the core's zone */
zone = zone_ptr(core, data);
g_assert(zone != NULL);

#if DEBUG_ATOM_BONDS
/*
printf("current core [%s] [%f %f %f] :  zone = [%d %d %d]\n",
core->label, core->x[0], core->x[1], core->x[2],
zone->grid[0], zone->grid[1], zone->grid[2]);
*/
#endif

/* setup scan limits */
ARR3SET(start, zone_grid(zone));
ARR3SET(stop, start);
/* standard - add a single zone around the edge */
VEC3SUB(start, 1, 1, 1);
VEC3ADD(stop, 1, 1, 1);

/* correct for the isolated (cartesian) case (NB: no periodic wrap around) */
for (i=data->periodic ; i<=2 ; i++)
  {
  if (start[i] < 0)
    start[i] = 0;
  if (stop[i] >= data->zone_div[i])
    stop[i] = data->zone_div[i]-1;
  }

#if DEBUG_ATOM_BONDS
printf("start: %d %d %d\n", start[0], start[1], start[2]);
printf(" stop: %d %d %d\n", stop[0], stop[1], stop[2]);
#endif

/* base core's constrained position */
ARR3SET(x1, core->x);
fractional_clamp(x1, m1, data->periodic);

/* zone sweep */
for (c=start[2] ; c<=stop[2] ; c++)
  {
  for (b=start[1] ; b<=stop[1] ; b++)
    {
    for (a=start[0] ; a<=stop[0] ; a++)
      {
/* the current zone */
#if DEBUG_ATOM_BONDS
printf("[%2d %2d %2d] : ", a, b, c);
#endif

/* periodic image translation */
      VEC3SET(pic, 0.0, 0.0, 0.0);
      VEC3SET(g, a, b, c);
/* constrain zone indexed */
      for (i=3 ; i-- ; )
        while (g[i] < 0)
          {
          g[i] += data->zone_div[i];
          pic[i] -= 1.0;
          }
      for (i=3 ; i-- ; )
        while (g[i] >= data->zone_div[i])
          {
          g[i] -= data->zone_div[i];
          pic[i] += 1.0;
          }
/* convert [a,b,c] to a valid zone index */
      zi = g[2]*data->zone_div[1]*data->zone_div[0]; 
      zi += g[1]*data->zone_div[0]; 
      zi += g[0];

/* this is the actual zone + required xlat that gives the current zone */
#if DEBUG_ATOM_BONDS
printf("[%2d %2d %2d] (%9.4f %9.4f %9.4f) : zone = %2d \n",
        g[0], g[1], g[2], pic[0], pic[1], pic[2], zi);
#endif

g_assert(zi >= 0);
g_assert(zi < data->num_zones);

/* loop over cores in the corresponding zone */
      zone2 = data->zones[zi];
      for (list=zone_cores(zone2) ; list ; list=g_slist_next(list))
        {

    comp = (struct core_pak *) list->data;

    if (comp->status & DELETED)
      continue;

/* avoid double counting and self-bonding */
    if (core >= comp)
      continue;

/* calc bond cutoff*/
    r2cut = ri + comp->bond_cutoff;
    r2cut += BOND_FUDGE*r2cut;
    r2cut *= r2cut;

/* FIXME - this new scheme of clamping so we enforce which zone */
/* the core is in fixes a bug but it slows things down significantly */
/* compare core's constrained position */
    ARR3SET(x2, comp->x);
    fractional_clamp(x2, m2, data->periodic);

#if DEBUG_ATOM_BONDS
printf("[%s ?< %f] : ", comp->label, r2cut);
printf("m1 [%d %d %d] ", m1[0], m1[1], m1[2]);
printf("m2 [%d %d %d] ", m2[0], m2[1], m2[2]);
printf("x [%f %f %f] : ", pic[0], pic[1], pic[2]);
#endif

/* compare core positions */
    ARR3SUB(x2, x1);
    ARR3ADD(x2, pic);

/* cartesian separation */
    vecmat(data->latmat, x2);
    r2 = VEC3MAGSQ(x2);

      if (r2 < POSITION_TOLERANCE)
        {
#if DEBUG_ATOM_BONDS
printf("Ignoring atoms that are too close!\n");
#endif
        }
      else
        {
        flag = 0;
        if (r2 < r2cut)
          {
/* normal */
          status = NORMAL;
          type = BOND_SINGLE;
          flag++;
          }
        else
          {
/* hbond candidate */
          if (data->show_hbonds)
            status = NORMAL;
          else
            status = HIDDEN;
          type = BOND_HBOND;
          if (core->atom_code == 1)
            {
            switch (comp->atom_code)
              {
              case 7:
              case 8:
              case 9:
                if (r2 < 6.25)
                  flag++;
                break;
              }
            }
          if (comp->atom_code == 1)
            {
            switch (core->atom_code)
              {
              case 7:
              case 8:
              case 9:
                if (r2 < 6.25)
                  flag++;
                break;
              }
            }
/* don't include if they belong to the same mol */
/*FIXME - the  problem is that the mol calc is done AFTER the bond calc */
/*
          if (core->mol == comp->mol)
            flag = 0;
*/
          }
/* create bond? */
        if (flag)
          {
          bond = g_malloc(sizeof(struct bond_pak)); 
          data->bonds = g_slist_prepend(data->bonds, bond);

          bond->type = type;
          bond->status = status;

          bond->atom1 = core;
          bond->atom2 = comp;

/* get total xlat need to make this bond */
          ARR3SET(xlat, pic);
          ARR3ADD(xlat, m2);
          ARR3SUB(xlat, m1);
          ARR3SET(bond->pic, xlat);

/* set type if xlat is non zero */
          if (VEC3MAGSQ(xlat) > FRACTION_TOLERANCE)
            bond->periodic = BOND_SPLIT;
          else
            bond->periodic = BOND_NORMAL;

/* atoms reference the bond */
          core->bonds = g_slist_prepend(core->bonds, bond);
          comp->bonds = g_slist_prepend(comp->bonds, bond);
          }
        }
        }
      }
    }
  }
}

/***********************************************/
/* modify primary connectivity with user bonds */
/***********************************************/
void merge_ubonds(struct model_pak *data)
{
gint found, match;
GSList *list1, *list2;
struct bond_pak *bond1, *bond2;
   
/* checks */
g_assert(data != NULL);
if (!data->build_molecules)
  return;

/* search & modify if bond already exists */
for (list1=data->ubonds ; list1 ; list1=g_slist_next(list1))
  {
  bond1 = (struct bond_pak *) list1->data;
  found = 0;

  for (list2=data->bonds ; list2 ; list2=g_slist_next(list2))
    {
    bond2 = (struct bond_pak *) list2->data;

    match = 0;
    if (bond1->atom1 == bond2->atom1 && bond1->atom2 == bond2->atom2)
      match++;
    if (bond1->atom1 == bond2->atom2 && bond1->atom2 == bond2->atom1)
      match++;

/* bond already exists -> modify type */
    if (match)
      {
      if (bond1->status == DELETED)
        bond2->status = DELETED;
      else
        {
        bond2->type = bond1->type;
        bond2->status = NORMAL;
        }
      found++;
      break;
      }
    }

  if (!found)
    {
/* if not found, create a copy for the primary list */
    bond2 = dup_bond(bond1);
    data->bonds = g_slist_prepend(data->bonds, (gpointer) bond2);

/* update atom bond references */
    (bond2->atom1)->bonds = g_slist_prepend((bond2->atom1)->bonds, bond2);
    (bond2->atom2)->bonds = g_slist_prepend((bond2->atom2)->bonds, bond2);
    }
  }
}

/************************************************************/
/* force creation of a bond of specified type between i & j */
/************************************************************/
#define DEBUG_USER_BOND 0
void user_bond(struct core_pak *core1, struct core_pak *core2,
                            gint type, struct model_pak *data)
{
gint count, flag;
struct bond_pak *bdata;
GSList *ubond;

#if DEBUG_USER_BOND
printf("submit bond [%d]: %p - %p : ", type, core1, core2);
#endif
/* checks */
g_assert(core1 != NULL);
g_assert(core2 != NULL);
g_assert(data != NULL);
if (core1 == core2)
  return;

/* search... */
flag=0;
ubond = data->ubonds;
while (ubond)
  {
  bdata = (struct bond_pak *) ubond->data;

/* atom match */
  count=0;
  if (core1 == bdata->atom1)
    count++;
  if (core1 == bdata->atom2)
    count++;
  if (core2 == bdata->atom1)
    count++;
  if (core2 == bdata->atom2)
    count++;

/* ...and change if found */
  if (count > 1)
    {
#if DEBUG_USER_BOND
printf(" [modifying]\n");
#endif
    if (type == BOND_DELETE)
      bdata->status = DELETED;
    else
      {
      bdata->type = type; 
      bdata->status = NORMAL;
      }
    flag=1;
    break;
    }
  ubond = g_slist_next(ubond);
  }

/* not found - create */
if (!flag)
  {
#if DEBUG_USER_BOND
printf(" [creating]\n");
#endif
  bdata = g_malloc(sizeof(struct bond_pak));
  bdata->atom1 = core1;
  bdata->atom2 = core2;
  if (type == BOND_DELETE)
    {
    bdata->status = DELETED;
    bdata->type = BOND_SINGLE;
    }
  else
    {
    bdata->status = NORMAL;
    bdata->type = type; 
    }
  bdata->periodic = BOND_NORMAL;
  data->ubonds = g_slist_prepend(data->ubonds, (gpointer) bdata);
  }

/* update primary bond list */
merge_ubonds(data);
/* update molecuels */
calc_mols(data);
}

/***********************************/
/* make bonds by clicking on atoms */
/***********************************/
void make_bond(struct core_pak *core, gint type, struct model_pak *data)
{
static struct core_pak *first=NULL;

switch(data->state)
  {
  case 0:
    first = core;
    core->status |= SELECT;
    data->state++;
    break;

  case 1:
    first->status &= ~SELECT;
    user_bond(first, core, type, data);
  default:
    data->state=0;
    first = NULL;
  }
}

/*******************************************/
/* update connectivity for a list of atoms */
/*******************************************/
#define DEBUG_REDO_BONDS 0
void wipe_atom_bonds(struct core_pak *core, struct model_pak *data)
{
GSList *list;
struct bond_pak *bond;
struct core_pak *core1, *core2;

/* seek input core in the bond list */
list = core->bonds;
while (list)
  {
  bond = (struct bond_pak *) list->data;
  list = g_slist_next(list);

  core1 = bond->atom1;
  core2 = bond->atom2;
/* remove both core references and main list reference, then free it */
  core1->bonds = g_slist_remove(core1->bonds, bond);
  core2->bonds = g_slist_remove(core2->bonds, bond);
  data->bonds = g_slist_remove(data->bonds, bond);
  g_free(bond);
  }

g_assert(!g_slist_length(core->bonds));
core->bonds=NULL;
}

/****************************/
/* recalc for a single core */
/****************************/
void redo_atom_bonds(struct core_pak *core, struct model_pak *data)
{
wipe_atom_bonds(core, data);
calc_atom_bonds(core, data);
}

/****************************/
/* destroy all connectivity */
/****************************/
void wipe_bonds(struct model_pak *data)
{
GSList *list;
struct bond_pak *bond;
struct core_pak *core1, *core2;

list = data->bonds;
while (list)
  {
  bond = (struct bond_pak *) list->data;
  list = g_slist_next(list);

  core1 = bond->atom1;
  core2 = bond->atom2;
/* remove both core references and main list reference, then free it */
  core1->bonds = g_slist_remove(core1->bonds, bond);
  core2->bonds = g_slist_remove(core2->bonds, bond);
  data->bonds = g_slist_remove(data->bonds, bond);
  g_free(bond);
  }
data->bonds=NULL;
}

/*********/
/* BONDS */
/*********/
#define DEBUG_CALC_BONDS 0
void calc_bonds(struct model_pak *data)
{
GSList *list;
struct core_pak *core;

#if DEBUG_CALC_BONDS
printf("calc_bonds(%d)\n", data->cur_frame);
printf("calc_bonds(%d)\n", data->build_molecules);
#endif

g_assert(data->zones != NULL);

if (data->anim_fix)
  return;

/* TODO - delete polyhedra */
/* redo atom connectivity */
wipe_bonds(data);

for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  core->status &= ~ZEOL_HIDDEN;
  calc_atom_bonds(core, data);
  }

/* user bond modification */
merge_ubonds(data);

/* special building modes */
build_hbonds(data);

if (data->build_zeolite)
  build_zeolite(data);

/*
if (data->build_polyhedra)
  build_polyhedral(data);
*/

#if DEBUG_CALC_BONDS
dump_bonds(data);
dump_atom_bonds(data);
#endif
}

/*********************/
/* remove a molecule */
/*********************/
/* NB: must remove ALL core references */
void delete_mol(struct mol_pak *mol, struct model_pak *data)
{
GSList *list1;
struct core_pak *core;

g_assert(mol != NULL);
g_assert(data != NULL);

/* enumerate cores in the molecules */
for (list1=mol->cores ; list1 ; list1=g_slist_next(list1))
  {
  core = (struct core_pak *) list1->data;
  delete_core(core);
  }

delete_commit(data);
}

/************************************/
/* free molecule list plus its data */
/************************************/
void free_mol_list(struct model_pak *data)
{
GSList *list;
struct mol_pak *mol;

for (list=data->moles ; list ; list=g_slist_next(list))
  {
  mol = (struct mol_pak *) list->data;

  g_slist_free(mol->cores);
  g_free(mol);
  }

g_slist_free(data->moles);
data->moles=NULL;
}

/******************************/
/* compute molecule centroids */
/******************************/
#define DEBUG_CALC_MOL_CENT 0
void calc_mol_centroids(struct model_pak *data)
{
gint n;
gdouble scale;
GSList *list, *list2;
struct mol_pak *mol;
struct core_pak *core;

/* centroid calc & core->mol reference */
for (list=data->moles ; list ; list=g_slist_next(list))
  {
  mol = (struct mol_pak *) list->data;
  VEC3SET(mol->centroid, 0.0, 0.0, 0.0);
  
  n=0;
  for (list2=mol->cores ; list2 ; list2=g_slist_next(list2))
    {
    core = (struct core_pak *) list2->data;

    core->mol = mol;

#if DEBUG_CALC_MOL_CENT
    printf("[%2d]  [%4s]  [%10.4f %10.4f %10.4f]\n",
           core->label, core->x[0], core->x[1], core->x[2]);
#endif

    ARR3ADD(mol->centroid, core->x);
    n++;
    }

  if (n)
    {
    scale = 1.0 / (gdouble) n;
    VEC3MUL(mol->centroid, scale);
    }
  }
}

/*********************************/
/* sort molecule cores primitive */
/*********************************/
gint sort_core_order(gpointer ptr_core1, gpointer ptr_core2)
{
struct core_pak *core1 = ptr_core1, *core2 = ptr_core2;

if (core1->atom_order < core2->atom_order)
  return(-1);
return(1);
}

/*********************************************************************/
/* sort cores in molecules into the same order as the main core list */
/*********************************************************************/
void sort_mol_cores(struct model_pak *model)
{
GSList *list;
struct mol_pak *mol;

for (list=model->moles ; list ; list=g_slist_next(list))
  {
  mol = list->data;

  mol->cores = g_slist_sort(mol->cores, (gpointer) sort_core_order);
  }
}

/**************************/
/* compute molecule lists */
/**************************/
#define DEBUG_CALC_MOLS 0
void calc_mols(struct model_pak *data)
{
gint n=0;
guint i=0;
GSList *list, *blist, *mlist, *list1, *list2;
struct core_pak *core, *core2;
struct bond_pak *bond;
struct mol_pak *mol;

g_assert(data != NULL);

/* clean */
free_mol_list(data);

/* init */
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  core->status &= ~PRUNED;
/* NEW - record core order in list */
  core->atom_order = i++;
  }

/* build mol lists */
mlist=NULL;
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->status & PRUNED)
    continue;

/* init a new mol list */
  if (!mlist)
    {
    mlist = g_slist_append(mlist, core);
    core->status |= PRUNED;
    core->molecule = n;
    }

/* while cores are being added to mlist */
  do
    { 
    blist = NULL;
    for (list2=mlist ; list2 ; list2=g_slist_next(list2))
      {
      core = (struct core_pak *) list2->data;

/* get list of atoms bonded to the current core */
      for (list1=core->bonds ; list1 ; list1=g_slist_next(list1))
        {
        bond = (struct bond_pak *) list1->data;

/* don't include these types in connectivity tracing */
        if (bond->status == DELETED)
          continue;
        if (bond->type == BOND_HBOND)
          continue;
        if (bond->periodic == BOND_SPLIT)
          continue;

/* get the attached core */
        if (core == bond->atom1)
          {
          core2 = bond->atom2;
          }
        else
          {
          g_assert(core == bond->atom2);
          core2 = bond->atom1;
          }

/* only accumulate bonded, non-pruned atoms */
        if (core2->status & PRUNED)
          continue;

        blist = g_slist_prepend(blist, core2); 
        core2->status |= PRUNED;

/* deprec. */
        core2->molecule = n;
        }
      }
/* augment the current listing */
    if (blist)
      mlist = g_slist_concat(mlist, blist);
    }
  while (g_slist_length(blist));
  n++;

/* done - completed mol */
  mol = g_malloc(sizeof(struct mol_pak));
  mol->cores = mlist;
  mlist=NULL;
  data->moles = g_slist_prepend(data->moles, mol);

/* label the cores as belonging to the mol */
  for (list1=mol->cores ; list1 ; list1=g_slist_next(list1))
    {
    core = (struct core_pak *) list1->data;
    core->mol = mol;
    }
  }

#if DEBUG_CALC_MOLS
dump_mols(data);
#endif

calc_mol_centroids(data);

/* NEW - ensure core ordering in molecules is the same as in the main list */
sort_mol_cores(data);
}

/**********************************************/
/* join two fragments sharing a periodic bond */
/**********************************************/
/* direction +ve -> 2->1, else 1->2 */
#define DEBUG_JOIN_FRAG 0
void join_frag(struct bond_pak *bond, gint direction, struct model_pak *data)
{
gdouble pic[3];
GSList *list1, *list2;
struct mol_pak *mol1, *mol2;
struct core_pak *core, *core1, *core2;
struct bond_pak *bond2;

/* checks */
g_assert(data != NULL);
g_assert(bond != NULL);
g_assert(bond->periodic == BOND_SPLIT);

core1 = bond->atom1;
core2 = bond->atom2;

/* setup move direction */
if (direction > 0)
  {
  mol1 = core1->mol;
  mol2 = core2->mol;
  }
else
  {
  mol1 = core2->mol;
  mol2 = core1->mol;
  VEC3MUL(bond->pic, -1.0);
  }

g_assert(mol1 != mol2);

#if DEBUG_JOIN_FRAG
printf("merging %p and %p\n", mol1, mol2);
#endif

/* move all atoms in mol2 to mol1 */
for (list1=mol2->cores ; list1 ; list1=g_slist_next(list1))
  {
  core = (struct core_pak *) list1->data;

/* move core & shell (if any) */
  ARR3ADD(core->x, bond->pic);
  if (core->shell)
    {
    ARR3ADD((core->shell)->x, bond->pic);
    }

  core->mol = mol1;

/* update (merge or alter pic) all periodic bond(s) in mol2 */
  for (list2=core->bonds ; list2 ; list2=g_slist_next(list2))
    {
    bond2 = (struct bond_pak *) list2->data;
    if (bond == bond2)
      continue;

g_assert(bond2 != NULL);

/* FIXME - strictly, we should test if this heals more than */
/* just the current bond. would need to compare pics's to */
/* test this tho' ... */
    if (bond2->periodic == BOND_SPLIT)
      {
      if (core == bond2->atom1)
        {
/* another periodic bond to mol1 */
        if ((bond2->atom2)->mol == mol1)
          {
/* iff pic same -> merged, else not */
          ARR3SET(pic, bond->pic);
          ARR3ADD(pic, bond2->pic);
          if (VEC3MAGSQ(pic) < FRACTION_TOLERANCE)
            bond2->periodic = BOND_MERGED;
          }
        else
          {
/* periodic bond to different molecule, just update pic */
          ARR3ADD(bond2->pic, bond->pic)
          }
        }
      else
        {
/* another periodic bond to mol1 */
        if ((bond2->atom1)->mol == mol1)
          {
/* iff pic same -> merged */
          ARR3SET(pic, bond->pic);
          ARR3SUB(pic, bond2->pic);
          if (VEC3MAGSQ(pic) < FRACTION_TOLERANCE)
            bond2->periodic = BOND_MERGED;
          }
        else
          {
/* periodic bond to different molecule, just update pic */
          ARR3SUB(bond2->pic, bond->pic)
          }
        }
      }
    }
  }

/* merge the mols */
/* NB: don't recalc centroid until the end, as more atoms may be added */
mol1->cores = g_slist_concat(mol1->cores, mol2->cores);
data->moles = g_slist_remove(data->moles, mol2);
g_free(mol2);

/* mark the periodic bond as merged */
bond->periodic = BOND_MERGED;
VEC3SET(bond->pic, 0.0, 0.0, 0.0);
}

/************************/
/* accumulate molecules */
/************************/
#define DEBUG_UNFRAGMENT 0
void unfragment(struct model_pak *data)
{
gint m, n, direction, merge;
GSList *list1, *list2, *broken_bonds;
struct mol_pak *mol;
struct bond_pak *bond;
struct core_pak *core1, *core2;

/*
printf("--- unfrag start\n");
dump_bonds(data);
*/
if (!data->periodic)
  return;


/* preconstruct broken bond list */
broken_bonds = NULL;
for (list1=data->bonds ; list1 ; list1=g_slist_next(list1))
  {
  bond = (struct bond_pak *) list1->data;
  if (bond->type != BOND_HBOND && bond->periodic == BOND_SPLIT)
    broken_bonds = g_slist_prepend(broken_bonds, bond);  
  }


/* seek periodic bonds containing atoms in the current mol */
list1 = data->moles;
while (list1)
  {
  mol = (struct mol_pak *) list1->data;
  merge = 0;

/* search for a periodic bond that links to mol */
/*
  list2 = data->bonds;
*/

  list2 = broken_bonds;
  while (list2)
    {
    bond = (struct bond_pak *) list2->data;
    list2 = g_slist_next(list2);

/* ignore these */
/*
    if (bond->type == BOND_HBOND)
      continue;

    if (bond->periodic == BOND_SPLIT)
*/
      {
      core1 = bond->atom1;
      core2 = bond->atom2;

#if DEBUG_UNFRAGMENT
printf("examining: [%p:%p] - [%p:%p]\n", core1, core1->mol, core2, core2->mol);
#endif

/* two possibilities for connection */
/* direction +ve -> 2->1, else 1->2 */
      direction=0;      
      if (core1->mol == mol)
        direction++;
      if (core2->mol == mol)
        direction--;
/*
direction=1;
*/

      if (direction)
        {
        join_frag(bond, direction, data);
        merge++;
        }
      }
    }

/* mol hasn't changed, get next */
  if (!merge)
    list1 = g_slist_next(list1);
  }

/* post run checks */
n=0;
for (list1=data->bonds ; list1 ; list1=g_slist_next(list1))
  {
  bond = (struct bond_pak *) list1->data;

/* ignore these */
  if (bond->type == BOND_HBOND)
    continue;

/* unhealed bond? */
  if (bond->periodic == BOND_SPLIT)
    n++;

/* all merged bonds can now be fixed */
  if (bond->periodic == BOND_MERGED)
    {
    VEC3SET(bond->pic, 0, 0, 0);
    }
  }

if (n)
  {
#if DEBUG_UNFRAGMENT
printf("Remaining periodic bonds: %d\n", n);
show_text(WARNING, "Can't unfragment a continuously periodic structure.\n");
#endif

/* restore original bonding */
  pbc_constrain_atoms(data);
  calc_bonds(data);
  }
else
  {
/* NB: recalc centroids, as join_frag doesn't seem to to it properly */
  m = 0;
  for (list1=data->moles ; list1 ; list1=g_slist_next(list1))
    {
    mol = (struct mol_pak *) list1->data;
    VEC3SET(mol->centroid, 0.0, 0.0, 0.0);
    
    for (list2=mol->cores ; list2 ; list2=g_slist_next(list2))
      {
      core1 = (struct core_pak *) list2->data;
      core1->mol = mol;
      core1->molecule = m;
      ARR3ADD(mol->centroid, core1->x);
      }
    n = g_slist_length(mol->cores);
    if (n)
      {
      VEC3MUL(mol->centroid, 1.0/n);
      }
    else
      printf("unfrag() warning: empty molecule found.\n");
    m++;
    }
/* constrain */
  pbc_constrain_mols(data);
  }

calc_coords(REFRESH, data);

/*
printf("--- unfrag end\n");
dump_bonds(data);
*/

sort_mol_cores(data);

#if DEBUG_UNFRAGMENT
printf("[*] molecules: %d\n", g_slist_length(data->moles));
#endif
}

/*********************************/
/* standard connectivity refresh */
/*********************************/
void connect_refresh(void)
{
struct model_pak *model;

model = sysenv.active_model;

calc_bonds(model);
calc_mols(model);
calc_coords(REFRESH, model);
redraw_canvas(SINGLE);
}

/****************/
/* button hooks */
/****************/
void cb_connect_confine(GtkWidget *w, gint mode)
{
struct model_pak *model;

model = sysenv.active_model;

switch (mode)
  {
  case CORE:
    pbc_constrain_atoms(model);
    break;
  case MOL:
    pbc_constrain_mols(model);
    break;
  }

init_objs(CENT_COORDS, model);
calc_mols(model);
redraw_canvas(SINGLE);
}

/*********************/
/* unfragment update */
/*********************/
void cb_connect_unfrag(void)
{
struct model_pak *model;

model = sysenv.active_model;

unfragment(model);

/* atom colour/display updates */
if (model->colour_scheme == MOL)
  init_objs(REFRESH_COLOUR, model);

redraw_canvas(SINGLE);
}

/*****************************/
/* connectivity modification */
/*****************************/
void connect_page(GtkWidget *box)
{
GtkWidget *hbox, *vbox, *vbox1, *vbox2;
GtkWidget *frame;
struct model_pak *data;

/* do we have to do anything? */
data = sysenv.active_model;
if (!data)
  return;

/* split panel display */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(box), hbox, TRUE, TRUE, 0);

/* pane 1 */
vbox1 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(vbox1), 0);

frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1),frame,FALSE,FALSE,0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING/2);

gtksh_auto_check("Build molecules ",    connect_refresh, NULL,
                                         &data->build_molecules, vbox);

/*
gtksh_auto_check("Polyhedral bonding ", connect_refresh, NULL,
                                         &data->build_polyhedra, vbox);
*/

gtksh_auto_check("Zeolite bonding ",    connect_refresh, NULL,
                                         &data->build_zeolite, vbox);
gtksh_auto_check("Hydrogen bonding ",   connect_refresh, NULL,
                                         &data->build_hydrogen, vbox);

/* pane */
vbox2 = gtk_vbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);
gtk_container_set_border_width(GTK_CONTAINER(vbox2), 0);

/* frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2),frame,FALSE,FALSE,0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING/2);

gtksh_button_x("Add single bonds ",
               gtk_switch_mode, (gpointer) BOND_SINGLE,
               vbox);
gtksh_button_x("Add double bonds ",
               gtk_switch_mode, (gpointer) BOND_DOUBLE,
               vbox);
gtksh_button_x("Add triple bonds ",
               gtk_switch_mode, (gpointer) BOND_TRIPLE,
               vbox);
gtksh_button_x("Delete bonds ",
               gtk_switch_mode, (gpointer) BOND_DELETE,
               vbox);

/* frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2),frame,FALSE,FALSE,0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING/2);

gtksh_button_x("Constrain atoms to PBC ",
               cb_connect_confine, (gpointer) CORE, vbox);
gtksh_button_x("Constrain mols to PBC ",
               cb_connect_confine, (gpointer) MOL, vbox);
gtksh_button_x("Unfragment mols ",
               cb_connect_unfrag, NULL, vbox);
}

