/*****************************************************************************
     This file is part of FWatch. 

     Copright (C)2003  Frank Hemer <frank@hemer.org>

     FWatch is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     FWatch is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with FWatch; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*******************************************************************************/

/* $Id: fwatch.c,v 1.3 2003/05/18 23:00:23 frank Exp $ */ 

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <sys/syscall.h>
#include <linux/file.h>
#include <linux/poll.h>
#include <linux/utime.h>

#ifndef LINUX_VERSION_CODE
#  include <linux/version.h>
#endif

#include "fwatch.h"

MODULE_AUTHOR("Frank Hemer <frank@hemer.org>");
MODULE_DESCRIPTION("reports file changes to /dev/fwatch");
MODULE_SUPPORTED_DEVICE("reports file changes to /dev/fwatch");
MODULE_LICENSE("GPL");

/* #define DEBUG  */
/* #define DEBUG_FULLNAMES */


#ifndef VERSION_CODE
#  define VERSION_CODE(vers,rel,seq) ( ((vers)<<16) | ((rel)<<8) | (seq) )
#endif


/* set the version information, if needed */
#ifdef NEED_VERSION
static char kernel_version[] = UTS_RELEASE;
#endif


/* ring buffer */
#define BUF_SIZE        512
#define MAX_PATH        512
char *buffer[BUF_SIZE];
int queue_head = 0;
int queue_tail = 0;
static wait_queue_head_t fwatch_wait;


/* device info for the fwatch device */ 
static int fwatch_major = 40;
static int fwatch_device_open = 0;
static int fwatch_mode = 0;

/* address of original syscalls sysc*/
int (*original_open)(const char * filename, int flags, int mode); 
int (*original_rmdir)(const char * pathname); 
int (*original_mkdir)(const char * pathname, int mode); 
int (*original_symlink)(const char * oldname, const char * newname); 
int (*original_link)(const char * oldname, const char * newname);
int (*original_unlink)(const char * pathname); 
int (*original_create)(const char * pathname, int mode);
int (*original_rename)(const char * oldname, const char * newname); 
int (*original_close)(unsigned int);
int (*original_chown)(const char * filename, uid_t user, gid_t group);
#ifdef __NR_chown32
int (*original_chown32)(const char * filename, uid_t user, gid_t group);
#endif
int (*original_lchown)(const char * filename, uid_t user, gid_t group);
#ifdef __NR_lchown32
int (*original_lchown32)(const char * filename, uid_t user, gid_t group);
#endif
int (*original_fchown)(const char * filename, uid_t user, gid_t group);
int (*original_chmod)(const char* filename, mode_t mode);
int (*original_stat)(const char* filename, struct stat * statbuf);
int (*original_fstat)(unsigned int fd, struct stat * statbuf);
#ifdef __NR_stat64
int (*original_stat64)(const char* filename, struct stat64 * statbuf, long flags);
#endif
#ifdef __NR_fstat64
int (*original_fstat64)(unsigned long fd, struct stat64 * statbuf, long flags);
#endif
int (*original_write)(unsigned int, const char *, size_t);
#ifdef __NR_utime
int (*original_utime)(char *, struct utimbuf * times);
#endif
spinlock_t fwatch_lock = SPIN_LOCK_UNLOCKED; 
unsigned long lock_flags;

/* ============================================================================== */

__inline__ int fwatch_cpname(char * dest, const unsigned char * src, int len) {
  if (src) {
    dest[len--] = '\0';
    while (len >= 0) {
      dest[len] = (char)(src[len]);
      len--;
    }
  } else return -1;
  return 0;
}

__inline__ char * fwatch_strncpy(char * dest, const char * src, int cnt) {
  strncpy(dest,src,--cnt);
  dest[cnt] = '\0';
  return dest;
}

__inline__ char * fwatch_strncat(char * dest, const char * src, int cnt) {
  strncat(dest,src,--cnt);
  dest[cnt] = '\0';
  return dest;
}

#if LINUX_VERSION_CODE < VERSION_CODE(2,4,0)

__inline__ int fwatch_getFullPath(char *buf, unsigned long size)
{
	int error;
	struct dentry *pwd = current->fs->pwd;
	*buf='\0'; 
	error = -ENOENT;
	/* Has the current directory has been unlinked? */
	if (pwd->d_parent == pwd || !list_empty(&pwd->d_hash)) {
		char *page = (char *) __get_free_page(GFP_USER);
		error = -ENOMEM;
		if (page) {
			unsigned long len;

			char * cwd = d_path(pwd, page, PAGE_SIZE);
			error = -ERANGE;
			len = PAGE_SIZE + page - cwd;
			if (len <= size) {
				error = len;
				strncat(buf, cwd, size); 
			}
			free_page((unsigned long) page);
		}
	}
	return error;
}
#else
__inline__ int fwatch_getFullPath(char *buf, unsigned long size)
{
	int error;
	struct vfsmount *pwdmnt, *rootmnt;
	struct dentry *pwd, *root;
	char *page = (char *) __get_free_page(GFP_USER);
#ifdef DEBUG
	printk("IN fwatch_getFullPath\n");
#endif

	if (!page)
#ifdef DEBUG
	printk("OUT fwatch_getFullPath nomem\n");
#endif
		return -ENOMEM;

	read_lock(&current->fs->lock);
	pwdmnt = mntget(current->fs->pwdmnt);
	pwd = dget(current->fs->pwd);
	rootmnt = mntget(current->fs->rootmnt);
	root = dget(current->fs->root);
	read_unlock(&current->fs->lock);

	error = -ENOENT;
	/* Has the current directory has been unlinked? */
	spin_lock_irqsave(&dcache_lock, lock_flags);
	if (pwd->d_parent == pwd || !list_empty(&pwd->d_hash)) {
		unsigned long len;
		char * cwd;

		cwd = __d_path(pwd, pwdmnt, root, rootmnt, page, PAGE_SIZE);
		spin_unlock_irqrestore(&dcache_lock, lock_flags);

		error = -ERANGE;
		len = PAGE_SIZE + page - cwd;
		if (len <= size) {
			error = len;
			strncat(buf, cwd, size); 
		}
	} else
	  spin_unlock_irqrestore(&dcache_lock, lock_flags);
	dput(pwd);
	mntput(pwdmnt);
	dput(root);
	mntput(rootmnt);
	free_page((unsigned long) page);
#ifdef DEBUG
	printk("OUT fwatch_getFullPath\n");
#endif
	return error;
}
#endif

static __inline__ int do_revalidate(struct dentry *dentry)
{
	struct inode * inode = dentry->d_inode;
	if (inode->i_op && inode->i_op->revalidate)
		return inode->i_op->revalidate(dentry);
	return 0;
}

/* ============================================================================== */

int fwatch_get_from_queue(fwatch_data_t * data) {
#ifdef DEBUG
  printk("IN get_from_queue\n");
#endif
  spin_lock(&fwatch_lock);
  if (queue_head == queue_tail) { 
    spin_unlock( &fwatch_lock);
#ifdef DEBUG
    printk("OUT get_from_queue: empty\n");
#endif
    return 1;
  }
  memcpy(data,buffer[queue_head],sizeof(fwatch_data_t));
  kfree(buffer[queue_head]); 
  queue_head++;
  if(queue_head == BUF_SIZE) queue_head=0;

  spin_unlock( &fwatch_lock);
#ifdef DEBUG
  printk("OUT get_from_queue\n");
#endif
  return 0;
}

__inline__ int fwatch_put_to_queue(fwatch_data_t * data) {
#ifdef DEBUG
  printk("IN put_to_queue\n");
#endif
  spin_lock(&fwatch_lock);
  if( (queue_tail + 1) == queue_head ||
      ( ((queue_tail + 1) == BUF_SIZE) &&
	( queue_head == 0 ) )) { 

    kfree(buffer[queue_head]);
    queue_head++;
    if ( queue_head == BUF_SIZE ) { 
      queue_head = 0; 
    }
    data->mask &= FWATCH_OVERFLOW;
  }

  buffer[queue_tail] = kmalloc(sizeof(fwatch_data_t), GFP_KERNEL);
  if ( ! buffer[queue_tail] ) { 
    spin_unlock( &fwatch_lock);
    printk("OUT put_to_queue: nomem\n");
    return -ENOMEM; 
  }

  memcpy(buffer[queue_tail],data,sizeof(fwatch_data_t));
  queue_tail++;

  if ( queue_tail == BUF_SIZE ) {
    queue_tail=0;
  }
  wake_up_interruptible(&fwatch_wait);
  spin_unlock( &fwatch_lock);
#ifdef DEBUG
  printk("OUT put_to_queue\n");
#endif
  return 1;
}

__inline__ int fwatch_get_data_for_file(struct file * f, fwatch_data_t * data) {

    struct dentry * tmp;
    tmp = f->f_dentry;      
    char filename[512];
    filename[0] = '\0';

    int error = do_revalidate(tmp);
    if (!error) {
      data->st_dev = tmp->d_inode->i_dev;
      data->st_ino = tmp->d_inode->i_ino;
#ifdef FWATCH_HAVENAMES
      {
	int len = (int)tmp->d_name.len;
	fwatch_cpname(data->name,tmp->d_name.name,(len >= FWATCH_MAXMSGSIZE) ? FWATCH_MAXMSGSIZE-1 : len);
      }
#endif
      if (S_ISREG(tmp->d_inode->i_mode)) {
	data->mask = FWATCH_ISFILE;
	if (atomic_read(&tmp->d_count)) {
	  tmp = dget(tmp->d_parent);
	} else {
	  tmp = dget_locked(tmp->d_parent);
	}
	if (tmp) {
	  data->st_pdev = tmp->d_inode->i_dev;
	  data->st_pino = tmp->d_inode->i_ino;
#ifdef FWATCH_HAVENAMES
	  {
	    int len = (int)tmp->d_name.len;
	    fwatch_cpname(data->pname,tmp->d_name.name,(len >= FWATCH_MAXMSGSIZE) ? FWATCH_MAXMSGSIZE-1 : len);
	  }
#endif
	} else {
	  data->st_pdev = -1;
	  data->st_pino = -1;
#ifdef FWATCH_HAVENAMES
	  strcpy(data->pname,"has no parent");
#endif
	}
	return 0;
      } else if (S_ISDIR(tmp->d_inode->i_mode)) {
	if (atomic_read(&tmp->d_count)) {
	  tmp = dget(tmp->d_parent);
	} else {
	  tmp = dget_locked(tmp->d_parent);
	}
	data->mask = FWATCH_ISDIR;
	if (tmp) {
	  data->st_pdev = tmp->d_inode->i_dev;
	  data->st_pino = tmp->d_inode->i_ino;
#ifdef FWATCH_HAVENAMES
	  {
	    int len = (int)tmp->d_name.len;
	    fwatch_cpname(data->pname,tmp->d_name.name,(len >= FWATCH_MAXMSGSIZE) ? FWATCH_PATHSIZE-1 : len);
	  }
#endif
	} else {
	  data->st_pdev = -1;
	  data->st_pino = -1;
#ifdef FWATCH_HAVENAMES
	  strcpy(data->pname,"dir has no parent");
#endif
	}
	return 0;

/*       } else { */
/* 	data->mask = FWATCH_NONE; */
/* 	data->st_pdev = -1; */
/* 	data->st_pino = -1; */
/* 	strcpy(data->pname,"not parent of file\0"); */
      }
    }
    return 1;
}

__inline__ int fwatch_get_data_for_name(const char * name, fwatch_data_t * data) {

  struct nameidata nd;
  int error;

#ifdef DEBUG
  printk("IN get_data\n");
#endif
  spin_lock(&dcache_lock);
  error = user_path_walk(name, &nd);
  if (!error) {
    struct dentry * tmp = nd.dentry;
    error = do_revalidate(tmp);
    if (!error) {
      data->st_dev = tmp->d_inode->i_dev;
      data->st_ino = tmp->d_inode->i_ino;
#ifdef FWATCH_HAVENAMES
      {
	int len = (int)tmp->d_name.len;
	fwatch_cpname(data->name,tmp->d_name.name,(len >= FWATCH_MAXMSGSIZE) ? FWATCH_MAXMSGSIZE-1 : len);
      }
#endif
      if (S_ISREG(tmp->d_inode->i_mode)) {
	if (atomic_read(&tmp->d_count)) {
	  tmp = dget(tmp->d_parent);
	} else {
	  tmp = dget_locked(tmp->d_parent);
	}
	data->mask = FWATCH_ISFILE;
	if (tmp) {
	  data->st_pdev = tmp->d_inode->i_dev;
	  data->st_pino = tmp->d_inode->i_ino;
#ifdef FWATCH_HAVENAMES
	  {
	    int len = (int)tmp->d_name.len;
	    fwatch_cpname(data->pname,tmp->d_name.name,(len >= FWATCH_MAXMSGSIZE) ? FWATCH_MAXMSGSIZE-1 : len);
	  }
#endif
	} else {
	  data->st_pdev = -1;
	  data->st_pino = -1;
#ifdef FWATCH_HAVENAMES
	  strcpy(data->pname,"file has no parent");
#endif
	}
	error = 0;
      } else if (S_ISDIR(tmp->d_inode->i_mode)) {
	if (atomic_read(&tmp->d_count)) {
	  tmp = dget(tmp->d_parent);
	} else {
	  tmp = dget_locked(tmp->d_parent);
	}
	data->mask = FWATCH_ISDIR;
	if (tmp) {
	  data->st_pdev = tmp->d_inode->i_dev;
	  data->st_pino = tmp->d_inode->i_ino;
#ifdef FWATCH_HAVENAMES
	  {
	    int len = (int)tmp->d_name.len;
	    fwatch_cpname(data->pname,tmp->d_name.name,(len > FWATCH_MAXMSGSIZE) ? FWATCH_MAXMSGSIZE-1 : len);
	}
#endif
	} else {
	  data->st_pdev = -1;
	  data->st_pino = -1;
#ifdef FWATCH_HAVENAMES
	  strcpy(data->pname,"dir has no parent");
#endif
	}
	error = 0;
      } else {
/*              data->mask = FWATCH_NONE; */
/*              data->st_pdev = -1; */
/*              data->st_pino = -1; */
/*              strcpy(data->pname,"not parent of file\0"); */
	error = 1;
      }
    }
    path_release(&nd);
  } else {
/*     data->mask = -1; */
/*     data->st_dev = -1; */
/*     data->st_ino = -1; */
/*     data->st_pdev = -1; */
/*     data->st_pino = -1; */
/*     strcpy(data->name,"reading inode failed\0"); */
/*     data->pname[0] = '\0'; */
    error = 1;
  }
  spin_unlock(&dcache_lock);
#ifdef DEBUG
  printk("OUT get_data\n");
#endif
  return error;
}

#ifdef FWATCH_HAVENAMES
#  ifdef DEBUG_FULLNAMES
__inline__ void fwatch_fill_with_fullNames( const char * oldname, const char * newname, const int operation, fwatch_data_t * data) {

  char fulloldname[FWATCH_PATHSIZE]; 
  char fullnewname[FWATCH_PATHSIZE]; 
  long len1=0; 
  long len2=0; 

#ifdef DEBUG
  printk("IN log_filename\n");
#endif

  fulloldname[0]='\0'; 
  fullnewname[0]='\0'; 

  if (!oldname) oldname = data->name;

  if ( *oldname == '/' ) {
    fwatch_strncpy(fulloldname, oldname, FWATCH_PATHSIZE);
  } else {
    len1 = fwatch_getFullPath(fulloldname, FWATCH_PATHSIZE);
    if (len1 < 0) {
      printk("fwatch: fwatch_getFullPath error (%ld)", len1);
      return;
    }
    len1 = strlen( fulloldname);
    len2 = strlen( oldname ); 
    if ( len1 + len2 < FWATCH_PATHSIZE - 1 ) {
      strcat(fulloldname, "/");
      strcat(fulloldname, oldname); 
    } else if (len1 < FWATCH_PATHSIZE - 2) {
      strcat(fulloldname, "/");
      fwatch_strncat(fulloldname, oldname, FWATCH_PATHSIZE- len1 - 1);
    }
  }
  
  if ( newname != NULL ) {
    if ( *newname == '/' ) {
      fwatch_strncpy( fullnewname, newname, FWATCH_PATHSIZE);
    } else {
      len1 = fwatch_getFullPath(fullnewname, FWATCH_PATHSIZE);
      if (len1 < 0) {
	printk("fwatch: fwatch_getFullPath error (%ld)", len1);
	return;
      } else {
	len1 = strlen( fullnewname);
	len2 = strlen( newname );
	if ( len1 + len2 < FWATCH_PATHSIZE - 1 ) {
	  strcat(fullnewname, "/"); 
	  strcat(fullnewname, newname); 
	} else if (len1  < FWATCH_PATHSIZE - 2) {
	  strcat(fullnewname, "/");
	  fwatch_strncat(fullnewname, newname, FWATCH_PATHSIZE- len1 - 1);
	}
      }
    }
  }

  switch ( operation ) {
    case FWATCH_OPENW:
      strcpy( data->name, fulloldname ); 
      break; 
	  
    case FWATCH_OPENR: 
      strcpy(data->name, fulloldname); 
      break; 
    
    case FWATCH_RMDIR: 
      strcpy(data->name, fulloldname); 
      break; 
    
    case FWATCH_MKDIR: 
      strcpy(data->name, fulloldname); 
      break; 
    
    case FWATCH_SYMLINK: 
      strcpy(data->name, oldname); 
      strcat(data->name, " -> "); 
      strcat(data->name, fullnewname); 
      break; 
    
    case FWATCH_LINK: 
      strcpy(data->name, oldname); 
      strcat(data->name, " -> "); 
      strcat(data->name, fullnewname); 
      break; 
    
    case FWATCH_UNLINK: 
      strcpy(data->name, fulloldname); 
      break; 
    
    case FWATCH_CREATE: 
      strcpy(data->name, fulloldname); 
      break;

    case FWATCH_CLOSE: 
      strcpy(data->name, fulloldname); 
      break; 
    
    case FWATCH_RENAME: 
      strcpy(data->name, fulloldname); 
      strcat(data->name, " -> "); 
      strcat(data->name, fullnewname); 
      break; 
    
    case FWATCH_CHOWN:
      strcpy(data->name, fulloldname);
      break;
    
    case FWATCH_CHMOD:
      strcpy(data->name, fulloldname);
      break;

    case FWATCH_STAT:
      strcpy(data->name, fulloldname);
      break;
    
    case FWATCH_FSTAT:
      strcpy(data->name, fulloldname);
      break;

    case FWATCH_UTIME:
      strcpy(data->name, fulloldname);
      break;

  default: 
    data->name[0]='\0'; 
  }

#ifdef DEBUG
  printk("OUT log_filename\n");
#endif
}
#  endif
#endif

__inline__ void fwatch_handle_data( const char * from, const char * to, const int operation, fwatch_data_t * data) {

#ifdef FWATCH_HAVENAMES
#  ifdef DEBUG_FULLNAMES
    fwatch_fill_with_fullNames( from, to, operation, data);
#  endif
#endif
  data->func = operation;
  fwatch_put_to_queue(data);
  return;

} 

/* ============================================================================== */

extern int fwatch_rename( const char * oldname, const char * newname ) 
{ 
  int ret; 
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN rename\n");
#endif
  ret = original_rename( oldname, newname ); 
  if ( ret >= 0 ) 
    { 
      if (!fwatch_get_data_for_name(newname,&data) ) {
	fwatch_handle_data( oldname, newname, FWATCH_RENAME, &data ); 
      }
    }
#ifdef DEBUG
  printk("OUT rename\n");
#endif
  return ret; 
}
    
extern int fwatch_open( const char * filename, int flags, int mode)
{
  int ret; 
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN open\n");
#endif
  ret=original_open( filename, flags, mode); 
  if ( ret >= 0 
       && ( flags & O_WRONLY ||
	    flags & O_RDWR ))
	{ 
	  if (!fwatch_get_data_for_name(filename,&data) ) {
	    fwatch_handle_data(filename, NULL, FWATCH_OPENW, &data); 
	  }
	}

#ifdef DEBUG
  printk("OUT open\n");
#endif
  return ret; 

}

extern int fwatch_close( unsigned int fd ) 
{
  struct file *f=NULL; 
  int ret; 
  fwatch_data_t data;

#ifdef DEBUG
  printk("IN close\n");
#endif

  spin_lock(&dcache_lock);

  f = fcheck(fd);
  if ( f && (f->f_mode & FMODE_WRITE) ) { 
    if (!fwatch_get_data_for_file(f,&data) ) {
      fwatch_handle_data(NULL, NULL, FWATCH_CLOSE, &data); 
    }
  }

  spin_unlock(&dcache_lock);
  ret=original_close(fd); 

#ifdef DEBUG
  printk("OUT close\n");
#endif

  return ret; 
}
  
extern int fwatch_rmdir( const char * pathname )
{ 
  int ret; 
  fwatch_data_t data;
  int dirfile;
#ifdef DEBUG
  printk("IN rmdir\n");
#endif

  dirfile = !fwatch_get_data_for_name(pathname,&data);
  ret = original_rmdir( pathname ); 
  if ( ret >= 0 ) 
    { 
      if (dirfile) {
	fwatch_handle_data(pathname, NULL, FWATCH_RMDIR, &data); 
      }
    }
#ifdef DEBUG
  printk("OUT rmdir\n");
#endif

  return ret; 
}

extern int fwatch_mkdir( const char * pathname, int mode)
{
  int ret; 
  fwatch_data_t data;
  int dirfile;
#ifdef DEBUG
  printk("IN mkdir\n");
#endif

  ret = original_mkdir( pathname, mode); 
  dirfile = !fwatch_get_data_for_name(pathname,&data);
  if ( ret >= 0 ) 
	{ 
	  if (dirfile) {
	    fwatch_handle_data(pathname, NULL, FWATCH_MKDIR, &data);
	  }
	}
#ifdef DEBUG
  printk("OUT mkdir\n");
#endif

  return ret; 
}

extern int fwatch_symlink(const char * oldname, 
				const char * newname)
{ 
  int ret; 
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN symlink\n");
#endif

  ret = original_symlink( oldname, newname ); 
  if ( ret >= 0 ) 
	{ 
	  if (!fwatch_get_data_for_name(newname, &data) ) {
	    fwatch_handle_data(oldname, newname, 
				      FWATCH_SYMLINK, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT symlink\n");
#endif

  return ret; 
}


extern int fwatch_link( const char * oldname, 
			      const char * newname )
{
  int ret; 
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN link\n");
#endif

  ret = original_link(oldname, newname); 
  if ( ret >= 0 ) 
	{ 
	  if (!fwatch_get_data_for_name(newname, &data) ) {
	    fwatch_handle_data(newname, oldname, 
				      FWATCH_LINK, &data); 
	  }
	}
  #ifdef DEBUG
  printk("OUT link\n");
#endif

  return ret; 
}

extern int fwatch_unlink( const char * pathname )
{ 
  int ret;
  fwatch_data_t data;
  int dirfile;
#ifdef DEBUG
  printk("IN unlink\n");
#endif

  dirfile = !fwatch_get_data_for_name(pathname,&data);
  ret = original_unlink( pathname ); 
  if ( ret >= 0 ) 
	{ 
	  if (dirfile) {
	    fwatch_handle_data(pathname, NULL, 
				      FWATCH_UNLINK, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT unlink\n");
#endif

  return ret; 
}


extern int fwatch_create ( const char *pathname, int mode)
{ 
  int ret; 
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN create\n");
#endif

  ret = original_create( pathname, mode ); 
  if ( ret >= 0 ) 
	{
	  if (!fwatch_get_data_for_name(pathname,&data) ) {
	    fwatch_handle_data(pathname, NULL, 
				      FWATCH_CREATE, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT create\n");
#endif

  return ret; 
}

extern int fwatch_chown( const char * filename, uid_t user, gid_t group )
{
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN chown\n");
#endif

  ret = original_chown( filename, user, group );
  if ( ret >= 0 )
    {
	  if (!fwatch_get_data_for_name(filename, &data) ) {
	    fwatch_handle_data(filename, NULL, 
				      FWATCH_CHOWN, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT chown\n");
#endif

  return ret;  
}

#ifdef __NR_chown32
extern int fwatch_chown32( const char * filename, uid_t user, gid_t group )
{
  long ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN chown32\n");
#endif

  ret = original_chown32( filename, user, group );
  if ( ret >= 0 )
    {
	  if (!fwatch_get_data_for_name(filename, &data) ) {
	    fwatch_handle_data(filename, NULL, 
				      FWATCH_CHOWN, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT chown32\n");
#endif

  return ret;  
}
#endif

extern int fwatch_lchown( const char * filename, uid_t user, gid_t group )
{
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN lchown\n");
#endif

  ret = original_lchown( filename, user, group );
  if ( ret >= 0 )
    {
	  if (!fwatch_get_data_for_name(filename, &data) ) {
	    fwatch_handle_data(filename, NULL, 
				      FWATCH_CHOWN, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT lchown\n");
#endif

  return ret;  
}


#ifdef __NR_lchown32
extern int fwatch_lchown32( const char * filename, uid_t user, gid_t group )
{
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN lchown32\n");
#endif

  ret = original_lchown32( filename, user, group );
  if ( ret >= 0 )
    {
	  if (!fwatch_get_data_for_name(filename, &data) ) {
	    fwatch_handle_data(filename, NULL, 
				      FWATCH_CHOWN, &data); 
	  }
	}
#ifdef DEBUG
  printk("OUT lchown32\n");
#endif

  return ret;  
}
#endif

extern int fwatch_chmod( const char * filename, mode_t mode )
{
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN chmod\n");
#endif

  ret = original_chmod( filename, mode );
  if ( ret >= 0 )
    {
	  if (!fwatch_get_data_for_name(filename, &data) ) {
	    fwatch_handle_data(filename, NULL, 
				      FWATCH_CHMOD, &data); 
	  }
    }
#ifdef DEBUG
  printk("OUT chmod\n");
#endif

  return ret;
}

extern int fwatch_stat(const char* filename, struct stat * statbuf) {
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN stat\n");
#endif

  ret = original_stat( filename, statbuf );
  if ( !ret ) {
    if (!fwatch_get_data_for_name(filename, &data) ) {
      fwatch_handle_data(filename, NULL, 
			 FWATCH_STAT, &data); 
    }
  }
#ifdef DEBUG
  printk("OUT stat\n");
#endif

  return ret;
  
}

extern int fwatch_fstat(unsigned int fd, struct stat * statbuf) {

  struct file *f=NULL; 
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN fstat\n");
#endif

  spin_lock(&dcache_lock);
  f = fcheck(fd);
  ret = original_fstat( fd, statbuf );
  if ( f && (!ret) ) {
    if (!fwatch_get_data_for_file(f, &data) ) {
      fwatch_handle_data(NULL, NULL, 
			 FWATCH_FSTAT, &data); 
    }
  }
  spin_unlock(&dcache_lock);
#ifdef DEBUG
  printk("OUT stat\n");
#endif

  return ret;
}

extern int fwatch_stat64(const char* filename, struct stat64 * statbuf, long flags) {
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN stat\n");
#endif

  ret = original_stat64( filename, statbuf, flags);
  if ( !ret ) {
    if (!fwatch_get_data_for_name(filename, &data) ) {
      fwatch_handle_data(filename, NULL, 
			 FWATCH_STAT, &data); 
    }
  }
#ifdef DEBUG
  printk("OUT stat\n");
#endif

  return ret;
  
}

extern int fwatch_fstat64(unsigned long fd, struct stat64 * statbuf, long flags) {

  struct file *f=NULL; 
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN fstat\n");
#endif

  spin_lock(&dcache_lock);
  f = fcheck(fd);
  ret = original_fstat64( fd, statbuf, flags);
  if ( f && (!ret) ) {
    if (!fwatch_get_data_for_file(f, &data) ) {
      fwatch_handle_data(NULL, NULL, 
			 FWATCH_FSTAT, &data); 
    }
  }
  spin_unlock(&dcache_lock);
#ifdef DEBUG
  printk("OUT stat\n");
#endif

  return ret;
}

extern ssize_t fwatch_write(unsigned int fd, const char * buf, size_t count) {
  return original_write(fd,buf,count);
}

extern long fwatch_utime(char * filename, struct utimbuf * times) {
  int ret;
  fwatch_data_t data;
#ifdef DEBUG
  printk("IN utime\n");
#endif

  ret = original_utime( filename, times);
  if ( !ret ) {
    if (!fwatch_get_data_for_name(filename, &data) ) {
      fwatch_handle_data(filename, NULL, 
			 FWATCH_UTIME, &data); 
    }
  }
#ifdef DEBUG
  printk("OUT utime\n");
#endif

  return ret;
  
}

/* ============================================================================== */

int fwatch_addorreleasefn( int add, int mode) {
  extern long sys_call_table[]; 
  if (mode & FWATCH_OPENW) {
    if (add) {
      original_open = (int (*)(const char *, int, int))(sys_call_table[__NR_open]); 
      sys_call_table[__NR_open] = (unsigned long)fwatch_open; 
    } else {
      sys_call_table[__NR_open] = (unsigned long)original_open; 
    }
  }
  if (mode & FWATCH_OPENR) {
    if (add) {
    } else {
    }
  }
  if (mode & FWATCH_MKDIR) {
    if (add) {
      original_mkdir = (int (*)(const char *, int))(sys_call_table[__NR_mkdir]); 
      sys_call_table[__NR_mkdir] = (unsigned long)fwatch_mkdir; 
    } else {
      sys_call_table[__NR_mkdir] = (unsigned long)original_mkdir; 
    }
  }
  if (mode & FWATCH_RMDIR) {
    if (add) {
      original_rmdir = (int (*)(const char *))(sys_call_table[__NR_rmdir]); 
      sys_call_table[__NR_rmdir] = (unsigned long)fwatch_rmdir;
    } else {
      sys_call_table[__NR_rmdir] = (unsigned long)original_rmdir; 
    }
  }
  if (mode & FWATCH_SYMLINK) {
    if (add) {
      original_symlink = (int (*)(const char *, const char *))(sys_call_table[__NR_symlink]); 
      sys_call_table[__NR_symlink] = (unsigned long)fwatch_symlink; 
    } else {
      sys_call_table[__NR_symlink] = (unsigned long)original_symlink; 
    }
  }
  if (mode & FWATCH_LINK) {
    if (add) {
      original_link = (int (*)(const char *, const char *))(sys_call_table[__NR_link]);
      sys_call_table[__NR_link] = (unsigned long)fwatch_link; 
    } else {
      sys_call_table[__NR_link] = (unsigned long)original_link; 
    }
  }
  if (mode & FWATCH_UNLINK) {
    if (add) {
      original_unlink = (int (*)(const char *))(sys_call_table[__NR_unlink]); 
      sys_call_table[__NR_unlink] = (unsigned long)fwatch_unlink; 
    } else {
      sys_call_table[__NR_unlink] = (unsigned long)original_unlink; 
    }
  }
#ifdef __NR_creat
  if (mode & FWATCH_CREATE) {
    if (add) {
      original_create = (int (*)(const char *, int))(sys_call_table[__NR_creat]);  
      sys_call_table[__NR_creat] = (unsigned long)fwatch_create; 
    } else {
      sys_call_table[__NR_creat] = (unsigned long)original_create; 
    }
  }
#endif
  if (mode & FWATCH_CLOSE) {
    if (add) {
      original_close = (int (*)(unsigned int))(sys_call_table[__NR_close]);
      sys_call_table[__NR_close] = (unsigned long)fwatch_close; 
    } else {
      sys_call_table[__NR_close] = (unsigned long)original_close;
    }
  }
  if (mode & FWATCH_RENAME) {
    if (add) {
      original_rename = (int (*)(const char *, const char *))(sys_call_table[__NR_rename]);
      sys_call_table[__NR_rename] = (unsigned long)fwatch_rename; 
    } else {
      sys_call_table[__NR_rename] = (unsigned long)original_rename;
    }
  }
  if (mode & FWATCH_CHOWN) {
    if (add) {
      original_chown = (int (*)(const char *, uid_t, gid_t))(sys_call_table[__NR_chown]);
      sys_call_table[__NR_chown] = (unsigned long)fwatch_chown;
#ifdef __NR_chown32
      original_chown32 = (int (*)(const char *, uid_t, gid_t))(sys_call_table[__NR_chown32]);
      sys_call_table[__NR_chown32] = (unsigned long)fwatch_chown32;
#endif
      original_lchown = (int (*)(const char *, uid_t, gid_t))(sys_call_table[__NR_lchown]);
      sys_call_table[__NR_lchown] = (unsigned long)fwatch_lchown;
#ifdef __NR_lchown32
      original_lchown32 = (int (*)(const char *, uid_t, gid_t))(sys_call_table[__NR_lchown32]);
      sys_call_table[__NR_lchown32] = (unsigned long)fwatch_lchown32;
#endif
      original_fchown = (int (*)(const char *, uid_t, gid_t))(sys_call_table[__NR_fchown]);
      sys_call_table[__NR_fchown] = (unsigned long)fwatch_lchown;
    } else {
      sys_call_table[__NR_chown] = (unsigned long)original_chown;
#ifdef __NR_chown32
      sys_call_table[__NR_chown32] = (unsigned long)original_chown32;
#endif
      sys_call_table[__NR_lchown] = (unsigned long)original_lchown;
#ifdef __NR_lchown32
      sys_call_table[__NR_lchown32] = (unsigned long)original_lchown32;
#endif
      sys_call_table[__NR_fchown] = (unsigned long)original_fchown;
    }
  }
  if (mode & FWATCH_CHMOD) {
    if (add) {
      original_chmod = (int (*)(const char *, mode_t))(sys_call_table[__NR_chmod]);
      sys_call_table[__NR_chmod] = (unsigned long)fwatch_chmod;
    } else {
      sys_call_table[__NR_chmod] = (unsigned long)original_chmod;
    }
  }
  if (mode & FWATCH_STAT) {
    if (add) {
      original_stat = (int (*)(const char*, struct stat *))(sys_call_table[__NR_stat]);
      sys_call_table[__NR_stat] = (unsigned long)fwatch_stat;
#ifdef __NR_stat64
      original_stat64 = (int(*)(const char*, struct stat64 *, long))(sys_call_table[__NR_stat64]);
      sys_call_table[__NR_stat64] = (unsigned long)fwatch_stat64;
#endif
    } else {
      sys_call_table[__NR_stat] = (unsigned long)original_stat;
#ifdef __NR_stat64
      sys_call_table[__NR_stat64] = (unsigned long)original_stat64;
#endif
    }
  }
  if (mode & FWATCH_FSTAT) {
    if (add) {
      original_fstat = (int (*)(unsigned int, struct stat *))(sys_call_table[__NR_fstat]);
      sys_call_table[__NR_fstat] = (unsigned long)fwatch_fstat;
#ifdef __NR_fstat64
      original_fstat64 = (int(*)(unsigned long, struct stat64 *, long))(sys_call_table[__NR_fstat64]);
      sys_call_table[__NR_fstat64] = (unsigned long)fwatch_fstat64;
#endif
    } else {
      sys_call_table[__NR_fstat] = (unsigned long)original_fstat;
#ifdef __NR_fstat64
      sys_call_table[__NR_fstat64] = (unsigned long)original_fstat64;
#endif
    }
  }
  if (mode & FWATCH_WRITE) {
    if (add) {
      original_write = (int(*)(unsigned int, const char *, size_t))(sys_call_table[__NR_write]);
      sys_call_table[__NR_write] = (unsigned long) fwatch_write;
    } else {
      sys_call_table[__NR_write] = (unsigned long) original_write;
    }
  }
  return 0;
  if (mode & FWATCH_UTIME) {
    if (add) {
      original_utime = (int(*)(char * filename, struct utimbuf * times))(sys_call_table[__NR_utime]);
      sys_call_table[__NR_utime] = (unsigned long) fwatch_utime;
    } else {
      sys_call_table[__NR_utime] = (unsigned long) original_utime;
    }
  }
  return 0;
}

int fwatch_switch_mode( int mode) {

  extern long sys_call_table[]; 
  spin_lock(&sys_call_table);
  fwatch_addorreleasefn(0,fwatch_mode);
  fwatch_addorreleasefn(1,mode);
  fwatch_mode = mode;
  spin_unlock(&sys_call_table);
  return 0;
}

int fwatch_ioctl( struct inode * inode, struct file * file, unsigned int ioctl_num, unsigned long ioctl_param) {

  switch (ioctl_num) {

    case FWATCH_EMERGENCYUNLOAD:
      printk("fwatch: EMERGENCYUNLOAD: MOD_IN_USE == %d\n", MOD_IN_USE);
      while(MOD_IN_USE > 1) MOD_DEC_USE_COUNT;
      break;
    case FWATCH_SETMODE:
      fwatch_switch_mode(ioctl_param);
      break;
    default: printk("fwatch: No such ioctl param: %lu\n",ioctl_param);
      break;
  }

  return 0;
}

int fwatch_open_dev( struct inode *in, struct file * fi ) { 
  if (fwatch_device_open) {
    return -EBUSY;
  }
  fwatch_device_open++;
  MOD_INC_USE_COUNT;
  return 0; 
}

int fwatch_close_dev( struct inode *in, struct file * fi ) {
  MOD_DEC_USE_COUNT; 
  fwatch_device_open--;
  return 0; 
}

ssize_t fwatch_read_dev( struct file *filep, char *buf, size_t count, loff_t *f_pos ) { 
  int mycount;
  int copied;
  fwatch_data_t data;

  if ( count > sizeof(fwatch_data_t) ) { 
    mycount = sizeof(fwatch_data_t); 
  } else { 
    mycount = count; 
  }

  if (fwatch_get_from_queue(&data)) {
    return 0;
  }

  if ( (copied = copy_to_user(buf, &data, mycount)) ) { 
    /*        printk("fwatch:  copy_to_user failed: %d\n",copied);  */
    return mycount-copied; 
  }
  return mycount; 
}

static unsigned int fwatch_poll(struct file *file, struct poll_table_struct *table) {

  spin_lock(&fwatch_lock);
  if (queue_head == queue_tail) {
    poll_wait(file, &fwatch_wait, table);
    spin_unlock(&fwatch_lock);
    return 0;
  }
  return POLLIN | POLLRDNORM;

}

static struct file_operations fwatch_fop = { 
  read: fwatch_read_dev, 
  poll: fwatch_poll, 
  ioctl: fwatch_ioctl,
  open: fwatch_open_dev, 
  release: fwatch_close_dev, 
}; 

int init_module() {

  printk("fwatch: module loaded, version: 0.1 alpha\n");

  init_waitqueue_head(&fwatch_wait);
  if ( register_chrdev(fwatch_major, "fwatch", &fwatch_fop)) { 
    return -EIO; 
  }
  return 0; 
}

void cleanup_module() { 

  printk("fwatch: unloading -- freeing all resources\n"); 
  fwatch_addorreleasefn(0,fwatch_mode);

  unregister_chrdev(fwatch_major, "fwatch"); 
}
