
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <time.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdint.h>

#include <abz/error.h>

#include "clients.h"
#include "gui.h"
#include "io.h"

#define ARRAYSIZE(x) (sizeof (x) / sizeof ((x)[0]))

static void hline (int x1,int x2,int y)
{
   int x;
   out_gotoxy (x1,y);
   for (x = x1; x <= x2; x++) out_putc (line.hl);
}

static void vline (int x,int y1,int y2)
{
   int y;
   for (y = y1; y <= y2; y++)
	 {
		out_gotoxy (x,y);
		out_putc (line.vl);
	 }
}

static void showtime (int width,int height)
{
   time_t tp = time (NULL);
   struct tm *tm = localtime (&tp);
   out_setcolor (COLOR_WHITE,COLOR_BLACK);
   out_gotoxy (width - 10,height - 2);
   out_printf ("%02u:%02u:%02u",tm->tm_hour,tm->tm_min,tm->tm_sec);
}

static void showrate (int x,int y,double rate,const struct bandwidth *bw)
{
   char buf[13];
   int i,n = 0;
   const char *suffix[] = { "", "k", "m" };

   out_setattr (ATTR_BOLD);

   if (rate <= bw->rate)
	 out_setcolor (COLOR_GREEN,COLOR_BLACK);
   else if (rate <= bw->peak)
	 out_setcolor (COLOR_YELLOW,COLOR_BLACK);
   else
	 out_setcolor (COLOR_RED,COLOR_BLACK);

   rate *= 8.0;
   for (i = 0; i < 3; i++)
	 if (rate > 1000.0)
	   rate /= 1000.0, n++;

   out_gotoxy (x,y);
   snprintf (buf,sizeof (buf),"%.1f%s",rate,suffix[n]);
   out_printf ("%11s",strlen (buf) <= 11 ? buf : "overflow");
}

static size_t frame_clients (const struct client *clients,size_t offset,size_t total,time_t period)
{
   int width = out_width (),height = out_height ();
   char buf[width - 31];
   const struct client *tmp;
   size_t i;
   struct rate avg;

   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_BLUE,COLOR_BLACK);

   hline (1,width - 2,0);
   hline (1,width - 2,2);
   hline (1,width - 2,height - 3);
   hline (1,width - 2,height - 1);

   vline (0,1,height - 2);
   vline (width - 29,1,height - 4);
   vline (width - 15,1,height - 4);
   vline (width - 1,1,height - 2);

   out_gotoxy (0,0);
   out_putc (line.tl);
   out_gotoxy (width - 29,0);
   out_putc (line.tt);
   out_gotoxy (width - 15,0);
   out_putc (line.tt);
   out_gotoxy (width - 1,0);
   out_putc (line.tr);

   out_gotoxy (0,2);
   out_putc (line.lt);
   out_gotoxy (width - 29,2);
   out_putc (line.ct);
   out_gotoxy (width - 15,2);
   out_putc (line.ct);
   out_gotoxy (width - 1,2);
   out_putc (line.rt);

   out_gotoxy (0,height - 3);
   out_putc (line.lt);
   out_gotoxy (width - 29,height - 3);
   out_putc (line.bt);
   out_gotoxy (width - 15,height - 3);
   out_putc (line.bt);
   out_gotoxy (width - 1,height - 3);
   out_putc (line.rt);

   out_gotoxy (0,height - 1);
   out_putc (line.bl);
   out_gotoxy (width - 1,height - 1);
   out_putc (line.br);

   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   out_gotoxy (2,1);
   out_puts ("Description");
   out_gotoxy (width - 21,1);
   out_puts ("Input");
   out_gotoxy (width - 8,1);
   out_puts ("Output");
   out_gotoxy (2,height - 2);
   out_puts ("Frogfoot Traffic Monitoring (press h for help)");

   showtime (width,height);

   for (tmp = clients, i = 0; i < offset; tmp = tmp->next, i++) ;

   if (tmp != clients)
	 {
		out_setcolor (COLOR_YELLOW,COLOR_BLACK);
		out_gotoxy (width - 29,3);
		out_putc ('-');
	 }

   for (i = 0; tmp != NULL && i <= height - 7; tmp = tmp->next, i++)
	 {
		strncpy (buf,tmp->name,sizeof (buf));
		buf[sizeof (buf) - 1] = '\0';

		out_gotoxy (2,i + 3);
		out_setattr (ATTR_OFF);
		out_setcolor (COLOR_CYAN,COLOR_BLACK);
		out_printf ("%-*s",width - 31,buf);

		if (strlen (tmp->name) > width - 32)
		  {
			 out_gotoxy (width - 34,i + 3);
			 out_setattr (ATTR_BOLD);
			 out_setcolor (COLOR_YELLOW,COLOR_BLACK);
			 out_puts (" ...");
		  }

		if (!client_calc_rate (&avg,tmp,period))
		  {
			 showrate (width - 27,i + 3,avg.input,&tmp->input);
			 showrate (width - 13,i + 3,avg.output,&tmp->output);
		  }
	 }

   out_setattr (ATTR_BOLD);

   if (tmp != NULL)
	 {
		out_setcolor (COLOR_YELLOW,COLOR_BLACK);
		out_gotoxy (width - 29,height - 4);
		out_putc ('+');
	 }

   out_setcolor (COLOR_WHITE,COLOR_BLACK);
   out_gotoxy (width - 22,height - 2);
   out_printf ("%3u-%u/%u",offset,offset + i,total);

   out_gotoxy (width - 33,height - 2);
   out_printf ("%3lu%c avg",
			   period < 60 ? period : period / 60,
			   period < 60 ? 's' : 'm');

   out_refresh ();

   return (offset + i);
}

static void frame_help ()
{
   int i,j,width = out_width (),height = out_height ();
   static const struct
	 {
		const char *key;
		const char *msg;
	 } help[] =
	 {
		{ "Up", "Scroll up (if possible)" },
		{ "Down", "Scroll down (if possible)" },
		{ "^L", "Redraw the screen" },
		{ "h or ?", "Print this list" },
		{ "5", "Show 5 second averages (default)" },
		{ "1", "Show 10 second averages" },
		{ "3", "Show 30 second averages" },
		{ "6", "Show 1 minute averages" },
		{ "r", "Reset packet counters" },
		{ "q", "Quit" }
	 };
   static const char *msg[] =
	 {
		"Rates are displayed in bits per second. Rates displayed in green are",
		"lower than the expected bitrate. Rates displayed in yellow exceed the",
		"expected bitrate, but are still lower than the peak bitrate. If rates",
		"exceed the peak (or if no peak was defined) they are coloured red."
	 };

   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_BLUE,COLOR_BLACK);

   hline (1,width - 2,0);
   hline (1,width - 2,height - 3);
   hline (1,width - 2,height - 1);

   vline (0,1,height - 2);
   vline (width - 1,1,height - 2);

   out_gotoxy (0,0);
   out_putc (line.tl);
   out_gotoxy (width - 1,0);
   out_putc (line.tr);

   out_gotoxy (0,height - 3);
   out_putc (line.lt);
   out_gotoxy (width - 1,height - 3);
   out_putc (line.rt);

   out_gotoxy (0,height - 1);
   out_putc (line.bl);
   out_gotoxy (width - 1,height - 1);
   out_putc (line.br);

   showtime (width,height);

   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   out_gotoxy (2,2);
   out_puts ("Interactive commands");
   out_gotoxy (2,height - 2);
   out_puts ("Frogfoot Traffic Monitoring (press any key to continue)");

   for (i = 0; i < ARRAYSIZE (help); i++)
	 {
		out_setattr (ATTR_BOLD);
		out_setcolor (COLOR_WHITE,COLOR_BLACK);
		out_gotoxy (4,4 + i);
		out_printf ("%-9s",help[i].key);

		out_setattr (ATTR_OFF);
		out_setcolor (COLOR_CYAN,COLOR_BLACK);
		out_puts (help[i].msg);
	 }

   out_setattr (ATTR_BOLD);
   out_setcolor (COLOR_MAGENTA,COLOR_BLACK);
   out_gotoxy (2,5 + i);
   out_puts ("Caveats");

   out_setattr (ATTR_OFF);
   out_setcolor (COLOR_CYAN,COLOR_BLACK);
   for (i += 7, j = 0; j < ARRAYSIZE(msg); j++, i++)
	 {
		out_gotoxy (4,i);
		out_puts (msg[j]);
	 }

   out_refresh ();
}

static int finished = 0;

static void catchme (int sig)
{
   finished = 1;
}

/*
 * Monitor clients. Return 0 if successful, -1 otherwise.
 * You can check the error message with abz_get_error().
 */
int gui (struct client *clients)
{
   struct timeval tv;
   fd_set rfds;
   const struct client *tmp;
   size_t head = 0,tail = 0,total = 0;
   time_t period = 5;
   enum { CLIENTS, HELP } prevframe = CLIENTS,curframe = CLIENTS;
   int c = 0,input = 0;

   FD_ZERO (&rfds);

   for (tmp = clients; tmp != NULL; tmp = tmp->next) total++;

   signal (SIGINT,catchme);
   signal (SIGTERM,catchme);
   signal (SIGSTOP,SIG_IGN);
   signal (SIGTSTP,SIG_IGN);

   io_open ();

   while (!finished)
	 {
		tv.tv_sec = 1, tv.tv_usec = 0;
		FD_SET (STDIN_FILENO,&rfds);

		if (curframe != prevframe)
		  out_clear ();

		if (curframe == CLIENTS)
		  tail = frame_clients (clients,head,total,period);
		else
		  frame_help ();

		switch (select (STDIN_FILENO + 1,&rfds,NULL,NULL,&tv))
		  {
		   case -1:
			 input = 0;
			 if (errno != EINTR)
			   {
				  abz_set_error ("failed to poll stdin: %s",strerror (errno));
				  input = -1;
			   }
			 break;

		   case 0:
			 input = 0;
			 break;

		   default:
			 input = 1;
			 c = in_getc ();
		  }

		prevframe = curframe;

		if (input < 0)
		  break;

		if (input)
		  {
			 if (curframe == CLIENTS)
			   {
				  if (c == KEY_UP)
					{
					   if (head) head--; else out_beep ();
					}
				  else if (c == KEY_DOWN)
					{
					   if (tail < total) head++; else out_beep ();
					}
				  else if (c == KEY_CTRL('l') || c == KEY_CTRL('L'))
					out_clear ();
				  else if (c == 'h' || c == '?')
					curframe = HELP;
				  else if (c == '5')
					period = 5;
				  else if (c == '1')
					period = 10;
				  else if (c == '3')
					period = 30;
				  else if (c == '6')
					period = 60;
				  else if (c == 'r')
					{
					   if (client_reset (clients) < 0)
						 break;
					}
				  else if (c == 'q')
					finished = 1;
				  else out_beep ();
			   }
			 else curframe = CLIENTS;
		  }

		if (client_update (clients) < 0)
		  break;
	 }

   io_close ();

   signal (SIGINT,SIG_DFL);
   signal (SIGTERM,SIG_DFL);
   signal (SIGSTOP,SIG_DFL);
   signal (SIGTSTP,SIG_DFL);

   return (finished - 1);
}

