/*
 *	fhist - file history and comparison tools
 *	Copyright (C) 2000, 2002 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to deliver output to files
 */

#include <ac/errno.h>
#include <ac/stdio.h>

#include <fcheck.h>
#include <mem.h>
#include <output/file.h>
#include <output/private.h>
#include <output/stdout.h>
#include <error_intl.h>


typedef struct output_file_ty output_file_ty;
struct output_file_ty
{
	output_ty	inherited;
	char		*filename;
	FILE		*deeper;
};


static void
destructor(output_ty *fp)
{
	output_file_ty	*this;

	this = (output_file_ty *)fp;
	fflush_and_check(this->deeper, this->filename);
	fclose_and_check(this->deeper, this->filename);
	mem_free(this->filename);
}


static const char *
filename(output_ty *fp)
{
	output_file_ty	*this;

	this = (output_file_ty *)fp;
	return this->filename;
}


static long
otell(output_ty *fp)
{
	output_file_ty	*this;

	this = (output_file_ty *)fp;
	return ftell(this->deeper);
}


static void
oputc(output_ty *fp, int c)
{
	output_file_ty	*this;

	this = (output_file_ty *)fp;
	if (putc(c, this->deeper) == EOF)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set_charstar(scp, "File_Name", this->filename);
		fatal_intl(scp, i18n("write \"$filename\": $errno"));
		/* NOTREACHED */
	}
}


static void
owrite(output_ty *fp, const void *data, size_t len)
{
	output_file_ty	*this;

	this = (output_file_ty *)fp;
	if (fwrite(data, 1, len, this->deeper) == EOF)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set_charstar(scp, "File_Name", this->filename);
		fatal_intl(scp, i18n("write \"$filename\": $errno"));
		/* NOTREACHED */
	}
}


static void
oflush(output_ty *fp)
{
	output_file_ty	*this;

	this = (output_file_ty *)fp;
	if (fflush(this->deeper))
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_errno_set(scp);
		sub_var_set_charstar(scp, "File_Name", this->filename);
		fatal_intl(scp, i18n("write \"$filename\": $errno"));
		/* NOTREACHED */
	}
}


static output_vtbl_ty vtbl =
{
	sizeof(output_file_ty),
	destructor,
	filename,
	otell,
	oputc,
	output_generic_fputs,
	owrite,
	oflush,
	"file",
};


static output_ty *
output_file_open(const char *fn, const char *mode)
{
	output_ty	*result;
	output_file_ty	*this;

	if (!fn || !*fn)
		return output_stdout();
	result = output_new(&vtbl);
	this = (output_file_ty *)result;
	this->deeper = fopen_and_check(fn, mode);
	this->filename = mem_copy_string(fn);
	return result;
}


output_ty *
output_file_text_open(const char *fn)
{
	return output_file_open(fn, "w");
}


output_ty *
output_file_binary_open(const char *fn)
{
	return output_file_open(fn, "wb");
}
