/***************************************************************************
                          FLSqlCursor.h  -  description
                             -------------------
    begin                : Fri Jul 27 2001
    copyright            : (C) 2001,2002 by Federico Albujer Zornoza
    email                : mail@infosial.com
 ***************************************************************************/

/***************************************************************************
 *                                            				   *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FLSQLCURSOR_H
#define FLSQLCURSOR_H

#include <qvariant.h>
#include <qsqlcursor.h>
#include <qsqlindex.h>
#include <qobject.h>
#include <qmessagebox.h>
#include <qdatetime.h>

class FLTableMetaData;
class FLRelationMetaData;
class FLAction;

/**  Esta clase es una extensin de la clase QSqlCursor
    	para la necesidades especificas de FacturaLUX.

    	Hereda de QSqlCursor, y aade ciertas funcionalidades.
    	Esta clase se utiliza para gestionar los registros de
    	una sola tabla o una consulta de ella. Tambien mantiene
    	los metadatos (ver FLTableMetaData) de dicha tabla.
    	
    	El funcionamiento de esta clase radica en el modo de acceso
    	en el que se encuentre, INSERTAR,MODIFICAR,BORRAR,NAVEGAR.
    	Se mantiene siempre un registro completo del cursor, en un buffer,
    	dependiendo del modo de acceso, este buffer contendr un registro
    	listo para INSERTAR en blanco, para MODIFICAR con el contenido
    	del registro actual del cursor, para BORRAR con el contenio del registro
    	actual listo para borrar, o para NAVEGAR con el contenido del registro
    	actual en mdo de slo lectura.
    	
    	Las acciones asociadas a INSERTAR,MODIFICAR, o BORRAR, son llevadas
    	a cabo cuando se enva el contenido del buffer de nuevo al cursor, con FLSqlCursor::commitBuffer(),
    	esta accin lleva implicita un chequeo de integridad para que se realize, ver FLSqlCursor::checkIntegrity().
    	
    	@autor	Federico Albujer Zornoza
    	@version 0.4*/
class FLSqlCursor:public QObject, public QSqlCursor
{
	Q_OBJECT

  	public:

  	/** constructor.

      	@param cR	Cursor relacionado con este. Al indicar que este cursor depende de otro,
                        	el contenido del cursor est en funcin del valor actual del campo por el que se
                        	establece la relacin del cursor relacionado.
      	@param r	Relacin que indica como se relacionan los dos cursores. */
  	FLSqlCursor (const QString & name = QString::null, bool autopopulate=true,
				QSqlDatabase* db=0, FLSqlCursor * cR = 0, FLRelationMetaData * r = 0);

  	/** destructor.  */
  	~FLSqlCursor ();

  	/** Constates para indicar el tipo de acceso al cursor */
  	enum Mode
  	{
    		/** Insertar, en este modo el buffer se prepara para crear un nuevo registro */
    		INSERT = 0,
    		/** Edicin, en este modo el buffer se prepara para editar el registro activo */
    		EDIT = 1,
    		/** Borrar, en este modo el buffer se prepara para borrar el registro activo */
    		DEL = 2,
     		/** Navegacion, en este modo solo se puede visualizar el buffer */
    		BROWSE = 3
	};

	/** Para obtener los metadatos de la tabla.
	
		@return	Objeto FLTableMetaData con los metadatos de la tabla asociada al cursor */
	FLTableMetaData *metadata () const
	{
    		return metadata_;
	}

	/** Para obtener el modo de acceso actual del cursor.

		@return	Constante FLSqlCursor::Mode que define en que modo de acceso esta preparado el buffer del cursor */
	int modeAccess () const
	{
    		return modeAccess_;
	}

	/** Para obtener el filtro principal del cursor.

		@return Cadena de texto con el filtro principal. */
	QString mainFilter() const
	{
		return mainFilter_;
	}

	/** Para obtener la accion asociada al cursor.

		@return  Objeto FLAction */
	FLAction * action()
	{
		return action_;
	}

	/** Establece la accion asociada al cursor.

		@param  a Objeto FLAction */
	void setAction(FLAction * a);

	/** Establece el filtro principal del cursor.

		@param f Cadena con el filtro, corresponde con una clausura WHERE. */
	void setMainFilter(const QString & f);

	/** Establece el modo de acceso para el cursor.

		@param m	Constante FLSqlCursor::Mode que indica en que modo de acceso se quiere establecer el cursor */
	void setModeAccess (const int m);

	/** Establece el valor de un campo del buffer con un valor.

		@param fN	Nombre del campo
		@param v	Valor a establecer para el campo */
	void setValueBuffer (const QString & fN, const QVariant & v);

	/** Devuelve el valor de un campo del buffer.
	
		@param fN	Nombre del campo */
	QVariant valueBuffer (const QString & fN) const;

	/** Establece el valor de FLSqlCursor::edition.
	
		@param b	TRUE o FALSE */
	void setEdition(const bool b)
	{	
		edition=b;
	}

       /** Establece el valor de FLSqlCursor::browse.

		@param b	TRUE o FALSE */
	void setBrowse(const bool b)
	{
		browse=b;
	}
	/** Dice si un campo est deshabilitado.
		
		Un campo estar deshabilitado, porque est clase le dar un valor automticamente.
		Estos campos son lo campos que estn en una relacin con otro cursor, por lo que
		su valor lo toman del campo forneo con el que se relacionan. El objeto FLReceiver pasa
		a ser propiedad del objeto FLSqlCursor, por lo tanto al destruir el objeto FLSqlCursor se
		borrar automticamente el objeto FLReceiver.
		
		@param fN	Nombre del campo a comprobar.
		@return	TRUE si est deshabilitado y FALSE en caso contrario. */
	bool fieldDisabled(const QString &fN);

	/** Inicia una transaccin.
	
		Si ya hay una transaccin en curso no hace nada.
        @return TRUE si la operacin tuvo exito */
	bool transaction();

	/** Deshace las operaciones de una transaccin y la acaba.
	
		@return TRUE si la operacin tuvo exito */
	bool rollback();

	/** Hace efectiva la transaccin y la acaba.

        @return TRUE si la operacin tuvo exito*/
	bool commit();

   /** Abre el formulario asociado a la tabla origen en el modo indicado.

		@param m Modo de apertura (FLSqlCursor::Mode)
      @param cont Indica que se abra el formulario de edicin de registros con el botn de
                          aceptar y continuar */
	void openFormInMode (int m, bool cont=true);

	/** Copia el contenido del FLSqlCursor::buffer_ actual en FLSqlCursor::bufferCopy_.

		Al realizar esta copia se podra comprobar posteriormente si el buffer actual y la copia realizada
		difieren mediante el metodo FLSqlCursor::isModifiedBuffer(). */
	void updateBufferCopy();

	/** Indica si el contenido actual del buffer difiere de la copia guardada.

		Ver FLSqlCursor::bufferCopy_ .
		@return	TRUE si el buffer y la copia son distintas, FALSE en caso contrario. */
	bool isModifiedBuffer();

	public slots:
	
	/** Refresca el contenido del cursor.
	
		Si no se ha indicado cursor relacionado obtiene el cursor completo, segun la consulta
		por defecto. Si se ha indicado que depende de otro cursor con el que se relaciona,
		el contenido del cursor depender del valor del campo que determina la relacin.
      Si se indica el nombre de un campo se considera que el buffer slo ha cambiado en ese
      campo y as evitar repeticiones en el refresco.

      @param    fN  Nombre del campo de buffer que ha cambiado.*/
	void refresh (QString fN=QString::null);

	/** Refresca el buffer segun el modo de acceso establecido.
	
		Lleva informacion del cursor al buffer para editar o navegar, o prepara el buffer para
		insertar o borrar. */
	void refreshBuffer ();

	/** Redefinicion del mtdo seek() de QSqlCursor.
	
		Este mtodo simplemente invoca al mtodo seek() original de QSqlCursor() y refresca
		el buffer con el metodo FLSqlCursor::refreshBuffer(). */
	bool seek (int i, bool relative = FALSE);

	/** Redefinicion del mtdo next() de QSqlCursor.
		
		Este mtodo simplemente invoca al mtodo next() original de QSqlCursor() y refresca el
		buffer con el metodo FLSqlCursor::refreshBuffer(). */
	bool next ();

	/** Redefinicion del mtdo prev() de QSqlCursor.
	
		Este mtodo simplemente invoca al mtodo prev() original de QSqlCursor() y refresca
		el buffer con el metodo FLSqlCursor::refreshBuffer(). */
	bool prev ();

	/** Redefinicion del mtdo first() de QSqlCursor.
	
		Este mtodo simplemente invoca al mtodo first() original de QSqlCursor() y refresca el
		buffer con el metodo FLSqlCursor::refreshBuffer(). */
	bool first ();

	/** Redefinicion del mtdo last() de QSqlCursor.
	
		Este mtodo simplemente invoca al mtodo last() original de QSqlCursor() y refresca el
		buffer con el metodo FLSqlCursor::refreshBuffer(). */
	bool last ();

	/** Redefinicion del mtdo del() de QSqlCursor.
	
		Este mtodo invoca al mtodo del() original de QSqlCursor() y comprueba si hay borrado
		en cascada, en caso afirmativo borrar tambin los registros relacionados en cardinalidad 1M . */
	int del ( bool invalidate = true);

   /** Redefinicion del mtodo select() de QSqlCursor */
   bool select (const QString & filter, const QSqlIndex & sort = QSqlIndex());

	/** Abre el formulario de edicion de registro definido en los metadatos (FLTableMetaData) listo
		para insertar un nuevo registro en el cursor. */
	void insertRecord ();

	/** Abre el formulario de edicion de registro definido en los metadatos (FLTableMetaData) listo
		para editar el registro activo del cursor. */
	void editRecord ();

	/** Abre el formulario de edicion de registro definido en los metadatos (FLTableMetaData) listo
		para slo visualizar el registro activo del cursor. */
	void browseRecord ();

	/** Borra, pidiendo confirmacion, el registro activo del cursor. */
	void deleteRecord ();

	/** Realiza la accin asociada a elegir un registro del cursor, por defecto se abre el formulario de
		edicin de registro,llamando al mtodo FLSqlCursor::editRecord(), si la bandera FLSqlCursor::edition
		indica TRUE, si indica FALSE este mtodo no hace nada */
	void chooseRecord ();

	/** Manda el contenido del buffer al cursor, o realiza la accin oportuna para el cursor.
	
		Todos los cambios realizados en el buffer se hacen efectivos en el cursor al invocar este mtodo.
		La manera de efectuar estos cambios viene determinada por el modo de acceso establecido para
		el cursor, ver FLSqlCursor::Mode, si el modo es editar o insertar actualiza con los nuevos valores de
		los campos del registro, si el modo es borrar borra el registro, y si el modo es navegacion no hace nada.
		Antes de nada tambien comprueba la integridad referencial invocando al mtodo FLSqlCursor::checkIntegrity().

		@return	TRUE si se ha podido entregar el buffer al cursor, y FALSE si ha fallado la entrega. */
	bool commitBuffer ();

	/** Buffer con un registro del cursor.
	
		Segn el modo de acceso FLSqlCursor::Mode establecido para el cusor, este buffer contendr
		el registro activo de dicho cursor listo para insertar,editar,borrar o navegar. */
	QSqlRecord * buffer_;   

	/** Copia del buffer.

		Aqui se guarda una copia del FLSqlCursor::buffer_ actual mediante el metodo FLSqlCursor::updateBufferCopy().*/
	QSqlRecord * bufferCopy_;

	/** Metadatos de la tabla asociada al cursor. */

	FLTableMetaData *metadata_;

	/** Mantiene el modo de acceso actual del cursor, ver FLSqlCursor::Mode. */
	int modeAccess_;

	/** Cursor relacionado con el este. */
	FLSqlCursor *cursorRelation;

	/** Relacin que determina como se relaciona con el cursor relacionado. */
	FLRelationMetaData *relation;

	/** Esta bandera cuando es TRUE indica que se abra el formulario de edicin de regitros en
		 modo edicin, y cuando es FALSE se consulta la bandera FLSqlCursor::browse. Por defecto esta
       bandera est a TRUE */
	bool edition;

   /** Esta bandera cuando es TRUE y la bandera FLSqlCuror::edition es FALSE, indica que se
       abra el formulario de edicin de registro en modo visualizacin, y cuando es FALSE no hace
       nada. Por defecto esta bandera est a TRUE */
   bool browse;

	/** Indica si hay una transaccin abierta */
	static int transaction_;

   /** Filtro principal para el cursor.

       Este filtro persiste y se aplica al cursor durante toda su existencia,
       los filtros posteriores, siempre se ejecutaran unidos con 'AND' a este. */
   QString mainFilter_;

   /** Accion asociada al cursor, esta accion pasa a ser propiedad de FLSqlCursor, que ser el
       encargado de destruirla */
   FLAction * action_;

	/** Realiza comprobaciones de intregidad.
	
		Se comprueba la integridad referencial al intentar borrar, tambien se comprueba la no duplicidad de
		claves primarias y si hay nulos en campos que no lo permiten cuando se inserta o se edita.
		Si alguna comprobacion falla muestra un cuadro de dilogo con el tipo de fallo encontrado y el metodo
		devuelve FALSE.
		
		@return	TRUE si se ha podido entregar el buffer al cursor, y FALSE si ha fallado alguna comprobacion
				de integridad. */
	bool checkIntegrity ();
   
	signals:

	/** Indica que se ha cargado un nuevo buffer */
	void newBuffer ();

	/** Indica ha cambiado un campo del buffer, junto con la seal se enva el nombre del campo que
	     ha cambiado. */
	void bufferChanged ( QString );

	/** Indica que se ha actualizado el cursor */
	void cursorUpdated ();

	/** Indica que se ha elegido un registro, mediante doble clic sobre l o bien pulsando la tecla Enter */
	void recordChoosed ();        
};

#endif
