#include <gtk/gtk.h>
#include <stdlib.h>
#include "entity.h"
#include "gtk-widget-attr.h"


static gint
rendgtk_timer_callback (gpointer data)
{
    ENode *node = data;
    gchar *function;

    GTimeVal *time;
    EBuf *usec_string;
    EBuf *sec_string;

    time = enode_get_kv (node, "rendgtk-timer-gtimeval");
    g_get_current_time (time);

    usec_string = ebuf_new ();	/* Bigger than 64bit */
    ebuf_sprintf (usec_string, "%ld", time->tv_usec);
    enode_attrib (node, "__usec-last", usec_string);	/* Time in microsecs */

    sec_string = ebuf_new ();	/* Bigger than 64bit */
    ebuf_sprintf (sec_string, "%ld", time->tv_sec);
    enode_attrib (node, "__sec-last", sec_string);	/* Time in seconds */

    function = enode_attrib_str (node, "action", NULL);

    enode_call_ignore_return (node, function, "");

    return (TRUE);
}

static void
rendgtk_timer_clear_tags (ENode * node)
{
    gint tag;

    tag = GPOINTER_TO_INT (enode_get_kv (node, "rendgtk-timer-timeout-tag"));

    if (tag > 0)
	gtk_timeout_remove (tag);

    enode_set_kv (node, "rendgtk-timer-timeout-tag", NULL);

    tag = GPOINTER_TO_INT (enode_get_kv (node, "rendgtk-timer-idle-tag"));

    if (tag > 0)
	gtk_idle_remove (tag);

    enode_set_kv (node, "rendgtk-timer-idle-tag", NULL);
}

static void
rendgtk_timer_percent_elapsed_attr_get (ENode * node, gchar * attr)
{
    GTimeVal *time;
    EBuf *last_sec_ebuf;
    EBuf *last_usec_ebuf;
    EBuf *return_ebuf;

    EBuf *interval_ebuf;
    gint interval;

    glong last_sec;
    glong last_usec;
    glong seconds_difference;
    glong microseconds_difference;
    gfloat percentage;

    time = enode_get_kv (node, "rendgtk-timer-gtimeval");
    g_get_current_time (time);

    interval_ebuf = enode_attrib (node, "interval", NULL);
    interval = atoi (interval_ebuf->str);

    last_sec_ebuf = enode_attrib (node, "__sec-last", NULL);
    last_usec_ebuf = enode_attrib (node, "__usec-last", NULL);
    if (ebuf_empty (last_sec_ebuf)) {
	last_sec_ebuf = enode_attrib (node, "__sec-started", NULL);
	last_usec_ebuf = enode_attrib (node, "__usec-started", NULL);
	if (ebuf_empty (last_sec_ebuf)) {
	    enode_attrib_quiet (node, "__percent-elapsed",
				ebuf_new_with_str (""));
	    return;
	}
    }


    last_sec = atol (last_sec_ebuf->str);
    seconds_difference = time->tv_sec - last_sec;

    last_usec = atol (last_usec_ebuf->str);
    microseconds_difference = (1000000 * seconds_difference)
	+ time->tv_usec - last_usec;
    percentage = (float) (microseconds_difference / 1000) / interval;

    return_ebuf = ebuf_new ();
    ebuf_sprintf (return_ebuf, "%2f", percentage);
    enode_attrib_quiet (node, "__percent-elapsed", return_ebuf);
}

static void
rendgtk_timer_sec_until_attr_get (ENode * node, gchar * attr)
{
    GTimeVal *time;
    EBuf *last_sec_ebuf;
    EBuf *return_ebuf;

    glong last_sec;
    glong seconds_difference;

    time = enode_get_kv (node, "rendgtk-timer-gtimeval");
    g_get_current_time (time);

    last_sec_ebuf = enode_attrib (node, "__sec-last", NULL);
    if (ebuf_empty (last_sec_ebuf)) {
	enode_attrib_quiet (node, "__sec-until", ebuf_new_with_str (""));
	return;
    }
    last_sec = atol (last_sec_ebuf->str);
    seconds_difference = time->tv_sec - last_sec;

    return_ebuf = ebuf_new ();
    ebuf_sprintf (return_ebuf, "%ld", seconds_difference);
    enode_attrib_quiet (node, "__sec-until", return_ebuf);
}

static void
rendgtk_timer_usec_until_attr_get (ENode * node, gchar * attr)
{
    GTimeVal *time;
    EBuf *last_sec_ebuf;
    EBuf *last_usec_ebuf;
    EBuf *return_ebuf;

    glong last_sec;
    glong last_usec;
    glong seconds_difference;
    glong microseconds_difference;

    time = enode_get_kv (node, "rendgtk-timer-gtimeval");
    g_get_current_time (time);

    last_sec_ebuf = enode_attrib (node, "__sec-last", NULL);
    last_usec_ebuf = enode_attrib (node, "__usec-last", NULL);
    if (ebuf_empty (last_usec_ebuf)) {
	enode_attrib_quiet (node, "__usec-until", ebuf_new_with_str (""));
	return;
    }
    last_sec = atol (last_sec_ebuf->str);
    seconds_difference = time->tv_sec - last_sec;

    last_usec = atol (last_usec_ebuf->str);
    microseconds_difference = (1000000 * seconds_difference)
	+ time->tv_usec - last_usec;

    return_ebuf = ebuf_new ();
    ebuf_sprintf (return_ebuf, "%ld", microseconds_difference);
    enode_attrib_quiet (node, "__usec-until", return_ebuf);
}

static gint
rendgtk_timer_interval_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    gint intval;
    gint id;

    intval = erend_get_integer (value);

    /* remove old timer */
    rendgtk_timer_clear_tags (node);

    if (intval < 0)
	return (TRUE);

    /* If less then 50, we setup an idle instead.. keeps gtk from going nutso 
     */
    if (intval < 50) {
	id = gtk_idle_add (rendgtk_timer_callback, node);
	enode_set_kv (node, "rendgtk-timer-idle-tag", GINT_TO_POINTER (id));
	return (TRUE);
    }

    id = gtk_timeout_add (intval, rendgtk_timer_callback, node);
    enode_set_kv (node, "rendgtk-timer-timeout-tag", GINT_TO_POINTER (id));

    return (TRUE);
}

static void
rendgtk_timer_destroy (ENode * node)
{
    rendgtk_timer_clear_tags (node);
}

void
rendgtk_timer_render (ENode * node)
{
    GTimeVal *time;
    EBuf *usec_string;
    EBuf *sec_string;

    time = g_new0 (GTimeVal, 1);
    g_get_current_time (time);
    enode_set_kv (node, "rendgtk-timer-gtimeval", time);

    usec_string = ebuf_new ();
    ebuf_sprintf (usec_string, "%ld", time->tv_usec);
    enode_attrib (node, "__usec-started", usec_string);	/* Time in microsecs */

    sec_string = ebuf_new ();
    ebuf_sprintf (sec_string, "%ld", time->tv_sec);
    enode_attrib (node, "__sec-started", sec_string);	/* Time in seconds */

    enode_attribs_sync (node);
}


void
timer_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_timer_render;
    element->destroy_func = rendgtk_timer_destroy;
    element->tag = "timer";
    element->description =
	"Create a new timer that will call a function every 'interval' milliseconds.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "interval";
    e_attr->description =
	"Sepcify the time interval before calling the action function.";
    e_attr->value_desc = "integer";
    e_attr->possible_values = "-1,*";
    e_attr->set_attr_func = rendgtk_timer_interval_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__usec-started";
    e_attr->description = "The microsecond the timer started";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__sec-started";
    e_attr->description = "The second the timer started";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__usec-last";
    e_attr->description =
	"The microsecond the timer was last active, this is useless unless you get __sec-last too";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__sec-last";
    e_attr->description = "The second the timer was last active";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__sec-until";
    e_attr->description = "The seconds until the timer is active";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    e_attr->get_attr_func = rendgtk_timer_sec_until_attr_get;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__usec-until";
    e_attr->description = "The microseconds until the timer is active";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    e_attr->get_attr_func = rendgtk_timer_usec_until_attr_get;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "__percent-elapsed";
    e_attr->description = "The percentage the timer has elapsed";
    e_attr->value_desc = "integer";
    e_attr->set_attr_func = NULL;
    e_attr->get_attr_func = rendgtk_timer_percent_elapsed_attr_get;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "action";
    e_attr->description =
	"Set to the function that will be called when a timeout occurs.";
    e_attr->value_desc = "function";
    e_attr->value_desc = "(timer_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);
}
