/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: game.cc,v 1.45.2.4 2003/10/01 22:38:57 dheck Exp $
 */
#include "display.hh"
#include "game.hh"
#include "menus.hh"
#include "options.hh"
#include "video.hh"
#include "sound.hh"
#include "help.hh"
#include "system.hh"

#include "px/sdl.hh"

#include <algorithm>
#include <fstream>
#include <cassert>

using namespace enigma;
using namespace px;
using namespace std;

//----------------------------------------
// Data types
//----------------------------------------
namespace
{
    class GameMenu : public gui::Menu {
    public:
        GameMenu(int zoomxpos_, int zoomypos_);
        virtual ~GameMenu();
    private:
        bool on_event (const SDL_Event &e);
        void on_action(gui::Widget *w);
        void draw_background(px::GC &gc);

        gui::Widget *resume, *restart, *options, *abort;
        px::Surface *zoomed;
        int          zoomxpos, zoomypos; // position to be zoomed
    };

    class Game {
    public:
        Game();
        px::Surface *create_preview(LevelPack *lp, int ilevel);
        void run(LevelPack *lp, int ilevel);
        void quit();
        void finish_level();
        void restart_level();
        void restart_game();
        int get_current_level() const { return icurrent_level; }
    private:
	// Private types.
        enum State {
            LEVELINIT,          // initialize level
            INGAME,             // game is currently running
            LEVELFINISHED,      // level finished; proceed to next one
            PLAYERDEAD,         // player is dead; restart level shortly
            NOMORELIVES,        // no player has any lives left, restart
            STARTGAME,          // start the current level (when called from level menu)
            RESTARTGAME,        // restart the current level (new lives)
            RELOADLEVEL,        // reload current level (resets objects)
            LEVELINFO,          // display level info
            ABORT               // quit the game immediately
        };


        // Private methods.
        void handle_events();
        void on_keydown(SDL_Event &e);
        void on_mousebutton(SDL_Event &e);
        void update_mouse_button_state();
        void change_state(State newstate);
        void tick(double dtime);
        void show_menu();
        void show_help();
        bool load_level(int ilevel);
        void advance_level(int mode = 0);

        // Private variables.
        State       state;
        LevelPack  *level_pack;
        unsigned    icurrent_level;
        Uint32      last_tick_time;
        double      actor_dead_dtime;
        double      current_state_dtime;
        double      overall_level_time;
        px::Screen *screen;
    };
}


//----------------------------------------
// GameMenu impl
//----------------------------------------

GameMenu::GameMenu(int zoomxpos_, int zoomypos_)
: resume(new gui::TextButton("Resume Level", this)),
  restart(new gui::TextButton("Restart Level", this)),
  options(new gui::TextButton("Options", this)),
  abort(new gui::TextButton("Abort Level", this)),
  zoomed(0),
  zoomxpos(zoomxpos_),
  zoomypos(zoomypos_)
{
    add(resume,     Rect(0,0,150,40));
    add(restart,    Rect(0,45,150,40));
    add(options,    Rect(0,90,150,40));
    add(abort,      Rect(0,135,150,40));
    center(video::GetScreen());
}

GameMenu::~GameMenu() {
    delete(zoomed);
}

void
GameMenu::draw_background(px::GC &gc) 
{
    if (!zoomed) {
        const Rect& game_area   = display::GetGameArea();
        int         part_width  = game_area.w/3;
        int         part_height = (part_width*480)/640;

        if (part_height > game_area.h) {
            part_height = game_area.h/3;
            part_width  = (part_height*640)/480;
            assert(part_width <= game_area.w);
        }

        // randomly choose ball offset
        int x, y;
        for (int trials = 5; trials; --trials) {
            x = IntegerRand(0, 5);
            y = IntegerRand(0, 3);

            // try to avoid menu-ball overlap:
            if (x<2 || x>3 || y<1 || y>2 || (trials == 1)) {
                int ax = zoomxpos-game_area.x;
                int ay = zoomypos-game_area.y;

                // upper left corner of part
                x = ax/32-1-x;
                y = ay/32-1-y;

                // ensure part is inside game_area
                x = max(0, min(x, (game_area.w-part_width)/32-1));
                y = max(0, min(y, (game_area.h-part_height)/32-1));

                // adjust to game fields
                x = x*32+24;
                y = y*32+16;

                break;
            }
        }

        // Be sure to redraw everything, or actors may appear on top
        // of the stones (actors are drawn in one pass and only
        // clipped to the screen boundary).
        display::RedrawAll(video::GetScreen());
        
        // get the selected part from screen
//         SDL_Surface *back = video::GetScreen()->get_surface();
        Rect     src_area(game_area.x+x, game_area.y+y, part_width, part_height);
        Surface *src = Grab(video::GetScreen()->get_surface(), src_area);

        // zoom multiple times for softer image
        const double stepsize = 0.3;
        for (double zoom = 0.4; zoom < 0.9; zoom += stepsize) {
            int      sx  = int(zoom*640+0.5);
            int      sy  = int(zoom*480+0.5);
            Surface *tmp = src->zoom(sx, sy);

            delete src;
            src = tmp;
        }
        zoomed = src->zoom(640, 480);
        delete src;
    }

    px::blit(gc, 0,0, zoomed);
}

bool
GameMenu::on_event (const SDL_Event &e) {
    if (e.type == SDL_MOUSEBUTTONDOWN
        && e.button.button == SDL_BUTTON_RIGHT)
    {
        Menu::quit();
        return true;
    }
    return false;
}

void
GameMenu::on_action(gui::Widget *w) {
    if (w == resume) {
        Menu::quit();
    }
    else if (w == abort) {
        enigma::QuitGame();
        Menu::quit();
    }
    else if (w == restart)
    {
        enigma::RestartGame();
        Menu::quit();
    }
    else if (w == options)
    {
        GUI_OptionsMenu(zoomed);
        if (options::MustRestartLevel) {
            options::MustRestartLevel = false;
            enigma::RestartGame();
        }
        Menu::quit();
    }
}




//======================================================================
// MAIN PROGRAM
//======================================================================

//----------------------------------------
// Level management
//----------------------------------------
namespace enigma
{
    vector<LevelPack *> LevelPacks;
}

void enigma::RegisterLevelPack (LevelPack *lp)
{
    LevelPacks.push_back(lp);
}

//----------------------------------------------------------------------
// GAME IMPLEMENTATION
//----------------------------------------------------------------------

Game::Game()
: state(INGAME),
  icurrent_level(0),
  last_tick_time(0),
  overall_level_time(0),
  screen (0)
{
}

void Game::quit() {
    change_state(ABORT);
}

void Game::finish_level() {
    change_state(LEVELFINISHED);
}

void Game::restart_level() {
    change_state(RELOADLEVEL);
}

void Game::restart_game() {
    change_state(RESTARTGAME);
}

void Game::change_state(State newstate)
{
    if (state == newstate) return;
    switch (newstate) {
    case NOMORELIVES:
        sound::PlaySound("exit");
        break;

    case LEVELFINISHED:
        if (state==INGAME) {
            sound::PlaySound("finished");
            player::LevelFinished(); // remove player-controlled actors

	    //int level_time = int(overall_level_time+current_state_dtime+.5);
            int level_time = int(current_state_dtime + .5); // [ant]: why to add overall_level_time?
            if (options::SetLevelTime (level_pack->get_name(),
                                       level_pack->get_info(icurrent_level)->filename,
                                       options::Difficulty,
                                       level_time))
            {
                display::GetStatusBar()->show_text("New best time!", display::TEXT_STATIC);
            } else {
                display::GetStatusBar()->show_text("Level finished!", display::TEXT_STATIC);
            }

            if (options::LevelStatusChanged) {
                options::Save();    // save options (Enigma sometimes crashes when loading next level)
            }
        }
        else
            return;             // do not enter state
        break;
    case PLAYERDEAD:
        actor_dead_dtime = 0;
//        display::ShowText("You lost", display::TEXT_STATIC);
        break;

    case LEVELINIT:
        if (state == LEVELINFO) break; // inventory already updated in state LEVELINFO
        // fall-through
    case LEVELINFO:
        player::GetInventory(player::CurrentPlayer())->redraw(); // draw inventory
        break;

    default: break;
    }

    state = newstate;
    overall_level_time  += current_state_dtime;
    current_state_dtime  = 0;
}

void
Game::advance_level(int mode)
    // mode == 0 -> advance according to options::SkipSolvedLevels
    // mode == 1 -> advance to next level
    // mode == 2 -> advance to next unsolved level

{
    bool     skip_solved = mode == 2 || (mode == 0 && options::SkipSolvedLevels);
    unsigned next_level  = NextLevel(level_pack, icurrent_level, HighestAvailableLevel(level_pack), skip_solved, true);

    if (next_level) {
        change_state(load_level(next_level) ? LEVELINFO : ABORT);
    }
    else {
        change_state(ABORT); // no next level in level pack
    }
}

static string displayedLevelInfo (const LevelInfo *info)
{
    string displayed = (info->name.empty()) ?
        "Another nameless level" : string("\"")+info->name+"\"";
    if (!info->author.empty())
        displayed += "  by "+info->author;
    return displayed;
}

void
Game::tick(double dtime)
{
    current_state_dtime += dtime;
    switch (state) {
    case STARTGAME:
    case RESTARTGAME:
        player::NewGame(2, level_pack->needs_twoplayers()); // two virtual players
        change_state(load_level(icurrent_level)
                     ? (state == STARTGAME ? LEVELINFO : LEVELINIT)
                     : ABORT);
        break;

    case LEVELINFO: {
        // show level information (name, author, etc.)
        string disp = displayedLevelInfo(level_pack->get_info(icurrent_level));
        display::GetStatusBar()->show_text(disp.c_str(),
                                           display::TEXT_5SECONDS,
                                           true); // may interrupt
        change_state(LEVELINIT);
        break;
    }
    case LEVELINIT:  {
        const int  BUFSIZE = 50;
        const LevelInfo *info    = level_pack->get_info(icurrent_level);
        char       buffer[BUFSIZE+1];

        snprintf(buffer, BUFSIZE, "Enigma - %s ", info->name.c_str());
        video::SetCaption(buffer);

        change_state(INGAME);
        break;
    }
    case INGAME:
        handle_events();
        world::Tick(dtime);
        player::Tick(dtime);
        display::Tick(dtime);
        display::Redraw(screen);
        break;

    case PLAYERDEAD:
        actor_dead_dtime += dtime;
        if (actor_dead_dtime <= 0.5) {
            handle_events();
            player::Tick(dtime);
            world::Tick(dtime);
            display::Tick(dtime);
            display::Redraw(screen);
        } else {
            change_state (RESTARTGAME);
        }
        break;

    case NOMORELIVES:
        if (current_state_dtime >= 2.0) {
            change_state(RESTARTGAME);
        }
        break;

    case RELOADLEVEL:
        if (current_state_dtime >= 1.0) {
            change_state(load_level(icurrent_level) ? LEVELINIT : ABORT);
        } else {
            world::Tick(dtime);
            display::Tick(dtime);
            display::Redraw(screen);
            handle_events();
        }
        break;

    case LEVELFINISHED:
        if (current_state_dtime <= 2.5) {
            handle_events();
            player::Tick(dtime);
            world::Tick(dtime);
            display::Tick(dtime);
            display::Redraw(screen);
        } else {
            advance_level();
        }
        break;

    default:
        break;
    }
}

bool
Game::load_level(int ilevel)
{
    // first set default compatibility mode
    // (may be overidden by load_level (from lua))
    {
        const LevelInfo *info     = level_pack->get_info(ilevel);
        enigma::GameCompatibility = info->type;
    }

    if (level_pack->load_level (ilevel))
    {
        icurrent_level=ilevel;

        display::FocusReferencePoint();

        GC gc(video::BackBuffer());
        display::DrawAll(gc);
        ShowScreen(video::TM_PUSH_RANDOM, video::BackBuffer());
        sdl::FlushEvents();
        update_mouse_button_state();
        last_tick_time = SDL_GetTicks();
        return true;
    }
    else
        return false;
}

px::Surface *
Game::create_preview(LevelPack *lp, int ilevel)
{
    px::Surface *img = 0;

    sound::TempDisableSound();
    player::NewGame(2, lp->needs_twoplayers()); // two virtual players
    if (lp->load_level (ilevel))
    {
        GC gc(video::BackBuffer());
        display::DrawAll(gc);
        img = video::BackBuffer();
    }
    sound::TempReEnableSound();
    return img;
}

namespace {
    class Avg {
        enum {N=20};
        double dtimes[N];
        int idx;
    public:
        Avg() : idx(0) { fill(dtimes, dtimes+N, 0.010); }
        void add(double dtime) {
            dtimes[idx++] = dtime;
            idx = idx % N;
        }
        double get() {
            double s=0;
            for (int i=0; i<N; s+=dtimes[i++]);
            return s/N;
        }
    };
}

void
Game::run (LevelPack *lp, int ilevel)
{
    screen = video::GetScreen();
    static Avg average_dtime;

    level_pack = lp;

    icurrent_level = ilevel;

    sdl::TempInputGrab grab(options::Nograb ? SDL_GRAB_OFF : SDL_GRAB_ON);
    video::HideMouse();

    sound::FadeoutMusic();
    if (options::InGameMusic) {
        sound::PlayMusic( options::LevelMusicFile.c_str());
    } else {
        sound::StopMusic();
    }

    double dtime       = 0;
    state              = STARTGAME;
    overall_level_time = 0;

    while (state != ABORT)
    {
        last_tick_time=SDL_GetTicks();

        tick(dtime);

        int sleeptime = 10 - (SDL_GetTicks()-last_tick_time);
        if (sleeptime > 0)
            ; //SDL_Delay(sleeptime);

        dtime=(SDL_GetTicks()-last_tick_time)/1000.0;

        double adtime = average_dtime.get();
        average_dtime.add(dtime);
        if (abs(adtime-dtime)/adtime < 0.1) {
            // less than 10% deviation from average frame time?
            dtime = adtime;
        }

	if (dtime > 500.0) /* Time has done something strange, perhaps
			      run backwards */
            dtime = 0.0;
	else if (dtime > 0.5)
            dtime = 0.5;
    }
    video::ShowMouse();
}

void
Game::handle_events()
{
    SDL_Event e;
    while (SDL_PollEvent(&e))
    {
        switch (e.type) {
        case SDL_KEYDOWN:
            on_keydown(e);
            break;
        case SDL_MOUSEMOTION:
            world::SetMouseForce (options::MouseSpeed *
                                  V2 (e.motion.xrel, e.motion.yrel));
            break;
        case SDL_MOUSEBUTTONDOWN:
        case SDL_MOUSEBUTTONUP:
            on_mousebutton(e);
            break;
        case SDL_ACTIVEEVENT:
            // if( e.active.gain == 0) show_menu();
            break;
        case SDL_QUIT:
            change_state(ABORT);
            break;
        }
    }
}

void
Game::on_mousebutton(SDL_Event &e)
{
    if (e.button.state == SDL_PRESSED)
    {
        if (e.button.button == 1) {
            // left mousebutton -> activate first item in inventory
            player::ActivateItem();
        }
        else if (e.button.button == 3) {
            // right mousebutton -> rotate inventory
            display::GetStatusBar()->hide_text();
            player::RotateInventory();
        }
	else if (e.button.button == 4) {
            // wheel up -> rotate inventory
            display::GetStatusBar()->hide_text();
            player::RotateInventory();
	}
	else if (e.button.button == 5) {
	    // wheel down -> inverse rotate inventory
            display::GetStatusBar()->hide_text();
            player::RotateInventory(-1);
	}
    }
    update_mouse_button_state();
}

void
Game::update_mouse_button_state()
{
    int b = SDL_GetMouseState(0, 0);
    player::InhibitPickup((b & SDL_BUTTON(1)) || (b & SDL_BUTTON(3)));
}

static void set_mousespeed(double spd)
{
    int ms = int(spd+.5);

    if      (ms<options::minMouseSpeed) ms = options::minMouseSpeed;
    else if (ms>options::maxMouseSpeed) ms = options::maxMouseSpeed;

    options::MouseSpeed = int(ms+.5);

    char msg[200];
    sprintf(msg, "Mouse speed: %d", ms);
    display::GetStatusBar()->show_text(msg, display::TEXT_2SECONDS);
}

void
Game::on_keydown(SDL_Event &e)
{
    switch (e.key.keysym.sym) {
    case SDLK_ESCAPE:
        show_menu();
        break;

    case SDLK_LEFT: set_mousespeed(options::MouseSpeed - 1); break;
    case SDLK_RIGHT: set_mousespeed(options::MouseSpeed + 1); break;

    case SDLK_F1:
        show_help();
        break;

    case SDLK_F2:
        break;

    case SDLK_F3:
        if (e.key.keysym.mod & KMOD_SHIFT) {
            change_state(RESTARTGAME);
        }
        else {
            player::Suicide();
        }
        break;

    case SDLK_F4:
        advance_level(1); // next level (ignores options::SkipSolvedLevels)
        break;

    case SDLK_F5:
        advance_level(2); // next unsolved level
        break;

    case SDLK_F10:
        {
            string fname = level_pack->get_info(icurrent_level)->filename + ".bmp";
            video::Screenshot(fname.c_str());
        }
        break;
    case SDLK_x:
        if (e.key.keysym.mod & KMOD_ALT) {
            change_state(ABORT);
        }
        break;
    default:
        break;
    }

    if (options::WizardMode > 0.0) {
        switch (e.key.keysym.sym) {
        case SDLK_f:
            options::ShowFPS = !options::ShowFPS;
            break;
        case SDLK_l:
            // Reload current level or abort if this fails
            change_state(load_level(icurrent_level) ? LEVELINIT : ABORT);
            break;
        case SDLK_t:
            // Darken the current screen; useful for debugging screen updates
            TintRect(screen->get_surface(), screen->size(), 0,0,0, 200);
            screen->update_all();
            screen->flush_updates();
            break;
        case SDLK_g:
            if (e.key.keysym.mod & KMOD_ALT) {
                display::ReloadModels();
                change_state(load_level(icurrent_level) ? LEVELINIT : ABORT);
            }
            break;
        case SDLK_1: ToggleFlag(display::SHOW_FLOOR); break;
        case SDLK_2: ToggleFlag(display::SHOW_ITEMS); break;
        case SDLK_3: ToggleFlag(display::SHOW_SHADES); break;
        case SDLK_4: ToggleFlag(display::SHOW_STONES); break;
        case SDLK_5: ToggleFlag(display::SHOW_SPRITES); break;
        default:
            break;
        }
    }
}

static const char *helptext_ingame[] = {
    "Left mouse button:",       "Activate/drop leftmost inventory item",
    "Right mouse button:",      "Rotate inventory items",
    "Escape:",                  "Show game menu",
    "F1:",                      "Show this help",
    "F3:",                      "Kill current marble",
    "Shift+F3:",                "Restart the current level",
    "F4:",                      "Skip to next level",
    "F5:",                      "Skip to next unsolved level",
    "F10:",                     "Make screenshot",
    "Left arrow:",              "Decrease mouse speed",
    "Right arrow:",             "Increase mouse speed",
//    "Alt+x:",                   "Return to level menu",
    "Alt+Return:",              "Switch between fullscreen and window",

    0
};

void
Game::show_help()
{
    sdl::TempInputGrab grab(SDL_GRAB_OFF);

    video::ShowMouse();

    displayHelp(screen, helptext_ingame, 200);

    video::HideMouse();
    update_mouse_button_state();
    last_tick_time = SDL_GetTicks();
    display::RedrawAll(screen);
}

void
Game::show_menu()
{
    sdl::TempInputGrab grab(SDL_GRAB_OFF);

    video::ShowMouse();
    {
        int x, y;
        display::GetReferencePointCoordinates(&x, &y);
        GameMenu(x, y).manage(screen);
    }
    video::HideMouse();
    update_mouse_button_state();
    last_tick_time = SDL_GetTicks();
    if (state != ABORT)
        display::RedrawAll(screen);
}


//----------------------------------------------------------------------
// VARIABLES / FUNCTIONS
//----------------------------------------------------------------------

namespace
{
    Game game_inst;
}

bool     enigma::ConserveLevel     = false;
bool     enigma::AllowTogglePlayer = false;
bool     enigma::ShowMoves         = false;
GameType enigma::GameCompatibility = GAMET_ENIGMA;
double   enigma::Brittleness       = 0;
double   enigma::SlopeForce        = 0;
double   enigma::SlopeForce2       = 0;
double   enigma::FrictionFactor    = 0;
double   enigma::ElectricForce     = 0;
double   enigma::BumperForce       = 0;


void enigma::GameReset()
{
    enigma::ConserveLevel     = true;
    enigma::AllowTogglePlayer = true;
    enigma::ShowMoves         = false;
    enigma::Brittleness       = 0.5;
    enigma::SlopeForce        = 25.0;
    enigma::SlopeForce2       = 25.0;
    enigma::FrictionFactor    = 1.0;
    enigma::ElectricForce     = 15.0;
    enigma::BumperForce       = 500.0;
}

px::Surface *
enigma::LevelPreview (LevelPack *lp, unsigned levelidx)
{
    return game_inst.create_preview (lp, levelidx);
}

unsigned
enigma::StartGame (LevelPack *lp, unsigned levelidx)
{
    if (lp->get_info(levelidx)->filename != "todo") {
        game_inst.run (lp, levelidx);
        levelidx = game_inst.get_current_level();
    }
    return levelidx;
}

unsigned
enigma::NextLevel (LevelPack *lp, unsigned levelidx, unsigned max_available, 
                   bool skip_solved, bool skip_todo) 
{
    // returns 0 if none found
    unsigned found = 0;
    unsigned size  = lp->size();

    while (!found) {
        ++levelidx;
        if (levelidx >= size)
            break; // none found

        const LevelInfo *info = lp->get_info(levelidx);

        if (skip_todo) { // skip "todo" levels
            if (info->filename == "todo") {
                printf("Skipping 'todo' level\n");
                continue;
            }
//             if (info->name == "") {
//                 printf("Skipping unnamed level '%s'\n", info->filename.c_str());
//                 continue;
//             }
        }

//        bool available = levelidx <= max_available;
        bool available = true;  // level locking disabled for the time being!

        if (skip_solved || !available) {
            options::LevelStatus *ls = options::GetLevelStatus(lp->get_name(), info->filename);
            if (ls && ls->finished >= options::Difficulty) { // already solved
                if (skip_solved)
                    continue;   // skip solved
                available = true; // otherwise force available
            }
        }

        if (available)
            found = levelidx;
    }

    return found;
}

unsigned
enigma::HighestAvailableLevel(LevelPack *lp) 
{
    if( options::WizardMode || // No restriction in WizardMode
        lp->get_name() == "TestLevels") // and test-levelpack
        return lp->size()-1;

    // Otherwise the player can choose out of 10 unsolved levels
    unsigned max_available = 10-1; // level numbers start at 0

    for( unsigned i = 0; i < lp->size() && i <= max_available; i++) {
        const LevelInfo *levelinfo = lp->get_info(i);
        if (options::LevelStatus *ls=options::GetLevelStatus(lp->get_name(), levelinfo->filename)) {
            if (ls->finished != 0) {
                max_available++;  // for each solved level, an additional level is available;
            }
        }
    }
    return max_available;
}

static options::LevelStatus *
GetStatus (LevelPack *lp, size_t index)
{
    return options::GetLevelStatus (lp->get_name(), 
                                    lp->get_info(index)->filename);
}

bool 
enigma::LevelIsLocked (LevelPack *lp, size_t index)
{
    return false;               // For now, do not lock any levels

    unsigned max_available = HighestAvailableLevel (lp);
    int finished = 0;

    if (options::LevelStatus *ls = GetStatus (lp, index))
        finished = ls->finished;

    return !(index <= max_available || finished >= options::Difficulty);
}

unsigned
enigma::CountSolvedLevels (LevelPack *lp)
{
    unsigned cnt = 0;
    for (unsigned i=0; i<lp->size(); ++i)
    {
	if (options::LevelStatus *ls = GetStatus(lp, i))
	    if (ls->finished)
		++cnt;
    }
    return cnt;
}



void enigma::FinishLevel() {
    game_inst.finish_level();
}

void enigma::RestartLevel() {
    game_inst.restart_level();
}

void enigma::RestartGame() {
    game_inst.restart_game();
}

void enigma::QuitGame() {
    game_inst.quit();
}


void enigma::SetCompatibility(const char *version) {
    static const char *versionName[GAMET_COUNT+1] = {
        "enigma", // same indices as enum GameType
        "oxyd1",
        "per.oxyd",
        "oxyd.extra",
        "oxyd.magnum",
        0
    };

    GameType type = GAMET_UNKNOWN;
    for (int v = 0; v<GAMET_COUNT; ++v) {
        if (0 == strcmp(version, versionName[v])) {
            type = GameType(v);
            break;
        }
    }

    if (type == GAMET_UNKNOWN) {
        fprintf(stderr, "Invalid compatibility mode '%s' (ignored. using enigma behavior)\n", version);
        fprintf(stderr, "Valid modes:");
        for (int v = 0; v<GAMET_COUNT; ++v)
            fprintf(stderr, " %s", versionName[v]);
        fprintf(stderr, "\n");
        type = GAMET_ENIGMA;
    }

    GameCompatibility = type;
}
