/** Check a decaf program for semantic errors.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include "decaf.h"
#include "parsetree.h"
#include "symboltable.h"
#include "util.h"
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>


/** Name of this program. */
const char* program = "decafc";


/** Number of errors encountered. */
static int errors;


/** Prints the specifed error message and its context. */
#define err(message) { \
	errors++; \
	printf( "%s near ", message); \
	print_lexemes( tree); \
	putchar( '\n'); }


/** Called when an unexpected production is encountered. */
#define unexpected_token() die( "unexpected token %s", \
		get_key( &decaf_symbols, $token))


/** Symbol table. */
SymbolTable symbols;


/** Checks the operands of an arithmetic operator. */
static int
arith_op( const Tree* tree, int lhs, int rhs)
{
	if( lhs != INT)
		err( "expected an integer expression on the left side");
	if( rhs != INT)
		err( "expected an integer expression on the right side");
	return INT;
}


/** Checks the operands of a relative operator. */
static int
rel_op( const Tree* tree, int lhs, int rhs)
{
	if( lhs != INT)
		err( "expected an integer expression on the left side");
	if( rhs != INT)
		err( "expected an integer expression on the right side");
	return BOOL;
}


/** Checks the operands of an equality operator. */
static int
eq_op( const Tree* tree, int lhs, int rhs)
{
	if( lhs & (ARRAY | METHOD))
		err( "expected a basic type on the left side");
	if( rhs & (ARRAY | METHOD))
		err( "expected a basic type on the right side");
	if( lhs != rhs)
		err( "left side does not agree with right side");
	return BOOL;
}


/** Checks the operands of a conditional operator. */
static int
cond_op( const Tree* tree, int lhs, int rhs)
{
	if( lhs != BOOL)
		err( "expected a boolean expression on the left side");
	if( rhs != BOOL)
		err( "expected a boolean expression on the right side");
	return BOOL;
}


/** A function to check a binary operator. */
typedef int (*BinaryOp)( const Tree* tree, int lhs, int rhs);


/** Returns the type of this binary operator and checks the types of
 * its operands. */
static BinaryOp
bin_op( const Tree* tree, const void* in)
{
	switch( $token) {
		case BIN_OP: // arith_op | rel_op | eq_op | cond_op
			return $(1,bin_op);
		case ARITH_OP: return arith_op;
		case REL_OP: return rel_op;
		case EQ_OP: return eq_op;
		case COND_OP: return cond_op;
		default: unexpected_token();
	}
}


/** Checks that this expression is a basic type. */
static int
check_basic( const Tree* tree, int type)
{
	if( type & (ARRAY | METHOD))
		err( "expected a basic type");
	return type;
}


/** Checks the operand of a integer unary operator. */
static int
check_integer( const Tree* tree, int operand)
{
	if( operand != INT)
		err( "expected an integer expression");
	return INT;
}


/** Checks the operand of a boolean unary operator. */
static int
check_boolean( const Tree* tree, int operand)
{
	if( operand != BOOL)
		err( "expected a boolean expression");
	return BOOL;
}


/** A function to check a unary operator. */
typedef int (*UnaryOp)( const Tree* tree, int operand);


/** Returns the type of this unary operator and checks the type of its
 * operand. */
static UnaryOp
unary_op( const Tree* tree, const void* in)
{
	(void)in;
	switch( $token) {
		case MINUS: return check_integer;
		case NOT: return check_boolean;
		default: unexpected_token();
	}
}


/** Returns this symbol. */
static Symbol*
symbol( const Tree* tree, const void* in)
{
	(void)in;
	switch( $token) {
		case ID:
			{ Symbol* s = find_symbol( &symbols, $lexeme);
				if( s == NULL)
					errors++;
				return s;
			}
		default: unexpected_token();
	}
}


static int type( const Tree* tree, const void* in);


/** Inherited attributes of expressions. */
typedef struct {
	/** Method being called. */
	Symbol* call;

	/** Which argument. */
	int arg;
} CallAttr;


/** Checks the arguments of this method call. */
static void
call( const Tree* tree, CallAttr* in)
{
#define $3 $(3,call)
	switch( $token) {
	case OPTIONAL_EXPR_LIST:
		$all(call);
		return;
	case EXPR_LIST: // expr
		if( in->arg < in->call->count && in->arg < PARAMETERS &&
				$(1,type) != in->call->parameters[in->arg])
			err( "parameter type does not agree");
		in->arg++;
		if( $length > 1) {
			// expr COMMA expr_list
			$3;
		}
		return;
	default: unexpected_token(); }
#undef $3
}


/** Returns the type of this expression. */
static int
type( const Tree* tree, const void* in)
{
#define $1 $(1,type)
#define $2 $(2,type)
#define $3 $(3,type)
#define $4 $(4,type)
	switch( $token) {
	case OPTIONAL_EXPR: switch( $length) {
		case 0: // epsilon
			return VOID;
		case 1: // expr
			return $1;
		default: unexpected_token();
	}
	case METHOD_CALL: switch( $length) {
		int id;
		case 4: // ID LPAREN optional_expr_list RPAREN
			id = $1;
			if( !(id & METHOD)) {
				err( "expected a method");
				return 0;
			}
			{ CallAttr In, *in = &In;
				in->call = $(1,symbol);
				in->arg = 0;
				$(3,call);
				if( in->arg != in->call->count)
					err( "number of parameters does not agree");
			}
			return id & ~METHOD;
		case 5: // CALLOUT LPAREN STRING_CONSTANT callout_arg_list RPAREN
			$4;
			return VOID;
		default: unexpected_token();
		}
	case CALLOUT_ARG_LIST:
		$all(type);
		return 0;
	case COMMA:
		return 0;
	case CALLOUT_ARG: // expr | STRING_CONSTANT
		return check_basic( tree, $(1,type));
	case LVALUE: switch( $length) {
		case 1: // ID
			return $1;
		case 4: // ID LBRACKET expr RBRACKET
			{ int t = $1;
				if( !(t & ARRAY)) {
					err( "expected an array");
					return 0;
				}
				check_integer( tree, $3);
				return t & ~ARRAY;
			}
		default: unexpected_token();
		}
	case EXPR: switch( $length) {
		case 1: // LVALUE | METHOD_CALL | CONSTANT
			return $1;
		case 2:
			// MINUS expr | NOT expr
			return $(1,unary_op)(tree, $2);
		case 3:
			if( $(1,token) == LPAREN)
				// LPAREN expr RPAREN
				return $2;
			else
			if( $(2,token) == BIN_OP)
				// expr bin_op expr
				return $(2,bin_op)( tree, $1, $3);
			else
		default: unexpected_token();
		}
	case CONSTANT: // INT_CONSTANT | CHAR_CONSTANT | bool_constant
		return $1;
	case INT_CONSTANT:
		errno = 0;
		strtol( $lexeme, NULL, 0);
		if( errno == ERANGE)
			err( "integer constant is out of range");
		return INT;
	case CHAR_CONSTANT: return INT;
	case BOOL_CONSTANT: return BOOL;
	case STRING_CONSTANT: return STRING_CONSTANT;
	case ID: {
		Symbol* symbol = find_symbol( &symbols, $lexeme);
		if( symbol == NULL) {
			errors++;
			return 0;
		}
		return top( &symbol->type); }
	default: unexpected_token();
	}
#undef $1
#undef $2
#undef $3
#undef $4
}


/** Returns this type declaration. */
static int
type_decl( const Tree* tree, const void* in)
{
#define $1 $(1,type_decl)
	switch( $token) {
		case TYPE: // INT | BOOL
			return $1;
		case INT: case BOOL: case VOID:
			return $token;
		default: unexpected_token();
	}
#undef $1
}


/** Inherited attributes. */
typedef struct {
	/** Method being compiled. */
	Symbol* method;

	/** Current block in scope. */
	int block;

	/** Type of the next symbol. */
	int type;

	/** Number of loops. */
	int loop;
} Attr;


/** Adds fields and methods to the symbol table. */
static void
add_members( const Tree* tree, Attr* in)
{
#define $1 $(1,add_members)
#define $2 $(2,add_members)
#define $3 $(3,add_members)
#define $4 $(4,add_members)
#define $5 $(5,add_members)
	switch( $token) {
	case PROGRAM:
		// CLASS class_name LBRACE field_decls method_decls RBRACE
		in->type = CLASS; $2; $4; $5; return;
	case FIELD_DECL:
		switch( $length) {
		case 3: // type field_list SEMICOLON
			in->type = $(1,type_decl); $2; return;
		case 5: // type ID ASSIGN constant SEMICOLON
			in->type = $(1,type_decl); $2;
			if( $(4,type) != in->type)
				err( "left side does not agree with right side");
			return;
		default: unexpected_token();
		}
	case FIELD:
		switch( $length) {
		case 1: // ID
			$1; return;
		case 4: // ID LBRACKET INT_CONSTANT RBRACKET
			in->type |= ARRAY; $1; in->type &= ~ARRAY; return;
		default: unexpected_token();
		}
	case METHOD_DECL:
		// type ID LPAREN optional_param_list RPAREN block |
		// VOID ID LPAREN optional_param_list RPAREN block
		in->type = METHOD | $(1,type_decl); $2;
		in->method = $(2,symbol); $4; return;
	case PARAM: // type ID
		if( in->method->count < PARAMETERS)
			in->method->parameters[in->method->count] = $(1,type_decl);
		else
			err( "maximum of four parameters");
		in->method->count++;
		return;
	case ID:
		add_symbol( &symbols, $lexeme, in->block, in->type);
		return;
	default: $all( add_members);
	}
#undef $1
#undef $2
#undef $3
#undef $4
#undef $5
}


static void check_types( const Tree* tree, Attr* in);


/** Checks the types of this statement. */
static void
check_statement( const Tree* tree, Attr* in)
{
	switch( $token) {
	case ASSIGNMENT_LIST:
		$all( check_statement);
	case COMMA:
		return;
	case ASSIGNMENT: // lvalue ASSIGN expr
		eq_op( tree, $(1,type), $(3,type));
		return;
	case METHOD_CALL:
		$$(type);
		return;
	case BREAK:
	case CONTINUE:
		if( in->loop == 0)
			err( "loop control statement is not within a loop");
		return;
	case STATEMENT: switch( $length) {
		case 1: // block
			$(1,check_types);
			return;
		case 2: // assignment SEMICOLON | method_call SEMICOLON |
			// BREAK SEMICOLON | CONTINUE SEMICOLON
			$(1,check_statement);
			return;
		case 3: // RETURN optional_expr SEMICOLON
			{ int t = top( &in->method->type);
				if( $(2,type) != (t & ~METHOD))
					err( "return type does not agree");
			}
			return;
		case 5: // WHILE LPAREN expr RPAREN block
			check_boolean( tree, $(3,type));
			in->loop++;
			$(5,check_types);
			in->loop--;
			return;
		case 6: // IF LPAREN expr RPAREN block optional_else_clause
			check_boolean( tree, $(3,type));
			$(5,check_types);
			$(6,check_types);
			return;
		case 9: // FOR LPAREN assignment_list SEMICOLON expr SEMICOLON
				//   assignment_list RPAREN block
			$(3,check_statement);
			check_boolean( tree, $(5,type));
			$(7,check_statement);
			in->loop++;
			$(9,check_types);
			in->loop--;
			return;
		default: unexpected_token();
		}
	default: unexpected_token();
	}
}


/** Adds these symbols to the current scope. */
static void
scope( const Tree* tree, Attr* in)
{
	switch( $token) {
		case PARAM: // type ID
			in->type = $(1,type_decl);
			$(2,scope);
			return;
		case VAR_DECL: // type id_list SEMICOLON
			in->type = $(1,type_decl);
			$(2,scope);
			return;
		case ID:
			add_symbol( &symbols, $lexeme, in->block, in->type);
			return;
		default: $all( scope);
	}
}


/** Removes these symbols from the current scope. */
static void
unscope( const Tree* tree, Attr* in)
{
	if( $token == ID)
		remove_symbol( &symbols, $lexeme);
	else
		$all( unscope);
}


/** Checks this parse tree for type errors. */
static void
check_types( const Tree* tree, Attr* in)
{
	switch( $token) {
		case METHOD_DECL:
			// type ID LPAREN optional_param_list RPAREN block |
			// VOID ID LPAREN optional_param_list RPAREN block
			in->method = $(2,symbol);
			in->block++;
			$(4,scope);
			in->block--;
			$(6,check_types);
			$(4,unscope);
			return;
		case BLOCK: // LBRACE var_decls statements RBRACE
			in->block++;
			$(2,scope);
			$(3,check_types);
			$(2,unscope);
			return;
		case STATEMENT:
			$$(check_statement);
			return;
		default: $all( check_types);
	}
}


/** Checks the type of main. */
static void
check_main( SymbolTable* symbols)
{
	// check for main
	Symbol* symbol = find_symbol( symbols, "main");
	if( symbol != NULL) {
		if( top( &symbol->type) != (METHOD | VOID)) {
			puts( "main must be a method that returns void");
			errors++;
		}
		if( symbol->count != 0) {
			puts( "method main must take no parameters");
			errors++;
		}
	} else
		errors++;
}


/** Checks this parse tree for semantic errors. */
static void
check_decaf( const Tree* tree)
{
	clear_symboltable( &symbols);
	{ Attr In, *in = &In;
		in->block = 1;
		in->loop = 0;
		$$(add_members);
		$$(check_types);
	}
	check_main( &symbols);
}


/** Parses the Decaf program given on standard input and checks it for
 * semantic errors. */
int
main()
{
	errors = 0;
	check_decaf( read_parse_tree( stdin, &decaf_symbols));
	if( errors > 0)
		exit( EXIT_FAILURE);
	return 0;
}
