#!/usr/bin/env perl

# Some tests for 'darcs add'

use Test::More qw/no_plan/;; 
use strict;
use vars qw/$DARCS/;

die 'darcs not found' unless $ENV{DARCS} || (-x "$ENV{PWD}/../darcs");
$DARCS = $ENV{DARCS} || "$ENV{PWD}/../darcs";

`rm -rf temp1`;
`mkdir temp1`;
 chdir 'temp1';
`$DARCS init`;

###

my $test_name = 'Make sure that messages about directories call them directories.';
`mkdir foo.d`;
`mkdir oof.d`;
`$DARCS add foo.d`;
`$DARCS add oof.d`;
# Try adding the same directory when it's already in the repo 
like(`$DARCS add foo.d`, qr/directory/,$test_name);
like(`$DARCS add foo.d oof.d`, qr/directories/,$test_name);

###

$test_name = 'Make sure that messages about files call them files.';
`touch bar`;
`touch baz`;
`$DARCS add bar`;
`$DARCS add baz`;
like(`$DARCS add bar`, qr/following file is/, $test_name);
like(`$DARCS add bar baz`, qr/following files are/, $test_name);

###

$test_name = 'Make sure that messages about both files and directories say so.';
like(`$DARCS add bar foo.d`, qr/files and directories/, $test_name);

###

sub maybe_locale_encode {
    my ($str) = @_;
    my ($estr) = $str;

    if ($ENV{DARCS_USE_ISPRINT}) {
        use locale;
        $estr =~ s/[\\]/\\\\/g;
        $estr =~ s/([\s])/sprintf("\\\\%d\\\\", ord($1))/eg;    
        $estr =~ s/([^\w.\\])/sprintf("\\\\%02x.*", ord($1))/eg;
    } else {
        $estr =~ s/[\\]/\\\\/g;
        $estr =~ s/([\s])/sprintf("\\\\%d\\\\", ord($1))/eg;    
        $estr =~ s/([^\w.\\])/sprintf("\\\\%02x.*", ord($1))/eg;
    }
    return $estr;
}

sub test_add_file {
    my ($test_name, $filename) = @_;

    # generate some random ascii content
    my @chars = ("A" .. "Z", "a" .. "z", 0 .. 9);
    my $content = join("", @chars[ map { rand @chars } ( 1 .. 8 ) ]);    

    open(STUPIDFILE, ">$filename");
    print STUPIDFILE "$content\n";
    close(STUPIDFILE);

    # some file systems (such as windows') won't allow many characters in
    # file names, so skip the test if the file can't be created.
    if (-e $filename) {
        # encode the filename in the same format darcs uses for console output
        my ($encoded_fn) = maybe_locale_encode($filename);
        
        like(`$DARCS add "$filename"`, qr/^$/, $test_name);

        `$DARCS whatsnew --no-summary` =~ qr/addfile .*$encoded_fn/ ||
            fail("whatsnew missing addfile with '$filename'");
        
        `$DARCS whatsnew --no-summary` =~ qr/hunk .*$encoded_fn 1/ ||
            fail("whatsnew missing hunk with '$content'");
        
        `$DARCS record --all -A me -m 'add file'` =~ qr/^Finished recording/ ||
            fail("record failed to add file '$filename'");
        
        $? == 0 || fail("return code != 0");
    } else {
        ok(1, "skipped '$test_name', can't create '$filename'");
    }
}

# If DARCS_USE_ISPRINT is set, we should be runing on a system with a
# locale that works with darcs, and we now have a chanse to test that code.

if ($ENV{DARCS_USE_ISPRINT}) {

    print ("DARCS_USE_ISPRINT is set: run extra tests\n");
    test_add_file('Check unusual filename handling', "\03.ltxt");
    test_add_file('Check big char filename handling', "\xAA.ltxt");

    # We allways can, and should, test the default ascii-is-safe mode.
    print ("Turn DARCS_USE_ISPRINT off for the rest of the tests\n");
    $ENV{DARCS_USE_ISPRINT} = "0";
}

test_add_file('Check unusual filename handling', "\03.txt");
test_add_file('Check big char filename handling', "\xAA.txt");

### (no locale encoding available from here)

test_add_file('Check whitespace filename handling', "space tab\tnewline\n.txt");

chdir '../';
`rm -rf temp1`;
ok((!-d 'temp1'), 'temp1 directory was deleted');




