/**************************************************************************/
/*                                                                        */
/*  cpbk - a mirroring utility for backing up your files                  */
/*  Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>            */
/*  Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>                   */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2, or (at your option)   */
/*  any later version.                                                    */
/*                                                                        */
/*  This program is distributed in the hope that it will be useful,       */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/*  GNU General Public License for more details.                          */
/*                                                                        */
/*  You should have received a copy of the GNU General Public License     */
/*  along with this program; if not, write to the                         */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,          */
/*  Boston, MA 02111-1307, USA.                                           */
/*                                                                        */
/**************************************************************************/

/* $Id: fileop.c,v 1.6 2001/03/30 13:16:36 gm Exp $ */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <dirent.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <utime.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "intl.h"
#include "strlib.h"
#include "variable.h"
#include "proto.h"


static void scan_file(FileData *file_data, const char *path_dir);
static int comp_file_name(const void *file_state1, const void *file_state2);
static int comp_isdir(const void *file_state1, const void *file_state2);
static int comp_path_name(const void *path_state1, const void *path_state2);
static int get_num_dirs(FileState *file_state, int num_files);
static bool is_symlink_different(const char *src_path_file, const char *dest_path_file);
static void copy_special_file(Order *order, const char *src, const char *dest);
static void copy_symbolic_link(Order *order, const char *src, const char *dest);
static void copy_regular_file(Order *order, const char *src, const char *dest);
static bool is_hole(char *buffer, int bytes);


/* --- PUBLIC FUNCTIONS --- */

char *canonicalize_path(char *path_dir, const char *dir_type)
{
	char *temp;
	char *absolute_path;
	char *cwd;

	temp = str_malloc(PATH_MAX);
	if (realpath(path_dir, temp) == NULL || temp[0] != '/') { /* realpath() failed or it may not return absolute path for relative path. */
		cwd = str_malloc(PATH_MAX);
		if (getcwd(cwd, PATH_MAX) == NULL) {
			err_printf(ALWAYS_EXIT, "getcwd", _("Cannot get current working directory.\n"));
		}
		absolute_path = str_concat(cwd, "/", path_dir, NULL);
		if (*(absolute_path + strlen(absolute_path) - 1) == '/') { /* remove trailing slash */
			*(absolute_path + strlen(absolute_path) - 1) = '\0';
		}
	} else {
		absolute_path = str_dup(temp);
	}
	free(temp);
	free(path_dir);

	if (strcmp(absolute_path, "/") == 0 || strcmp(absolute_path, "//") == 0) {
		absolute_path[0] = '\0';
	}

	return (absolute_path);
}


int mkdir_parents(const char *path_dir, mode_t mode)
{
	char *temp;
	char *dirptr;

	if (access(path_dir, F_OK) == 0) {
		return (0);
	}

	temp = str_dup(path_dir);
	
	dirptr = temp + 1;
	while (dirptr != NULL) {
		if ((dirptr = strchr(dirptr, '/')) != NULL) {
			*dirptr = '\0';
		}
		if (access(temp, F_OK) == -1) {
			if (mkdir(temp, mode) == -1) {
				err_printf(CAN_PROCEED, "mkdir", NULL);
				return (-1);
			}
		}
		if (dirptr != NULL) {
			*dirptr = '/';
			dirptr++;
		}
	}
	return (0);
}


void check_valid_dir(const char *path_dir, const char *dir_type)
{
	struct stat status;

	if (lstat((path_dir[0] == '\0') ? "/" : path_dir, &status) == -1) {
		err_printf(ALWAYS_EXIT, "lstat", _("Cannot get file status: %s\n"), (path_dir[0] == '\0') ? "/" : path_dir);
	}

	if (!S_ISDIR(status.st_mode)) {
		err_printf(ALWAYS_EXIT, NULL, _("The argument specified as %s directory is not a directory: %s\n"), dir_type, (path_dir[0] == '\0') ? "/" : path_dir);
	}
}


void get_file_data(FileData *file_data, const char *top_dir)
{
	file_data->num_paths = 0;
	file_data->path_state = NULL;

	scan_file(file_data, top_dir);

#if 0
	output_file_data(file_data, 0);
#endif
}


void sort_file_state(FileState *file_state, int num_files)
{
	int num_dirs;

	num_dirs = get_num_dirs(file_state, num_files);

	/* files move to the front */
	qsort(file_state, num_files, sizeof(FileState), comp_isdir);

	/* sort files by alphabetical order */
	qsort(file_state, num_files - num_dirs, sizeof(FileState), comp_file_name);

	/* sort directories by alphabetical order */
	qsort(file_state + num_files - num_dirs, num_dirs, sizeof(FileState), comp_file_name);
}


void sort_path_state(PathState *path_state, int num_paths)
{
	/* sort by alphabetical order */
	qsort(path_state, num_paths, sizeof(PathState), comp_path_name);
}


void output_file_data(const FileData *file_data, int nest_level)
{
	int i, j;

	for (i = 0; i < file_data->num_paths; i++) {
		printf("%3d %s\n", file_data->path_state[i].num_files, (file_data->path_state[i].path[0] == '\0') ? "/" : file_data->path_state[i].path);
		for (j = 0; j < file_data->path_state[i].num_files; j++) {
			printf("%ld %ld %ld %8ld %d %d %o %d %s\n",
			       file_data->path_state[i].file_state[j].status.st_atime,
			       file_data->path_state[i].file_state[j].status.st_mtime,
			       file_data->path_state[i].file_state[j].status.st_ctime,
			       file_data->path_state[i].file_state[j].status.st_size,
			       file_data->path_state[i].file_state[j].status.st_uid,
			       file_data->path_state[i].file_state[j].status.st_gid,
			       file_data->path_state[i].file_state[j].status.st_mode,
			       file_data->path_state[i].file_state[j].isdir,
			       file_data->path_state[i].file_state[j].name);
		}
	}
}


void free_file_data(FileData *file_data)
{
	int i, j;

	for (i = 0; i < file_data->num_paths; i++) {
		for (j = 0; j < file_data->path_state[i].num_files; j++) {
			free(file_data->path_state[i].file_state[j].name);
		}
		free(file_data->path_state[i].path);
		free(file_data->path_state[i].file_state);
	}
	free(file_data->path_state);
}


bool is_src_newer(FileState *src_stat, FileState *dest_stat, const char *src_dir, const char *dest_dir)
{
	char *src_path_file;
	char *dest_path_file;

	if (config.force) {
		return (TRUE);
	}

	src_path_file = str_concat(src_dir, "/", src_stat->name, NULL);
	dest_path_file = str_concat(dest_dir, "/", dest_stat->name, NULL);

	if (S_ISLNK(src_stat->status.st_mode)) {
		return (is_symlink_different(src_path_file, dest_path_file));
	}

	if (config.file_size_check && src_stat->status.st_size != dest_stat->status.st_size) {
		return (TRUE);
	}
	if (config.inode_ctime_check && src_stat->status.st_ctime > dest_stat->status.st_ctime) {
		return (TRUE);
	}

	if (src_stat->status.st_mtime > dest_stat->status.st_mtime) {
		return (TRUE);
	}

	return (FALSE);
}


bool is_src_mode_changed(FileState *src_stat, FileState *dest_stat, const char *src_dir, const char *dest_dir)
{
	char *src_path_file;
	char *dest_path_file;

	src_path_file = str_concat(src_dir, "/", src_stat->name, NULL);
	dest_path_file = str_concat(dest_dir, "/", dest_stat->name, NULL);

	if (src_stat->status.st_uid != dest_stat->status.st_uid ||
	    src_stat->status.st_gid != dest_stat->status.st_gid) {
		if (!am_i_root()) {
			err_printf(CAN_PROCEED, NULL, _("\nOwner of `%s' is changed.\nBut there is no permission to change the owner.\n"), src_path_file);
		} else {
			return (TRUE);
		}
	}
	if (src_stat->status.st_mode != dest_stat->status.st_mode) {
		return (TRUE);
	}

	return (FALSE);
}


void copy_file(Order *order, const char *src, const char *dest)
{
	if (config.simulate) {
		return;
	}

	if (strcmp(src, dest) == 0) {
		err_printf(CAN_PROCEED, NULL, _("Source file and destination file are identical. Order canceled.\n"));
		return;
	}
	if (S_ISCHR(order->status.st_mode) || S_ISBLK(order->status.st_mode) || S_ISFIFO(order->status.st_mode) || S_ISSOCK(order->status.st_mode)) {
		copy_special_file(order, src, dest);
	} else if (S_ISLNK(order->status.st_mode)) {
		copy_symbolic_link(order, src, dest);
	} else {
		copy_regular_file(order, src, dest);
	}
}


void remove_file(Order *order, const char *dest)
{
	if (config.simulate) {
		return;
	}

	if (unlink(dest) == -1) {
		err_printf(CAN_PROCEED, "unlink", _("Cannot remove file `%s'.\n"), dest);
	}
}


void make_dir(Order *order, const char *dest)
{
	struct utimbuf newtime;

	if (config.simulate) {
		return;
	}

	if (mkdir(dest, order->status.st_mode & 07777) == -1) {
		err_printf(CAN_PROCEED, "mkdir", _("Cannot make directory `%s'.\n"), dest);
	}

	if (am_i_root()) {
		if (lchown(dest, order->status.st_uid, order->status.st_gid) == -1) {
			err_printf(CAN_PROCEED, "lchown", _("Cannot change ownership of directory `%s'.\n"), dest);
		}	
	}

	/* Set time on directory to same as source directory */
	newtime.actime = order->status.st_atime;
	newtime.modtime = order->status.st_mtime;

	if (utime(dest, &newtime) == -1) {
		err_printf(CAN_PROCEED, "utime", _("Cannot change access and modification times of an inode: %s.\n"), dest);
	}	
}


void remove_dir(Order *order, const char *dest)
{
	if (config.simulate) {
		return;
	}

	if (rmdir(dest) == -1) {
		err_printf(CAN_PROCEED, "rmdir", _("Cannot remove directory `%s'.\n"), dest);
	}
}


void change_mode(Order *order, const char *dest)
{
	struct utimbuf newtime;

	if (config.simulate) {
		return;
	}

	if (chmod(dest, order->status.st_mode & 07777) == -1) {
		err_printf(CAN_PROCEED, "chmod", _("Cannot change access permission of `%s'.\n"), dest);
	}	

	if (am_i_root()) {
		if (lchown(dest, order->status.st_uid, order->status.st_gid) == -1) {
			err_printf(CAN_PROCEED, "lchown", _("Cannot change ownership of `%s'.\n"), dest);
		}	
	}

	/* Set time on directory to same as source directory */
	newtime.actime = order->status.st_atime;
	newtime.modtime = order->status.st_mtime;

	if (utime(dest, &newtime) == -1) {
		err_printf(CAN_PROCEED, "utime", _("Cannot change access and modification times of an inode: %s.\n"), dest);
	}	
}


bool am_i_root(void)
{
	return (getuid() == 0);
}


/* --- PRIVATE FUNCTIONS --- */

static void scan_file(FileData *file_data, const char *path_dir)
{
	FileState *file_state = NULL;
	DIR *dir;
	struct dirent *ent;
	struct stat status;
	bool isdir;
	int num_paths;
	int num_files;
	char *path_file;

	update_progress_bar();

	dir = opendir((path_dir[0] == '\0') ? "/" : path_dir);

	num_paths = file_data->num_paths;
	file_data->num_paths++;

	file_data->path_state = str_realloc(file_data->path_state, sizeof(PathState) * (num_paths + 1));
	file_data->path_state[num_paths].path = str_dup(path_dir);
	file_data->path_state[num_paths].file_state = NULL;
	num_files = 0;

	if (dir == NULL) {
		if (!config.quiet) printf(_("%c Error\n"), '\b');
		err_printf(CAN_PROCEED, "opendir", _("Cannot open directory `%s'.\n"), (path_dir[0] == '\0') ? "/" : path_dir);
		file_data->path_state[num_paths].num_files = 0;
		return;
	}

	while ((ent = readdir(dir)) != NULL) {
		if (strcmp(ent->d_name, ".") == 0 || strcmp(ent->d_name, "..") == 0) {
			continue;
		}

		path_file = str_concat(path_dir, "/", ent->d_name, NULL);

		if (is_exclude_path(path_file, path_file)) {
			continue;
		}

		if (lstat(path_file, &status) == -1) {
			if (!config.quiet) printf(_("%c Error\n"), '\b');
			err_printf(CAN_PROCEED, "lstat", _("Cannot get file status: %s\n"), path_file);
			continue;
		}

		isdir = S_ISDIR(status.st_mode);

		file_state = str_realloc(file_state, (num_files + 1) * sizeof(FileState));
		file_state[num_files].name = str_dup(ent->d_name);
		file_state[num_files].isdir = isdir;
		file_state[num_files].status = status;

		if (isdir) {
			scan_file(file_data, path_file);
		}

		num_files++;

		free(path_file);
	}

	if (closedir(dir) == -1) {
		if (!config.quiet) printf(_("%c Error\n"), '\b');
		err_printf(CAN_PROCEED, "closedir", _("Cannot close directory `%s'.\n"), (path_dir[0] == '\0') ? "/" : path_dir);
	}

	sort_file_state(file_state, num_files);

	file_data->path_state[num_paths].num_files = num_files;
	file_data->path_state[num_paths].file_state = file_state;
}


static int comp_file_name(const void *file_state1, const void *file_state2)
{
	return (strcmp(((FileState *) file_state1)->name, ((FileState *) file_state2)->name));
}


static int comp_isdir(const void *file_state1, const void *file_state2)
{
	if (!((FileState *) file_state1)->isdir && ((FileState *) file_state2)->isdir) {
		return (-1);
	}
	if (((FileState *) file_state1)->isdir && !((FileState *) file_state2)->isdir) {
		return (1);
	}
	return (0);
}


static int comp_path_name(const void *path_state1, const void *path_state2)
{
	return (strcmp(((PathState *) path_state1)->path, ((PathState *) path_state2)->path));
}


static int get_num_dirs(FileState *file_state, int num_files)
{
	int i;
	int num_dirs = 0;

	for (i = 0; i < num_files; i++) {
		if (file_state[i].isdir) {
			num_dirs++;
		}
	}
	return (num_dirs);
}


static bool is_symlink_different(const char *src_path_file, const char *dest_path_file)
{
	int symsize;
        char src_symbuf[PATH_MAX + 1];
        char dest_symbuf[PATH_MAX + 1];

	if ((symsize = readlink(src_path_file, src_symbuf, PATH_MAX)) == -1) {
		err_printf(CAN_PROCEED, "readlink", _("Cannot read value of symbolic link `%s'.\n"), src_path_file);
		return (FALSE);
	}
	src_symbuf[symsize] = '\0';

	if ((symsize = readlink(dest_path_file, dest_symbuf, PATH_MAX)) == -1) {
		return (TRUE);
	}
	dest_symbuf[symsize] = '\0';

	return (strcmp(src_symbuf, dest_symbuf));
}


static void copy_special_file(Order *order, const char *src, const char *dest)
{
	struct utimbuf newtime;

	if (am_i_root() || S_ISFIFO(order->status.st_mode)) {
		if (access(dest, F_OK) == 0 && unlink(dest) == -1) {
			err_printf(CAN_PROCEED, "unlink", _("Cannot remove file `%s'.\n"), dest);
			return;
		}
		if (mknod(dest, order->status.st_mode, order->status.st_rdev) == -1) {
			err_printf(CAN_PROCEED, "mknod", _("Cannot make special file `%s'.\n"), dest);
			return;
		}
		if (am_i_root()) {
			if (chown(dest, order->status.st_uid, order->status.st_gid) == -1) {
				err_printf(CAN_PROCEED, "chown", _("Cannot change ownership of file `%s'.\n"), dest);
			}	
		}

		/* Set time on file to same as source file */
		newtime.actime = order->status.st_atime;
		newtime.modtime = order->status.st_mtime;

		if (utime(dest, &newtime) == -1) {
			err_printf(CAN_PROCEED,"utime", _("Cannot change access and modification times of an inode: %s\n"), dest);
		}	
	} else {
		err_printf(CAN_PROCEED, NULL, _("Copy only permitted by root for this file type: %s\n"), dest);
	}
}


static void copy_symbolic_link(Order *order, const char *src, const char *dest)
{
	int symsize;
        char symbuf[PATH_MAX + 1];

	if (!is_symlink_different(src, dest)) {
		return;
	}

	unlink(dest);

	if ((symsize = readlink(src, symbuf, PATH_MAX)) == -1) {
		err_printf(CAN_PROCEED, "readlink", _("Cannot read value of symbolic link `%s'.\n"), src);
		return;
	}
	symbuf[symsize] = '\0';

	if (symlink(symbuf, dest) == -1) {
		err_printf(CAN_PROCEED, "symlink", _("Cannot create symbolic link `%s'.\n"), dest);
		return;
	}
	if (am_i_root()) {
		if (lchown(dest, order->status.st_uid, order->status.st_gid) == -1) {
			err_printf(CAN_PROCEED, "lchown", _("Cannot change ownership of file `%s'.\n"), dest);
		}	
	}
	/* No need to do anything else with symlinks,
	   because permissions are ignored by system and
	   times cannot be set with utimes(). */
}


static void copy_regular_file(Order *order, const char *src, const char *dest)
{
	bool has_hole = FALSE;
	char *copy_buf;
	int src_fd;
	int dest_fd;
	size_t read_bytes;
	size_t total_bytes = 0;
	struct utimbuf newtime;

	/* If we get this far and it is not a regular file, then we cannot copy it.
	   It is probably a UNIX Socket. We will get to those soon. */
	if (!S_ISREG(order->status.st_mode)) {
		err_printf(CAN_PROCEED, NULL, _("`%s' is not a regular file.\n"), src);
		return;
	}

	/* Check if size is what it should be compared to how many blocks
	   that we have. */
	if ((size_t)((order->status.st_size / order->status.st_blksize) + 1) > (size_t)order->status.st_blocks) {
		has_hole = TRUE;
	}

	if ((src_fd = open(src, O_RDONLY)) == -1) {
		err_printf(CAN_PROCEED, "open", _("Cannot open source file `%s'.\n"), src);
		return;
	}

	if (access(dest, F_OK) == 0 && unlink(dest) == -1) {
		err_printf(CAN_PROCEED, "unlink", _("Cannot remove old destination file `%s'.\n"), dest);
		if (close(src_fd) == -1) {
			err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
		}
		return;
	}
	if ((dest_fd = creat(dest, order->status.st_mode & 07777)) == -1) {
		err_printf(CAN_PROCEED, "open", _("Cannot create destination file `%s'.\n"), dest);
		if (close(src_fd) == -1) {
			err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
		}
		return;
	}

	copy_buf = str_malloc(BUFSIZE);

	while ((read_bytes = read(src_fd, copy_buf, BUFSIZE)) > 0) {
		total_bytes += read_bytes;

		if (has_hole && is_hole(copy_buf, read_bytes)) { /* Check if data is a hole. */
			if (read_bytes == BUFSIZE) { /* lseek read_bytes to rebuild the hole */
				if (lseek(dest_fd, read_bytes, SEEK_CUR) == -1) {
					err_printf(CAN_PROCEED, "lseek", _("Cannot reposition file offset: %s\n"), dest);
					if (close(src_fd) == -1) {
						err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
					}
					if (close(dest_fd) == -1) {
						err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), dest);
					}
					free(copy_buf);
					return;
				}
			} else {
				if (ftruncate(dest_fd, total_bytes) == -1) { /* ftrunct total bytes read to rebuild the hole */
					err_printf(CAN_PROCEED, "ftruncate", _("Cannot truncate file `%s'.\n"), dest);	
					if (close(src_fd) == -1) {
						err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
					}
					if (close(dest_fd) == -1) {
						err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), dest);
					}
					free(copy_buf);
					return;
				}
			}
		} else {
			if (write(dest_fd, copy_buf, read_bytes) == -1) {
				err_printf(CAN_PROCEED, "write", _("Cannot write file `%s'.\n"), dest);	
				if (close(src_fd) == -1) {
					err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
				}
				free(copy_buf);
				return;
			}
		}
	}

	/* Error check if read returned an error. */
	if (read_bytes == -1) {
		err_printf(CAN_PROCEED, "read", _("Cannot read file `%s'.\n"), src);
		if (close(src_fd) == -1) {
			err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
		}
		if (close(dest_fd) == -1) {
			err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), dest);
		}
		free(copy_buf);
		return;
	}

	if (am_i_root()) {
		if (fchown(dest_fd, order->status.st_uid, order->status.st_gid) == -1) {
			err_printf(CAN_PROCEED, "fchown", _("Cannot change ownership of file `%s'.\n"), dest);
		}	
	}

	/* Set time on file to same as source file */
	newtime.actime = order->status.st_atime;
	newtime.modtime = order->status.st_mtime;

	if (utime(dest, &newtime) == -1) {
		err_printf(CAN_PROCEED, "utime", _("Cannot change access and modification times of an inode: %s.\n"), dest);
	}	

	if (close(src_fd) == -1) {
		err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), src);
	}
	if (close(dest_fd) == -1) {
		err_printf(CAN_PROCEED, "close", _("Cannot close file `%s'.\n"), dest);
	}
					
	free(copy_buf);
}


static bool is_hole(char *buffer, int bytes)
{
	int i;

	for (i = 0; i < bytes; i++) {
		if (buffer[i] != 0) {
			return (FALSE);
		}
	}
	return (TRUE);
}
