/* bmark.c */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <setjmp.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include "config.h"
#include "bmark.h"
#include "main.h"
#include "sysinfo.h"
#include "trivial.h"
#include "programs.h"

extern int opt_prog_bar;

#define TMP_FILE "tmp"
static pid_t pid;	/* we need to get to it from the signal handler */

unsigned long bmark(struct load *ld)
{
	int status, tmpfd;
	struct bench_results res;
	struct rusage ru;
	struct timeval before, after;
	sigset_t sigset;
	struct sigaction sa;
	struct stat statbuf;

	ld->child_pid=0;
	if(pipe(ld->pipe)==-1){
		printsys("error creating pipe\n");
	}
	if((pid=fork())==-1){
		printsys("fork error\n");
	} else if(pid){		/* parent, returns to caller in main.c */
		ld->child_pid=pid;
		return pid;
	}
	/* child */

	if((tmpfd=open(TMP_FILE, O_RDWR|O_CREAT|O_TRUNC))==-1){
		printsys("Could not open temporary file\n");
	}
	if(dup2(tmpfd, TMP_FD)==-1){
		printsys("error dup2ing tmpfd\n");
	}
	close(tmpfd);

	restore_signals();

	/* fill a structure to block all irrelevant signals */
	sigfillset(&sigset);
	sigdelset(&sigset, SIGCHLD);
	sigdelset(&sigset, SIGUSR1);

	/* pass it to the kernel */
	sigprocmask(SIG_SETMASK, &sigset, NULL);


	/* prepare sigaction struct */
	sa.sa_handler=&sig_usr1;
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGUSR1);
	sa.sa_flags=SA_RESTART;

	gettimeofday(&before, NULL);
	if((pid=fork())==-1){
		printf("fork error\n");
		return 0;
	} else if(!pid){	/* child, is waited for in bmark() */
		unblock_signals();
		ld->do_load();
		exit(0);
	}
	/* parent */

	/*
	 * we wait for the child, but if we get
	 * SIGUSR1, we kill it from the signal handler
	 * and return here to wait for it again
	 */
	sigaction(SIGUSR1, &sa, NULL);
	if(wait4(pid, &status, 0, &ru)!=pid){
		printsys("bmark: error waiting for child\n");
	}
	gettimeofday(&after, NULL);
	if(stat(TMP_FILE, &statbuf)==-1){
		printsys("could not get loadruns");
	}
	if(unlink(TMP_FILE)){
		printsys("could not unlink file %s\n", TMP_FILE);
	}
	res.loadruns=statbuf.st_size;
	printd("read loadruns=%lu\n", res.loadruns);
	res.utime=ru.ru_utime.tv_sec;
	res.systime=ru.ru_stime.tv_sec;
	res.elapsed=after.tv_sec - before.tv_sec;
	res.major_faults=ru.ru_majflt;
	res.minor_faults=ru.ru_minflt;
	printd("child: about to write data to parent\n");
	if(write(ld->pipe[1], &res, sizeof(struct bench_results))!=
			sizeof(struct bench_results)){
		printf("error transferring data to parent\n");
		exit(1);
	}
	printd("child: wrote it\n");
	exit(0);
}

/*
 * we get SIGUSR1 by the parent when kernel compilation
 * has finished and we must kill the child that generates
 * the load
 */
void sig_usr1(int whatever)
{
	printd("pid=%d, got SIGUSR1, killing process %d\n", getpid(), pid);
	if(kill(pid, SIGTERM) && (errno!=ESRCH)){
		printsys("could not kill process %lu\n", pid);
	}
	whatever=0;
	return;
}

struct bench_results compile_kernel(struct load *ld)
{
	pid_t pid;
	int status, ret, perc;
	unsigned long time=0, estimate;
	struct rusage ru;
	struct bench_results res;
	struct timeval before, after;

	estimate=est_elapsed(ld->logfile);
	if(opt_prog_bar){
		if(estimate){
			printf("estimated progress according to previous runs:\n");
		} else {
			printf("no previous results for estimate...\n");
		}
	}
	estimate *= 4;		/* we sleep for 1/4 sec */

	block_signals();	/* we need to unblock them in the child, too */
	gettimeofday(&before, NULL);
	if((pid=fork())<0){
		printf("fork error, aborting\n");
		exit(1);
	} else if(!pid){	/* child */
		int fd;
		int cpus;
		char buf[20];

		unblock_signals();
		if(get_cpus(&cpus)){
			printw("no info on number of cpus, using default=1\n");
			cpus=1;
		}
		if((fd=open("/dev/null", O_WRONLY))==-1){
			printsys("could not open /dev/null\n");
		}
		close(STDOUT_FILENO);
		close(STDERR_FILENO);
		if(dup2(fd, STDOUT_FILENO)==-1){
			printsys("error redirecting stdout to /dev/null\n");
		}
		if(dup2(fd, STDERR_FILENO)==-1){
			printsys("error redirecting stderr to /dev/null\n");
		}
		snprintf(buf, 20, "%d", 4*cpus);
		if(execlp(MAKE_BIN, MAKE_BIN, "bzImage", "-j", buf, NULL)){
			printsys("failed execing make bzImage\n");
		}
	}
	harvester_add_child(pid);
	unblock_signals();
	printd("waiting for child with pid %d\n", pid);
	if(opt_prog_bar){
		while((ret=wait4(pid, &status, WNOHANG, &ru))!=pid){
			if(ret==-1){
				printsys("wait failed\n");
			}
			usleep(250000);	/* 1/4 sec */
			time+=1;
			if(estimate){
				perc=(100*time)/estimate;
				if(perc>100){
					perc=100;
					estimate=0;
					print_prog_bar(100);
					printf("\nalmost done ...\n");
				} else {
					print_prog_bar(perc);
				}
			}
		}
	} else if(wait4(pid, &status, 0, &ru)!=pid){
		printsys("could not wait for pid %d\n", pid);
	}
	gettimeofday(&after, NULL);
	harvester_del_child(pid);
	if(estimate && opt_prog_bar){
		print_prog_bar(100);
		printf("\n");
	}
	res.utime=ru.ru_utime.tv_sec;
	res.systime=ru.ru_stime.tv_sec;
	res.elapsed=after.tv_sec - before.tv_sec;
	res.major_faults=ru.ru_majflt;
	res.minor_faults=ru.ru_minflt;
	return res;
}
