;;; -*- Mode: LISP; Syntax: ANSI-Common-Lisp; Package: CL-USER; -*-
;;;
;;; Filename:    socket-internal.lisp
;;; Author:      Jochen Schmidt <jsc@dataheaven.de>
;;; Description: An implementation for simple buffering
;;;              and the primitives for doing character
;;;              and byte oriented I/O
;;;

(in-package :ssl-internal)

(eval-when (:compile-toplevel :load-toplevel :execute)
(defparameter *random-entropy* 256)
)

(defun ssl-socket-rand-seed ()
  (let ((seed (make-byte-vector 256)))
    (dotimes (i *random-entropy*)
      (setf (bvref seed i) (random *random-entropy*)))
    (rand-seed seed *random-entropy*)
    (free-byte-vector seed)))

(eval-when (:compile-toplevel :load-toplevel :execute)
(defparameter *buffer-size* 4096)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Internal socket-representation with simple buffer handling ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defclass simple-buffering-mixin ()
  ((input-avail :accessor ssl-socket-input-avail :initform 0 :type fixnum)
   (input-offset :accessor ssl-socket-input-offset :initform 0 :type fixnum)
   (input-buffer :accessor ssl-socket-input-buffer
                 :initform (make-byte-vector 4096))
   (output-offset :accessor ssl-socket-output-offset :initform 0 :type fixnum)
   (output-buffer :accessor ssl-socket-output-buffer
                  :initform (make-byte-vector 4096))))

(defclass ssl-socket-mixin () ;(simple-buffering-mixin)
  ((fd :accessor ssl-socket-fd :initarg :fd)
   (method :accessor ssl-socket-method :initarg :method)
   (ctx :accessor ssl-socket-ctx :initarg :ctx)
   (handle :accessor ssl-socket-handle :initarg :handle)
   (rsa-privatekey-file :accessor ssl-socket-rsa-privatekey-file
                        :initarg :rsa-privatekey-file)
   (certificate-file :accessor ssl-socket-certificate-file
                     :initarg :certificate-file)))

(defun ssl-socket-read-byte (sock)
  (declare (optimize (speed 3) (safety 1)))
  (handler-case
      (progn
        (unless (< (ssl-socket-input-offset sock) (ssl-socket-input-avail sock))
          (fill-input-buffer sock))
        (let ((e (bvref (ssl-socket-input-buffer sock) (ssl-socket-input-offset sock))))
          (incf (ssl-socket-input-offset sock))
          e))
    (ssl-error-zero-return (condition)
         (declare (ignore condition))
         (return-from ssl-socket-read-byte :eof))))

(defun ssl-socket-read-char (sock)
  (let ((ret (ssl-socket-read-byte sock)))
    (if (eq :eof ret)
        ret
      (code-char ret))))

(defun ssl-socket-write-byte (byte sock)
  (with-slots (output-offset output-buffer) sock
    (when (>= output-offset *buffer-size*)
      (flush-output-buffer sock))
    (prog1
        (setf (bvref output-buffer output-offset) byte)
      (incf output-offset))))

(defun ssl-socket-write-char (char sock)
  (ssl-socket-write-byte (char-code char) sock))

(defun fill-input-buffer (sock)
  (setf (ssl-socket-input-avail sock)
        (ensure-ssl-funcall (ssl-socket-handle sock) 'ssl-read 
                            '(:sleep-time 0.5)
                            (ssl-socket-handle sock)
                            (ssl-socket-input-buffer sock)
                            *buffer-size*))
  (setf (ssl-socket-input-offset sock) 0))

(defun flush-output-buffer (sock)
  (with-slots (output-offset output-buffer) sock
    (when (> output-offset 0)
      (ensure-ssl-funcall (ssl-socket-handle sock) 'ssl-write 
                          '(:sleep-time 0.5)
                          (ssl-socket-handle sock)
                          output-buffer
                          output-offset)
      (setf output-offset 0))))

(defun ssl-socket-internal-close (sock)
  (with-slots (fd input-buffer output-buffer) sock
              #+lispworks
              (comm::close-socket fd)
              (free-byte-vector input-buffer)
              (free-byte-vector output-buffer)))

;; EOF
