/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkMultiRowSel Copyright (C) 2000 Stefan Ondrejicka <ondrej@idata.sk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gtkmultirowsel.h"
#include "gtkselbox.h"
#include <gtk/gtkviewport.h>
#include <gtk/gtksignal.h>
#include <stdlib.h>

#define BETWEEN(v, r1, r2) (((v) >= (r1)) && ((v) <= (r2)))

static void	gtk_multirowsel_class_init	(GtkMultiRowSelClass	*klass);
static void	gtk_multirowsel_init		(GtkMultiRowSel		*multirowsel);
static gint	gtk_multirowsel_button_press	(GtkWidget		*widget,
						 GdkEventButton		*event);
static gint	gtk_multirowsel_button_release	(GtkWidget		*widget,
						 GdkEventButton		*event);
static gint	gtk_multirowsel_motion		(GtkWidget		*widget, 
						 GdkEventMotion		*event);
static void	gtk_multirowsel_realize		(GtkWidget		*widget);
static gint	gtk_multirowsel_expose		(GtkWidget		*widget,
						 GdkEventExpose		*event);
static void	gtk_multirowsel_add		(GtkContainer		*container,
						 GtkWidget *widget);

guint gtk_multirowsel_get_type ()
{
        static guint multirowsel_type = 0;

        if (!multirowsel_type)
        {
                GtkTypeInfo multirowsel_info =
                {
                        "GtkMultiRowSel",
                        sizeof (GtkMultiRowSel),
                        sizeof (GtkMultiRowSelClass),
                        (GtkClassInitFunc) gtk_multirowsel_class_init,
                        (GtkObjectInitFunc) gtk_multirowsel_init,
                        NULL,
                        NULL,
			NULL,
                };

                multirowsel_type = gtk_type_unique (gtk_multirow_get_type (), &multirowsel_info);
        }

        return multirowsel_type;
}

static void gtk_multirowsel_class_init (GtkMultiRowSelClass *klass)
{
        GtkWidgetClass *widget_class;
	GtkContainerClass *container_class;

        widget_class = (GtkWidgetClass*) klass;
        container_class = (GtkContainerClass*) klass;

	widget_class->button_press_event = gtk_multirowsel_button_press;
	widget_class->button_release_event = gtk_multirowsel_button_release;
	widget_class->motion_notify_event = gtk_multirowsel_motion;
	widget_class->realize = gtk_multirowsel_realize;
	widget_class->expose_event = gtk_multirowsel_expose;

	container_class->add = gtk_multirowsel_add;
}

static void gtk_multirowsel_init (GtkMultiRowSel *multirowsel)
{
	multirowsel->sel_gc = NULL;
	multirowsel->selecting = FALSE;
	multirowsel->viewport_parent = FALSE;

	multirowsel->redrawing = FALSE;
	multirowsel->selection_mode = GTK_SELECTION_MULTIPLE;

	GTK_WIDGET_SET_FLAGS (multirowsel,  GTK_RECEIVES_DEFAULT);
	GTK_WIDGET_UNSET_FLAGS (multirowsel, GTK_NO_WINDOW);

}

GtkWidget* gtk_multirowsel_new (
gint ncols)
{
        GtkMultiRow *multirowsel;

        multirowsel = gtk_type_new (gtk_multirowsel_get_type ());

	gtk_multirow_set_number_of_columns(GTK_MULTIROW(multirowsel), ncols);

        return GTK_WIDGET (multirowsel);
}

static void gtk_multirowsel_draw_sel_box(GtkMultiRowSel *multirowsel)
{
	GtkWidget *widget = GTK_WIDGET(multirowsel);
	gint x,y,w,h;

	x = MIN(multirowsel->sel_box.x1, multirowsel->sel_box.x2);
	y = MIN(multirowsel->sel_box.y1, multirowsel->sel_box.y2);
	w = abs(multirowsel->sel_box.x2 - multirowsel->sel_box.x1);
	h = abs(multirowsel->sel_box.y2 - multirowsel->sel_box.y1);

	gdk_draw_rectangle(widget->window, multirowsel->sel_gc, FALSE,
		x, y, w, h);
}

static void gtk_multirowsel_unselect_all(GtkMultiRowSel *mrs)
{
	GtkBoxChild *child;
	GList *children;

        children = GTK_BOX(mrs)->children;
        while (children)
        {
		child = children->data;
		children = children->next;

		gtk_selbox_set_selected(GTK_SELBOX(child->widget), FALSE);
	}
}

static void gtk_multirowsel_select(GtkMultiRowSel *mrs, GdkEvent *event)
{
	GtkBoxChild *child;
	GList *children;
	GdkRectangle srec;
	GdkRectangle r,tr;

	srec.x = MIN(mrs->sel_box.x1, mrs->sel_box.x2);
	srec.y = MIN(mrs->sel_box.y1, mrs->sel_box.y2);
	srec.width = abs(mrs->sel_box.x1 - mrs->sel_box.x2);
	srec.height = abs(mrs->sel_box.y1 - mrs->sel_box.y2);

	if (!srec.width)
		srec.width++;

	if (!srec.height)
		srec.height++;

        children = GTK_BOX(mrs)->children;
        while (children)
        {
		child = children->data;
		children = children->next;

		if (GTK_WIDGET_VISIBLE (child->widget))
		{
			r.x = child->widget->allocation.x;
			r.y = child->widget->allocation.y;
			r.width = child->widget->allocation.width;
			r.height = child->widget->allocation.height;

			if (gdk_rectangle_intersect(&srec, &r, &tr))
				gtk_selbox_set_selected(GTK_SELBOX(child->widget),
					(((GdkEventButton *)event)->state & GDK_CONTROL_MASK) ? !GTK_SELBOX(child->widget)->selected : TRUE);
 		}
	}
}

static gint gtk_multirowsel_button_press (
GtkWidget		*widget,
GdkEventButton		*event)
{
	GtkMultiRowSel *mrs = GTK_MULTIROWSEL(widget);

	if (event->button == 1)
	{
		if (event->window != widget->window)
		{
			gint wx,wy,ex,ey;

			gdk_window_get_deskrelative_origin(event->window, &ex, &ey);
			gdk_window_get_deskrelative_origin(widget->window, &wx, &wy);

			event->x = event->x + ex - wx;
			event->y = event->y + ey - wy;
		}

		mrs->sel_box.x1 = event->x;
		mrs->sel_box.y1 = event->y;
		mrs->sel_box.x2 = event->x;
		mrs->sel_box.y2 = event->y;

		if (!(event->state & GDK_CONTROL_MASK) ||
			(mrs->selection_mode != GTK_SELECTION_MULTIPLE))
		{
			gtk_multirowsel_unselect_all(mrs);
		}

		if (mrs->selection_mode == GTK_SELECTION_SINGLE)
		{
			gtk_multirowsel_select(mrs, (GdkEvent *)event);
		}
		else
		{
			gdk_pointer_grab(widget->window,
				FALSE,
				GDK_POINTER_MOTION_MASK |
				GDK_BUTTON_RELEASE_MASK,
				NULL, NULL, event->time);

			mrs->selecting = TRUE;

			gdk_gc_set_clip_rectangle (mrs->sel_gc, NULL);
			gtk_multirowsel_draw_sel_box(mrs);
		}
	}
	return FALSE;
}

static gint gtk_multirowsel_button_release (
GtkWidget		*widget,
GdkEventButton		*event)
{
	GtkMultiRowSel *mrs = GTK_MULTIROWSEL(widget);

	if (mrs->selection_mode != GTK_SELECTION_MULTIPLE)
		return FALSE;

	if (event->button == 1 && mrs->selecting)
	{
		mrs->selecting = FALSE;

		gdk_pointer_ungrab(event->time);
		gdk_flush();
		gtk_multirowsel_select(mrs, (GdkEvent *)event);
		gdk_gc_set_clip_rectangle (mrs->sel_gc, NULL);
		gtk_multirowsel_draw_sel_box(mrs);
	}

	return FALSE;
}

static void gtk_multirowsel_scroll (mrs, event)
GtkMultiRowSel	*mrs;
GdkEventMotion	*event;
{
	GtkAdjustment *ha,*va;
	gint xo = 0, yo = 0;
	GtkViewport *vp;

	if (!mrs->viewport_parent)
		return;

	vp =  GTK_VIEWPORT(GTK_WIDGET(mrs)->parent);

	ha = vp->hadjustment;
	va = vp->vadjustment;

	if ((event->y - va->value) > va->page_size)
	{
		yo = MIN(event->y, va->upper) - va->value - va->page_size;
	}
	else if ((event->y > 0) && (event->y < va->value))
	{
		yo = MAX(event->y, 0) - va->value;
	}

	if ((event->x - ha->value) > ha->page_size)
	{
		xo = MIN(event->x, ha->upper) - ha->value - ha->page_size;
	}
	else if ((event->x > 0) && (event->x < ha->value))
	{
		xo = MAX(event->y, 0) - ha->value;
	}


	if (yo)
	{
		gtk_adjustment_set_value(va , va->value + yo);
		gtk_adjustment_value_changed(va);
	}

	if (xo)
	{
		gtk_adjustment_set_value(ha , ha->value + xo);
		gtk_adjustment_value_changed(ha);
	}
}

static gint gtk_multirowsel_motion (
GtkWidget		*widget, 
GdkEventMotion		*event)
{
	GtkMultiRowSel *mrs = GTK_MULTIROWSEL(widget);

	if (mrs->selection_mode != GTK_SELECTION_MULTIPLE)
		return FALSE;

	if (mrs->selecting)
	{
		if (mrs->sel_box.x2 != event->x ||
			mrs->sel_box.y2 != event->y)
		{
			gdk_gc_set_clip_rectangle (mrs->sel_gc, NULL);
			gtk_multirowsel_draw_sel_box(mrs);
			mrs->redrawing = TRUE;

			gtk_multirowsel_scroll(mrs, event);
			mrs->sel_box.x2 = event->x;
			mrs->sel_box.y2 = event->y;
			gdk_gc_set_clip_rectangle (mrs->sel_gc, NULL);
			gtk_multirowsel_draw_sel_box(mrs);
			mrs->redrawing = TRUE;
		}
	}

	return FALSE;
}

static void
gtk_multirowsel_realize (GtkWidget *widget)
{
	GdkWindowAttr attributes;
	gint attributes_mask;
	gint border_width;
	GtkMultiRowSel *multirowsel = GTK_MULTIROWSEL(widget);

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	border_width = GTK_CONTAINER (widget)->border_width;

	attributes.x = widget->allocation.x + border_width;
	attributes.y = widget->allocation.y + border_width;
	attributes.width = widget->allocation.width - 2*border_width;
	attributes.height = widget->allocation.height - 2*border_width;

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);
	attributes.event_mask = gtk_widget_get_events (widget)
		| GDK_BUTTON_MOTION_MASK
		| GDK_POINTER_MOTION_MASK
		| GDK_BUTTON_PRESS_MASK
		| GDK_BUTTON_RELEASE_MASK
		| GDK_EXPOSURE_MASK
		| GDK_ENTER_NOTIFY_MASK
		| GDK_LEAVE_NOTIFY_MASK;

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
	gdk_window_set_user_data (widget->window, widget);

	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
	if (!multirowsel->sel_gc)
	{
		GdkGCValues values;

		values.foreground = (widget->style->white.pixel==0 ?
			widget->style->black:widget->style->white);
		values.function = GDK_XOR;
		values.subwindow_mode = GDK_INCLUDE_INFERIORS;
		multirowsel->sel_gc = gdk_gc_new_with_values (widget->window,
                                          &values,
                                          GDK_GC_FOREGROUND |
                                          GDK_GC_FUNCTION |
                                          GDK_GC_SUBWINDOW);
		gdk_gc_set_line_attributes (multirowsel->sel_gc, 1, GDK_LINE_ON_OFF_DASH, 0,0);
		gdk_gc_set_line_attributes (multirowsel->sel_gc, 1, GDK_LINE_SOLID, 0,0);
 
	}

	multirowsel->viewport_parent = GTK_IS_VIEWPORT(widget->parent);

	gdk_window_show(widget->window);
}

static gint gtk_multirowsel_expose (GtkWidget *widget, GdkEventExpose *event)
{
	GtkMultiRowSel *mrs = GTK_MULTIROWSEL(widget);

	if (mrs->selecting && !mrs->redrawing)
	{
		gdk_gc_set_clip_rectangle (mrs->sel_gc, &event->area);
		gtk_multirowsel_draw_sel_box(mrs);
	}
	return TRUE;
}

#if 0
static void gtk_multirowsel_child_expose(GtkWidget *widget, GdkEventExpose *event, GtkMultiRowSel *mrs)
{
	if (mrs->selecting && mrs->redrawing)
	{
		GdkRectangle area;

		area = event->area;

		area.x += widget->allocation.x;
		area.y += widget->allocation.y;

		gdk_gc_set_clip_rectangle (mrs->sel_gc, &area);
		gtk_multirowsel_draw_sel_box(mrs);
		printf("area %d %d\n", area.x, area.y);
	}
}
#endif

static void gtk_multirowsel_add (GtkContainer *container, GtkWidget *widget)
{
#if 0
	gtk_signal_connect_after(GTK_OBJECT(widget), "expose_event",
		GTK_SIGNAL_FUNC(gtk_multirowsel_child_expose), container);
#endif
	g_return_if_fail (GTK_IS_SELBOX (widget));
	gtk_box_pack_start_defaults (GTK_BOX (container), widget);
}


GSList *gtk_multirowsel_get_selection(GtkMultiRowSel *mrs)
{
	GtkBoxChild *child;
	GList *children;
	GSList *rv = NULL;

        children = GTK_BOX(mrs)->children;
        while (children)
        {
		child = children->data;
		children = children->next;

		if (GTK_SELBOX(child->widget)->selected)
		{
			rv = g_slist_append(rv, child->widget);
			gtk_widget_ref(child->widget);
		}
	}

	return rv;
}

void gtk_multirowsel_kill_selection(GSList *selection)
{
	GSList *ptr;

	for (ptr = selection; ptr; ptr = g_slist_remove_link(ptr, ptr))
	{
		gtk_widget_unref(GTK_WIDGET(ptr->data));
	}
}

GtkWidget *gtk_multirowsel_get_selected(GtkMultiRowSel *mrs)
{
	GtkBoxChild *child;
	GList *children;

        children = GTK_BOX(mrs)->children;
        while (children)
        {
		child = children->data;
		children = children->next;

		if (GTK_SELBOX(child->widget)->selected)
		{
			return child->widget;
		}
	}

	return NULL;
}

void gtk_multirowsel_set_selected_all(GtkMultiRowSel *mrs, gboolean selected)
{
	GtkBoxChild *child;
	GList *children;

	g_return_if_fail(mrs->selection_mode == GTK_SELECTION_MULTIPLE);

        children = GTK_BOX(mrs)->children;
        while (children)
        {
		child = children->data;
		children = children->next;

		gtk_selbox_set_selected(GTK_SELBOX(child->widget), selected);
		
	}
}

void gtk_multirowsel_set_selection_mode(GtkMultiRowSel *mrs,GtkSelectionMode mode)
{
	g_return_if_fail(!(mode != GTK_SELECTION_SINGLE && mode != GTK_SELECTION_MULTIPLE));
	if (mrs->selection_mode != mode)
	{
		gtk_multirowsel_set_selected_all(mrs, FALSE);
		mrs->selection_mode = mode;
	}
}

