#ifndef __TGDB_COMMAND_H__
#define __TGDB_COMMAND_H__

/*! 
 * \file
 * tgdb_command.h
 *
 * \brief
 * This interface is intended to allow TGDB to represent a single command.
 */

#include "tgdb_types.h"
#include "tgdb_client_command.h"

/**
 * The current command type. TGDB is capable of having any commands of this
 * type in it's queue.
 *
 * I don't know what this should look like completly.
 */
enum tgdb_command_choice {

	/**
	 * A command from the front end
	 */
	TGDB_COMMAND_FRONT_END,

	/**
	 * A command from the console
	 */
	TGDB_COMMAND_CONSOLE,

	/**
	 * A command from readline
	 */
	TGDB_COMMAND_READLINE,

	/**
	 * A command from a client of TGDB
	 */
	TGDB_COMMAND_TGDB_CLIENT,

	/**
	 * ???
	 */
	TGDB_COMMAND_TGDB_CLIENT_PRIORITY
};

/**
 * TGDB can do several actions for the client. The client is allowed to ask
 * TGDB to perform such action's through this interface.
 */
enum tgdb_command_action_choice {

	/**
	 * This is relevant when there is no TGDB action.
	 */
	TGDB_COMMAND_ACTION_NONE,

	/**
	 * This tells TGDB to change the current prompt.
	 */
	TGDB_COMMAND_ACTION_CONSOLE_SET_PROMPT,

	/**
	 * This tells TGDB to redisplay the current prompt.
	 */
	TGDB_COMMAND_ACTION_CONSOLE_REDISPLAY_PROMPT
};

/* This is here to add new client_command/command faster */

/**
 * This is the main tgdb queue command.
 * It will be put into a queue and run.
 * For each TGDB command generated by a client, the client command is
 * with it.
 */
struct tgdb_queue_command {
	/**
	 * The actual command to give.
	 */
	char *tgdb_command_data;
	
	/**
	 * The type of command this one is.
	 */
	enum tgdb_command_choice command_choice;

	/**
	 * What type of command is this?
	 */
	enum tgdb_command_action_choice action_choice;

	/**
	 * The client command associated with this command.
	 */
	struct tgdb_client_command *client_command;
};

/**
 * Creates a new command and initializes it 
 *
 * \param tgdb_command_data
 * The data needed to run the command
 *
 * \param command_choice
 * The type of command to run.
 *
 * \param action_choice
 * The type of action TGDB would like to perform.
 *
 * \param tcc
 * This is the client command associated with the TGDB command.
 *
 * @return
 * Always is successfull, will call exit on failed malloc
 */
struct tgdb_queue_command *tgdb_command_create (    
		const char *tgdb_command_data,
        enum tgdb_command_choice command_choice, 
        enum tgdb_command_action_choice action_choice,
		struct tgdb_client_command *tcc );
        
/** 
 * This will free a TGDB queue command.
 * These are the commands given by TGDB to the debugger.
 * This is a function for debugging. 
 *
 * \param item
 * The command to free
 */
void tgdb_command_destroy ( void *item);

/**
 * This will print a TGDB queue command to stderr.
 * These are the commands given by TGDB to the debugger.
 * This is a function for debugging. 
 *
 * \param item
 * The command to print
 */
void tgdb_command_print ( void *item );

#endif
