/*
 *  CaSU - communications & status utilities.
 *  Copyright (C) 1992, 1993, 1994 Luke Mewburn <lm@rmit.edu.au>
 *	incorporating:
 *	   flon - lists your friends who are logged on.
 *	   to - send a short message to a friend
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "casu.h"

/*
 * 	errmesg
 * print a message to stderr, in a consistant format.
 * output is of the form:
 *	progname: 'm2' - m1.\n		# m2 is ! NULL
 *	progname: m1.\n			# m2 is NULL
 */
void
errmesg(m1, m2)
    char *m1, *m2;
{
    fprintf(stderr, "%s: ", progname);
    if (m2)
	fprintf(stderr, "'%s' - ", m2);
    fprintf(stderr, "%s.\n", m1);
} /* errmesg */


/*
 * 	errexit
 *
 * use errmesg to print a message and exit the program.
 */
void
errexit(m1, m2)
    char *m1, *m2;
{
    errmesg(m1, m2);
    exit(1);
} /* errexit */


/*
 * 	strnc0py
 *
 * as strncpy, but always put in the \0 (assume array is siz+1 long)
 */
char *
strnc0py(s1, s2, n)
    char *s1, *s2;
    size_t n;
{
    s1[n] = '\0';
    return strncpy(s1, s2, n);
} /* strnc0py */


/*
 *	get_username
 *
 * find out the username of the user...
 */
char *
get_username()
{
    struct passwd	*pwbufr;
    char		*myname;
    int			my_uid;
    static char		namebuf[UT_NAMESIZE+1];

    myname=getlogin();
    if (myname == NULL)		/* attempt to get name another way */
    {
	my_uid = (int) getuid();
	if (pw_list != NULL)
	{
	    int lp;
	    for (lp = 0; lp < pw_count; lp++)
		if (pw_list[lp].uid == my_uid)
		    break;
	    if (lp != pw_count)
		myname = pw_list[lp].username;
	}
	else	/* don't use inbuilt passwd file - USE_GETPWENT does this too */
	{
	    if ((pwbufr = getpwuid(my_uid)) != NULL)
		myname = pwbufr->pw_name;
	}
    }
    if (myname == NULL)
	myname = "";
    strnc0py(namebuf, myname, UT_NAMESIZE);
    return (namebuf);
} /* get_username */


/*
 *	convert_realname
 *
 * converts the GCOS field at tfb to a full name, replacing
 * occurances of `&' with name. If uid matches the last request,
 * it will return the last converted entry (cache!). It returns
 * a pointer to an internal static buffer
 */
char *
convert_realname(tfb, name, uid)
    char *tfb, *name;
    int  uid;
{
    static char	realnam[BUFSIZ + 1];
    static int	last_uid = -1;

    char	*namep, *final;
    int		len, namelen;

    if (last_uid == uid)
	return realnam;
    final = strchr(tfb, ',');
    if (final != NULL)
	*final = '\0';
    if (strchr(tfb, '&') == NULL)
	return tfb;
    final = tfb;
    namelen = strlen(name) - 1;
    for (namep=tfb, len=1; *namep && len < BUFSIZ ; namep++, len++)
    {
	if (*namep=='&')
	{
	    if (len + namelen >= BUFSIZ)
		break;		/* safety net for copy below */
	    len += namelen;
	}
    }
    final = namep;
    for (namep=tfb, len=0; namep < final ; namep++, len++)
    {
	if (*namep=='&')
	{
	    strcpy(&realnam[len], name);
	    if (islower(realnam[len]))
		realnam[len] = toupper(realnam[len]);
	    len += namelen;
	}
	else
	    realnam[len] = *namep;
    }
    realnam[len]='\0';
    last_uid = uid;
    return realnam;
} /* convert_realname */
