/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_entities.pike,v 1.86.2.3 2004/03/23 13:50:27 bertrand Exp $
 */
//
//! module: CAMAS: CAMAS Entities
//!  This module handle CAMAS Entities used in layouts.
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PARSER
//! cvs_version: $Id: camas_entities.pike,v 1.86.2.3 2004/03/23 13:50:27 bertrand Exp $
//
#include <module.h>
inherit "module";
inherit "caudiumlib";
#include <camas/globals.h>      // MB_DISPLAYNAME_IDX
#include <camas/screens.h>	// For screennames
#include <camas/msg.h>		// MSG() Language macros
#include <camas/utf.h>		// Deal with internal charset

constant cvs_version   = "$Id: camas_entities.pike,v 1.86.2.3 2004/03/23 13:50:27 bertrand Exp $";
constant module_type   = MODULE_PARSER;
constant module_name   = "CAMAS: CAMAS Entities";
constant module_doc    = "This module handle CAMAS Entities used in layouts."
  			 "<br /><b>This module is automatically "
			 "selected if you select \"CAMAS: Main module\".</b>";
constant module_unique = 1;
constant thread_safe   = 1;

// Threads safing..
#if constant(thread_create)
object global_lock = Thread.Mutex();
#endif

void create()
{
#ifdef CAMAS_DEBUG
  defvar("debug",0,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
#endif
  defvar("showpass",0,"Allow &amp;camasmsg.passwordcontent; entity", TYPE_FLAG,
         "Setting this to YES will allow the entity &amp;camasmsg."
         "passwordcontent; to have password inside. <b>Warning</b> : this "
         "be an open hole in your security policy if you set this to Yes.");
}

void start(int cnt, object conf)
{
  module_dependencies(conf,
    ({ 
      "camas_tags",
    }));
}

string status()
{
  string out = "The following entities are available:<br/><table border=\"1\">";
  array(object) scopes = query_scopes();
  foreach(scopes, object scope)
  {
    string scopename = scope->name;
    array(string) scopefuns = sort(indices(scope) - ({ "get" }));
    foreach(scopefuns, string fun)
    {
      out += "<tr><td>&amp;"+scopename+"."+fun+";</td></tr>";
    }
  }
  out += "</table>";
  return out;
}

//! entity_scope: camas
//!  This scope contains CAMAS data
class CamasScope
{
  inherit "scope";
  string name = "camas";
  mixed ret = -1;
  

  array(string)|string get(string entity, object id)
  {
    CDEBUG("&camas."+entity+"; called.");

    if(!CSESSION)
      return "&amp;camas.*; entities works only on CAMAS pages";

    function fun = this_object()[entity];
    if(fun && functionp(fun))
      fun(id);

    if(ret == -1)
    {
      CDEBUG("Invalid entity &amp;"+name+"."+entity+";");
      return "&"+name+"."+entity+";";
    }
    
		if(ret)
      return ({ ret });
  }
	
	void currentscreen(object id)
	{
		//! entity: currentscreen
		//!  The name of the current screen
		if(id->misc->camas->status)
			ret = screennames[id->misc->camas->status][0];
		else
			ret = "";
	}

  void fullversion(object id) {
    //! entity: fullversion
    //!  The full version of CAMAS, eg 1.2.11-DEVEL
    object camas_main = id->conf->get_provider("camas_main");
    if(!objectp(camas_main))
      ret = "CAMAS main module not present";
    ret = camas_main->camas_version;
  }

  void version(object id) {
    //! entity: version
    //!  The version of CAMAS, eg 1.2
    object camas_main = id->conf->get_provider("camas_main");
    if(!objectp(camas_main))
      ret = "CAMAS main module not present";
    ret = camas_main->__camas_version__;
  }

  void build(object id) {
    //! entity: build
    //!  The build of CAMAS, eg 11
    object camas_main = id->conf->get_provider("camas_main");
    if(!objectp(camas_main))
      ret = "CAMAS main module not present";
    ret = camas_main->__camas_build__;
  }

  void state(object id) {
    //! entity: state
    //!  The state of CAMAS, eg DEVEL
    object camas_main = id->conf->get_provider("camas_main");
    if(!objectp(camas_main))
      ret = "CAMAS main module not present";
    ret = camas_main->__camas_state__;
  }
	
}

//! entity_scope: camasmsg
//!  This scope contains translated message from CAMAS Language module
//!  used inside camas layouts 
class CamasMsgScope
{
  inherit "scope";
  string name = "camasmsg";
  object auth_module;
  object layout_mngr;
  object features;
  mixed ret = -1;

  array(string)|string get(string entity, object id)
  {
    CDEBUG("&camasmsg."+entity+"; called.");

    if(!CSESSION)
      return "&amp;camasmsg.*; entities works only on CAMAS pages";

    auth_module = id->conf->get_provider("camas_auth"); // Auth module
    layout_mngr = id->conf->get_provider("camas_layout_manager"); // Layouts
    features = id->conf->get_provider("camas_features"); // CAMAS features

    function fun = this_object()[entity];
    if(fun && functionp(fun))
      fun(id);

    if(ret == -1)
    {
      CDEBUG("Invalid entity &amp;"+name+"."+entity+";");
      return "&"+name+"."+entity+";";
    }
    if(ret) return ({ ret });
    return 0;
  }

  // *** All Screens ****
  void action(object id) {
    //! entity: action
    //!  The last user's action
    ret = CSESSION->lastaction;
  }

  void loginname(object id) {
    //! entity: loginname
    //!  The login name of the user
    ret = CSESSION->login;
  }

  void passwordcontent(object id) {
    //! entity: passwordcontent
    //!  This is the real password of the user, be sure to know what you
    //!  are doing when using this
    if(QUERY(showpass)) 
      ret = CSESSION->passwd;
    else
      ret = "*****";
  }

  void charset(object id) {
    //! entity: charset
    //!  The charset of the current user
    ret = CSESSION->charset;
  }

  void username(object id) {
    //! entity: username
    //!  The real of the user
    ret = CSESSION->name;
  }

  void language(object id) {
    //! entity: language
    //!  The language of the user
    ret = CSESSION->language;
  }

  void organization(object id) {
    //! entity: organization
    //!  The organization of the user
    ret = CSESSION->organization;
  }

  void layout_www(object id) {
    //! entity: layout_www
    //!  The base path for retrieving files from {layout_fs_path}/_www/
    //!  Used for getting files from the layout via http
    ret = layout_mngr->query_location() + HTTP_ENCODE_URL(CAMAS.Tools.current_layout(id))+"/";
  }

  void layout_name(object id) {
    //! entity: layout_name
    //!  The current layout name
    ret = CAMAS.Tools.current_layout(id);
  }

  void next_url(object id) {
    //! entity: next_url
    //!  The CAMAS nexturl ie: http://server.domain.tld/mountpoint
    //! note: going there may log you out
    ret = CSESSION->nexturl;
  }

  void status(object id) {
    //! entity: status
    //!  The CAMAS status, mainly for debug pruposes
    if(id->misc->camas->status)
      ret = (string)id->misc->camas->status;
    else
      ret = (string)CSESSION->status;
  }
    
    // *** LOGIN Screen ***
  void login(object id) {
    //! entity: login
    //!  <b>Login Page</b> : Returns the username/login string from
    //!  language module aka message named "M_LOGIN".
    ret = MSG(M_LOGIN);
  }
  void password(object id) {
    //! entity: password
    //!  <b>Login Page</b> : Returns the password string from language
    //!  module aka message named "M_PASSWORD".
    ret = MSG(M_PASSWORD);
  }
  void emaildomain(object id) {
    //! entity: emaildomain
    //!  <b>Login Page</b> : Returns the domain name if set from the camas
    //!  auth module when email login is used and set there.
    if(objectp(auth_module) && auth_module->getemaillogin()) {
      if(stringp(auth_module->getdnsname()))
        ret = "@"+auth_module->getdnsname();
    }
    else ret = "";
  }
  void imapserver(object id) {
    //! entity: imapserver
    //! <b>Login Page</b> : Returns the imap server from language
    //! module aka message named "M_IMAPSERVER".
    ret = MSG(M_IMAPSERVER);
  }
  void port(object id) {
    //! entity: port
    //! <b>Login Page</b> : Returns the port from language
    //! module aka message named "M_PORT".
    ret = MSG(M_PORT);
  }
  void userinterface(object id) {
    //! entity: userinterface
    //!  <b>Login Page</b> : Returns the "chose layout" message from the
    //!  language module when more than one layout is configured in the
    //!  camas layout manager.<br />
    //!  The message named in language module is "M_USERINTERFACE".
    if(objectp(layout_mngr) && (sizeof(layout_mngr->list_layouts())>1))
      ret = MSG(M_USERINTERFACE);
    else ret = "";
  }
  void nologinerrors(object id) {
    //! entity: loginerrors
    //!  <b>Login Page</b> :Return an error message when it is needed. <br/>
    //!  Messages taken from language module can be "M_NOLOGIN", "M_LOGINERROR"
    //!  or "M_IMAPERROR" depending of what kind of error happened.
    if(CSESSION->status == LOGINFAILED)
      ret = MSGA(M_NOLOGIN, ({ (CSESSION->loginerror) ? CSESSION->loginerror : MSG(M_LOGINERROR) }));
    else
    {
      if (CSESSION->status == LOGINFAILEDIMAP)
        ret = MSG(M_IMAPERROR);
      else ret = "";
    }
  }

    // *** ADDRESSBOOK, EDITADDRESS, IMPORTADDRESS screens ***
  void addressbook(object id) {
    //! entity: addressbook
    //!   The "addressbook" word
    ret = MSG(M_ADDRESSBOOKTITLE);
  }

  void noaddress(object id) {
    //! entity: noadress
    //!  Message to display when there is no address in addressbook
    ret = MSG(M_NOADDRESSES);
  }

  void addrbook_name(object id) {
    if(CSESSION->recipientfield != "toccbcc")
      ret = MSG(M_INDEXNAMERECIPIENT);
    else
      ret = MSG(M_INDEXNAMEEDIT); 
  }
  
  void mail(object id) {
    ret = MSG(M_ADDRESS);
  }

  void surname(object id) {
    ret = MSG(M_SURNAME);
  }

  void nickname(object id) {
    ret = MSG(M_NICKNAME);
  }

  void homephone(object id) {
    ret = MSG(M_HOMEPHONE);
  }

  void workphone(object id) {
    ret = MSG(M_WORKPHONE);
  }

  void title(object id) {
    ret = MSG(M_TITLE);
  }

  void company(object id) {
    ret = MSG(M_COMPANY);
  }

  void department(object id) {
    ret = MSG(M_DEPARTMENT);
  }

  void homepostaladdress(object id) {
    ret = MSG(M_HOMEPOSTALADDRESS);
  }

  void workpostaladdress(object id) {
    ret = MSG(M_WORKPOSTALADDRESS);
  }

  void workcity(object id) {
    ret = MSG(M_WORKCITY);
  }

  void workstate(object id) {
    ret = MSG(M_WORKSTATE);
  }

  void workcountry(object id) {
    ret = MSG(M_WORKCOUNTRY);
  }

  void pager(object id) {
    ret = MSG(M_PAGER);
  }

  void mobile(object id) {
    ret = MSG(M_MOBILE);
  }

  void uri(object id) {
    ret = MSG(M_URI);
  }

  void description(object id) {
    ret = MSG(M_DESCRIPTION);
  }

  void postalcode(object id) {
    ret = MSG(M_POSTALCODE);
  }

    // *** ATTACHMENTS screen ***
  void attachmentsheader(object id) {
    //! entity: attachmentsheader
    //!  Title for attachments screen
    ret = MSG(M_ATTACHMENTSHEADER);
  }

  void addfileattachment(object id) {
    ret = MSG(M_ADDFILEASATTACHMENT);
  }

  void uploadattachment(object id) {
    ret = MSG(M_UPLOADATTACHMENT);
  }

  void noattachfiles(object id) {
    ret = MSG(M_NOATTACHFILES);
  }

    // *** COMPOSE screen ***
  void originalmessage(object id) {
    if(CSESSION->originalmessage)
      ret = CSESSION->originalmessage;
    else
      ret = "<!-- CESSION-&gt;originalmessage does not exist -->";
  }
  void compose(object id) {
    //! entity: compose
    //! <b>Compose Page</b>: The compose name
    ret = MSG(M_NEWMAIL);
  }

  void fromfield(object id) {
    //! entity: fromfield
    //!  <b>Compose Page</b>: Return the From: field message from the
    //!  language module
    ret = MSG(M_FROM);
  }

  void tofield(object id) {
    //! entity: tofield
    //!  <b>Compose Page</b>: Return the To: field message from the
    //!  language module
    ret = MSG(M_TO);
  }

  void ccfield(object id) {
    //! entity: ccfield
    //!  <b>Compose Page</b>: Return the Cc: field message from the
    //!  language module
    // don not output CC translation if there is no CC in the mail
    ret = MSG(M_CC);
  }

  void bccfield(object id) {
    //! entity: bccfield
    //!  <b>Compose Page</b>: Return the Bcc: field message from the
    //!  language module
    ret = MSG(M_BCC);
  }

  void subjectfield(object id) {
    //! entity: subjectfield
    //!  <b>Compose Page</b>: Return the Subject: field message from the
    //!  language module
    ret = MSG(M_SUBJECT);
  }

  void mdn(object id) {
    //! entity: mdn
    //!  <b>Compose Page</b>: Return the mdn message from the
    //!  language module
    ret = MSG(M_MDN);
  }

  void sendblind(object id) {
    //! entity: sendblind
    //!  <b>Compose Page</b>: return the send blind (recipients don't see each other when in To:)
    //!  from the language module
    if(features->QUERY(userblindsend) && (int)CSESSION->blindsend)
      ret = MSG(M_SENDBLIND);
    else
      ret = "<!-- You must enable this feature in your setup and in CAMAS: Features to get it -->\n";
  }

  void bcc2admin(object id) {
    //! entity: bcc2admin
    //!  <b>Compose Page</b>: Return the bcc2admin message from the
    //!  language module
    if(sizeof(CSESSION->bcc2admin)){
      ret = MSG(M_CHECKBCC);
    }
    else
    {
      ret = "<!-- Default message to admin not set-->";
    }
  }

  void attachments(object id) {
    //! entity: attachments
    //!  Message for adding attachments
    if(features->QUERY(attachments))
    {
      ret = MSG(M_ATTACHMENTS);
    }
    else
    {
      ret = "<!-- Attachments sending is not allowed -->";
      CDEBUG("Attachments sending is not allowed");
    }
  }

  void fromaddr(object id) {
    //! entity: fromaddr
    //!  From real address (without first and last name), use only in COMPOSE screen
    //! screen : compose
    string to = CSESSION->to;
    if(to)
    {
      sscanf(to, "%*s<%s>", to);
      ret = CAMAS.Tools.fix_header(to);
    }
    else
      ret = "";
  }

  void fromfull(object id) {
    //! entity: fromfull
    //!  From full name and address, use only in COMPOSE screen
    //! screen : compose
    ret = HTML_ENCODE_STRING (CSESSION->name+" <"+CSESSION->address+">");
  }
    
    // *** DIALOGBOX screen ***
  void dialogmessage(object id) {
    //! entity: dialogmessage
    //!  Message for dialogbox screen
    //! screen : dialogbox
    ret = CAMAS.Tools.fixstring (CSESSION->dialogtext);
  }

    // *** EDITADDRESS screen ***
  void editname(object id) {
    //! entity: editname
    //!  Message
    ret = MSG(M_INDEXNAME);
  }

  void editaddress(object id) {
    //! entity: address
    ret = MSG(M_ADDRESSFORMAT);
  }

    // *** EDITADDRESSFILTER screen ***
  void filter(object id) {
    filters(id);
  }
  void filters(object id) {
    //! entity: filters
    //!  The "Mail filter" words
    ret = MSG(M_FILTERBOOKTITLE);
  }

  void filterfield(object id) {
    //! entity: filterfield
    //!  Title for the mail header on which apply the filter
    ret = MSG(M_INDEXFIELDFILTER);
  }
    
  void filterexpression(object id) {
    //! entity: filterexpression
    //!  Title for the text to filter on
    ret = MSG(M_INDEXNAMEFILTER);
  }
    
  void filterfolder(object id) {
    //! entity: filterfolder
    //!  Title for the folder to put the mails that match the filter expression
    ret = MSG(M_FILTERFOLDER);
  }
    
    // *** IMPORTADDRESS screen ***
  void importaddressbooktitle(object id) {
    //! entity: importaddressbooktitle
    //!  Title for the addressbook screen
    ret = MSG(M_IMPORTADDRESSBOOKTITLE);
  }

  void uploadaddressbook(object id) {
    //! entity: uploadaddressbook
    //!  The upload adddressbook help string
    ret = MSG(M_UPLOADADDRESSBOOK);
  }

  void pinelocation(object id) {
    //! entity: pinelocation
    //!  Help string for pine location
    ret = MSG(M_PINELOCATION);
  }

  void destinationimport(object id) {
    //! entity: destinationimport
    //!  String select the destination to import to
    ret = MSG(M_DESTINATIONIMPORT);
  }

  // **** IMPORTADDRESS2 screen ***
  void selectmatchingfields(object id) {
    //! entity: selectmatchingfields
    //!  String select two matching fields
    ret = MSG(M_SELECTMATCHINGFIELDS);
  }

  void importedfields(object id) {
    //! entity: importedfields
    //!  String imported fields
    ret = MSG(M_IMPORTEDFIELDS);
  }

  void availablefields(object id) {
    //! entity: availablefields
    //!  String available fields
    ret = MSG(M_AVAILABLEFIELDS);
  }

    // *** MAILINDEX screen ***
  void index(object id) {
    //! entity: index
    //!  The "current mailbox" words
    ret = MSG(M_CURRMAILBOX);
  }

  void inbox(object id) {
    //! entity: inbox
    //!  The 'Inbox' string
    ret = MSG(M_INBOX);
  }

  void current_mailbox(object id) {
    //! entity: current_mailbox
    //!  The name of current mailbox
    ret = CAMAS.FolderTools.translate_frommbox (id, CSESSION->mailbox);
  }

  void current_mailboxroot(object id) {
    //! entity: the current mailbox root
    //!  The current mailbox root
    ret = CAMAS.FolderTools.getmailboxroot(CSESSION->mailbox, CSESSION);
  }

  void checknewmail(object id) {
    //! entity: checknewmail
    //!  The "check new mail" words
    ret = MSG(M_CHECKNEWMAIL);
  }

  void selectall(object id) {
    //! entity: selectall
    //! <b>Mailindex page</b>: Returns the markall message from language
    //! module aka message named "M_MARKALL";
    //! Requires a JavaScript-enabled browser
    if(CSESSION->searchstring && sizeof(CSESSION->searchstring) && CSESSION->nothingfound)
    {
      CDEBUG("&camasmsg.selectall; : we are in the case of a search giving no result");
    }
    else
    {
      if(sizeof(CSESSION->mails) && id->supports->javascript)
      {
        ret = MSG(M_MARKALL);
      }
      else
      {
        ret = "<!-- &camasmsg.selectall; : either there is no mail, either browser do not support javascript -->";
        CDEBUG("&camasmsg.selectall; : either there is no mail, either browser do not support javascript");
      }
    }
  }
    
  void searchsentence(object id) {
    //! entity: seachsentence
    //!  Message introducing search result
    //!  This message won't be display if trying to find the history of someone
    if( ((CSESSION->searchstring && strlen (CSESSION->searchstring))
       || CSESSION->stickyfilter) && !(CSESSION->historysearch))
    {
      ret = MSG(M_SEARCHSHOWINGMAIL);
    }
    else
    {
      ret = "";
      CDEBUG("&camasmsg.searchsentence; : there is no search string or coming from history, thus "
"there can be no result");
    }
  }
    
  void searchstrings(object id) {
    //! entity: searchstrings
    //!  Message with the string which was searched and where it was
    if((CSESSION->searchstring && strlen (CSESSION->searchstring)) || CSESSION->stickyfilter)
    {
      ret = CAMAS.Tools.fixstring(CSESSION->searchstring);
    }
    else
    {
      CDEBUG("&camasmsg.searchstrings; : there is no search string, thus there can be no result");
      ret = "<!-- &camasmsg.searchsentence; : there is no search string or coming from history, thus "
        "there can be no result -->";
    }
  }
    
  void nosearchresult(object id) {
    //! entity: nosearchresult
    //!  Message to display is the search gave no result
    if(CSESSION->searchstring && strlen (CSESSION->searchstring) && CSESSION->nothingfound)
    {
      ret = MSG(M_NOSEARCHRESULTS);
    }
    else
    {
      CDEBUG("&camasmsg.nosearchresult; : either there is no search string, either there is a result");
ret = "";
    }
  }
 
  void numfield(object id) {
    //! entity : numfield
    //!  Title for the number column
    //! screen : MAILINDEX
    ret = MSG(M_NUMBER);
  }

  void receivedfield(object id) {
    //! entity: receivedfield
    //!  Title for the received column
    //! screen: MAILINDEX
    ret = MSG(M_RECEIVED);
  }
    
  void markfield(object id) {
    //! entity: markfield
    //!  Title for the mark column
    //! screen : MAILINDEX
    ret = MSG(M_MARKFLAG);
  }

  void newfield(object id) {
    //! entity: newfield
    //!  Title for the new column
    //! screen : MAILINDEX
    ret = MSG(M_NEWFLAG);
  }
    
  void answeredfield(object id) {
    //! entity: answeredfield
    //!  Title for the answered column
    //! screen : MAILINDEX
    ret = MSG(M_ANSWEREDFLAG);
  }

  void sizefield(object id) {
    //! entity: sizefield
    //!  Title for the size column
    //! screen : MAILINDEX
    ret = MSG(M_SIZE);
  }

  void threadfield(object id) {
    //! entity: threadfield
    //!  Title for the thread column
    ret = MSG(M_THREADSHOW);
  }

  void historyfield(object id) {
    //! entity: historyfield
    //!  Title for the history column
    ret = MSG(M_HISTFLAG);
  }
  
  void historyof(object id) {
    //! entity: historyof
    //!  Message that will be displayed if the user search for history of a mail
    if( CSESSION->searchstring && strlen (CSESSION->searchstring) && CSESSION->historysearch)
      ret = MSG(M_HISTFLAG) + " " + MSG(M_OF);
    else
    {
      CDEBUG("&camasmsg.history; : either there is no search string, either you are not called from history");
      ret = "";
    }
  }
    
  void nomail(object id) {
    //! entity: nomail
    //!  Message to display when there is no mail
    ret = MSG (M_NOMAILS);
  }

  void mailnotifywindow(object id) {
    //! entity: mailnotifywindow
    //!  
    ret = MSG(M_MAILNOTIFYWINDOW)+"...";
  }

  void logout(object id) {
    //! entity: logout
    //!  "logout" message
    ret = MSG (M_LOGOUT);
  }

    // *** FILTERBOOK screen ***
  void nofilters(object id) {
    nofilter(id);
  }
  void nofilter(object id) {
    //! entity: nofilter
    //!  Message to be displayed when no filters have been set
    ret = MSG(M_NOADDRESSESFILTER);
  }

    // *** LOGOUT screen ***
  void logoutmsg(object id) {
    //! entity: logoutmsg
    //!  Translation of "You have been logged out"
    ret = MSG(M_LOGOUTMSG);
  }

  void logoutclose(object id) {
    //! entity: logoutclose
    //!  Translation of "Close this browser window to make sure no one can read your mail"
    ret = MSG(M_LOGOUTCLOSE);
  }

  void logoutreload(object id) {
    //! entity: logoutreload
    //!  Translation of "Reload this page to login"
    ret = MSG(M_LOGOUTRELOAD);
  }

    // *** READMAIL screen ***
  void datefield(object id) {
    //! entity: datefield
    //!  label for date field
    ret = MSG(M_DATE);
  }

  void to(object id) {
    //! entity: to
    //!  data for to field
    if(CSESSION->cmail && CSESSION->cmail->headers)
      ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers->to);
    else
      ret = "";
  }

  void cc(object id) {
    //! entity: cc
    //!  data for the cc field
    ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers->cc);
  }

  void from(object id) {
    //! entity: from
    //!  data for from field
    if(CSESSION->cmail && CSESSION->cmail->headers)
      ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers->from);
    else
      ret = "";
  }

  void subject(object id) {
    //! entity: subject
    //!  data for subject field
    if(CSESSION->cmail && CSESSION->cmail->headers)
      ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers->subject);
    else
      ret = "";
  }

  void date(object id) {
    //! entity: date
    //!  data for date field
    if(CSESSION->cmail && CSESSION->cmail->headers)
      ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers->date);
    else
      ret = "";
  }

  void messageid(object id) {
    //! entity: messageid
    //!  the message id of the current email
    if(CSESSION->cmail && CSESSION->cmail->headers)
      ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers["message-id"]);
    else
      ret = "";
  }
   
  void useragent(object id) {
    //! entity: useragent
    //!  the user-agent header in the current email
    if(CSESSION->cmail && CSESSION->cmail->headers)
      ret = CAMAS.Tools.fix_header(CSESSION->cmail->headers["user-agent"]);
    else
      ret = "";
  }
    
    // *** SEARCHMAIL screen ***
  void searchmailtitle(object id) {
    //! entity: searchmailtitle
    //!  Message for searching mails
    ret = MSG(M_SEARCHMAILTITLE);
  }

  void nosearchremember(object id) {
    //! entity: nosearchremember
    //!  Message for not remembering previous search
    if (CSESSION->searchstring && sizeof(CSESSION->searchstring) > 0)
    {
      ret = MSG(M_NOSEARCHREMEMBER);
    }
    else
    {
      ret = "";
    }
  }

  void searchallmbox(object id) {
    //! entity: searchallmbox
    //!  Message search in all mailboxes (only available on headers and when cache is feeded)
    if(CSESSION->cachefeeded)
      ret = MSG(M_SEARCHALLMBOX);
    else
      ret = "";
  }

  void stickyfilter(object id) {
    //! entity: stickyfilter
    //!  Message for applying search to all mailboxes/folders when switching to them
    ret = MSG(M_STICKYFILTER);
  }

  void searchcachenotfeeded(object id) {
    //! entity: cachenotfeeded
    //!  Message for warning the user he can't search in multiple mailboxes and fast search
    //!  is not available yet (the cache is not feeded yet)
    if(!CSESSION->cachefeeded)
      ret = MSG(M_SEARCHCACHENOTFEEDED);
    else
      ret = "";
  }
    
    // *** SEARCHLDAP screen ***
  void searchldaptitle(object id) {
    //! entity: searchldaptitle
    //!  Message for searching LDAP addressbook
    ret = MSG(M_NAMECONT);
  }

    // *** SETUP ***
  void setup(object id) {
    //! entity: setup
    //!  Localisation of the "setup" word
    ret = MSG(M_PREFS);
  }

    // *** SPELLCHECK screen ***
  void nospellingerrors(object id) {
    //! entity: nospellingerrors
    //!  Message when there is no errors in the text to spellcheck
    ret = MSG(M_NOSPELLINGERRORS);
  }

  void spellsuggestions(object id) {
    //! entity: spellsuggestions
    //!  Message for announcing spelling suggestions
    ret = MSG(M_SPELLSUGGESTIONS);
  }

  void manualcorrection(object id) {
    //! entity: manualcorrection
    //!  Message for manual spell correction
    ret = MSG(M_MANUALCORRECTION);
  }

  // *** Folderlist screen ***
  //! entity: folderlist
  //!  The "Mailboxes" word
  void folderlist(object id) {
    ret = MSG(M_MAILBOXES);
  }
  
  // *** Files screen ***
  //! entity: files
  //!  The "files" word
  void files(object id) {
    ret = MSG(M_FILES);
  }
}

array (object) query_scopes()
{
  return  ({
             CamasScope(),
             CamasMsgScope(),
           });
}


/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: showpass
//! Setting this to YES will allow the entity &amp;camasmsg.passwordcontent; to have password inside. <b>Warning</b> : this be an open hole in your security policy if you set this to Yes.
//!  type: TYPE_FLAG
//!  name: Allow &amp;camasmsg.passwordcontent; entity
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

