
/*
 *  Diverse SLab audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#ifdef linux
/*
 * This may want to be /usr/lib/oss/include/sys/soundcard.h if someone has
 * purchased the 4Front drivers?
 */
#include <sys/soundcard.h>
audio_buf_info bufferInfo;
#endif

#include <unistd.h>
#include <fcntl.h>

/*
 * Audio device structure format definitions.
 */
#include "slabrevisions.h"
#include "slabaudiodev.h"

int *
getAudioOSpace2(audioDev, devID)
duplexDev *audioDev;
{
#ifdef linux
#ifdef ALSA_SUPPORT
	if ((audioDev->siflags & AUDIO_ALSA) == 0)
#endif
	{
		ioctl(audioDev->fd, SNDCTL_DSP_GETOSPACE, &bufferInfo);
		return((int *) &bufferInfo);
	}
#endif
	return((int *) 0);
}

int
ossAudioInit(audioDev, devID, fragSize)
duplexDev *audioDev;
int devID;
{
#ifdef linux
	int trackfile, i, j, results, data = 0, mode, index;

#ifdef SUBFRAGMENT
#ifdef SLAB_SUBF_IOCTL
	if (((audioDev->cflags & SLAB_SUBF_IOCTL) == 0) &&
		(audioDev->cflags & SLAB_SUBFRAGMENT))

	setscheduler(audioDev);
#else
	if (audioDev->cflags & SLAB_SUBFRAGMENT)
#endif
	{
		newInitAudioDevice2(audioDev, devID, fragSize);
		return(0);
	}
#endif

/* printf("old init\n"); */
	/*
	 * Set the device to DUPLEX. This will fail on a non-duplex device, but
	 * we test for this case below.
	 */
	if (audioDev->flags == SLAB_FULL_DUPLEX)
	{
		/*
		 * I don't want to use dup anymore, just copy the fd since we might
		 * use it intermittently. This should be put under flag control:
		 * if duplicate flag is set, use dup, if not, just copy
		audioDev->fd2 =
			fcntl(audioDev->fd, F_DUPFD, 
			audioDev->fd);
		audioDev->fd2 = audioDev->fd;
		 */
		if ((audioDev->cflags & SLAB_FDUP) == 0)
			audioDev->fd2 = audioDev->fd;
		else
			audioDev->fd2 = fcntl(audioDev->fd, F_DUPFD, audioDev->fd);
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("ioctl(%i, SNDCTL_DSP_SETDUPLEX, 0)\n", audioDev->fd);
#endif
		if (ioctl(audioDev->fd, SNDCTL_DSP_SETDUPLEX, 0) < 0)
			printf("Failed to set duplex on device %s\n", audioDev->devName);
		if (audioDev->fd2 < 0)
			printf("Failed to duplicate input channel on %s\n",
				audioDev->devName);
	} else {
		audioDev->fd2 = audioDev->fd;
	}
#ifndef ALSA_SUPPORT
	checkAudioCaps2(audioDev, devID, audioDev->fd);
#endif
	/*
	 * 2 Fragments. The fragment size is the power of 2 needed to give the
	 * adiodSegmentSize, or log2(adiodSegmentSize).
	 *
	 * If we are doing interpolation the fragment sizes need to be double for
	 * the extra interpolated data. This is not the case as of rel 1.0.....
	 */
	switch(audioDev->OSegmentSize) {
		case 4096:
			data = 0x0002000b;
			break;
		case 8192:
			data = 0x0002000c;
			break;
		case 16384:
			data = 0x0002000d;
			break;
		/*
		 * This is 8192 blockSampleSize, where internally we deal with ints or
		 * floats, but the IO buffer is in shorts.
		 */
		case 32768:
		default:
			data = 0x0002000e;
			break;
		case 65536:
			data = 0x0002000f;
			break;
	}
	/*
	 * MARK HACK This may be an issue?
	 */
	audioDev->OWorkSpaceSize = audioDev->OSegmentSize;
	/*
	 * Set up the partial write operations.
	 * The newInit routines will accept the device default fragSize.
	 */
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SNDCTL_DSP_SETFRAGMENT, &%08x)\n",
			audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_SETFRAGMENT, &data)) == 0)
	{
#ifdef DEBUG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("set audio fragment: %x\n", data);
		else
			printf("set audio frag failed: %x, result = %i\n", data, results);
#else
		;
#endif
	}

#ifdef SUBFRAGMENT
	if (audioDev->cflags & SLAB_SUBFRAGMENT)
	{
		if (audioDev->fragBuf != (char *) NULL)
			free(audioDev->fragBuf);
		audioDev->fragBuf = (char *) 0;
		/*
		 * If we are actually a subfragmented device then we need to add
		 * the fragSize and fragBuf.
		 */
		audioDev->fragSize = audioDev->OSegmentSize;

		audioDev->fragBuf = (char *) malloc(audioDev->fragSize);
		bzero(audioDev->fragBuf, audioDev->fragSize);
	}
#endif

	/*
	 * Reset audio device. The docs say we must reset before changing the
	 * sampling parameters. No comments are made regarding the duplex config
	 * after a reset. For now, will take this, but may need to reset duplex
	 * capability again AFTER the reset.
	if ((results = ioctl(audioDev->fd, SOUND_PCM_RESET, &data)) == 0)
#ifdef DEBUG
		printf("Reset audio device: %i\n", data);
#else
	;
#endif
	else
		printf("sound device reset failed: %i\n", results);
	 */

	/*
	 * Check capabilities.
	 */
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SNDCTL_DSP_GETCAPS, &%i)\n", audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_GETCAPS, &data)) == 0)
	{
		audioDev->genCaps = data;
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			if ((audioDev->genCaps & SNDCTL_DSP_SETTRIGGER) != 0)
				printf("Device %s does support SNDCTL_SET_TRIGGER\n",
					audioDev->devName);
			else
				printf("Device %s does NOT support SNDCTL_SET_TRIGGER\n",
					audioDev->devName);
		}
#endif
		/*
		 * Take this out, we only need to check the options here.
		 */
		if ((data & DSP_CAP_DUPLEX) != 0)
		{
#ifdef IOCTL_DBG
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("ioctl(%i, DSP_CAP_DUPLEX, &%x)\n", audioDev->fd, data);
#endif
			if (ioctl(audioDev->fd, DSP_CAP_DUPLEX, &data) < 0)
#ifdef IOCTL_DBG
			{
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("Failed to set Duplex\n");
			} else { 
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("Set to Duplex\n");
			}
#else
				;
#endif
		}
	}

	/*
	 * Get current bits (0x10 = 16 bit, 0x8 = 8 bit
	 */
	data = 0;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SND_CTL_DSP_GETFMTS, &%i)\n", audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_GETFMTS, &data)) == 0)
#ifdef IOCTL_DBG
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Current audio formats: %x\n", data);
	}
#else
		;
#endif
	else
		printf("Get sound format failed: %x: %i\n", results, data);

	/*
	 * Set to required resolution.
	 */
	if (audioDev->cflags & SLAB_8_BIT_OUT)
	{
		data = 8; /* THIS SHOULD USE THE soundcard.h DEFINITION */
	} else
		data = 16;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SOUND_PCM_WRITE_BITS, &%i)\n", audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SOUND_PCM_WRITE_BITS, &data)) == 0)
#ifdef IOCTL_DBG
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	Set write resolution: %i\n", data);
	}
#else
		;
#endif
	else
		printf("Set resolution failed: %i\n", results);

	if (audioDev->cflags & SLAB_8_BIT_IN)
		data = 8;
	else
		data = 16;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SOUND_PCM_READ_BITS, &%i)\n", audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SOUND_PCM_READ_BITS, &data)) == 0)
#ifdef IOCTL_DBG
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	Set read resolution: %i\n", data);
	}
#else
		;
#endif
	else
		printf("Set resolution failed: %i\n", results);

	/*
	 * Set to stereo
	 */
	data = 1;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SNDCTL_DSP_STEREO, &%i)\n", audioDev->fd, data);
#endif
	mode = SNDCTL_DSP_STEREO;
	if ((results = ioctl(audioDev->fd, mode, &data)) == 0)
#ifdef IOCTL_DBG
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	Set to stereo: %i\n", data);
	}
#else
		;
#endif
	else
		printf("Set stereo failed: %i\n", results);

	/*
	 * Set to AUDIO_RATE Hz
	 */
	data = audioDev->writeSampleRate;

	if ((results = ioctl(audioDev->fd, SOUND_PCM_WRITE_RATE, &data)) == 0)
#ifdef IOCTL_DBG
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	Set audio write rate to %i\n", data);
	}
#else
		;
#endif
	else
		printf("Set line frequency failed: %i\n", results);

	/*
	 * Save the actual physical line rate for conversions later
	 */
	audioDev->writeSampleRate = data;

	/*
	 * Write rate may be different to read rate, and physical IO speeds
	 * may be different from disk recorded rate.....
	 */
	data = audioDev->readSampleRate;

	if ((results = ioctl(audioDev->fd, SOUND_PCM_READ_RATE, &data)) == 0)
#ifdef IOCTL_DBG
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	Set audio read rate to %i\n", data);
	}
#else
		;
#endif
	else
		printf("Set line frequency failed: %i\n", results);

	/*
	 * Save the actual physical line rate for conversions later
	 */
	audioDev->readSampleRate = data;

#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
	{
		int *bufferData;

		bufferData = getAudioOSpace2(audioDev, devID);
		if (bufferData != 0)
			printf("Dev %i TBufs %i, FBufs %i, FBts 0x%x, FSze 0x%x\n",
				devID, bufferData[1], bufferData[0], bufferData[3],
					bufferData[2]);
	}
#endif
#endif /* linux from start of rubroutine */

	return(0);
}

#ifdef SUBFRAGMENT
/*
 * This algo will not support different RW rates over a single audio device.
 * The older init routines should do that.
 */
int
newInitAudioDevice2(audioDev, devID, fragSize)
duplexDev *audioDev;
int devID;
{
#ifdef linux
	int trackfile, i, j, results, data = 0, mode, index;

	/*
	 * This is what we are going to go for.
	 */
	fragSize = 1024;

#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("newInitAudioDevice2(%x, %i, %i)\n", audioDev, devID, fragSize);
#endif

	/*
	 * Clear out this buffer.
	 */
	if (audioDev->fragBuf != (char *) NULL)
	{
		free(audioDev->fragBuf);
		audioDev->fragBuf = (char *) NULL;
	}

	/*
ioctl(5, SNDCTL_DSP_GETCAPS, 0xbffff27c) = 0 - perhaps ensure options is ok.
ioctl(5, SNDCTL_DSP_SETDUPLEX, 0x8128870) = 0
ioctl(5, SNDCTL_DSP_SETFMT, 0xbffff278) = 0
ioctl(5, SNDCTL_DSP_STEREO, 0xbffff274) = 0
ioctl(5, SNDCTL_DSP_SPEED, 0xbffff268)  = 0
ioctl(5, SNDCTL_DSP_GETBLKSIZE, 0xbffff264) = 0 - output fragment size.
ioctl(5, SNDCTL_DSP_GETOSPACE, 0xbffff280) = 0 - optional.
	 */
#ifndef ALSA_SUPPORT
	checkAudioCaps2(audioDev, devID, audioDev->fd);
#endif
	if (fragSize != 0)
	{
		int data;

		switch(fragSize) {
			case 1024:
				data = 0x0040000a;
				break;
			case 2048:
				data = 0x0020000b;
				break;
			case 4096:
				data = 0x0010000c;
				break;
			case 8192:
				data = 0x0008000d;
				break;
			case 16384:
				data = 0x0004000e;
				break;
			/*
			 * This is 8192 blockSampleSize, where internally we deal with
			 * ints or floats, but the IO buffer is in shorts.
			 */
			case 32768:
			default:
				data = 0x0004000f;
				break;
		}
		if ((results = ioctl(audioDev->fd, SNDCTL_DSP_SETFRAGMENT, &data)) == 0)
#ifdef IOCTL_DBG
		{
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("ioctl(%i, SNDCTL_DSP_SETFRAGMENT, %x)\n",
					audioDev->fd, data);
		}
#else
			;
#endif
		else
			printf("ioctl(%i, SNDCTL_DSP_SETFRAGMENT, %x): failed\n",
				audioDev->fd, data);
	}

	/*
	 * Check capabilities.
	 */
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SNDCTL_DSP_GETCAPS, &0x%x)\n", audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_GETCAPS, &data)) == 0)
	{
		audioDev->genCaps = data;
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			if ((audioDev->genCaps & SNDCTL_DSP_SETTRIGGER) != 0)
				printf("Device %s does support SNDCTL_SET_TRIGGER\n",
					audioDev->devName);
			else
				printf("Device %s does NOT support SNDCTL_SET_TRIGGER\n",
					audioDev->devName);
		}
#endif
		/*
		 * Take this out, we only need to check the options here.
		 */
		if (data & DSP_CAP_DUPLEX)
		{
#ifdef IOCTL_DBG
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("ioctl(%i, SNDCTL_DSP_SETDUPLEX, &0x%x)\n",
					audioDev->fd, data);
#endif
			if (ioctl(audioDev->fd, SNDCTL_DSP_SETDUPLEX, &data) < 0)
#ifdef DEBUG
				printf("Failed to set Duplex\n");
			else
				printf("Set to Duplex\n");
#else
				;
#endif
		}
	}

	if ((audioDev->cflags & SLAB_FDUP) == 0)
		audioDev->fd2 = audioDev->fd;
	else
		audioDev->fd2 = fcntl(audioDev->fd, F_DUPFD, audioDev->fd);

	/*
	 * Set to required resolution.
	 */
	if (audioDev->cflags & (SLAB_8_BIT_OUT|SLAB_8_BIT_IN))
	{
		data = 8; /* THIS SHOULD USE THE soundcard.h DEFINITION */
	} else
		data = 16;

#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SNDCTL_DSP_SETFMT, &%i)\n", audioDev->fd, data);
#endif
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_SETFMT, &data)) == 0)
	{
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Set audio format: %i\n", data);
#else
		;
#endif
	} else
		printf("Set resolution failed: %i\n", results);

	/*
	 * Set to stereo
	 */
	data = 1;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("ioctl(%i, SNDCTL_DSP_STEREO, &%i)\n", audioDev->fd, data);
#endif
	mode = SNDCTL_DSP_STEREO;
	if ((results = ioctl(audioDev->fd, mode, &data)) == 0)
	{
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Set to stereo: %i\n", data);
#else
		;
#endif
	} else
		printf("Set stereo failed: %i\n", results);

	/*
	 * Set to AUDIO_RATE Hz
	 */
	data = audioDev->writeSampleRate;
#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
	{
		printf("ioctl(%i, SNDCTL_DSP_SPEED, &%i)\n", audioDev->fd, data);
/*		printf("SNDCTL_DSP_SPEED equates to %x\n", SNDCTL_DSP_SPEED); */
	}
#endif
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_SPEED, &data)) == 0)
	{
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Set audio readwrite rate to %i\n", data);
#else
		;
#endif
	} else
		printf("Set line frequency failed: %i\n", results);

	/*
	 * Save the actual physical line rate for conversions later
	 */
	audioDev->writeSampleRate = data;

	/*
	 * Save the actual physical line rate for conversions later
	 */
	audioDev->readSampleRate = data;

	/*
	 * Finally get the output fragment size. We can have an awkward problem if
	 * this fragment size is not the same as the primary device - with sync IO
	 * where we readBuffer/writeBuffer, having different sizes very soon puts 
	 * the audio daemon in to a lock state. To overcome this, and to support
	 * eventual realtime mixing (where we will specify a small buffersize
	 * anyway) we should add a "desired buffer size" param, or have the adiod
	 * routines cater for differences.
	 */
	data = 0;
	if ((results = ioctl(audioDev->fd, SNDCTL_DSP_GETBLKSIZE, &data)) == 0)
#ifdef IOCTL_DBG
	{
		audioDev->fragSize = data;

		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("ioctl(%i, SNDCTL_DSP_GETBLKSIZE, &0): %i bytes\n",
				audioDev->fd, audioDev->fragSize);
		audioDev->fragBuf = (char *) malloc(audioDev->fragSize);
	}
#else
		;
#endif
	else
		printf("Get frag size failed: %i\n", results);

#ifdef IOCTL_DBG
	if (audioDev->cflags & SLAB_AUDIODBG)
	{
		int *bufferData;

		bufferData = getAudioOSpace2(audioDev, devID);
		if (bufferData != 0)
			printf("Dev %i TBufs %i, FBufs %i, FBts 0x%x, FSze 0x%x\n",
			devID, bufferData[1], bufferData[0], bufferData[3],
				bufferData[2]);
	}
#endif

#endif /* linux */

#ifdef SUN
	int results = 0;

	AUDIO_INITINFO(&audioInfo);

	if ((results = ioctl(audioDev->fd, AUDIO_GETDEV, &results)) < 0)
	{
		perror("Failed to set audio dev info");
		return(-2);
	}

#ifdef DEBUG
	switch(results) {
		default:
			printf("Device unknown\n");
			break;
		case 1:
			printf("Device AMD\n");
			break;
		case 2:
			printf("Device DBRI\n");
			break;
		case 3:
			printf("Device CODEC\n");
			break;
	}
#endif

	/*
	 * These are the preferred parameters
	 */
	audioInfo.play.sample_rate = 44100;
	audioInfo.record.sample_rate = 44100;

	audioInfo.play.encoding = AUDIO_ENCODING_LINEAR;
	audioInfo.record.encoding = AUDIO_ENCODING_LINEAR;

	audioInfo.play.channels = 2;
	audioInfo.record.channels = 2;

	audioInfo.play.precision = 16;
	audioInfo.record.precision = 16;

	audioInfo.play.port = AUDIO_LINE_IN;
	audioInfo.record.port = AUDIO_LINE_IN;

	if ((results = ioctl(audioDev->fd, AUDIO_SETINFO, &audioInfo)) < 0)
	{
		/*
		 * We may only have AMD capabilities. Check it out.
		 */
		audioInfo.play.sample_rate = 8000;
		audioInfo.record.sample_rate = 8000;

		audioInfo.play.encoding = AUDIO_ENCODING_ALAW;
		audioInfo.record.encoding = AUDIO_ENCODING_ALAW;

		audioInfo.play.channels = 1;
		audioInfo.record.channels = 1;

		audioInfo.play.precision = 8;
		audioInfo.record.precision = 8;

		audioInfo.play.port = 1;
		audioInfo.record.port = 1;

		if ((results = ioctl(audioDev->fd, AUDIO_SETINFO, &audioInfo)) < 0)
		{
			/*
			 * This this happened then we do not have any capabilities
			 */
			printf("Failed to get audio info\n");
			return(-2);
		}
	}

#ifdef DEBUG
	printf("play sampleRate %i\n", audioInfo.play.sample_rate);
	printf("record sampleRate %i\n", audioInfo.record.sample_rate);

	printf("play encoding %i\n", audioInfo.play.encoding);
	printf("record encoding %i\n", audioInfo.record.encoding);

	printf("play channels %i\n", audioInfo.play.channels);
	printf("record channels %i\n", audioInfo.record.channels);

	printf("play precision %i\n", audioInfo.play.precision);
	printf("record precision %i\n", audioInfo.record.precision);

	printf("play port %i\n", audioInfo.play.port);
	printf("record port %i\n", audioInfo.record.port);

	printf("play avail_ports %i\n", audioInfo.play.avail_ports);
	printf("record avail_ports %i\n", audioInfo.record.avail_ports);
#endif /* DEBUG */
	/*
	 * Move the required variables over to the controlBuffer, so that the
	 * code can converge to these values
	 */
	controlBuffer->trackDesc.resolution = audioInfo.record.precision;
	controlBuffer->trackDesc.sampleRate =
	controlBuffer->trackDesc.writeSampleRate = audioInfo.play.sample_rate;
#endif /* SUN */

	return(0);
}
#endif /* SUB_FRAGMENT */

#ifndef ALSA_SUPPORT
/*
 * Needs to be ALSAfied - ie, binned if we have alsa - this is GUI stuff.
 */
checkAudioCaps2(audioDev, devID, fd)
duplexDev *audioDev;
int devID, fd;
{
	int i, stereodevs = 0;

#ifdef linux
#ifdef DEBUG
	printf("checkAudioCaps2(%i, %i)\n", devID, fd);
#endif

#ifdef ALSA_SUPPORT
	if (audioDev->siflags & AUDIO_ALSA)
		retutn;
#endif

    if (ioctl(fd, SOUND_MIXER_READ_STEREODEVS, &stereodevs) == -1)
	{
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
#endif
			printf("Failed to get stereo capabilities: %08x\n", stereodevs);
	} else {
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Capabilities: %08x\n", stereodevs);
#endif
		audioDev->stereoCaps = stereodevs;
	}


#ifdef DEBUG
	for (i = 1; i < 131072; i = i << 1) {
		if (i & stereodevs)
			printf("Found stereo dev %08x\n", i);
	}
#endif

	stereodevs = 0;
    if (ioctl(fd, SOUND_MIXER_READ_DEVMASK, &stereodevs) == -1)
	{
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
#endif
			printf("Failed to get audio capabilities: %08x\n", stereodevs);
	} else {
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Mono Capabilities: %08x\n", stereodevs);
#endif
		audioDev->monoCaps = stereodevs;
	}


	stereodevs = 0;
    if (ioctl(fd, SOUND_MIXER_READ_RECMASK, &stereodevs) == -1)
	{
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
#endif
			printf("Failed to get record capabilities: %08x\n", stereodevs);
	} else {
#ifdef IOCTL_DBG
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Record Caps: %08x\n", stereodevs);
#endif
		audioDev->recordCaps = stereodevs;
	}

#ifdef DEBUG
	for (i = 1; i < 131072; i = i << 1) {
		if (i & stereodevs)
			printf("Found mono dev %08x\n", i);
	}
#endif /* DEBUG */
#endif /* linux */
#ifdef SUN
	/*
	 * Fixed set of capabilities for Sun Sparcs.
	 */
	if (audioInfo.play.sample_rate == 8000)
	{
		audioDev->stereoCaps = 0;
		audioDev->monoCaps = SLAB_VOL_MIC_MASK|SLAB_VOL_LINE_MASK;
	} else {
		audioDev->stereoCaps = SLAB_VOL_MIC_MASK|SLAB_VOL_LINE_MASK;
		audioDev->monoCaps = 0;
	}
#endif /* SUN */
}
#endif /* ALSA_SUPPORT */

