
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <unistd.h>
#include "pthread.h"

#include "bristol.h"

void inthandler();

int exitReq = 0;

extern void *audioThread();
extern void *midiThread();

pthread_t spawnThread();

audioMain audiomain;

/*
 * We are going to create, initially, two threads. One will field MIDI events,
 * and translate the (for now) raw data into bristol MIDI messages. The messages
 * are queued, and then fielded by the audio thread. The audio thread reacts to
 * the midi events and starts sound generation operations. The MIDI thread will
 * remain responsible for MIDI parameter management, but not real-time messages.
 *
 * It could make sense to move the threads requests into the midi library for
 * use with other programmes?
 */
main(int argc, char **argv)
{
	int retcode, count = 0, policy, argCount = 1;
	pthread_t audiothread, midithread;
	void * retval;

	bzero(&audiomain, sizeof(audioMain));
	audiomain.iosize = BUFSZE;
	audiomain.preload = 4;
	// default to OSS until ALSA goes 1.0
	audiomain.flags = BRISTOL_MIDI_OSS|BRISTOL_OSS;
	audiomain.samplerate = 44100;

	while (argc > argCount)
	{
		if (strcmp(argv[argCount], "-mididev") == 0)
		{
			if (++argCount <= argc)
				audiomain.mididev = argv[argCount];
		}

		if (strcmp(argv[argCount], "-midi") == 0)
		{
			if (++argCount <= argc)
			{
				/*
				 * Note that some of these types of interface to not actually
				 * apply to MIDI at the moment. OSS and ALSA are rawmidi specs.
				 * SEQ is ALSA sequencer support. The others may or may not
				 * provide midi messaging, however it is possible to have jack
				 * audio drivers with ALSA rawmidi interfacing, for example.
				 */
				if (strcmp(argv[argCount], "oss") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_OSS;
				else if (strcmp(argv[argCount], "alsa") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_ALSA;
				else if (strcmp(argv[argCount], "slab") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_SLAB;
				else if (strcmp(argv[argCount], "seq") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_SEQ;
				else if (strcmp(argv[argCount], "jack") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_JACK;
				else if (strcmp(argv[argCount], "ladsp") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_MIDIMASK)
						|BRISTOL_MIDI_LADSP;
			}
		}

		if (strcmp(argv[argCount], "-audiodev") == 0)
		{
			if (++argCount <= argc)
				audiomain.audiodev = argv[argCount];
		}

		if (strcmp(argv[argCount], "-audio") == 0)
		{
			if (++argCount <= argc)
			{
				if (strcmp(argv[argCount], "oss") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_OSS;
				else if (strcmp(argv[argCount], "alsa") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_ALSA;
				else if (strcmp(argv[argCount], "slab") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_SLAB;
				else if (strcmp(argv[argCount], "jack") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_JACK;
				else if (strcmp(argv[argCount], "ladsp") == 0)
					audiomain.flags = (audiomain.flags & ~BRISTOL_AUDIOMASK)
						|BRISTOL_LADSP;
			}
		}

		/*
		 * This is the TCP control port connection only.
		 */
		if (strcmp(argv[argCount], "-port") == 0)
		{
			if (++argCount <= argc)
				audiomain.port = atoi(argv[argCount]);
		}

		if (strcmp(argv[argCount], "-oss") == 0)
			audiomain.flags = (audiomain.flags &
				~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK))
				|BRISTOL_OSS|BRISTOL_MIDI_OSS;

		if (strcmp(argv[argCount], "-seq") == 0)
			audiomain.flags |= BRISTOL_MIDI_SEQ;

		if (strcmp(argv[argCount], "-alsa") == 0)
			audiomain.flags = (audiomain.flags &
				~(BRISTOL_AUDIOMASK|BRISTOL_MIDIMASK))
				|BRISTOL_ALSA|BRISTOL_MIDI_ALSA;

		if ((strcmp(argv[argCount], "-rate") == 0) && (argc < argCount))
			audiomain.samplerate = atoi(argv[argCount++ + 1]);

		if ((strcmp(argv[argCount], "-bufsize") == 0) && (argc > argCount))
			audiomain.iosize = atoi(argv[argCount++ + 1]);

		if ((strcmp(argv[argCount], "-preload") == 0) && (argc > argCount))
			audiomain.preload = atoi(argv[argCount++ + 1]);

		argCount++;
	}

	audiomain.atStatus = audiomain.mtStatus = BRISTOL_EXIT;
	audiomain.atReq = audiomain.mtReq = BRISTOL_OK;

	/*
	 * Create two threads, one for midi, then one for audio.
	 *
	 * We cannot create the audio thread yet, since we do not know how many 
	 * voices to create - this is a function of the midi start requests from
	 * some controller.
	 */
	printf("spawning midi thread\n");
	midithread = spawnThread(midiThread, SCHED_RR);

	signal(SIGINT, inthandler);
//	signal(SIGSEGV, inthandler);

	printf("parent going into idle loop\n");

	while (!exitReq)
	{
		/*
		 * We should wait for children here, and restart them. We should also 
		 * monitor any bristol system sysex messages, since they will be used
		 * to request we start the audio thread.
		 */
		if (audiomain.atReq & BRISTOL_REQSTART)
		{
			/*
			 * See if the audio thread is active, if not, start it. May 
			 * eventually support multiple audio threads.
			 */
			if (audiomain.atStatus == BRISTOL_EXIT)
			{
				printf("spawning audio thread\n");
				audiothread = spawnThread(audioThread, SCHED_FIFO);
			}
		}
		sleep(1);
	}

	sleep(1);

	printf("parent exiting\n");

	return(0);
}

pthread_t spawnThread(void * (*threadcode)(void *), int despolicy)
{
	int retcode;
	pthread_t thread;
	struct sched_param schedparam;
	int policy;

	/*
	 * Create thread, detach it.
	 */
	retcode = pthread_create(&thread, NULL, threadcode, &audiomain);
	if (retcode != 0)
		fprintf(stderr, "create a failed %d\n", retcode);
	retcode = pthread_detach(thread);
	if (retcode != 0)
		fprintf(stderr, "detach failed %d\n", retcode);

	/*
	 * See if we need to reschedule it. This may fail if we do not have root
	 * permissions (or if the audio library configures RRpri99 before we make
	 * this request?)
	 */
	if (pthread_getschedparam(thread, &policy, &schedparam) == 0)
	{
		if (policy != despolicy)
		{
			policy = despolicy;
			if (pthread_setschedparam(thread, policy, &schedparam) == 0)
				printf("Rescheduled thread to desired %i\n", despolicy);
			else
				printf("Could not reschedule thread to %i\n", despolicy);
		} else
			printf("Did not reschedule thread\n");
	} else
		printf("Could not get thread schedule\n");

	return(thread);
}

void
inthandler()
{
	audiomain.atReq = BRISTOL_REQSTOP;
	exitReq = 1;
	bristolMidiTerminate();
}

alterAllNotes(Baudio *baudio)
{
	bristolVoice *voice = audiomain.playlist;

	while (voice != NULL)
	{
		if (voice->baudio == baudio)
			doNoteChanges(voice);

		voice = voice->next;
	}
}

