/***************************************************************************
                konqueror.cpp  -  Konqueror Class Implementation
                             -------------------
    begin                : Sun Aug 24 2003
    copyright            : (C) 2003 by Ken Schenke
    email                : kenschenke at yahoo dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#include <ctype.h>
#include <stdlib.h>
#include <stack>
#include <qfile.h>
#include <qdir.h>
#include <qfiledialog.h>

#include "konqueror.h"
#include "browserlist.h"
#include "xmlparser.h"
#include "xbelwrite.h"

// for access()
#ifdef _WIN32
#include <io.h>
#else
#include <unistd.h>
#endif

/***************************************************************************
 *                                                                         *
 *   Constants                                                             *
 *                                                                         *
 ***************************************************************************/

#define SAXSTATE_XBEL			(SAXSTATE_USER+1)
#define SAXSTATE_FOLDER			(SAXSTATE_USER+2)
#define SAXSTATE_BOOKMARK		(SAXSTATE_USER+3)
#define SAXSTATE_TITLE			(SAXSTATE_USER+4)

/***************************************************************************
 *                                                                         *
 *   XML Parser Call-back Functions                                        *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartXbel(void *, const xmlChar *, const xmlChar **, short);
static bool xmlStartBookmark(void *, const xmlChar *, const xmlChar **, short);
static bool xmlStartFolder(void *, const xmlChar *, const xmlChar **, short);
static void xmlEndFolder(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndBookmark(void *, const xmlChar *, const xmlChar *, short, short);
static void xmlEndTitle(void *, const xmlChar *, const xmlChar *, short, short);

static ELEMHANDLER xmlHandlers[] = {
	{
		SAXSTATE_XBEL,
		"xbel",
		xmlStartXbel,
		NULL,
	},{
		SAXSTATE_FOLDER,
		"folder",
		xmlStartFolder,
		xmlEndFolder,
	},{
		SAXSTATE_BOOKMARK,
		"bookmark",
		xmlStartBookmark,
		xmlEndBookmark,
	},{
		SAXSTATE_TITLE,
		"title",
		NULL,
		xmlEndTitle,
	},
	// this element marks the end of the list
	{ 0, NULL, NULL, NULL }
};

/***************************************************************************
 *                                                                         *
 *   Structure Definitions                                                 *
 *                                                                         *
 ***************************************************************************/

typedef struct {
	std::stack<BkFolder *>	*folders;
	BkBookmark	*bookmark;
	BRWSNUM		browserOrd;
} SAXDATA;

/***************************************************************************
 *                                                                         *
 *   Function Prototypes                                                   *
 *                                                                         *
 ***************************************************************************/

static	void	xmlStartDocument(void *);

/***************************************************************************
 *                                                                         *
 *   xmlStartDocument()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called by the SAX2 parsing engine before any      *
 *      others.  It uses the opportunity to initialize state data.  It     *
 *      also pushes the current state on to a state stack.  This stack is  *
 *      pushed when a new XML element is encountered and popped at the end *
 *      of the element.  This allows the callback functions to always      *
 *      know where they are in the XML document.                           *
 *                                                                         *
 ***************************************************************************/

static void xmlStartDocument(void *ctx)
{
	SAXDATA *data = (SAXDATA *)ctx;

	data->bookmark = NULL;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartXbel()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *              (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **    (not used)                                     *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <xbel> element is encountered.    *
 *      All it really does is verify the <xbel> element is occuring in the *
 *      correct part of the XML document.                                  *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartXbel(void *, const xmlChar *, const xmlChar **, short prevState)
{
	if(prevState != SAXSTATE_STARTEND)
		return true;

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartBookmark()                                                    *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **atts                                               *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <bookmark> element is             *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartBookmark(void *ctx, const xmlChar *, const xmlChar **atts, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	BkBookmark bookmark;

	int order =
		data->folders->top()->folders() +
		data->folders->top()->bookmarks() +
		data->folders->top()->separators() + 1;
	data->bookmark = &data->folders->top()->addBookmark(bookmark);
	data->bookmark->setOrder(data->browserOrd, order);

	// this bookmark belongs to Konqueror

	data->bookmark->setBrowserFound(data->browserOrd);
	data->bookmark->setBrowserSaved(data->browserOrd);

	// save some attributes in the element

	for(int i=0; atts[i]; i+=2)
	{
		if(!xmlStrcmp(atts[i], (const xmlChar *)"href"))
		{
			data->bookmark->setUrl(QString::fromUtf8((const char *)atts[i+1]));
			data->bookmark->setValidField(BKVALID_URL);
		}
		else
		{
			data->bookmark->setAttr(
				(const char *)atts[i],
				(const char *)atts[i+1],
				data->browserOrd);
		}
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlStartFolder()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar **atts                                               *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      true if there was an error, false otherwise                        *
 *   Description:                                                          *
 *      This function is called when the <folder> element is encountered.  *
 *                                                                         *
 ***************************************************************************/

static bool xmlStartFolder(void *ctx, const xmlChar *, const xmlChar **atts, short)
{
	SAXDATA	*data = (SAXDATA *)ctx;

	BkFolder folder, *child;
	int order =
		data->folders->top()->folders() +
		data->folders->top()->bookmarks() +
		data->folders->top()->separators() + 1;
	child = &data->folders->top()->addFolder(folder);
	child->setOrder(data->browserOrd, order);
	data->folders->push(child);

	// this folder belongs to Konqueror

	data->folders->top()->setBrowserFound(data->browserOrd);
	data->folders->top()->setBrowserSaved(data->browserOrd);
	
	// save some attributes in the element

	for(int i=0; atts[i]; i+=2)
	{
		if(!xmlStrcmp(atts[i], (const xmlChar *)"folded"))
			;	// ignore
		else
		{
			data->folders->top()->setAttr(
				(const char *)atts[i],
				(const char *)atts[i+1],
				data->browserOrd);
		}
	}

	return false;
}

/***************************************************************************
 *                                                                         *
 *   xmlSAXendFolder()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      short state                                                        *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <folder> element is    *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndFolder(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_FOLDER);
	ASSERT(data->folders->size() > 0);

	data->folders->pop();
}

/***************************************************************************
 *                                                                         *
 *   xmlEndBookmark()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const xmlChar *     (not used)                                     *
 *      short state                                                        *
 *      short               (not used)                                     *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <bookmark> element is  *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndBookmark(void *ctx, const xmlChar *, const xmlChar *, short state, short)
{
	SAXDATA *data = (SAXDATA *)ctx;

	ASSERT(state == SAXSTATE_BOOKMARK);
	
	data->bookmark = NULL;
}

/***************************************************************************
 *                                                                         *
 *   xmlEndTitle()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      void *ctx                                                          *
 *      const xmlChar *     (not used)                                     *
 *      const char *pChars                                                 *
 *      short               (not used)                                     *
 *      short prevState                                                    *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      This function is called when the end of the <title> element is     *
 *      encountered.                                                       *
 *                                                                         *
 ***************************************************************************/

static void xmlEndTitle(void *ctx, const xmlChar *, const xmlChar *pChars, short, short parentState)
{
	SAXDATA *data = (SAXDATA *)ctx;

	if(pChars)
	{
		if(parentState == SAXSTATE_FOLDER)
		{
			ASSERT(data->folders->size() > 0);
			data->folders->top()->setTitle(
				QString::fromUtf8((const char *)pChars));
			data->folders->top()->setValidField(BKVALID_TITLE);
		}
		else if(parentState == SAXSTATE_BOOKMARK)
		{
			ASSERT(data->bookmark != NULL);
			data->bookmark->setTitle(
				QString::fromUtf8((const char *)pChars));
			data->bookmark->setValidField(BKVALID_TITLE);
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   Konqueror::AreBookmarksValid()                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &bookmarks                                           *
 *   Return:                                                               *
 *      true if valid, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called to verify the filename given in the first  *
 *      first parameter is most likely a Konqueror bookmark file.          *
 *      It first attempts to open the file and then reads the first few    *
 *      lines and matches them against what it thinks the they should be.  *
 *                                                                         *
 ***************************************************************************/

bool Konqueror::AreBookmarksValid(const QString &bookmarks)
{
	QFile   file(bookmarks);

	if(file.open(IO_ReadOnly) == false)
		return false;
	QTextStream stream(&file);

	QString line = stream.readLine().stripWhiteSpace();
	if(!line.startsWith("<!DOCTYPE xbel>"))
		return false;

	line = stream.readLine().stripWhiteSpace();
	return line.startsWith("<xbel folded=\"no\"");
}

/***************************************************************************
 *                                                                         *
 *   Konqueror::BrowseForBookmarks()                                       *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &                                                  *
 *      QString &bookmarks                                                 *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      true if the user did not select a bookmark file, false otherwise   *
 *   Description:                                                          *
 *      This function is called to present the user with a file dialog box *
 *      allowing them to select the location of their bookmark file.       *
 *                                                                         *
 ***************************************************************************/

bool Konqueror::BrowseForBookmarks(const BridgeCfg &, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
	{
#if defined(Q_WS_X11)
		bookmarks = QDir::homeDirPath() + "/.kde/share/apps/konqueror";
#elif defined(Q_WS_WIN)
		bookmarks = QDir::homeDirPath();
#else
#error "Must define default path for Konqueror Bookmarks File"
#endif
	}
	bookmarks = QFileDialog::getOpenFileName(
		bookmarks, "Bookmark File (bookmarks.xml)",
		parent, "Get Bookmarks File", "Select Your Bookmarks File");

	return (bookmarks == QString::null);
}

/***************************************************************************
 *                                                                         *
 *   Konqueror::classFactory()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      the newly allocated class                                          *
 *   Description:                                                          *
 *      This is a static member function of the Konqueror class.  Its job  *
 *      is to allocate an instance of the Konqueror class for the caller.  *
 *                                                                         *
 ***************************************************************************/

BrowserBk *Konqueror::classFactory(void)
{
	return new Konqueror;
}

/***************************************************************************
 *                                                                         *
 *   Konqueror::DetectBrowser()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &                                                  *
 *      QStringList &path                                                  *
 *   Return:                                                               *
 *      true if an installation of Konqueror was found, false otherwise    *
 *   Description:                                                          *
 *      This function attempts to locate Konqueror's bookmark database on  *
 *      the system.                                                        *
 *                                                                         *
 ***************************************************************************/

#ifdef Q_WS_X11
bool Konqueror::DetectBrowser(const BridgeCfg &, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  It would be something like:
	// /home/ken/.kde/share/apps/konqueror

	basePath = QDir::homeDirPath() + "/.kde/share/apps/konqueror";

	// With the base path defined, see if it exists

	QDir base(basePath);
	if(!base.exists())
		return false;

	QString filename(basePath);
	filename += "/bookmarks.xml";

	if(access(QDir::convertSeparators(filename), 0) == 0)
		paths.append(filename);
	
	return paths.size() > 0;
}
#endif

/***************************************************************************
 *                                                                         *
 *   Konqueror::readBookmarks()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &filename                                            *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to read Konqueror's bookmark file.          *
 *                                                                         *
 ***************************************************************************/
 
void Konqueror::readBookmarks(const QString &filename, BRWSNUM browserOrd)
				throw(BkException)
{
	SAXDATA		data;

	data.folders = new std::stack<BkFolder *>;
	data.folders->push(&m_Root);
	data.browserOrd = browserOrd;

	// parse the file

	ParseXmlDocument(
		filename,
		&data,
		xmlHandlers,
		xmlStartDocument,
		NULL);

	if(data.folders->size() != 1)
		BKEXCEPT("XML Parser has Inconsistent Internal State");
	data.folders->pop();

	delete data.folders;
}

/***************************************************************************
 *                                                                         *
 *   Konqueror::saveBookmarks()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &filename                                            *
 *      BkFolder &root                                                     *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function begins the process of saving the bookmark tree,      *
 *      specified in the second parameter to the file, specified in the    *
 *      first parameter.                                                   *
 *                                                                         *
 ***************************************************************************/

void Konqueror::saveBookmarks(const QString &filename, BkFolder &root, BRWSNUM browserOrd)
		throw(BkException)
{
	QFile	out;

	// open the file for writing

	out.setName(filename);
	if(out.open(IO_WriteOnly | IO_Translate) == false)
		BKEXCEPT("Unable to Open Konqueror Bookmark File for Output");

	// set up a text stream

	QTextStream t(&out);
	t.setEncoding(QTextStream::UnicodeUTF8);

	// write out Konqueror's boilerplate header

	t
		<< "<!DOCTYPE xbel>" << endl
		<< "<xbel folded=\"no\">" << endl;

	saveFolder(root, t, 0, browserOrd);

	// finish up

	t << "</xbel>" << endl;
}

void Konqueror::saveFolder(BkFolder &folder, QTextStream &t, int levels, BRWSNUM browserOrd)
	throw(BkException)
{
	int	i;
	QString	temp;
	std::vector<SortedNodeList> nodes;
	std::vector<SortedNodeList>::iterator it;
	
	SortNodes(folder, nodes, browserOrd);
	
	for(it=nodes.begin(); it!=nodes.end(); ++it)
	{
		if(it->m_type == NODETYPE_FOLDER)
		{
			// if this node has been deleted from one of the
			// browsers, skip it...
		
			if(it->m_fit->isDeleted())
				continue;
				
			// Has the user requested to ignore this node?
			// If so, we should skip it if it's not already stored in Konqueror
		
			if(it->m_fit->ignore() && !it->m_fit->browserFound(browserOrd))
				continue;
			
			// save the folder
		
			// write out the title and other attributes

			for(i=0; i<levels+1; i++)
				t << ' ';
			t << "<folder folded=\"no\"";
			for(i=0; i<it->m_fit->nAttrs(); i++)
			{
				QString	name, value;
				BRWSNUM browser;

				it->m_fit->attr(i, name, value, browser);

				if(browser != browserOrd)
					continue;
	
				t
					<< " "
					<< name
					<< "=\""
					<< value
					<< "\"";
			}
			t << '>' << endl;

			for(i=0; i<levels+2; i++)
				t << ' ';
			EncodeString(it->m_fit->title(), temp);
			t << "<title>" << temp << "</title>" << endl;
			
			saveFolder(*(it->m_fit), t, levels+1, browserOrd);
		
			for(i=0; i<levels+1; i++)
				t << ' ';
			t << "</folder>" << endl;
			
			it->m_fit->setBrowserFound(browserOrd);
			it->m_fit->setBrowserSaved(browserOrd);
		}
		else if(it->m_type == NODETYPE_BOOKMARK)
		{
			// if this node has been deleted from one of the
			// browsers, skip it...
		
			if(it->m_bit->isDeleted())
				continue;
		
			// Has the user requested to ignore this node?
			// If so, we should skip it if it's not already stored in Konqueror
		
			if(it->m_bit->ignore() && !it->m_bit->browserFound(browserOrd))
				continue;
			
			// store the bookmark
		
			saveBookmark(*(it->m_bit), t, levels+1, browserOrd);
			
			it->m_bit->setBrowserFound(browserOrd);
			it->m_bit->setBrowserSaved(browserOrd);
		}
		else if(it->m_type == NODETYPE_SEPARATOR)
			continue;	// not supported in Konqueror
		else
			BKEXCEPT("Internal Error: Unrecognized Bookmark Type");
	}
}

void Konqueror::saveBookmark(const BkBookmark &bkmark, QTextStream &t, int levels, BRWSNUM browserOrd)
	throw(BkException)
{
	int	i;
	QString	temp;

	EncodeString(bkmark.url(), temp);
	for(i=0; i<levels; i++)
		t << ' ';
	t << "<bookmark href=\"" << temp << '\"';

	for(i=0; i<bkmark.nAttrs(); i++)
	{
		QString	name, value;
		BRWSNUM browser;

		bkmark.attr(i, name, value, browser);

		if(browser != browserOrd)
			continue;

		t
			<< " "
			<< name
			<< "=\""
			<< value
			<< "\"";
	}
	t << '>' << endl;
	
	EncodeString(bkmark.title(), temp);
	for(i=0; i<levels+1; i++)
		t << ' ';
	t << "<title>" << temp << "</title>" << endl;
	
	for(i=0; i<levels; i++)
		t << ' ';
	t << "</bookmark>" << endl;
}
