/***************************************************************************
                browserbk.cpp  -  BrowserBk Class Implementation
                             -------------------
    begin                : Sun Sep 15 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kenschenke at yahoo dot com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

#include "browserbk.h"
#include "browserlist.h"
#include <qdir.h>
#include <qmessagebox.h>

// for access()
#ifdef _WIN32
#include <io.h>
#else
#include <unistd.h>
#endif

#include <algorithm>

using namespace std;

/***************************************************************************
 *                                                                         *
 *   BrowserBk::areBookmarksReadonly()                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &bookmarks                                           *
 *   Return:                                                               *
 *      true if bookmark data is readonly, false otherwise                 *
 *   Description:                                                          *
 *      This virtual function checks to see if the browser's bookmark      *
 *      database is read only.  Since most browsers store their bookmarks  *
 *      in one file (with the notable exception of Internet Explorer),     *
 *      this function doesn't usually have to over-ridden.                 *
 *                                                                         *
 ***************************************************************************/

bool BrowserBk::AreBookmarksReadonly(const QString &bookmarks)
{
	return (access(QDir::convertSeparators(bookmarks), 0) != 0);
}

/***************************************************************************
 *                                                                         *
 *   BrowserBk::DetectBrowser()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &                                                  *
 *      const QStringList &                                                *
 *   Return:                                                               *
 *      always returns false, indicating no detection                      *
 *   Description:                                                          *
 *      This virtual function is normally over-ridden in order to detect   *
 *      the presence of a browser on the system.                           *
 *                                                                         *
 ***************************************************************************/

bool BrowserBk::DetectBrowser(const BridgeCfg &, QStringList &)
{
	return false;
}

/***************************************************************************
 *                                                                         *
 *   BrowserBk::findDifferencesX()                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &mFolder                                                  *
 *      BkFolder &bFolder                                                  *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function does the real work behind comparing a browser's      *
 *      bookmark database, represented by the bFolder parameter with the   *
 *      master database, represented by the mFolder parameter.  It         *
 *      compares one folder at a time.  If a sub-folder is encountered,    *
 *      the function calls itself recursively.  For each node encountered, *
 *      it calls compareBookmarks() to note any differences.               *
 *                                                                         *
 ***************************************************************************/

void BrowserBk::findDifferencesX(BkFolder &mFolder, const BkFolder &bFolder)
			throw(BkException)
{
	bool		equal;
	BookmarkLst::iterator bookmark;
	FolderLst::iterator folder;
	BookmarkLst::const_iterator bit;
	FolderLst::const_iterator fit;
	SeparatorLst::const_iterator sit;

	// Walk through the bookmark sub-tree for the browser.  For each
	// node, attempt to find a match in the master tree.  If a match is
	// found, indicate in the master node that the browser also has the node.
	// If a match is not found, add a copy of the browser's bookmark to the
	// master tree.

	for(bit = bFolder.BookmarksBegin();
		bit != bFolder.BookmarksEnd();
		++bit)
	{
		bookmark = find(mFolder.BookmarksBegin(), mFolder.BookmarksEnd(), bit->url());
		if(bookmark != mFolder.BookmarksEnd())
		{
			// merge the browser's copy of the bookmark into
			// the master list's bookmark

			bookmark->setBrowserFound(*bit);

			// save this bookmark's attributes into the
			// master tree's node

			for(int i=0; i<bit->nAttrs(); i++)
			{
				QString name, value;
				BRWSNUM browser;

				bit->attr(i, name, value, browser);
				bookmark->setAttr(name, value, browser);
			}
			
			// save this bookmark's order into the master tree's node
			
			bookmark->copyOrder(*bit);

			// compare the master bookmark with the browser's
			
			equal = true;
			if(bit->isFieldValid(BKVALID_DESC))
			{
				if( !bookmark->isFieldValid(BKVALID_DESC)
				 || bookmark->desc() != bit->desc())
				{
					bookmark->setDiffDesc(bit->desc());
					equal = false;
				}
			}
			if(bit->isFieldValid(BKVALID_TITLE))
			{
				if( !bookmark->isFieldValid(BKVALID_TITLE)
				 || bookmark->title() != bit->title())
				{
					bookmark->setDiffTitle(bit->title());
					equal = false;
				}
			}

			if(equal)
				bookmark->setBrowserSaved(*bit);
		}
		else
			mFolder.addBookmark(*bit);
	}

	// Do the same for the folders, this time calling this function recursively

	for(fit = bFolder.FoldersBegin();
		fit != bFolder.FoldersEnd();
		++fit)
	{
		folder = find(mFolder.FoldersBegin(), mFolder.FoldersEnd(), fit->title());
		if(folder != mFolder.FoldersEnd())
		{
			// save this folders's attributes into the
			// master tree's node

			for(int i=0; i<fit->nAttrs(); i++)
			{
				QString name, value;
				BRWSNUM browser;

				fit->attr(i, name, value, browser);
				folder->setAttr(name, value, browser);
			}

			// save this folder's order into the master tree's node
			
			folder->copyOrder(*fit);

			// compare the descriptions

			if(fit->isFieldValid(BKVALID_DESC))
			{
				if( !fit->isFieldValid(BKVALID_DESC)
				 || folder->desc() != fit->desc())
					folder->setDiffDesc(fit->desc());
			}
			
			// check the sub-folder

			findDifferencesX(*folder, *fit);

			// merge the brower's copy of the folder into
			// the master list's folder

			folder->setBrowserFound(*fit);
			folder->setBrowserSaved(*fit);
		}
		else
			mFolder.addFolder(*fit);
	}

	// Since no attempt is made to merge separators between browsers, each
	// separator is simply added to the master tree where each browser can
	// later find its own separators in order to save them back out again.
	
	for(sit=bFolder.SeparatorsBegin(); sit!=bFolder.SeparatorsEnd(); ++sit)
		mFolder.addSeparator(*sit);
}

/***************************************************************************
 *                                                                         *
 *   BrowserBk::scrubFolder()                                              *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &folder                                                   *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function goes through the browser's bookmark and folder tree  *
 *      looking for bookmark titles and folder names with invalid          *
 *      characters that might not be valid as URL filenames for Internet   *
 *      Explorer.  When it finds one it changes the title so that it is    *
 *      a valid filename.                                                  *
 *                                                                         *
 ***************************************************************************/

void BrowserBk::scrubFolder(BkFolder &folder) throw(BkException)
{
	BookmarkLst::iterator bit;
	FolderLst::iterator fit;
	QString title;

	// look for invalid characters in the folder's title

	title = folder.title();
	scrubTitle(title, true);

	// if the title was changed, save it back to the folder
	
	if(title != folder.title())
		folder.setTitle(title);

	for(bit = folder.BookmarksBegin();
		bit != folder.BookmarksEnd();
		++bit)
	{
		// look for invalid characters in the bookmark's title

		title = bit->title();
		scrubTitle(title, false);

		// if the title was changed, save it back to the bookmark

		if(title != bit->title())
			bit->setTitle(title);
	}

	for(fit = folder.FoldersBegin();
		fit != folder.FoldersEnd();
		++fit)
	{
		scrubFolder(*fit);
	}
}

/***************************************************************************
 *                                                                         *
 *   BrowserBk::scrubTitle()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      QString &title                                                     *
 *   Return:                                                               *
 *      None.                                                              *
 *   Description:                                                          *
 *      This function looks for and removes invalid characters from the    *
 *      title passed by reference as the argument.                         *
 *                                                                         *
 ***************************************************************************/

void BrowserBk::scrubTitle(QString &title, bool isFolder)
{
	int i;

	// if this is a folder, Windows won't allow it to end with a period

	if(isFolder)
	{
		for(;;)
		{
			if(title.right(1) == ".")		// if it ends in a period...
				title.replace(title.length()-1, 1, "");		// remove it
			else
				break;
		}
	}

	// replace forward slashes with dashes

	for(;;)
	{
		i = title.find('/', 0);
		if(i >= 0)
			title.replace(i, 1, "-");
		else
			break;
	}

	// remove any of the following characters:
	//    :  colon
	//    *  asterisk
	//    ?  question mark
	//    "  double quote
	//    <  less-than
	//    >  greater-than
	//    |  vertical bar (pipe)
	//    \  back-slash

	for(;;)
	{
		if((i = title.find(':', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('*', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('?', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('\"', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('<', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('>', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('|', 0)) >= 0)
			title.replace(i, 1, "");
		else if((i = title.find('\\', 0)) >= 0)
			title.replace(i, 1, "");
		else
			break;
	}
}

/***************************************************************************
 *                                                                         *
 *   SortNodes()                                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      BkFolder &folder                                                   *
 *      vector<SortedNodeList> &list                                       *
 *      BRWSNUM browser                                                    *
 *   Return:                                                               *
 *      None.                                                              *
 *   Description:                                                          *
 *      This function sorts the folders and bookmarks contained within the *
 *      folder in the first argument according to the order for the        *
 *      browser in the third argument.  It does not modify the original    *
 *      bookmarks and folders.  It instead saves iterators for each of the *
 *      bookmarks and folders in a vector (the second argument).           *
 *                                                                         *
 ***************************************************************************/

void SortNodes(BkFolder &folder, vector<SortedNodeList> &l, BRWSNUM browser)
{
	FolderLst::iterator fit;
	BookmarkLst::iterator bit;
	SeparatorLst::iterator sit;
	
	for(fit=folder.FoldersBegin(); fit!=folder.FoldersEnd(); ++fit)
		l.push_back(SortedNodeList(fit, fit->order(browser)));
	
	for(bit=folder.BookmarksBegin(); bit!=folder.BookmarksEnd(); ++bit)
		l.push_back(SortedNodeList(bit, bit->order(browser)));
	
	for(sit=folder.SeparatorsBegin(); sit!=folder.SeparatorsEnd(); ++sit)
		l.push_back(SortedNodeList(sit, sit->order(browser)));
	
	sort(l.begin(), l.end());
}
