#include <sys/types.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "iobuf/iobuf.h"
#include "net/ipv4.h"
#include "str/str.h"

#include "dns.h"

#define MAX_USE_COUNT 10000
#define MAX_USE_TIME (10*60)

str dns_search = {0,0,0};
ipv4addr dns_servers[DNS_MAX_SERVERS];
unsigned dns_server_count = 0;

static time_t config_time;
static struct timeval load_time;
static int dns_use_count = MAX_USE_COUNT;

static int str_isprefix(const str* line, char* word, unsigned len)
{
  return memcmp(line->s, word, len) == 0 && isspace(line->s[len]);
}

static unsigned skip_space(const str* line, unsigned i)
{
  while (i < line->len && isspace(line->s[i])) ++i;
  return i;
}

static int parse_resolv_line(const str* line)
{
  unsigned i;
  const char* end;
  if (dns_server_count < DNS_MAX_SERVERS &&
      str_isprefix(line, "nameserver", 10)) {
    i = skip_space(line, 10);
    if ((end = ipv4_scan(line->s+i, &dns_servers[dns_server_count])) != 0 &&
	*end == 0)
      ++dns_server_count;
  }
  else if (str_isprefix(line, "domain", 6) ||
	   str_isprefix(line, "search", 6)) {
    i = skip_space(line, 6);
    if (i < line->len)
      if (!str_copyb(&dns_search, line->s+i, line->len-i)) return 0;
  }
  return 1;
}

int dns_load_config(void)
{
  const char* env;
  ibuf in;
  str line = {0,0,0};
  dns_server_count = 0;
  dns_use_count = 0;
  gettimeofday(&load_time, 0);
  if (ibuf_open(&in, "/etc/resolv.conf", 0)) {
    int result = 1;
    struct stat st;
    fstat(in.io.fd, &st);
    config_time = st.st_mtime;
    while (result && ibuf_getstr(&in, &line, LF)) {
      str_rstrip(&line);
      if (!parse_resolv_line(&line)) result = 0;
    }
    str_free(&line);
    ibuf_close(&in);
    if (!result) return 0;
  }
  else
    config_time = 0;
  if ((env = getenv("LOCALDOMAIN")) != 0)
    if (!str_copys(&dns_search, env)) return 0;
  if ((env = getenv("DNSCACHEIP")) != 0) {
    ipv4addr tmp;
    const char* end;
    if ((end = ipv4_scan(env, &tmp)) && *end == 0) {
      memcpy(dns_servers, &tmp, sizeof tmp);
      dns_server_count = 1;
    }
  }
  if (dns_server_count == 0) {
    memcpy(dns_servers, "\177\0\0\1", 4);
    dns_server_count = 1;
  }
  str_subst(&dns_search, ' ', 0);
  str_subst(&dns_search, '\t', 0);
  return 1;
}

int dns_try_reload_config(void)
{
  struct timeval now;
  if (dns_server_count == 0 || dns_use_count >= MAX_USE_COUNT)
    return dns_load_config();
  ++dns_use_count;
  gettimeofday(&now, 0);
  if (now.tv_sec - load_time.tv_sec >= MAX_USE_TIME) return dns_load_config();
  return 1;
}
