<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    layout-editor-defs.php                                  */
/* Author:      Paul Waite                                              */
/* Description: Definitions for content layout editing in webpages.     */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Record maintainer for perms settings */
include_once("recmaint-defs.php");

// ----------------------------------------------------------------------
/**
* Layouteditor
* A layouteditor is a utility class. It contains all of the methods
* required to edit a layout, so that the layout class can concentrate
* on the basics of layout acquisition and display. The constructor
* of a layouteditor must be passed a reference to the layout it is
* going to be providing editing services for.
* @package cm
*/
class layouteditor extends RenderableObject {
  // Public
  // Private
  /** The layout we are providing
      editing services for
      @access private */
  var $layout;
  // ....................................................................
  /**
  * Constructor
  * Create a new layouteditor object.
  * @param reference $layout A reference to the layout being edited
  */
  function layouteditor(&$layout) {
    // Find out if there is an Editor group(s) override..
    $this->layout = $layout;
  } // layouteditor
  // ....................................................................
  /**
  * Replicate the hosted layout as a new layout. Creates a brand new
  * layout in the database, with same data as this one. The end result
  * is that this current object becomes the new layout, and a duplicate
  * set of layout records exist in the database. The layout ID of this
  * new layout is, of course, updated to being a brand new one.
  * NOTES: The layout name is normally left null, which keeps the layout
  * in the same 'family' of layout versions. You can force the layout
  * name to be different, and this will create a new 'layout_set'
  * record of that name for you, if required.
  * @param string $layoutname New layout name. If null, keeps same name.
  */
  function replicate($layoutname="") {
    if ($this->layout->exists) {
      // If a layout name is specified make sure layout set exists..
      if ($layoutname != "") {
        $this->layout->layout_name = $layoutname;
        $q  = "SELECT * FROM ax_layout_set";
        $q .= " WHERE layout_name='" . addslashes($layoutname) . "'";
        $chkset = dbrecordset($q);
        if ($chkset->rowcount == 0) {
          $LSin = new dbinsert("ax_layout_set");
          $LSin->set("layout_name", $layoutname);
          $LSin->execute();
        }
      }

      // Save replicated layout as a brand new one..
      $orig_layid = $this->layout->layoutid;
      $this->layout->layoutid = get_next_sequencevalue("seq_layout_id", "ax_layout", "layout_id");
      $this->layout->exists = false;
      $this->layout->put();

      // Replicate all the blocks and adjust layout table references..
      $Blq = dbrecordset("SELECT block_id FROM ax_block WHERE layout_id=$orig_layid");
      if ($Blq->hasdata) {
        do {
          // Replicate block/blocklets into new block
          $blockid = $Blq->field("block_id");
          $b = new block($blockid);
          $b->layoutid = $this->layout->layoutid;
          $b->replicate();
          // Fix up the layout table block reference..
          reset($this->layout->layout_blocks);
          while (list($rowcol, $old_blockid) = each($this->layout->layout_blocks)) {
            if ($old_blockid == $blockid) {
              $bits = explode("|", $rowcol);
              $row = $bits[0];
              $col = $bits[1];
              $cell = $this->layout->layout_table->get_cell($row, $col);
              $cell->blockid = $b->blockid;
              $this->layout->layout_table->set_cell($row, $col, $cell);
              $this->layout->layout_blocks["$row|$col"] = $b->blockid;
            }
          }
        } while ($Blq->get_next());
      }

      // Replicate all metadata..
      $Mlq = dbrecordset("SELECT * FROM ax_layout_metadata WHERE layout_id=$orig_layid");
      if ($Mlq->hasdata) {
        do {
          $Mins = new dbinsert("ax_layout_metadata");
          $Mins->set("layout_id", $this->layout->layoutid);
          $Mins->set("element_id", $Mlq->field("element_id"));
          $Mins->set("schema_name", $Mlq->field("schema_name"));
          if ($Mlq->field("enc_scheme_id" != "")) {
            $Mins->set("enc_scheme_id", $Mlq->field("enc_scheme_id"));
          }
          $Mins->set("metadata_value", $Mlq->field("metadata_value"));
          $Mins->set("linked_uri", $Mlq->field("linked_uri"));
          $Mins->set("language", $Mlq->field("language"));
        } while ($Mlq->get_next());
      }

      // Update changes to layout table..
      $this->layout->put();
    }
  } // replicate
  // ....................................................................
  /**
  * Delete the hosted layout from the database. Afterwards, the current object
  * still exists as it was before this method was executed, but the
  * $this->layout->exists flag will have been reset to false.
  */
  function delete() {
    if ($this->layout->exists) {
      $external_transaction = transaction_open();
      if (!$external_transaction) {
        start_transaction();
      }
      // Remove blocks first..
      $lcq = new dbselect("ax_block");
      $lcq->fieldlist("block_id");
      $lcq->where("layout_id=" . $this->layout->layoutid);
      $lcq->execute();
      if ($lcq->hasdata) {
        do {
          $blockid = $lcq->field("block_id");
          $b = new block($blockid);
          $b->delete();
        } while ($lcq->get_next());
      }
      // Remove any layout set reference, but only if it
      // refers to the hosted layout alone. If versions exist then
      // it will refer to all versions, so don't remove..
      $q  = "SELECT l.layout_id";
      $q .= "  FROM ax_layout_set ls, ax_layout l";
      $q .= " WHERE ls.layout_name='" . $this->layout->layout_name . "'";
      $q .= "   AND l.layout_name=ls.layout_name";
      $q .= "   AND l.layout_id <> " . $this->layout->layoutid;
      $chk = dbrecordset($q);
      if ($chk->rowcount == 0) {
        $ldel = new dbdelete("ax_layout_set");
        $ldel->where("layout_name='" . $this->layout->layout_name . "'");
        $ldel->execute();
      }
      $ldel = new dbdelete("ax_layout");
      $ldel->where("layout_id=" . $this->layout->layoutid);
      $ldel->execute();
      if (!$external_transaction) {
        commit();
      }
      $this->layout->exists = false;
    }
  } // delete
  // ....................................................................
  /**
  * Render the layout editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function editform() {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $perm_groups, $perm_perms;

    $pwidth = "150px";

    // For button form submits with mode setting..
    $RESPONSE->add_script(
        "function layoutgo(mode) {\n"
      . " document.forms." . $this->layout->layoutfm . ".layoutmode.value=mode;\n"
      . " document.forms." . $this->layout->layoutfm . ".submit();\n"
      . "}\n"
      );

    // Layout table copy we will use..
    $Tlay = $this->layout->layout_table;
    $Tlay->setstyle("border-width:1px;border-style:dotted;border-color:#0000ff;padding:10px;");
    $Tlay->setborder(1);

    // Make an Axyl colour combobox..
    $ss = new stylesheet($RESPONSE->site_docroot . $RESPONSE->head->stylesheet);
    $colourCombo = new form_combofield("colours");
    $colourCombo->setclass("axcombo");
    $colourCombo->setstyle("width:$pwidth;");
    $colourCombo->additem("", "default colour");
    $TotColours = defaulted($ss->style("axylpalette", "total_colours"), 0);
    if ($TotColours > 0) {
      for ($c = 1; $c <= $TotColours; $c++) {
        $colour_style = $ss->style("axylpalette", "colour_$c");
        $colour_bits = explode(",", $colour_style);
        if (isset($colour_bits[0]) && isset($colour_bits[1])) {
          $colourCombo->additem($colour_bits[0], $colour_bits[1]);
        }
      }
    }
    // Initialise content..
    $s = "";

    // Buttons..
    $bnew     = new form_imagebutton("_new",      "", "", "$LIBDIR/img/_new.gif",     "New layout",     42, 15);
    $bdone    = new form_imagebutton("_done",     "", "", "$LIBDIR/img/_done.gif",    "Done",           57, 15);
    $bsave    = new form_imagebutton("_save",     "", "", "$LIBDIR/img/_save.gif",    "Save layout",    57, 15);
    $bpublish = new form_imagebutton("_publish",  "", "", "$LIBDIR/img/_publish.gif", "Publish",        57, 15);
    $brevert  = new form_imagebutton("_revert",   "", "", "$LIBDIR/img/_revert.gif",  "Revert",         57, 15);
    $bsplit   = new form_imagebutton("_split",    "", "", "$LIBDIR/img/_split.gif",   "Split",          15, 15);
    $bmgrow   = new form_imagebutton("_mergerow", "", "", "$LIBDIR/img/_arrowD.gif",  "Merge rows",     11, 11);
    $bmgcol   = new form_imagebutton("_mergecol", "", "", "$LIBDIR/img/_arrowR.gif",  "Merge columns",  11, 11);
    $bmgall   = new form_imagebutton("_mergeall", "", "", "$LIBDIR/img/_arrowRR.gif", "Merge all cols", 11, 11);
    $bdelete  = new form_imagebutton("_delete",   "", "", "$LIBDIR/img/_delete.gif",  "Delete block",   57, 15);
    $binsrow  = new form_imagebutton("_insrow",   "", "", "$LIBDIR/img/_chevR.gif",   "Insert row",     13,  9);
    $binscol  = new form_imagebutton("_inscol",   "", "", "$LIBDIR/img/_chevD.gif",   "Insert column",   9, 13);
    $bredx    = new form_imagebutton("_redx",     "", "", "$LIBDIR/img/_redx.gif",    "Delete",          9,  9);

    // Confirmation stuff
    $bsave->set_onclick("layoutgo('save');");
    $bdelete->set_confirm_text("Delete this block?");
    $bnew->set_confirm_text("This will DELETE the whole layout. Are you sure?");
    $bpublish->set_confirm_text("This will make this pending layout LIVE. Are you sure?");
    $brevert->set_confirm_text("This will revert to the previous Live layout. Current pending layout will be lost. Are you sure?");

    // Version access combo
    $versionCombo = new form_combofield("version_id");
    $versionCombo->setclass("axcombo");

    // Hidden fields..
    $layfm = new form_hiddenfield("edit_layoutform", $this->layout->layoutfm);
    $mode  = new form_hiddenfield("layoutmode",      $this->layout->mode);
    $elid  = new form_hiddenfield("edit_layoutid",   $this->layout->layoutid);
    $lver  = new form_hiddenfield("layout_version",  $this->layout->version);
    $merge = new form_hiddenfield("layout_action");

    // ..................................................................
    // KEYFIELD and RECORD MAINTAINER
    // Privileges listbox
    // Declared here so we can create the maintainer and register buttons
    // before they are used in the form.
    //
    // This is the keyfield listbox which controls the maintainance
    // process. It lists all records being maintained..
    $privs_listbox = new form_combofield("priv_group_id");
    $privs_listbox->setclass("axlistbox");
    // Make a new privs record maintainer, and attach the buttons..
    $privs_maintainer = new recmaintainer($this->layout->layoutfm, $privs_listbox, "privs_");
    // Register privs maintainer buttons..
    $privs_maintainer->register_button("save", $bsave); // Save button is common to all
    $privs_listbox->setstyle("width:$pwidth");
    $privs_listbox->size = 7;

    // Control table..
    $Ted = new table($this->layout->layoutfm);

    // ..................................................................
    // TOOLBAR
    $toolbar = array();

    // TOOLBAR: DONE button
    $toolbar[] = $bdone;

    // TOOLBAR: PUBLISH or REVERT buttons
    if ($RESPONSE->ismemberof_group_in("Editor")) {
      switch ($this->layout->version) {
        case VERSION_PENDING:
          $toolbar[] = $bpublish;
          break;
        case VERSION_LIVE:
          $toolbar[] = $brevert;
          break;
      } // switch
    }

    // TOOLBAR: NEW button
    if ($this->layout->user_can_edit()) {
      $toolbar[] = $bnew;
    }

    // TOOLBAR: SAVE button
    if ($this->layout->user_can_edit()) {
      $toolbar[] = $bsave;
    }

    // TOOLBAR: HEADING LABEL
    switch ($this->layout->version) {
      case VERSION_PENDING:   $hdg = "PENDING";  break;
      case VERSION_LIVE:      $hdg = "LIVE";     break;
      case VERSION_PREVIOUS:  $hdg = "PREVIOUS"; break;
      case VERSION_UNDEFINED: $hdg = "EDITING";  break;
      default: $hdg = "VERSION " . $this->layout->version;
    } // switch

    // TOOLBAR: Table
    $Tbar = new table("toolbar");
    //$Tbar->setwidth(500);
    $Tbar->tr("axtitle");
    $Tbar->th("<b>$verhdg</b> [" . $this->layout->layout_name . "]", "axtitle");
    $tools = "";
    foreach ($toolbar as $tool) {
      $tools .= $tool->render();
    }
    $Tbar->th($tools, "axtitle");
    $Tbar->th_css("text-align:right");
    $Ted->thead();
    $Ted->tr("axtitle");
    $Ted->td( $Tbar->render(), "axtitle" );
    $Ted->td_alignment("", "top");

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>LAYOUT SETTINGS</b>", "axhdg");
    $Ted->td_css("text-align:center");
    $Ted->td_colspan(2);
    // ..................................................................
    // Layout properties..
    $Trows = $Tlay->rowcount();
    $Tcols = $Tlay->cellcount();

    $gentxt = new form_textfield();
    $gentxt->clearstyle();
    $gentxt->setclass("axtxtbox");
    $gentxt->setstyle("width:35px;text-align:center;");

    $gentxt->setvalue($Tcols);
    $colsF = $gentxt->render("layout_cols");

    $gentxt->setvalue($Trows);
    $rowsF = $gentxt->render("layout_rows");

    $gentxt->setvalue($Tlay->cellpadding);
    $padF = $gentxt->render("layout_padding");

    $colourCombo->setvalue($Tlay->bgcolor);

    $showlastmod = new form_checkbox("show_last_modified");
    $showlastmod->checked = $this->layout->show_last_modified;

    $lastmodCombo = new form_combofield("format_last_modified");
    $lastmodCombo->setclass("axcombo");
    $lastmodCombo->setstyle("width:$pwidth;");
    $lastmodCombo->additem(NICE_FULLDATETIME,        "Mar 3rd 1999 1:30pm");
    $lastmodCombo->additem(NICE_DATE,                "Mar 3rd 1999");
    $lastmodCombo->additem(DAY_AND_DATE,             "Friday, 20th July 2001");
    $lastmodCombo->additem(DISPLAY_DATE_ONLY,        "31/12/1999");
    $lastmodCombo->additem(DISPLAY_DATE_FORMAT,      "31/12/1999 23:59");
    $lastmodCombo->additem(DISPLAY_TIMESTAMP_FORMAT, "31/12/1999 23:59:59");
    $lastmodCombo->additem(NICE_TIME_ONLY,           "1:30pm");
    $lastmodCombo->setvalue($this->layout->format_last_modified);

    $Tprop = new table("props");

    // Multi-language selector, or hidden field..
    $Tprop->tr("axbglite");
    if ($RESPONSE->multilang) {
      $langsCombo = new form_combofield("language");
      $langsCombo->setclass("axcombo");
      $langsCombo->setstyle("width:$ewidth;");
      $LQ = dbrecordset("SELECT * FROM ax_language ORDER BY display_order");
      $langsCombo->add_querydata($LQ, "lang_id", "lang_desc");
      $Tprop->td("Layout language:", "axfg");
    }
    else {
      $langsCombo = new form_hiddenfield("language");
      $Tprop->td("&nbsp;");
    }
    $langsCombo->setvalue($this->layout->language);
    $Tprop->td( $langsCombo->render() );
    $Tprop->td_colspan(3);

    $Tprop->setpadding(2);
    $Tprop->setstyle("padding-left:5px;padding-right:5px;");
    $Tprop->tr("axbgdark");
    $Tprop->td("Cols x Rows:", "axfg");
    $Tprop->td( $colsF . "&nbsp;x&nbsp;" . $rowsF );

    $Tprop->td("Padding:", "axfg");
    $Tprop->td_alignment("right");
    $Tprop->td( $padF );
    $Tprop->td_alignment("right");

    $Tprop->tr("axbglite");
    $Tprop->td("Background colour:", "axfg");
    $Tprop->td($colourCombo->render("background_colour"));
    $Tprop->td();
    $Tprop->td_colspan(2);

    $Tprop->tr("axbgdark");
    $Tprop->td("Show last modified:", "axfg");
    $Tprop->td($showlastmod->render());
    $Tprop->td_colspan(3);

    $Tprop->tr("axbglite");
    $Tprop->td("Last mod. format:", "axfg");
    $Tprop->td($lastmodCombo->render());
    $Tprop->td_colspan(3);

    $gentxt->setvalue($this->layout->prefix_last_modified);
    $gentxt->clearstyle();
    $gentxt->setstyle("width:$pwidth");
    $Tprop->tr("axbgdark");
    $Tprop->td("Last mod. prefix:", "axfg");
    $Tprop->td($gentxt->render("prefix_last_modified"));
    $Tprop->td_colspan(3);

    $gentxt->setvalue($this->layout->index_category);
    $gentxt->clearstyle();
    $gentxt->setstyle("width:$pwidth");
    $Tprop->tr("axbglite");
    $Tprop->td("Index category:", "axfg");
    $Tprop->td($gentxt->render("index_category"));
    $Tprop->td_colspan(3);

    $gentxt->setvalue($this->layout->layout_style);
    $gentxt->clearstyle();
    $gentxt->setstyle("width:$pwidth");
    $Tprop->tr("axbgdark");
    $Tprop->td("CSS style:", "axfg");
    $Tprop->td($gentxt->render("layout_style"));
    $Tprop->td_colspan(3);

    $gentxt->setvalue($this->layout->layout_table->width);
    $gentxt->clearstyle();
    $gentxt->setstyle("width:60;text-align:right;");
    $Tprop->tr("axbglite");
    $Tprop->td("Table width:", "axfg");
    $Tprop->td($gentxt->render("table_width"));
    $Tprop->td_colspan(3);

    // These properties only appear if we have plain cells..
    if ($this->layout->tot_plain > 0) {
      $tstylesCombo = new form_combofield("table_style");
      $tstylesCombo->setclass("axcombo");
      $tstylesCombo->setstyle("width:$cbowidth;");
      $tstylesCombo->additem("", "default style");
      $Totstyles = defaulted($ss->style("axyl_tablestyles", "total_styles"), 0);
      if ($Totstyles > 0) {
        for ($c = 1; $c <= $Totstyles; $c++) {
          $tstyle_style = $ss->style("axyl_tablestyles", "style_$c");
          $tstyle_bits = explode(",", $tstyle_style);
          if (isset($tstyle_bits[0]) && isset($tstyle_bits[1])) {
            $tstylesCombo->additem($tstyle_bits[0], $tstyle_bits[1]);
          }
        }
      }
      $tstylesCombo->setvalue($this->layout->layout_table->class);
      $Tprop->tr("axbgdark");
      $Tprop->td("Table style:");
      $Tprop->td($tstylesCombo->render());
      $Tprop->td_colspan(3);

      $autoj = new form_checkbox("table_autojustify");
      $autoj->setclass("axchkbox");
      $autoj->checked = $this->layout->layout_table->autojustify;
      $Tprop->tr("axbglite");
      $Tprop->td("Auto-justify:");
      $Tprop->td($autoj->render());
      $Tprop->td_colspan(3);

      $rowstr = new form_checkbox("table_rowstripes");
      $rowstr->setclass("axchkbox");
      $rowstr->checked = (implode(",", $this->layout->layout_table->rowstripes) != "");
      $Tprop->tr("axbgdark");
      $Tprop->td("Row striping:");
      $Tprop->td($rowstr->render());
      $Tprop->td_colspan(3);
    }

    // Render properties
    $Ted->tr("axbglite");
    $Ted->td( $Tprop->render() );
    $Ted->td_alignment("", "top");

    // ..................................................................
    // EDITING PRIVILEGES

    foreach($this->layout->privilege_groups as $group_id => $group_desc) {
      $privs_listbox->additem($group_id, $group_desc);
      // Populate maintainer data. The maintainer add_record method
      // requires an associative array keyed on listbox key id..
      $rec = array(
              "priv_group_id" => $group_id,
              "priv_editor"   => (isset($this->layout->privileges["editor|$group_id"])) ? "t" : "f",
              "priv_author"   => (isset($this->layout->privileges["author|$group_id"])) ? "t" : "f",
              "priv_entry"    => (isset($this->layout->privileges["entry|$group_id"]))  ? "t" : "f",
              );
      $privs_maintainer->add_record($group_id, $rec);
    }
    // Now set the defaults for each of the fields. These are
    // necessary for when a new record is created..
    $defaults = array(
              "priv_group_id" => 0,
              "priv_editor"   => "f",
              "priv_author"   => "f",
              "priv_entry"    => "f"
              );
    $privs_maintainer->add_defaults($defaults);

    $Tpriv = new table("layprivs");
    $Tpriv->setpadding(2);
    $Tpriv->setwidth("500");
    $Tpriv->setcss("margin-left:100px");
    $Tpriv->tr();
    $blurb  = "To change the access privileges for layout and block content editing for ";
    $blurb .= "this layout, highlight the group and select one or more privileges for it.";
    $blurb .= "Repeat for each group as required, then click the save button.";
    $Tpriv->td($blurb);
    $Tpriv->td_colspan(2);
    $Tpriv->tr();

    // The listbox field..
    $Tpriv->tr();
    $Tpriv->td( $privs_listbox->render() );
    $Tpriv->td_colspan(2);

    $genchk = new form_checkbox("");
    $genchk->setclass("axchkbox");
    $genchk->setvalue("yes");

    // Checkbox for each privilege
    // Editor
    $Fld = $genchk;
    $Fld->setname("priv_editor");
    $Fld->checked = isset($this->layout->privileges["editor|$group_id"]);
    $privs_maintainer->register_field($Fld);
    $Tpriv->tr("axbglite");
    $Tpriv->td($Fld->render());
    $Tpriv->td("Editor/Publisher");

    // Author
    $Fld = $genchk;
    $Fld->setname("priv_author");
    $Fld->checked = isset($this->layout->privileges["author|$group_id"]);
    $privs_maintainer->register_field($Fld);
    $Tpriv->tr("axbgdark");
    $Tpriv->td($Fld->render());
    $Tpriv->td("Author of content");

    // Entry
    $Fld = $genchk;
    $Fld->setname("priv_entry");
    $Fld->checked = isset($this->layout->privileges["entry|$group_id"]);
    $privs_maintainer->register_field($Fld);
    $Tpriv->tr("axbglite");
    $Tpriv->td($Fld->render());
    $Tpriv->td("Data entry");

    $Tpriv->set_width_profile("5%,95%");

    // Render privileges
    $Ted->tr("axhdg");
    $Ted->td("<b>LAYOUT PRIVILEGE SETTINGS</b>", "axhdg");
    $Ted->td_css("text-align:center");
    $Ted->td_colspan(2);
    $Ted->tr("axbgdark");
    $Ted->td( $Tpriv->render() . $privs_maintainer->render() );
    $Ted->td_alignment("", "top");
    $Ted->td_colspan(2);

    // ..................................................................
    // PLAIN-CELL PERMISSIONS
    if ($this->layout->tot_plain > 0) {
      $groups = new form_combofield("perm_groups", "", $perm_groups);
      $groups->multiselect = true;
      $groups->setclass("axlistbox");
      $groups->setstyle("width:150px;");
      $groups->set_size(5);
      $gps = dbrecordset("SELECT * FROM ax_group");
      if ($gps->hasdata) {
        do {
          $groups->additem($gps->field("group_desc"));
        } while ($gps->get_next());
      }
      $perms = new form_combofield("perm_perms", "", $perm_perms);
      $perms->multiselect = true;
      $perms->setclass("axlistbox");
      $perms->setstyle("width:150px;");
      $perms->set_size(5);
      $perms->additem(PERM_READ,   "READ");
      $perms->additem(PERM_UPDATE, "UPDATE");
      $perms->additem(PERM_CREATE, "CREATE");
      $perms->additem(PERM_DELETE, "DELETE");
      $perms->additem(PERM_NONE,   "NONE");

      $btnset = new form_imagebutton("_perm_set");
      $btnset->setimage("$LIBDIR/img/_set.gif", "Set permissions");
      $btnunset = new form_imagebutton("_perm_unset");
      $btnunset->setimage("$LIBDIR/img/_unset.gif", "Unset permissions");

      $Tperm = new table("perms");
      $Tperm->setpadding(2);
      $Tperm->setwidth("500");
      $Tperm->setcss("margin-left:100px");
      $Tperm->tr();
      $blurb  = "To set permissions for plain cells which have been selected below, choose ";
      $blurb .= "one or more groups, and select one or more access methods then click ";
      $blurb .= "the Set button. Unset clears permissions from the selected cells.";
      $Tperm->td($blurb);
      $Tperm->td_colspan(5);
      $Tperm->tr();
      $Tperm->td( "Group(s) " );
      $Tperm->td_alignment("", "top");
      $Tperm->td( $groups->render() );
      $Tperm->td_alignment("center", "top");
      $Tperm->td( " permitted to " );
      $Tperm->td_alignment("center", "top");
      $Tperm->td( $perms->render() );
      $Tperm->td_alignment("center", "top");
      $Tperm->td( $btnset->render() . "<br>" . $btnunset->render() );
      $Tperm->td_alignment("right", "top");

      // Render perms
      $Ted->tr("axhdg");
      $Ted->td("<b>PLAIN-CELL PERMISSIONS</b>", "axhdg");
      $Ted->td_css("text-align:center");
      $Ted->td_colspan(2);
      $Ted->tr("axbgdark");
      $Ted->td( $Tperm->render() );
      $Ted->td_alignment("", "top");
      $Ted->td_colspan(2);
    }

    // ..................................................................
    // WIDTH PROFILE
    if ($Tcols > 1) {
      $Ted->tr("axhdg");
      $Ted->td("<b>LAYOUT COLUMN WIDTHS PROFILE</b>", "axhdg");
      $Ted->td_css("text-align:center");
      $Ted->td_colspan(2);

      $prof = $Tlay->get_width_profile();
      $Tprf = new table("prfcols");
      $Tprf->setstyle("padding:10px;");
      $Tprf->tr();
      $gentxt->setcss("");
      $gentxt->setclass("axtxtbox");
      $gentxt->setcss("width:50px;text-align:center;");
      foreach ($prof as $width) {
        $gentxt->setvalue($width);
        $Tprf->td( $gentxt->render("width_profile[]") );
        $Tprf->td_alignment("center");
      }
      $Tprop = new table("profile");
      $Tprop->rowstripes("axyl_rowstripe_dark,axyl_rowstripe_lite");
      $Tprop->tr();
      $Tprop->td( $Tprf->render() );
      // Insert it in main table..
      $Ted->tr("axbglite");
      $Ted->td( $Tprop->render() );
      $Ted->td_alignment("", "top");
      $Ted->td_colspan(2);
    }

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>LAYOUT PLANNER</b>", "axhdg");
    $Ted->td_css("text-align:center");
    $Ted->td_colspan(2);

    // ..................................................................
    // BULK SETTING for CELL DEFINITION
    if ($this->layout->tot_empty > 0) {
      $bulkbtn = new form_imagebutton("_bulk_set");
      $bulkbtn->setimage("$LIBDIR/img/_set.gif", "Set all cells");
      $bulkbtn->set_onclick("bulk_set()");
      $bulkset = new form_combofield("layout_bulksetting");
      $bulkset->setclass("axcombo");
      $bulkset->setstyle("width:70px;");
      $bulkset->additem(EMPTY_CELL);
      $bulkset->additem(BLOCK_CONTENT, "Block");
      $bulkset->additem(WYSIWYG_EDITOR, "Wysiwyg");
      $bulkset->additem(PLAIN_CELL, "Cell");
      // Add the script it needs..
      $RESPONSE->body->add_script(
        "function bulk_set() {\n"
      . " var i,j;\n"
      . " ix=document.forms." . $this->layout->layoutfm . ".layout_bulksetting.selectedIndex;\n"
      . " if (ix != -1) {\n"
      . "  var bulkval=document.forms." . $this->layout->layoutfm . ".layout_bulksetting[ix].value;\n"
      . "  for (i=0; i < document." . $this->layout->layoutfm . ".length; i++) {\n"
      . "   var e=document." . $this->layout->layoutfm . ".elements[i];\n"
      . "   if (e.name == 'layout_newcell[]') {\n"
      . "    for (j=0; j < e.length; j++) {\n"
      . "     curval = e[j].value.substr(0,1);\n"
      . "     if (curval == bulkval) {\n"
      . "      e.selectedIndex = j;\n"
      . "      break;\n"
      . "     }\n"
      . "    }\n"
      . "   }\n"
      . "  }\n"
      . " }\n"
      . "}\n"
      );
      // Insert it in main table..
      $Ted->tr("axbgdark");
      $Ted->td( "Set all to&nbsp;" . $bulkset->render() . "&nbsp;" . $bulkbtn->render() );
      $Ted->td_alignment("", "top");
    }

    // Double-clicking left-most checkboxes..
    if ($this->layout->tot_plain > 0) {
      $RESPONSE->body->add_script(
        "function chkrow(fld) {\n"
      . " var newchk = !fld.checked;\n"
      . " var v = fld.value.split('|');\n"
      . " var row = v[0];\n"
      . " for (var i=0; i < document." . $this->layout->layoutfm . ".length; i++) {\n"
      . "  var e=document." . $this->layout->layoutfm . ".elements[i];\n"
      . "  if (e.name == 'layout_cellsel[]') {\n"
      . "   v = e.value.split('|');\n"
      . "   r = v[0];\n"
      . "   if (r == row) {\n"
      . "     e.checked = newchk;\n"
      . "   }\n"
      . "  }\n"
      . " }\n"
      . "}\n"
      );
    }

    // ..................................................................
    // The Layout Table..

    // This submits a generic layout request..
    $RESPONSE->body->add_script(
        "function layoutAction(val) {\n"
      . "  document.forms." . $this->layout->layoutfm . ".layout_action.value=val;\n"
      . "  document.forms." . $this->layout->layoutfm . ".submit();\n"
      . "}\n"
      );

    // Controls table..
    $Tti = new table("controls");
    $Tti->setstyle("vertical-align:top");
    $Tti->tbody("font-size:8pt;vertical-align:top;background:white;");
    // Row & col insert buttons
    $Tti->tr();
    $Tti->td("tt_00");
    $Tti->td_height(22);
    //$Tti->td_alignment("", "top");
    $Tti->td("tt_01");
    $Tti->td("tt_02");
    $Tti->td_alignment("right");
    // Merge buttons etc.
    $Tti->tr();
    $Tti->td("tt_10");
    $Tti->td("tt_11");
    $Tti->td_alignment("center");
    $Tti->td("tt_12");
    $Tti->td_alignment("right");
    // Permissions
    $Tti->tr();
    $Tti->td("tt_20");
    $Tti->td_colspan(3);
    $Tti->td_alignment("tt_20_align");
    $Tti->close_group();
    $Tti->set_width_profile("15%,70%,15%");
    $Ttis = $Tti->render();

    // Combo for each cell..
    $ccre = new form_combofield("layout_newcell[]");
    $ccre->setclass("axcombo");
    $ccre->setstyle("width:70px;");

    // Checkbox for each cell..
    $cchk = new form_checkbox("layout_cellsel[]");
    $cchk->setclass("axchkbox");

    // Populate our layout table with blocks..
    for ($r = 0; $r < $this->layout->tot_rows; $r++) {
      for ($c = 0; $c < $this->layout->tot_cols; $c++) {
        if ($Tlay->cell_exists($r, $c)) {

          // Get existing cell for population with controls etc..
          $cell = $Tlay->get_cell($r, $c);

          // If no block yet, offer the create checkbox and the
          // various merge/split controls..
          $rowmerge_controls = "";
          $colmerge_controls = "";
          $other_controls = "";
          $Tt = $Ttis;
          $Tt_cells = array();
          for ($ttr=0; $ttr < 3; $ttr++) {
            for ($ttc=0; $ttc < 3; $ttc++) {
              $ttid = "tt_" . $ttr . $ttc;
              $Tt_cells[$ttid] = "";
            }
          }

          // Add row & column modifying buttons..
          if ($c == 0) {
            $binsrow->set_onclick("layoutAction('insrow|$r|$c')");
            $btns = $binsrow->render();
            if ($this->layout->tot_rows > 1) {
              $bredx->set_onclick("layoutAction('delrow|$r|$c')");
              $bredx->settitle("Delete row");
              $btns .= "<br>" . $bredx->render();
            }
            $Tt_cells["tt_00"] = $btns;
          }
          if ($r == 0) {
            $binscol->set_onclick("layoutAction('inscol|$r|$c')");
            $btns = $binscol->render();
            if ($this->layout->tot_cols > 1) {
              $bredx->set_onclick("layoutAction('delcol|$r|$c')");
              $bredx->settitle("Delete column");
              $btns .= "<br>" . $bredx->render();
            }
            $Tt_cells["tt_02"] = $btns;
          }

          // If not defined, then offer defining controls..
          if (!isset($this->layout->layout_blocks["$r|$c"])) {

            // Cell creation checkbox..
            $ccre->clearitems();
            $ccre->additem(EMPTY_CELL);
            $ccre->additem(BLOCK_CONTENT  . "|$r|$c", "Block");
            $ccre->additem(WYSIWYG_EDITOR . "|$r|$c", "Wysiwyg");
            $ccre->additem(PLAIN_CELL     . "|$r|$c", "Plain");
            $other_controls .= $ccre->render();

            // Row merge controls
            if ($cell->colspan == 1 && $r < ($Tlay->visible_cellsincol($c) - 1)) {
              $ok = true;
              if ($Tlay->cell_exists($r + 1, $c)) {
                $nextcell = $Tlay->get_cell($r + 1, $c);
                if ($nextcell->rowspan > 1) $ok = false;
              }
              if ($ok) {
                $bmgrow->set_onclick("layoutAction('merge|row|$r|$c')");
                $rowmerge_controls .= $bmgrow->render();
              }
            }
            if ($cell->rowspan > 1) {
              $bsplit->set_onclick("layoutAction('split|row|$r|$c')");
              $rowmerge_controls .= $bsplit->render();
            }
            if ($rowmerge_controls == "") $rowmerge_controls = "&nbsp;";
            $Tt_cells["tt_10"] = $rowmerge_controls;

            // Column merge controls
            if ($cell->rowspan == 1 && $c < ($Tlay->visible_cellsinrow($r) - 1)) {
              $ok = true;
              if ($Tlay->cell_exists($r, $c + 1)) {
                $nextcell = $Tlay->get_cell($r, $c + 1);
                if ($nextcell->rowspan > 1) $ok = false;
              }
              if ($ok) {
                $bmgcol->set_onclick("layoutAction('merge|col|$r|$c')");
                $colmerge_controls .= $bmgcol->render();
              }
            }
            if ($cell->colspan > 1) {
              $bsplit->set_onclick("layoutAction('split|col|$r|$c')");
              $colmerge_controls .= $bsplit->render();
            }
            if ($colmerge_controls == "") $colmerge_controls = "&nbsp;";
            $Tt_cells["tt_12"] = $colmerge_controls;

            // Bulk column merge control..
            if ($c == 0) {
              $row = $this->layout->layout_table->get_row($r);
              if ($row && !$row->has_colspans()) {
                $bmgall->set_onclick("layoutAction('merge|allcols|$r|$c')");
                $Tt_cells["tt_20"] = $bmgall->render();
                $Tt = str_replace("tt_20_align", "right", $Tt);
              }
            }
          }
          else {
            // Cell is occupied, so we offer the delete option..
            $blockid = $this->layout->layout_blocks["$r|$c"];
            if ($this->layout->user_can_edit()) {
              if ($blockid != 0) {
                // Content managed cell..
                $bdelete->set_onclick("layoutAction('deletecell|$r|$c')");
                $other_controls .= ($cell->celltype == "w") ? "{wysiwyg}" : "{block}";
                $other_controls .= "<br>" . $bdelete->render();
              }
              else {
                // Plain cell..
                $bdelete->set_onclick("layoutAction('deletecell|$r|$c')");
                $other_controls .= "{plain}<br>" . $bdelete->render();
              }
            }
          }

          // Insert permissions controls & info..
          if (isset($cell->access)) {
            $cchk->setvalue("$r|$c");
            if ($c == 0) {
              $cchk->set_ondblclick("chkrow(this,'" . $this->layout->layoutfm . "')");
            }
            $other_controls .= "<br>" . $cchk->render();
            $Tt_cells["tt_20"] = $cell->access->dump();
            $Tt = str_replace("tt_20_align", "center", $Tt);
          }

          // Insert the miscellaneous controls..
          $Tt_cells["tt_11"] = $other_controls;

          // Plug in the cell content..
          foreach ($Tt_cells as $cellid => $cellcontent) {
            $Tt = str_replace($cellid, $cellcontent, $Tt);
          }
          $cell->setcontent( $Tt );
          $cell->setcontentcss( "vertical-align:top" );
          $Tlay->set_cell($r, $c, $cell);
        }
      }
    }
    $Ted->tr();
    $Ted->td( $Tlay->render() );
    $Ted->td_alignment("", "top");

    if ($this->layout->show_last_modified && $this->layout->last_modified != "") {
      $Ted->tr();
      $Ted->td($this->layout->last_modified, "axyl_lastmod");
    }

    $Ted->tr("axfoot");
    $Ted->td("", "axfoot");

    // ..................................................................
    // Finish off..
    $s .= "<form name=\"" . $this->layout->layoutfm . "\" method=\"post\">\n";
    $s .= $Ted->render()
        . $layfm->render()
        . $mode->render()
        . $elid->render()
        . $merge->render()
        . $lver->render();
    $s .= "</form>\n";

    debug_trace();
    // Return the html..
    return $s;
  } // editform
} // layouteditor class

// ----------------------------------------------------------------------
?>