/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Base process class, and general base classes.
 *
 *	by Tony Sideris	(06:36AM Aug 11, 2001)
 *================================================*/
#include "arson.h"

#include <qstringlist.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qdom.h>
#include <qdir.h>

#include <unistd.h>
#include <errno.h>

#include <kmessagebox.h>
#include <kregexp.h>
#include <klocale.h>

#ifdef ARSON_KDE3
#	include <kguiitem.h>
#endif

#include "encoderopts.h"
#include "tempfile.h"
#include "processmgr.h"
#include "process.h"
#include "konfig.h"
#include "isofs.h"
#include "logwnd.h"
#include "cdripper.h"
#include "mainwnd.h"

#ifdef HAVE_CONFIG_H
#	include "../config.h"
#	if STDC_HEADERS
#		include <string.h>
#	else
#		if !HAVE_STRCHR
#			define strrchr rindex
#			define strchr index
#		endif
		char *strchr(), *strrchr();
#		if !HAVE_MEMCPY
#			define memmove(d, s, n) bcopy ((s), (d), (n))
#			define memcpy(d, s, n) bcopy ((s), (d), (n))
#		endif
#	endif
#endif	//	HAVE_CONFIG_H

/*========================================================*/
/*	Make damn sure i'm not tempted to use the global
 *	configuration. All user congfigable stuff must
 *	come in through opts. (but only in debug mode
 *	so that --enable-final still will work).
 */
#ifdef ARSONDBG
#	undef ACONFIG
#endif	//	ARSONDBG

/*========================================================*/

ArsonProcessBase::ArsonProcessBase (const QString &type, RunMode runmode,
	Communication comm, int program, const ArsonConfig &cfg)
	: KProcess(), m_strType(type),
	m_runmode(runmode), m_comm(comm),
	m_program(program),
	m_outOut(NULL),
	m_outErr(NULL),
	m_bBuffer(true)
{
	QObject::connect(
		this, SIGNAL(receivedStdout(KProcess*,char*,int)),
		this, SLOT(slotGotStdout(KProcess*,char*,int)));
	QObject::connect(
		this, SIGNAL(receivedStderr(KProcess*,char*,int)),
		this, SLOT(slotGotStderr(KProcess*,char*,int)));

	if (program != ArsonConfig::PROGRAM_UNKNOWN)
	{
		const char *cmd = cfg.program(program)->path();

		if (cmd && *cmd)
			THIS << cmd;
	}

	ARSON_INSTANCE_INCR("ArsonProcessBase");
}

ArsonProcessBase::~ArsonProcessBase (void)
{
	ARSON_INSTANCE_DECR("ArsonProcessBase");
	
	delete m_outOut;
	delete m_outErr;
}

/*========================================================*/

ArsonProcessBase &ArsonProcessBase::appendArgList (const QStringList &sl)
{
	for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
		 it != end; ++it)
		THIS << (*it);

	return *this;
}

/*========================================================*/

void ArsonProcessBase::setOutput (Communication comm, KRegExp *arr, size_t count)
{
	if (comm & Stdout)
	{
		if (!(m_comm & Stdout))
			m_comm |= Stdout;

		if (m_outOut)
			delete m_outOut;

		m_outOut = new Output(arr, count);
	}

	if (comm & Stderr)
	{
		if (!(m_comm & Stderr))
			m_comm |= Stderr;

		if (m_outErr)
			delete m_outErr;

		m_outErr = new Output(arr, count);
	}

	if (comm & Stdin)
		m_comm |= Stdin;
}

/*========================================================*/
/*	Defered args are guarenteed to be last (in
 *	the order they were added).
 *========================================================*/

ArsonProcessBase &ArsonProcessBase::deferArgs (const QStringList &args)
{
	m_defered += args;
	return *this;
}

ArsonProcessBase &ArsonProcessBase::deferArg (const QString &arg)
{
	m_defered.append(arg);
	return *this;
}

ArsonProcessBase &ArsonProcessBase::resetDefered (void)
{
	m_defered.clear();
	return *this;
}

/*========================================================*/

bool ArsonProcessBase::execute (void)
{
	bool result = false;
	const char *bin = config().program(m_program)->path();

#ifdef ARSON_KDE3
	const QCString cmd (args()[0]);
#else
	const QCString cmd (args()->first());
#endif	//	ARSON_KDE3

	appendExtra();

	//	Apply the defered args
	appendArgList(m_defered);
	
	Trace("Executing %s process with command:\n%s\n",
		m_strType.latin1(),
		commandString().latin1());

	if (!bin || *bin == 0)
	{
		arsonErrorMsg(
			i18n("Path not set for %1! Please install and configure this program and make sure the path to it is properly set in the Programs page of arson's configuration dialog.")
			.arg(ArsonProgramDef::definition(m_program).programName()));

		return false;
	}

	if (!arsonIsExecutable(cmd))
	{
		if (!cmd.isEmpty())
			arsonErrorMsg(i18n("%1 is not executable!")
				.arg(cmd));

		return false;
	}

	ArsonLogWindow::Wnd()->log(commandString());
	return start(m_runmode, Communication(m_comm));
}

/*========================================================*/

void ArsonProcessBase::invalidate (const QString &errmsg)
{
#ifdef ARSON_KDE3
	clearArguments();
#else
	args()->clear();
#endif	//	ARSON_KDE3

	if (errmsg != QString::null)
		arsonErrorMsg(errmsg);
}

/*========================================================*/

KRegExp *ArsonProcessBase::findRegexp (ArsonProcess::Output *arr, const QString &str, int *pindex)
{
	if (arr)
		for (int index = 0; index < arr->count; index++)
			if (arr->re[index].match(str))
			{
				if (pindex) *pindex = index;
				return arr->re + index;
			}

	return NULL;
}

/*========================================================*/

QCString ArsonProcessBase::bufferInput (const char *ptr, int len)
{
	QCString input (ptr, len + 1);
	Assert(input.length() == len);

	if (buffering())
	{
		int pos;

		/*	If something was previously buffered, add
		 *	it to the start of this read, and clear
		 *	the buffer.
		 */
		if (!m_buffer.isEmpty())
		{
			input.prepend(m_buffer);
			m_buffer = QCString();
		}

		pos = QMAX(input.findRev('\r'), input.findRev('\n'));

		/*	If the current input does not end in a
		 *	complete line, chop off the end, and
		 *	save it for next time.
		 */
		if (pos != -1)
		{
			m_buffer = input.data() + pos + 1;
			input.truncate(pos + 1);
		}
		else
		{
			m_buffer.append(input);
			input = QCString();
		}
/*
		if ((pos = input.findRev('\r')) != -1 ||
			(pos = input.findRev('\n')) != -1)
		{
			int at = pos++;

			while (pos < input.length() && (input[pos] == '\r' || input[pos] == '\n'))
				pos++;

			m_buffer = input.data() + pos;
			input.truncate(at);
		}
		else
		{
			m_buffer.append(input);
			input = QCString();
		}
*/
	}
#ifdef ARSONDBG
	else { Trace("not buffering\n"); }
#endif

	return input;
}

/*========================================================*/

void ArsonProcessBase::handleOutput (const char *ptr, int len, bool error)
{
	const QCString input (bufferInput(ptr, len));
	static const QRegExp reLineSep("[\r\n]+");

	if (!input.isEmpty())
	{
		Output *po = error ? m_outErr : m_outOut;
		const QStringList sl = QStringList::split(reLineSep, input);

		for (uint index = 0; index < sl.count(); ++index)
		{
			KRegExp *pre;
			int event;
			const QString line (sl[index].stripWhiteSpace());

			if (line.isEmpty())
				continue;

//			Trace("line: %s\n", line.latin1());

			if ((pre = findRegexp(po, line, &event)))
			{
				if (error)
					onStderr(event, pre);
				else
					onStdout(event, pre);
			}
			else
				output(line, error);
		}
	}
}

/*========================================================*/

void ArsonProcessBase::slotGotStdout (KProcess *ptr, char *pbuf, int len)
{
	Trace("handling stdout\n");
	handleOutput(pbuf, len, false);
}

void ArsonProcessBase::slotGotStderr (KProcess *ptr, char *pbuf, int len)
{
	handleOutput(pbuf, len, true);
}

/*========================================================*/

QString ArsonProcessBase::quoteFile (const QString &filename)
{
	return filename;
}

void ArsonProcessBase::appendExtra (void)
{
	if (m_program != ArsonConfig::PROGRAM_UNKNOWN)
	{
		QStringList::Iterator it, end;
		QStringList sl = QStringList::split(
			QRegExp("[ \t]+"), config().program(m_program)->params());

		for (it = sl.begin(), end = sl.end(); it != end; ++it)
			appendParam(*it);
	}
}

/*========================================================*/

QString ArsonProcessBase::commandString (void)
{
	QString val;

#ifdef ARSON_KDE3
	const QValueList<QCString> &parms = args();
	int index;

	if (!parms.empty())
		for (index = 1, val = parms[0]; index < parms.count(); ++index)
			val.append(" ").append(parms[index]);
	
#else
	QStrList *pl = args();
	char *ptr = pl ? pl->first() : NULL;

	if (ptr)
		for (val = ptr, ptr = pl->next(); ptr; (ptr = pl->next()))
			val.append(" ").append(ptr);

#endif	//	ARSON_KDE3
	return val;
}

/*========================================================*/

bool ArsonProcessBase::successful (int *res) const
{
	const int code = exitStatus();

	if (res)
		*res = code;

	return (normalExit() && code == 0);
}

/*========================================================*/
/*	Base class for syncronous processes
 *========================================================*/

ArsonUtilityProcess::ArsonUtilityProcess (const ArsonConfig &cfg,
	Communication comm, const QString &name, int program)
	: ArsonProcessBase(name, Block, comm, program, cfg),
	m_config(cfg)
{
	//	Nothing...
}

bool ArsonUtilityProcess::execute (void)
{
	return ArsonProcessBase::execute() && successful();
}

/*========================================================*/
/*	Base process class implementation for asynchronous
 *	process classes
 *========================================================*/

ArsonProcess::ArsonProcess (const QString &typ, ArsonProcessMgr *pMgr, int program)
	: ArsonProcessBase(typ, NotifyOnExit, Communication(Stdout | Stderr), program, pMgr->opts()),
	m_pWriter(pMgr)
{
	Assert(m_pWriter != NULL);
	
	if (!pMgr->opts().program(program)->valid())
		throw ArsonError(
			i18n("Invalid or missing %1 program path!")
			.arg(type()));

	QObject::connect(
		this, SIGNAL(processExited(KProcess*)),
		this, SLOT(slotExited(KProcess*)));
}

ArsonProcess::~ArsonProcess (void)
{
	Trace("%s process object going bye-bye...\n\n",
		type().latin1());
}

/*========================================================*/

const ArsonConfig &ArsonProcess::config (void) const
{
	return m_pWriter->opts();
}

/*========================================================*/

void ArsonProcess::output (const QString &str, bool error)
{
	if (ArsonProcessUI *pUI = ui())
		pUI->output(str, error);
}

ArsonProcessUI *ArsonProcess::ui (void) const
{
	/*	Only allow this process to update the UI if
	 *	this process is the PRIMARY process (more
	 *	than one can be going at a time (like ripper
	 *	and encoder for example)).
	 */
	return (m_pWriter->process() == this)
		? m_pWriter->ui() : NULL;
}

/*========================================================*/

void ArsonProcess::slotExited (KProcess *ptr)
{
	int exitcode;
	ArsonProcessUI *pUI;

	/*	We must directly retrive the UI pointer, and not use
	 *	ui() because we NEED access to the UI, this shit is
	 *	important.
	 */
	if (m_pWriter && (pUI = m_pWriter->ui()))
	{
		const bool updUI = (ui() != NULL);

		if (successful(&exitcode))
		{
			if (updUI)
				pUI->setText(
					i18n("%1 process completed successfully...")
					.arg(type()));
		}
		else
			pUI->setText(
				i18n("%1 process exited with error code %2.")
				.arg(type())
				.arg(exitcode));

		if (updUI)
			pUI->setMaxProgress(0);
	}

	//	Notify the manager controlling this processs
	if (m_pWriter)
		m_pWriter->taskComplete(this);

	Trace("%s process got slotExit (writer=%p, ui=%p) ...\n",
		type().latin1(), m_pWriter,
		m_pWriter ? m_pWriter->ui() : 0);

	if (m_pWriter && m_pWriter->process() == ptr)
		m_pWriter->setProcess(NULL);

	delete ptr;
}

/*========================================================*/

bool ArsonProcess::execute (void)
{
	const bool res = ArsonProcessBase::execute();

	if (!res && m_pWriter)
		m_pWriter->taskComplete(this);

	return res;
}

/*========================================================*/
/*	Base class for all cd-writer processes
 *========================================================*/

ArsonWriterProcess::ArsonWriterProcess (ArsonProcessMgr *pWriter,
	int program, const QString &type)
	: ArsonProcess((type == QString::null) ? i18n("Writer") : type, pWriter, program),
	m_bSimulate(false)
{
	//	Nothing...
}

/*========================================================*/

int ArsonWriterProcess::commSetupDoneC (void)
{
	if (config().is(ArsonConfig::flagRenice))
	{
#ifdef ARSONDBG
		const int dbg =
#endif	//	ARSONDBG

		//	Renice the new process if needed
		nice(-(config().nice()));

#ifdef ARSONDBG
		if (dbg)	Trace("NICE FAILED: %s\n", strerror(errno));
		else 		Trace("RENICED the writer PROCESS...\n");
#endif	//	ARSONDBG
	}

	return ArsonProcess::commSetupDoneC();
}

/*========================================================*/

bool ArsonWriterProcess::execute (void)
{
	if (simulate())
		setupSimulate();

	else if (m_pWriter->opts().getBool(optEject))
		setupEject();

	return ArsonProcess::execute();
}

/*========================================================*/
/*	A base class for all 'cdrecord' processes.
 *	SHOULD CONTAIN NO AUDIO/DATA SPECIFIC CODE!
 *========================================================*/
/*
Last chance to quit, starting dummy write in 1 seconds.
Track 01:   7 of  17 MB written (fifo 100%).
Track 01: Total bytes read/written: 18259968/18259968 (8916 sectors).

Checking for a regexp match for:
Writing  time:   14.314s
Fixating...
*/

KRegExp ArsonCdrecordProcess::events[ArsonCdrecordProcess::_EVENT_MAX] = {
	"^Last chance to quit, starting[a-z ]+write in[ ]+([0-9]+) seconds(.*)",
	"^([0-9]+) seconds",
	"^Starting to write",
	"^Track[ ]+([0-9]+):[ ]+([0-9]+) of[ ]+([0-9]+) MB written",
	"^Track ([0-9]+):[ \t]*([0-9]+) MB written",
	"^Track [0-9]+: Total bytes read/written: ([0-9]+)/([0-9]+)",
	"^Fixating\\.\\.\\.",
	"^Re-load disk and hit",
	"^Blanking",
	"^Writing time:[\t ]+([0-9\\.]+)",
};

/*========================================================*/

ArsonCdrecordProcess::ArsonCdrecordProcess (ArsonProcessMgr *pWriter)
	: ArsonWriterProcess(pWriter, ArsonConfig::PROGRAM_CDRECORD),
	m_partialBase(0),
	m_trackNo(0)
{
	const bool audiomaster = pWriter->opts().getBool(optAudMaster);
	const long speed = pWriter->opts().getLong(optSpeed);

	setOutput(Communication(Stdout | Stdin), events, ARRSIZE(events));

	if (audiomaster && speed > 8)
		THIS << QString("speed=8");
	else if (pWriter->opts().hasOpt(optSpeed))
		THIS << QString("speed=%1").arg(speed);

	THIS << QString("dev=%1").arg(config().device())
		 << "-v";

	if (pWriter->opts().getBool(optByteSwap))
		THIS << "-swab";

	if (pWriter->opts().getBool(optNoFix))
		THIS << "-nofix";

	if (pWriter->opts().getBool(optOverBurn))
		THIS << "--overburn";

	if (pWriter->opts().getBool("cdrecorddao"))
		THIS << "--dao";

   appendParam(driverOpts());
}

/*========================================================*/

QString ArsonCdrecordProcess::driverOpts (void)
{
	QStringList sl;

	if (m_pWriter->opts().getBool(optAudMaster))
		sl.append("audiomaster");

	if (config().is(ArsonConfig::flagBurnProof))
	{
		const ArsonVersion *pv = ArsonProgramDef::definition(
			ArsonConfig::PROGRAM_CDRECORD).getVersion();

		/*	cdrecord 1.11 changed the "burnproof"
		 *	switch to "burnfree"
		 */
		if (pv && (*pv) > ArsonVersion(1, 10))
			sl.append("burnfree");
		else
			sl.append("burnproof");
	}

	if (config().is(ArsonConfig::flagForceSpeed))
		sl.append("forcespeed");

	return sl.isEmpty()
		? QString::null
		: (QString("driveropts=") + sl.join(","));
}

/*========================================================*/

void ArsonCdrecordProcess::onStdout (int type, KRegExp *pre)
{
	static bool locked = false;

	if (ArsonProcessUI *pUI = ui())
		switch (type)
		{
		case EVENT_PAUSE:
			if (QString(pre->group(2)).find("Operation starts") != -1)
				break;
			//	FALL THROUGH

		case EVENT_PAUSE2:
			pUI->setText(
				i18n("Last chance to quit, starting write in %1 seconds... Press Cancel to abort.")
				.arg(pre->group(1)));
			break;

		case EVENT_STARTING:
			pUI->setText(i18n("Starting write..."));
			break;

		case EVENT_MBWRITTEN:
			{
				const int written = QString(pre->group(2)).toUInt();
				const int total = QString(pre->group(3)).toUInt();

				pUI->setText(i18n("Writing track %1 ...").arg(pre->group(1)));
				pUI->setProgress(written, total);
			}
			break;

		case EVENT_MBPARTIAL:
			{
				const int track = QString(pre->group(1)).toInt();
				const uint mb = QString(pre->group(2)).toUInt();

				if (track > m_trackNo)
				{
					pUI->setText(
						i18n("Writing track %1...")
						.arg(track));

					m_partialBase = pUI->progress();
					m_trackNo = track;
				}

				pUI->setProgress(m_partialBase + mb);
			}
			break;

		case EVENT_TOTAL:
			pUI->setText(i18n("Wrote %1 of %2 bytes...")
				.arg(pre->group(1))
				.arg(pre->group(2)));
			pUI->setMaxProgress(0);
			break;

		case EVENT_FIXATING:
			pUI->setText(i18n("Fixating..."));
			pUI->setMaxProgress(-1);
			break;

		case EVENT_PROMPT:
			if (!locked)
			{
				locked = true;
				KMessageBox::information(ArsonFrame::getFrame(),
					i18n("cdrecord needs for you to reload the CDR device.\n") +
					i18n("Please do so, and then press Ok."),
					i18n("Reload Device"));

				writeStdin("\n", 1);
				locked = false;
			}
			break;

		case EVENT_BLANK:
			pUI->setMaxProgress(-1);
			pUI->setText(
				i18n("Blanking CDRW..."));
			break;

		case EVENT_WRITETIME:
			pUI->setProgress(pUI->maxProgress());
			break;
		}
}

/*========================================================*/

void ArsonCdrecordProcess::setupSimulate (void) { THIS << "-dummy"; }
void ArsonCdrecordProcess::setupEject (void) { THIS << "-eject"; }

/*========================================================*/
/*	Data-specific implementation of the CDRECORD
 *	process class (burns ISO images)
 *========================================================*/

ArsonCdrecordDataProcess::ArsonCdrecordDataProcess (ArsonProcessMgr *pWriter, const QString &path)
	: ArsonCdrecordProcess(pWriter)
{
	deferArg("-data").deferArg(quoteFile(path));
}

/*========================================================*/
/*	Device specific implementation of CDRECORD process
 *========================================================*/

ArsonCdrecordIsosizeProcess::ArsonCdrecordIsosizeProcess (ArsonProcessMgr *pWriter, const QString &device)
	: ArsonCdrecordProcess(pWriter)
{
	deferArg("-isosize").deferArg(quoteFile(device));
}

/*========================================================*/
/*	cdrecord blanking process
 *========================================================*/

ArsonCdrecordBlankProcess::ArsonCdrecordBlankProcess (ArsonProcessMgr *pMgr, uint how)
	: ArsonCdrecordProcess(pMgr)
{
	THIS << QString("blank=") + ((how == BLANK_DISK) ? "all" : "fast");
}

bool ArsonCdrecordBlankProcess::execute (void)
{
	if (ArsonProcessUI *pUI = ui())
		pUI->setText(i18n("Initializing device..."));

	return ArsonCdrecordProcess::execute();
}

/*========================================================*/
/*	Fixate a CD
 *========================================================*/

ArsonCdrecordFixateProcess::ArsonCdrecordFixateProcess (ArsonProcessMgr *pMgr)
	: ArsonCdrecordProcess(pMgr)
{
	THIS << "-fix";
}

bool ArsonCdrecordFixateProcess::execute (void)
{
	if (ArsonProcessUI *pUI = ui())
		pUI->setText(i18n("Fixating disk..."));
		
	return ArsonCdrecordProcess::execute();
}

/*========================================================*/
/*	cdrecord audio writing class
 *========================================================*/

ArsonCdrecordAudioProcess::ArsonCdrecordAudioProcess (ArsonProcessMgr *pMgr)
	: ArsonCdrecordProcess(pMgr)
{
	THIS << "-pad";
	deferArg("-audio");
}

/*========================================================*/

void ArsonCdrecordAudioProcess::addTrack (const QString &filename)
{
	deferArg(filename);
}

/*========================================================*/
/*	A base class for all 'cdrdao' processes.
 *	SHOULD CONTAIN NO AUDIO/DATA SPECIFIC CODE!
 *========================================================*/

/*	Single disk copy output:
Starting CD copy simulation at speed 16...

Track   Mode    Flags  Start                Length
------------------------------------------------------------
 1      AUDIO   0      00:00:32(    32)     03:55:70( 17695)
Leadout AUDIO   0      03:56:27( 17727)

Copying audio tracks 1-1: start 00:00:00, length 03:56:27 to "cddata12613.bin"...
Track 1...
Found 26 Q sub-channels with CRC errors.
Please insert a recordable medium and hit enter.

ERROR: Cannot determine disk status - hit enter to try again.

Turning BURN-Proof on
Writing track 01 (mode AUDIO/AUDIO)...
Wrote 39 of 39 MB (Buffer 100%).
Wrote 17727 blocks. Buffer fill min 93%/max 100%.
Flushing cache...
Simulation finished successfully.
CD copying finished successfully.
*/

KRegExp ArsonCdrdaoProcess::events[ArsonCdrdaoProcess::_EVENT_MAX] = {
	"^Pausing 10 seconds",
	"^Process can be aborted",
	"^Writing track ([0-9]+)",
	"^Wrote ([0-9]+) of ([0-9]+) MB",
	"^Flushing cache",
	"^Copying data track [0-9]+ \\([^)]+\\): start ([0-9]+):([0-9]+):([0-9]+), length ([0-9]+):([0-9]+):([0-9]+) .+",
	"^([0-9]+):([0-9]+):([0-9]+)",
	"^Copying audio tracks [0-9]+-[0-9]+: start ([0-9]+):([0-9]+):([0-9]+), length ([0-9]+):([0-9]+):([0-9]+) .+",
	"^Please insert a recordable medium and hit enter\\.",
	"^ERROR: Cannot determine disk status - hit enter to try again",
	"^Disk seems to be written - hit return to reload disk",
	"^Trying to unlock drive",
	"^Blanking",
};

/*========================================================*/

ArsonCdrdaoProcess::ArsonCdrdaoProcess (ArsonProcessMgr *pWriter,
	const char *tocfile, const char *oper, const QString &type)
	: ArsonWriterProcess(pWriter, ArsonConfig::PROGRAM_CDRDAO, type),
	m_bInPrompt(false)
{
	const QString soper (oper);
	
	setOutput(Communication(Stderr | Stdin), events, ARRSIZE(events));

	THIS << (oper ? oper : "write")
			<< "--device" << config().device();

	//	None of these are necessary if we're just unlocking...
	if (soper != "unlock")
	{
		if (pWriter->opts().hasOpt(optSpeed))
			THIS << "--speed" << QString::number(pWriter->opts().getLong(optSpeed));

		if (pWriter->opts().getBool(optOverBurn))
			THIS << "--overburn";

		if (pWriter->opts().getBool(optByteSwap))
			THIS << "--swap";

		if (pWriter->opts().getBool(optNoFix))
			THIS << "--multi";
	}

	if (soper == "read-cd" || soper == "copy")
	{
		const long sc = pWriter->opts().getLong("subchan");
		const char *scs[] = { "rw", "rw_raw" };

		if (pWriter->opts().getBool("raw"))
			THIS << "--read-raw";	

		if (sc > 0)
			THIS << "--read-subchan" << scs[sc - 1];
	}

	appendDriverSwitch();

	if (tocfile)
		deferArg(quoteFile(tocfile));
}

/*========================================================*/

int intFromPossibleHexString (const QString &str)
{
	if (str.left(2) == "0x")
		return str.mid(2).toInt(NULL, 16);

	return str.toInt(NULL, 16);
}

/*========================================================*/
/**
 *	Combine initial flag set with the "drvflags" option.
 *
 *	For example if the user has driver set to:
 *	generic-mmc:0x02
 *
 *	and the opt "drvflags" is set to 0x10 then set to
 *	--driver generic-mmc:0x12
 *
 *========================================================*/

void ArsonCdrdaoProcess::appendDriverSwitch (void)
{
	QString drv = config().driver();

	if (!drv.isEmpty())
	{
		int flags = 0;
		const int pos = drv.find(':');
		const QString drvflags (m_pWriter->opts().getString(optDrvFlags));

		if (pos != -1)
		{
			flags = intFromPossibleHexString(drv.mid(pos + 1));
			drv.truncate(pos);
		}

		if (!drvflags.isEmpty() && m_pWriter->opts().getBool(optCdText))
			flags |= intFromPossibleHexString(drvflags);

		if (flags)
			drv.append(QString().sprintf(":0x%x", flags));

		THIS << "--driver" << drv;
	}
}

/*========================================================*/

void ArsonCdrdaoProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_PAUSE:
			pUI->setProgress(0, 100);
			pUI->setText(i18n(
				"Beginning write in 10 seconds... "
				"press Cancel to abort."));
			break;

		case EVENT_CALIBRATE:
			pUI->setProgress(0, 100);
			pUI->setText(
				i18n("Initializing device..."));
			break;

		case EVENT_TRACKSTART:
			pUI->setText(pre->group(0));
			break;

		case EVENT_MBWRITTEN:
			{
				const uint written = QString(pre->group(1)).toUInt();
				const uint total = QString(pre->group(2)).toUInt();

				pUI->setProgress(written, total);
			}
			break;

		case EVENT_FIXATING:
			pUI->setMaxProgress(-1);
			pUI->setText(
				i18n("Flushing cache..."));
			break;

		case EVENT_COPYAUDIOLEN:
		case EVENT_COPYDATALEN:
			{
				const uint start = calcTime(
					QString(pre->group(1)).toUInt(),
					QString(pre->group(2)).toUInt(),
					QString(pre->group(3)).toUInt());

				const uint end = calcTime(
					QString(pre->group(4)).toUInt(),
					QString(pre->group(5)).toUInt(),
					QString(pre->group(6)).toUInt());

				pUI->setText(i18n("Beginning CD copy..."));
				pUI->setMaxProgress(end - start);
			}
			break;

		case EVENT_COPYSTATUS:
			{
				const uint pos = calcTime(
					QString(pre->group(1)).toUInt(),
					QString(pre->group(2)).toUInt(),
					QString(pre->group(3)).toUInt());

				pUI->setProgress(pos);
			}
			break;

		case EVENT_PROMPT:
			promptContinue(
				i18n("Read complete, please insert a recordable disk, and press Continue to write..."),
				i18n("Change Disk"),
				KMessageBox::QuestionYesNo,
				i18n("Continue"),
				i18n("Cancel"));
			break;

		case EVENT_RETRY:
			promptContinue(
				i18n("Cannot determine disk status, press Retry to try again (don't panic, this is common)..."),
				i18n("Disk Status"),
				KMessageBox::WarningContinueCancel,
				i18n("Retry"),
				i18n("Cancel"));
			break;

		case EVENT_RELOAD:
			promptContinue(
				i18n("Disk seems to be written, please change the disk (or at least open/close the device), and press Retry to continue..."),
				i18n("Change Disk"),
				KMessageBox::WarningContinueCancel,
				i18n("Retry"),
				i18n("Cancel"));
			break;
			
		case EVENT_UNLOCK:
			pUI->setText(
				i18n("Attempting to unlock device..."));
			break;

		case EVENT_BLANK:
			pUI->setMaxProgress(-1);
			pUI->setText(
				i18n("Blanking CDRW..."));
			break;
		}
}

/*========================================================*/

void ArsonCdrdaoProcess::setupSimulate (void) { THIS << "--simulate"; }
void ArsonCdrdaoProcess::setupEject (void) { THIS << "--eject"; }

/*========================================================*/

void ArsonCdrdaoProcess::promptContinue (const QString &msg,
	const QString &caption, int how,
	const QString &yes, const QString &no)
{
	if (!m_bInPrompt)
	{
		m_bInPrompt = true;
		
		const int res = KMessageBox::messageBox(
			ArsonFrame::getFrame(),
#ifdef ARSON_KDE3
			KMessageBox::DialogType
#else
			int
#endif	//	ARSON_KDE3
			(how),
			msg, caption,
#ifdef ARSON_KDE3
			KGuiItem(yes),
			KGuiItem(no)
#else
			yes, no
#endif	//	ARSON_KDE3
			);

		if (res == KMessageBox::Continue || res == KMessageBox::Yes)
			writeStdin("\n", 1);
		else
			m_pWriter->abort();

		m_bInPrompt = false;
	}
}

/*========================================================*/
/*	Parse a hours:mins:secs into secs
 *========================================================*/

uint ArsonCdrdaoProcess::calcTime (uint hrs, uint mins, uint secs)
{
	return (secs + (mins * 60) + ((hrs * 60) * 60));
}

/*========================================================*/
/*	Process fpr copying CDs using 'cdrdao copy'
 *========================================================*/

ArsonCdrdaoCopyProcess::ArsonCdrdaoCopyProcess (ArsonProcessMgr *pMgr,
	bool deleteImg, bool onTheFly, const char *srcdev)
	: ArsonCdrdaoProcess(pMgr, NULL, "copy")
{
	const QString &dest = config().device();

	if (dest != srcdev)
	{
		const QString &drv = config().srcDriver();

		THIS << "--source-device" << srcdev;

		if (drv != QString::null)
			THIS << "--source-driver" << drv;

		if (onTheFly)
			THIS << "--on-the-fly";
	}
	
	if (!deleteImg)
		THIS << "--keepimage";
}

/*========================================================*/
/*	Class for writing ISO files using cdrdao
 *========================================================*/

ArsonCdrdaoIsoProcess::ArsonCdrdaoIsoProcess (ArsonProcessMgr *pMgr, const char *isofile)
	: ArsonCdrdaoProcess(pMgr, NULL, NULL),
	m_tempbase(QString::null)
{
	const QString tocfile = createTocBin(isofile);

	if (tocfile != QString::null)
		deferArg(quoteFile(tocfile));
}

ArsonCdrdaoIsoProcess::~ArsonCdrdaoIsoProcess (void)
{
	if (m_tempbase != QString::null)
	{
		ArsonFileCleanup()
			<< m_tempbase + ".bin"
			<< m_tempbase + ".toc";
	}
}

/*========================================================*/

QString ArsonCdrdaoIsoProcess::createTocBin (const char *isofile)
{
	QString contents;
	const char *content_fmt = "CD_ROM\nTRACK MODE1\nCOPY\nDATAFILE \"%s\"\n";
	const QString temp = ArsonTempFile::tempFileName("iso", "iso");
	const QString binfile = temp + ".bin";
	const QString tocfile = temp + ".toc";
	QFile file (QFile::encodeName(tocfile));

	/*	Create a symlink named foo.bin
	 *	to the ISO file foo
	 */
	if (symlink(QFileInfo(isofile).absFilePath(), binfile))
	{
		Trace("Failed to create symlink %s -> %s\n",
			isofile, binfile.latin1());
		
		throw ArsonError(
			i18n("Failed to symlink %1 -> %2")
			.arg(isofile).arg(binfile));
	}

	/*	At this point at least ONE file
	 *	needs to be cleaned up...
	 */
	m_tempbase = temp;

	//	Write the temporary TOC file
	if (!file.open(IO_WriteOnly | IO_Truncate))
		throw ArsonError(
			i18n("Failed to create TOC file"));

	contents.sprintf(content_fmt,
		binfile.latin1());

	file.writeBlock(contents,
		contents.length());

	return tocfile;
}

/*========================================================*/
/*	A process to unlock the drive.
 *========================================================*/

ArsonCdrdaoUnlockProcess::ArsonCdrdaoUnlockProcess (ArsonProcessMgr *pMgr)
	: ArsonCdrdaoProcess(pMgr, NULL, "unlock", i18n("Unlock"))
{
	//	Nothing...
}

/*========================================================*/
/*	cdrdao blank process
 *========================================================*/

ArsonCdrdaoBlankProcess::ArsonCdrdaoBlankProcess (ArsonProcessMgr *pMgr, uint how)
	: ArsonCdrdaoProcess(pMgr, NULL, "blank", i18n("Blank"))
{
	THIS << "--blank-mode" << ((how == BLANK_DISK) ? "full" : "minimal");
}

bool ArsonCdrdaoBlankProcess::execute (void)
{
	if (ArsonProcessUI *pUI = ui())
		pUI->setText(
			i18n("Initializing device..."));

	return ArsonCdrdaoProcess::execute();
}

/*========================================================*/
/*	cdrdao read-cd wrapper
 *========================================================*/

ArsonCdrdaoReadCdProcess::ArsonCdrdaoReadCdProcess (ArsonProcessMgr *pMgr,
	const char *outfile)
	: ArsonCdrdaoProcess(pMgr, outfile, "read-cd", i18n("Read CD")),
	m_initDir(QDir::current().canonicalPath())
{
	QFileInfo fi (outfile);

	if (chdir(fi.dir().canonicalPath().latin1()))
		throw ArsonError(
			i18n("Failed to change to target directory."));

	THIS << "--datafile" << (fi.baseName() + ".bin");
}

ArsonCdrdaoReadCdProcess::~ArsonCdrdaoReadCdProcess (void)
{
	chdir(m_initDir.latin1());
}

/*========================================================*/
/*	Base class for all ripper processes (cdda2wav now...
 *	if we support cdparanoia in the future it's process
 *	class should derive from this).
 *========================================================*/

ArsonRipperProcess::ArsonRipperProcess (ArsonProcessMgr *pMgr,
	int program, int trackNo, const char *outfile)
	: ArsonProcess(i18n("Ripper"), pMgr, program)
{
//	appendExtra();
}

/*========================================================*/
/*	Base class for cdda2wav processes
 *========================================================*/

KRegExp ArsonCdda2WavProcess::events[ArsonCdda2WavProcess::_EVENT_MAX] = {
	"[ \t]*([0-9]+)[%]",
	"load cdrom please and press enter",
};

ArsonCdda2WavProcess::ArsonCdda2WavProcess (ArsonProcessMgr *pMgr,
	int trackNo, const char *outfile)
	: ArsonRipperProcess(pMgr, ArsonConfig::PROGRAM_CDDA2WAV, trackNo, outfile)
{
	if (const ArsonDevice *pd = config().devices().device(config().ripper().srcdev()))
	{
		setOutput(Stderr, events, ARRSIZE(events));

		(*this) << "-H"
				<< "-v" << "1";

		if (pd->scsiDevice())
			(*this) << "-I" << "generic_scsi"
					<< "-D" << QString(pd->id());
		else if (pd->ideDevice())
			(*this) << "-I" << "cooked_ioctl"
					<< "-D" << QString(pd->id());
		else
			(*this) << "-I" << "cooked_ioctl"
					<< "-D" << QString(pd->dev());

		if (trackNo >= 0)
		{
			Assert(outfile != NULL);
		
			deferArg("-t")
				.deferArg(QString::number(trackNo))
				.deferArg(quoteFile(outfile));
		}
	}
	else
		invalidate(
			i18n("Device not found: %1")
			.arg(config().ripper().srcdev()));
}

/*========================================================*/

#define RIPPERFMT(f,a)	case (f): arg=(a); break

void ArsonCdda2WavProcess::setOutputFormat (uint fmt)
{
	const char *arg = NULL;

	switch (fmt)
	{
		RIPPERFMT(RIPPERFMT_WAV, "wav");
		RIPPERFMT(RIPPERFMT_AU, "au");
		RIPPERFMT(RIPPERFMT_CDR, "cdr");
		RIPPERFMT(RIPPERFMT_AIFF, "aiff");
		RIPPERFMT(RIPPERFMT_AIFC, "aifc");
	}

	if (arg)
		(*this) << "-O" << arg;
}

/*========================================================*/

void ArsonCdda2WavProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_PERCENT:
			pUI->setProgress(QString(pre->group(1)).toInt(), 100);
			break;

		case EVENT_LOAD:
			break;
		}
}

/*========================================================*/
/*	cdparanoia class implementation
 *========================================================*/
/*
cdparanoia III release 9.8 (March 23, 2001)
(C) 2001 Monty <monty@xiph.org> and Xiphophorus

Report bugs to paranoia@xiph.org
http://www.xiph.org/paranoia/

Ripping from sector      32 (track  1 [0:00.00])
          to sector   18849 (track  1 [4:10.67])

outputting to test.wav

 (== PROGRESS == [                    >         | 018849 00 ] == :^D * ==)

Done.
*/

KRegExp ArsonCdparanoiaProcess::events[ArsonCdparanoiaProcess::_EVENT_MAX] = {
	"Ripping from sector[ \t]+([0-9]+)",
	"[ \t]+to sector[ \t]+([0-9]+)",
	"##: ([0-9-]+) \\[([a-z]+)\\] [@] ([0-9]+)"
};

ArsonCdparanoiaProcess::ArsonCdparanoiaProcess (ArsonProcessMgr *pMgr,
	int trackNo, const char *outfile)
	: ArsonRipperProcess(pMgr, ArsonConfig::PROGRAM_CDPARANOIA, trackNo, outfile),
	m_start(0), m_sectors(0)
{
	if (const ArsonDevice *pd = config().devices().device(config().ripper().srcdev()))
	{
		if (!pd->dev().isEmpty())
		{
			setOutput(Stderr, events, ARRSIZE(events));
			
			(*this) << "-e"
					<< "-d" << QString(pd->dev());

			deferArg(QString("%1-%2").arg(trackNo).arg(trackNo));
			deferArg(quoteFile(outfile));
		}
		else
			invalidate(
				i18n("Device path for SCSI device %1 is not set. Cdparanoia cannot be used with this device until it's device path is set in the configure dialog.").arg(pd->id()));
	}
	else
		invalidate(
			i18n("Device not found: %1")
			.arg(config().ripper().srcdev()));
}

/*========================================================*/

void ArsonCdparanoiaProcess::setOutputFormat (uint fmt)
{
	const char *arg = NULL;

	switch (fmt)
	{
		RIPPERFMT(RIPPERFMT_WAV, "wav");
		RIPPERFMT(RIPPERFMT_AIFF, "aiff");
		RIPPERFMT(RIPPERFMT_AIFC, "aifc");
	}

	if (arg)
		(*this) << QString("--output-") + arg;
}

/*========================================================*/

#define CD_CONSTANT		1176

void ArsonCdparanoiaProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_BEG:
			m_start = QString(pre->group(1)).toULong();
			Trace("Start: %u\n", m_start);
			break;

		case EVENT_END:
			m_sectors = QString(pre->group(1)).toULong() - m_start;
			Trace("Sectors: %u\n", m_sectors);
			break;

		case EVENT_PROGRESS:
			{
				const QCString oper (pre->group(2));

				if (oper == "wrote" || oper == "finished")
				{
					const ulong inpos = QString(pre->group(3)).toULong();

					pUI->setProgress(
						inpos - (m_start * CD_CONSTANT),
						m_sectors * CD_CONSTANT);
				}
			}
			break;
		}
}

/*========================================================*/
/*	readcd Output:
Capacity: 159203 Blocks = 318406 kBytes = 310 MBytes = 326 prMB
Sectorsize: 2048 Bytes
Capacity: 159203 Blocks = 318406 kBytes = 310 MBytes = 326 prMB
Sectorsize: 2048 Bytes
Copy from SCSI (0,0,0) disk to file 'test.iso'
end:    159203
addr:   159203 cnt: 998
Time total: 94.083sec
Read 318406.00 kB at 3384.3 kB/sec.
*/

KRegExp ArsonReadCdProcess::events[ArsonReadCdProcess::_EVENT_MAX] = {
	"end:[ \t]+([0-9]+)",
	"addr:[ \t]+([0-9]+) cnt:[ \t]+([0-9]+)",
};

ArsonReadCdProcess::ArsonReadCdProcess (ArsonProcessMgr *pMgr, const char *outfile, const char *dev)
	: ArsonProcess(i18n("readcd"), pMgr, ArsonConfig::PROGRAM_READCD),
	m_max(0)
{
	setOutput(Stderr, events, ARRSIZE(events));

	(*this) << quoteFile(QString("dev=") + (dev ? dev : config().device()))
			<< quoteFile(QString("f=") + outfile);
}

/*========================================================*/

void ArsonReadCdProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_LENGTH:
			pUI->setText(i18n("Preparing to read CD..."));
			m_max = QString(pre->group(1)).toInt();
			pUI->setMaxProgress(m_max);
			break;

		case EVENT_PROGRESS:
			{
				const int progress = QString(pre->group(1)).toInt();

				pUI->setProgress(progress);

				if (pUI->progress() == 0)
					pUI->setText(i18n("Reading CD..."));
			}
			break;
		}
}

void ArsonReadCdProcess::output (const QString &str, bool error)
{
	if (ArsonProcessUI *pUI = ui())
	{
		if (str.length() == 1 &&
			strchr("~-+,.", str[0]))
		{
			/*	Do something fancy to handle those wacky
			 *	retry codes (~-+, etc)
			 */
			//return;
		}
	}

	ArsonProcess::output(str, error);
}

/*========================================================*/
/*	Base class for all encoder processes
 *========================================================*/

ArsonEncoderProcess::ArsonEncoderProcess (int program, ArsonProcessMgr *pMgr)
	: ArsonProcess(i18n("Encoder"), pMgr, program),
	m_pRipper(NULL)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonEncoderProcess::kill (int signo)
{
	if (m_pRipper && m_pRipper->isRunning())
		m_pRipper->kill(signo);

	return ArsonProcess::kill(signo);
}

/*========================================================*/
/*	bladeenc status output:
 *	Status:   12.5% done, ETA 00:02:05          BATCH: 12.5% done, ETA 00:02:05
 *	Completed. Encoding time: 00:02:15 (1.51X)
 *========================================================*/

KRegExp ArsonBladeProcess::events[ArsonBladeProcess::_EVENT_MAX] = {
	"Status:[ \t]+([0-9]+)\\.([0-9]+)% done, ETA ([0-9]+):([0-9]+):([0-9]+)[ \t]+BATCH:[ \t]+([0-9]+).([0-9]+)% done, ETA ([0-9]+):([0-9]+):([0-9]+)",
	"Completed\\. Encoding time: ([0-9]+):([0-9]+):([0-9]+) \\(([0-9]+)\\.([0-9]+)X\\)",
};

ArsonBladeProcess::ArsonBladeProcess (ArsonProcessMgr *pMgr,
	const char *infile,
	const char *outfile)
	: ArsonEncoderProcess(ArsonConfig::PROGRAM_BLADEENC, pMgr)
{
	ArsonEncoderPresets presets;
	
	setOutput(Stdout, events, ARRSIZE(events));

	if (ArsonEncoderOpts *po = presets.opts(pMgr->opts().getString("quality")))
		po->applyTo(*this);

	deferArg(quoteFile(infile));
	deferArg(quoteFile(outfile));
}

void ArsonBladeProcess::onStdout (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_PERCENT:
			pUI->setProgress(QString(pre->group(1)).toInt(), 100);
			break;

		case EVENT_DONE:
			pUI->setProgress(100);
		}
}

/*========================================================*/
/*	LAME sample output:
LAME version 3.70 (www.sulaco.org/mp3) 
GPSYCHO: GPL psycho-acoustic and noise shaping model version 0.77. 
Using polyphase lowpass filter,  transition band:  15115 Hz - 15648 Hz
Encoding track-06.wav to track-06.wav.mp3
Encoding as 44.1 kHz 128 kbps j-stereo MPEG1 LayerIII (11.0x)  qval=5
    Frame          |  CPU/estimated  |  time/estimated | play/CPU |   ETA
  1150/  7787( 14%)| 0:00:12/ 0:01:22| 0:00:12/ 0:01:21|    2.4786| 0:01:09
*/

KRegExp ArsonLameProcess::events[ArsonLameProcess::_EVENT_MAX] = {
	"[ \t]*([0-9]+)/[ \t]*([0-9]+)[ \t]*\\([ \t]*([0-9]+)%\\)|",
};

ArsonLameProcess::ArsonLameProcess (ArsonProcessMgr *pMgr,
	const char *infile,
	const char *outfile)
	: ArsonEncoderProcess(ArsonConfig::PROGRAM_LAME, pMgr)
{
	ArsonEncoderPresets presets;

	setOutput(Stderr, events, ARRSIZE(events));

	THIS << "--nohist";		//	Always

	if (ArsonEncoderOpts *po = presets.opts(pMgr->opts().getString("quality")))
		po->applyTo(*this);

	deferArg(quoteFile(infile));
	deferArg(quoteFile(outfile));
}

/*========================================================*/

void ArsonLameProcess::setTag (const char *title, int trackNo,
	const char *artist, const char *album, const char *comment,
	const char *date, const char *genre)
{
#define LAMEARG(sw,arg)	if((arg)&&*((arg)))(*this)<<(sw)<<quoteFile((arg))
#define LAMELIMIT(var,lim)	if((var)&&qstrlen((var))>(lim)){Trace("WARNING: chopping %s\n", (var));(var)=NULL;}

	LAMELIMIT(comment, 30);
	LAMELIMIT(title, 30);
	LAMELIMIT(artist, 30);
	LAMELIMIT(album, 30);
	LAMELIMIT(date, 4);

	LAMEARG("--tt", title);
	LAMEARG("--ta", artist);
	LAMEARG("--tl", album);
	LAMEARG("--ty", date);
	LAMEARG("--tc", comment);
	LAMEARG("--tg", genre);

	if (trackNo != -1)
		THIS << "--tn" << QString::number(trackNo);
}

/*========================================================*/

void ArsonLameProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_STATUS:
			pUI->setProgress(QString(pre->group(3)).toInt(), 100);
			break;
		}
}

/*========================================================*/
/*	Ogg Vorbis encoder process
 *========================================================*/
/*	oggenc sample output:
Opening with wav module: WAV file reader
Encoding "track-06.ogg" [  2.4%] [ 2m27s remaining] |

Done encoding file "track-06.ogg"

	File length:  3m 23.0s
	Elapsed time: 2m 28.2s
	Rate:         1.3722
	Average bitrate: 133.3 kb/s

*/

KRegExp ArsonOggencProcess::events[ArsonOggencProcess::_EVENT_MAX] = {
	"\\[[ \t]*([0-9]+)\\.([0-9]+)%\\] \\[[ \t]*([0-9]+)m([0-9]+)s remaining\\]",
};

ArsonOggencProcess::ArsonOggencProcess (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile)
	: ArsonEncoderProcess(ArsonConfig::PROGRAM_OGGENC, pMgr)
{
	ArsonEncoderPresets presets;

	setOutput(Stderr, events, ARRSIZE(events));

	if (ArsonEncoderOpts *po = presets.opts(pMgr->opts().getString("quality")))
		po->applyTo(*this);

	(*this) << quoteFile(infile)
			<< "-o" << quoteFile(outfile);
}

/*========================================================*/

void ArsonOggencProcess::setTag (const char *title, int trackNo,
	const char *artist, const char *album, const char *comment,
	const char *date, const char *genre)
{
#define OGGARG(sw,arg)	if((arg)&&*((arg)))(*this)<<(sw)<<quoteFile((arg))

	if (comment && *comment)
		THIS << "-c" << QString().sprintf("comment=%s", comment);

//	OGGARG("-c", comment);
	OGGARG("-d", date);
	OGGARG("-t", title);
	OGGARG("-l", album);
	OGGARG("-a", artist);

	if (trackNo != -1)
		THIS << "-N" << QString::number(trackNo);
}

/*========================================================*/

void ArsonOggencProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_STATUS:
			pUI->setProgress(QString(pre->group(1)).toInt(), 100);
			break;
		}
}

/*========================================================*/
/*	General FLAC stuff
 *========================================================*/

#ifdef FLAC
namespace arson {
	namespace flac
	{
		enum {
			EVENT_PROGRESS,
			_EVENT_MAX,
		};

		KRegExp events[_EVENT_MAX] = {
			".+: +([0-9]+)[%] complete",
		};
	};
};
#endif	//	FLAC

/*========================================================*/
/*	FLAC encoder process
 *========================================================*/
#ifdef FLAC
ArsonFlacEncoderProcess::ArsonFlacEncoderProcess (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile)
	: ArsonEncoderProcess(ArsonConfig::PROGRAM_FLAC, pMgr)
{
	setOutput(Stderr, arson::flac::events, ARRSIZE(arson::flac::events));

	THIS << "-o" << quoteFile(outfile);
	deferArg(quoteFile(infile));
}

void ArsonFlacEncoderProcess::onStderr (int event, KRegExp *pre)
{
	ArsonProcessUI *pUI = ui();
	
	if (pUI && event == arson::flac::EVENT_PROGRESS)
		pUI->setProgress(QString(pre->group(1)).toInt(), 100);
}
#endif	//	FLAC
/*========================================================*/
/*	Normalize process class
 *========================================================*/
/*	normalize sample output:
Computing levels...
 track7.wav 100% done, ETA 00:00:00 (batch 100% done, ETA 00:00:00)
Applying adjustment of 1.419...
 track7.wav 100% done, ETA 00:00:00 (batch 100% done, ETA 00:00:00)
*/

KRegExp ArsonNormalizeProcess::events[ArsonNormalizeProcess::_EVENT_MAX] = {
	"Computing levels",
	"Applying adjustment",
	"batch[ \t]+([0-9]+)% done",
};

ArsonNormalizeProcess::ArsonNormalizeProcess (ArsonProcessMgr *pMgr)
	: ArsonProcess(i18n("normalize"), pMgr, ArsonConfig::PROGRAM_NORMALIZE),
	m_state(stateUnknown)
{
	setOutput(Stderr, events, ARRSIZE(events));

	THIS << ((pMgr->opts().getLong(optNormal) == NORMALIZE_BATCH)
		? "-b" : "-m");
}

/*========================================================*/

void ArsonNormalizeProcess::addTrack (const char *file)
{
	(*this) << quoteFile(file);
}

/*========================================================*/

void ArsonNormalizeProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_COMPUTING:
			m_state = stateCompute;
			break;

/*		Uncomment when/if normalize gets fixed...
 *		guess i should email the maintainer huh? i did...
 *		
		case EVENT_APPLYING:
			Trace("applying...\n");
			m_state = stateApply;
			break;
*/
		case EVENT_STATUS:
			{
				const int per = QString(pre->group(1)).toInt();

				/*	HACK: methinks thar be a bug in normalize... all of
				 *	it's output goes to stderr, except the "Applying ..."
				 *	message which goes to stdout, therefore i don't get
				 *	it, hence this little hack to to find the state.
				 */
				if (per == 0 && pUI->progress() == 100)
					m_state = stateApply;

				if (m_state == stateCompute)
					pUI->setText(
						i18n("Computing normalization for audio tracks..."));
				else
					pUI->setText(
						i18n("Applying normalization adjustment to audio tracks..."));

				pUI->setProgress(per, 100);
			}
			break;
		}
}

/*========================================================*/
/*	A base class for audio file decoders
 *========================================================*/

ArsonAudioDecoderProcess::ArsonAudioDecoderProcess (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile, int program)
	: ArsonProcess(i18n("audio decoder"), pMgr, program)
{
	//	Nothing...
}

/*========================================================*/
/*	Mpg123 processs class implementation
 *========================================================*/

KRegExp ArsonMpg123Process::events[ArsonMpg123Process::_EVENT_MAX] = {
	"Time: ([0-9]+):([0-9]+)\\.[0-9]+",
};

ArsonMpg123Process::ArsonMpg123Process (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile)
	: ArsonAudioDecoderProcess(pMgr, infile, outfile, ArsonConfig::PROGRAM_MPG123)
{
	setOutput(Stderr, events, ARRSIZE(events));

	THIS << "-v" << "--wav"
		 << quoteFile(outfile);

	deferArg(quoteFile(infile));
}

/*========================================================*/
/*	mpg123 verbose output...
 *	Frame# 12776 [    4],<BREAK>Time: 05:33.74 [00:00.10],
 */

void ArsonMpg123Process::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_TIME:
			{
				const int mins = QString(pre->group(1)).toInt();
				const int secs = QString(pre->group(2)).toInt();

				pUI->setProgress((mins * 60) + secs);
			}
			break;
		}
}

/*========================================================*/
/*	ogg123 sample output
Playing from file track6.ogg.
Device:   WAV file output
Author:   Aaron Holtzman <aholtzma@ess.engr.uvic.ca>
Comments: Sends output to a .wav file


Bitstream is 2 channel, 44100Hz
Encoded by: Xiphophorus libVorbis I 20010813

Time: 03:23.40 [00:00.00] of 03:23.40, Bitrate: 0.4
Done.
*/
#ifdef OGG

KRegExp ArsonOgg123Process::events[ArsonOgg123Process::_EVENT_MAX] = {
	"Time: ([0-9]+):([0-9]+)\\.([0-9]+)",
};

ArsonOgg123Process::ArsonOgg123Process (ArsonProcessMgr *pWriter,
	const char *infile, const char *outfile)
	: ArsonAudioDecoderProcess(pWriter, infile, outfile, ArsonConfig::PROGRAM_OGG123)
{
	setOutput(Communication(Stdout | Stderr), events, ARRSIZE(events));

	(*this) << "-v"
			<< "-o" << "byteorder:big"
			<< "-d" << "raw"
			<< "-f" << quoteFile(outfile);

	deferArg(quoteFile(infile));
}

/*========================================================*/

void ArsonOgg123Process::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_TIME:
			{
				const int mins = QString(pre->group(1)).toInt();
				const int secs = QString(pre->group(2)).toInt();

				pUI->setProgress((mins * 60) + secs);
			}
			break;
		}
}

#endif	//	OGG
/*========================================================*/
/*	Shorten class impl
 *========================================================*/

ArsonShortenProcess::ArsonShortenProcess (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile)
	: ArsonAudioDecoderProcess(pMgr, infile, outfile, ArsonConfig::PROGRAM_SHORTEN)
{
	THIS << "-x";

	deferArg(quoteFile(infile));
	deferArg(quoteFile(outfile));
}

/*========================================================*/

bool ArsonShortenProcess::execute (void)
{
	if (ArsonAudioDecoderProcess::execute())
	{
		if (ArsonProcessUI *pUI = ui())
			pUI->setMaxProgress(-1);

		return true;
	}

	return false;
}

/*========================================================*/
/*	FLAC class implementation
 *========================================================*/
#ifdef FLAC
ArsonFlacDecoderProcess::ArsonFlacDecoderProcess (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile)
	: ArsonAudioDecoderProcess(pMgr, infile, outfile, ArsonConfig::PROGRAM_FLAC)
{
	setOutput(Stderr, arson::flac::events, ARRSIZE(arson::flac::events));

	THIS << "-d" << "-o" << quoteFile(outfile);
	deferArg(quoteFile(infile));
}

/*========================================================*/

void ArsonFlacDecoderProcess::onStderr (int event, KRegExp *pre)
{
	ArsonProcessUI *pUI = ui();

	if (pUI && event == arson::flac::EVENT_PROGRESS)
		pUI->setProgress(QString(pre->group(1)).toInt(), 100);
}
#endif	//	FLAC
/*========================================================*/
/*	LAME decoder
 *========================================================*/
/*	Sample output:
input:  temp.mp3  (44.1 kHz, 2 channels, MPEG-1 Layer III)
output: temp.wav  (16 bit, Microsoft WAVE)
skipping initial 1105 samples (encoder+decoder delay)
Frame#  1667/1667   128 kbps  L  R
 */

KRegExp ArsonLameDecoderProcess::events[ArsonLameDecoderProcess::_EVENT_MAX] = {
	"Frame#[ \t]+([0-9]+)/([0-9]+).+",
//	"[^ios].+",		//	FIXME: Weird output... this HACK gobbles up all non-important output
};

ArsonLameDecoderProcess::ArsonLameDecoderProcess (ArsonProcessMgr *pMgr,
	const char *infile, const char *outfile)
	: ArsonAudioDecoderProcess(pMgr, infile, outfile, ArsonConfig::PROGRAM_LAME)
{
	setOutput(Stderr, events, ARRSIZE(events));

	THIS << "--decode";

	deferArg(quoteFile(infile));
	deferArg(quoteFile(outfile));
}

/*========================================================*/

void ArsonLameDecoderProcess::onStderr (int event, KRegExp *pre)
{
	ArsonProcessUI *pUI = ui();

	if (event == EVENT_PROGRESS && pUI)
	{
		pUI->setProgress(
			QString(pre->group(1)).toInt(),
			QString(pre->group(2)).toInt());
	}
}
	
/*========================================================*/
/*	Class implementation for class wrapping mkisofs
 *========================================================*/

ArsonMkisoProcess::ArsonMkisoProcess (ArsonProcessMgr *pWriter,
	const ArsonIsoFlags *pFlags, const QString &outfile, const QStringList &what)
	: ArsonProcess(i18n("mkisofs"), pWriter, ArsonConfig::PROGRAM_MKISOFS)
{
	QStringList::ConstIterator it, end;
	const QString label = pWriter->opts().getString(optIsoLabel);
	
	setOutput(Stderr, events, ARRSIZE(events));

	if (!label.isEmpty())
		THIS << "-V" << label;
	
	pFlags->applyTo(this, outfile);

	deferArgs(what);
}

/*========================================================*/

bool ArsonMkisoProcess::execute (void)
{
	if (ArsonProcessUI *pUI = ui())
		pUI->setText(
			i18n("Creating ISO image..."));

	return ArsonProcess::execute();
}

/*========================================================*/
/*	Sample mkisofs output:
tonys111:~/Projects/arson/arson$ mkiso temp.iso . temp >/dev/null
mkisofs: No such file or directory. File ./.#process.h is not readable - ignoring
Using _PROGRAM.000;1 for  /_programdlg.cpp (_programdlg.moc.cpp)
Using _CONFIGD.000;1 for  /_configdlg.cpp (_configdlg.moc.cpp)
Using _PROGRES.000;1 for  /_progressdlg.cpp (_progressdlg.moc.cpp)
Using _WAITDLG.000;1 for  /_waitdlg.cpp (_waitdlg.moc.cpp)
Using CDROM-LO.000;1 for  ./.xvpics/cdrom-lo-32.png (cdrom-lo-16.png)
 35.30% done, estimate finish Fri Oct 12 18:45:29 2001
 70.53% done, estimate finish Fri Oct 12 18:45:30 2001
Total extents actually written = 14192
Total translation table size: 0
Total rockridge attributes bytes: 15319
Total directory bytes: 14336
Path table size(bytes): 80
Max brk space used 1a344
14192 extents written (27 Mb)
 */

KRegExp ArsonMkisoProcess::events[_EVENT_MAX] = {
	"[ \t]*([0-9]+)\\.([0-9]+)% done, estimate finish (.+)",
};

void ArsonMkisoProcess::onStderr (int event, KRegExp *pre)
{
	if (ArsonProcessUI *pUI = ui())
		switch (event)
		{
		case EVENT_PROGRESS:
			pUI->setProgress(QString(pre->group(1)).toInt());
			break;
		}
}

/*========================================================*/
/*	VCD image generation process (vcdxbuild)
 *
++ WARN: initializing libvcd 0.7.11 [linux-gnu/i686]
++ WARN:  
++ WARN:  this is the UNSTABLE development branch!
++ WARN:  use only if you know what you are doing
++ WARN:  see http://www.hvrlab.org/~hvr/vcdimager/ for more information
++ WARN:  
#scan[sequence-00]: 0/678917092 ( 0%)          
#scan[sequence-00]: 6790728/678917092 ( 1%)          
#scan[sequence-00]: 13581456/678917092 ( 2%)          
#scan[sequence-00]: 20372184/678917092 ( 3%)          
   INFO: scanning mpeg sequence item #0 for scanpoints...
++ WARN: file `test.bin' exist already, will get overwritten!
++ WARN: file `test.toc' exist already, will get overwritten!
#write[1/2]: 0/292658 ( 0%)
   INFO: writing track 1 (ISO9660)...
#write[1/2]: 75/292658 ( 0%)
#write[1/2]: 150/292658 ( 0%)
#write[1/2]: 225/292658 ( 0%)
#write[1/2]: 300/292658 ( 0%)
#write[2/2]: 300/292658 ( 0%)
   INFO: writing track 2, MPEG1, NTSC SIF (352x240/30fps), audio[0]: l2/44.1kHz/224kbps/stereo ...
#write[2/2]: 375/292658 ( 0%)
#write[2/2]: 450/292658 ( 0%)
#write[2/2]: 525/292658 ( 0%)
 */

/*
KRegExp ArsonVcdxbuildProcess::events[_EVENT_MAX] = {
	"INFO: scanning mpeg sequence item [#]([0-9]+) for scanpoints",
	"[#]scan\\[sequence-[0-9]+\\]:[ \t]+[0-9]+/[0-9]+[ \t]+\\([ \t]*([0-9]+)%\\)",
	"[ \t]+INFO: writing track ([0-9]+)",
	"[#]write\\[[0-9]+/[0-9]+\\]:[ \t]+[0-9]+/[0-9]+[ \t]+\\([ \t]*([0-9]+)%\\)",
};
*/

ArsonVcdxbuildProcess::ArsonVcdxbuildProcess (ArsonProcessMgr *pWriter, const char *basefile)
	: ArsonProcess(i18n("VCD image"), pWriter, ArsonConfig::PROGRAM_VCDXBUILD),
	m_stage(stageUnknown), m_filebase(basefile)
{
	(*this) << "--progress" << "--gui"
			<< quoteFile(QString("--cdrdao-file=") + m_filebase);

	deferArg(quoteFile(m_filebase + ".xml"));
}

/*========================================================*/

bool ArsonVcdxbuildProcess::execute (void)
{
	ArsonUtilityProcess proc(config(),
		NoCommunication, QString::null,
		ArsonConfig::PROGRAM_VCDXGEN);
	const QString label = m_pWriter->opts().getString(optIsoLabel);

	if (!label.isEmpty())
		proc << "-l" << label;
	
	proc << "-t" << m_pWriter->opts().getString("type")
		 << "-o" << (m_filebase + ".xml");

//	Trace("adding %d mpg files to util process\n", count());
	
	for (int index = 0; index < count(); ++index)
		proc << quoteFile(m_mpgs[index]);
	
	if (!proc.execute() || !proc.successful())
		return false;

	Trace("executing vcdxbuild\n");
	return ArsonProcess::execute();
}

/*========================================================*/

void ArsonVcdxbuildProcess::output (const QString &str, bool error)
{
	if (!error)
	{
		QDomDocument doc;
		QDomElement root;

		//	Cheesy way to parse it but...
		doc.setContent(
			QString("<?xml version='1.0'?><vcdxbuild>")
			+ str
			+ "</vcdxbuild>"
			);

		root = doc.documentElement();

		/*	There should be only one... but iterate just incase...
		 *	When you assume, you make an ass out of u and me...
		 */
		for (QDomNode node = root.firstChild(); !node.isNull(); node = node.nextSibling())
		{
			QDomElement el = node.toElement();

			if (el.isNull())
				continue;

			handleOutputElement(el);
		}
	}
}

/*========================================================*/
/*	Handles XML output from vcdxbuild, and updates
 *	the UI accordingly.
 *========================================================*/

void ArsonVcdxbuildProcess::handleOutputElement (QDomElement &el)
{
	const QString tagName = el.tagName().lower();

	if (ArsonProcessUI *pUI = ui())
	{
		if (tagName == "progress")
			outputProgress(el, pUI);

		else if (tagName == "log")
			outputLog(el, pUI);
	}
}

/*========================================================*/

void ArsonVcdxbuildProcess::outputProgress (QDomElement &el, ArsonProcessUI *pUI)
{
	const QString oper = el.attribute("operation").lower();
	const uint pos = el.attribute("position").toUInt();
	const uint size = el.attribute("size").toUInt();

	if (oper == "scan")
	{
		const uint index = el.attribute("id")
			.replace(QRegExp("sequence-"), "")
			.toUInt();

		if (m_stage == stageUnknown || pos < pUI->progress())
			pUI->setText(
				i18n("Scanning video file %1")
				.arg(index + 1)
				);

		m_stage = stageScan;
	}

	else if (oper == "write")
	{
		if (m_stage == stageScan)
			pUI->setProgress(size);

		m_stage = stageWrite;
	}

	else
		return;

	pUI->setProgress(pos, size);
}

/*========================================================*/

void ArsonVcdxbuildProcess::outputLog (QDomElement &el, ArsonProcessUI *pUI)
{
	QDomText tel = el.firstChild().toText();
	const QString level = el.attribute("level").lower();

	if (tel.isText())
	{
		const QString text = tel.data();

		if (m_stage == stageWrite && level == "information")
			pUI->setText(text);
		else
			pUI->output(text, level != "error");
	}
}

/*========================================================*/
/*	MD5SUM check process
 *========================================================*/

ArsonMd5sumProcess::ArsonMd5sumProcess (ArsonProcessMgr *pMgr, const char *infile)
	: ArsonProcess(i18n("MD5 check"), pMgr, ArsonConfig::PROGRAM_MD5SUM),
	m_tempfile(ArsonTempFile::tempFileName("md5", "md5")),
	m_initDir(QDir::current().canonicalPath())
{
	QFile in (infile), out(m_tempfile);
	QCString md5file (m_tempfile.latin1());

	//	We need to be in the same directory as the md5 file.
	chdir(QFileInfo(infile).dir().canonicalPath().latin1());

	/*	Copy the md5 file to a tempfile, and tranlate to UNIX
	 *	text format just incase it's in Windows or Mac format
	 *	(which is likely).
	 */
	if (in.open(IO_ReadOnly) && out.open(IO_WriteOnly | IO_Truncate))
	{
		for (int ch = 0; (ch = in.getch()) != -1;)
			if (ch != '\r') out.putch(ch);

		Trace("wrote UNIX md5 file: %s\n",
			m_tempfile.latin1());
		
		out.close();
		in.close();
	}
	else
	{
		m_output.append(
			i18n("Warning: Failed to convert input MD5 file, may be in wrong format!"));
		md5file = infile;
	}

	THIS << "-c";
	deferArg(quoteFile(md5file));
}

ArsonMd5sumProcess::~ArsonMd5sumProcess (void)
{
	//	Restore the initial CWD
	chdir(m_initDir.latin1());
	QFile::remove(m_tempfile);
}

/*========================================================*/

bool ArsonMd5sumProcess::execute (void)
{
	if (ArsonProcessUI *pUI = ui())
		pUI->setMaxProgress(-1);

	return ArsonProcess::execute();
}

void ArsonMd5sumProcess::output (const QString &str, bool error)
{
	Trace("md2sum output: %s\n",
		str.latin1());
	
	m_output.append(str);
}

/*========================================================*/
/*	Determine a SHN file length (in seconds)
 *========================================================*/

KRegExp ArsonShnlenProcess::events[ArsonShnlenProcess::_EVENT_MAX] = {
	"([0-9]+):([0-9]+)\\.([0-9]+)[ \t]+([0-9]+)[ \t]+...[ \t]+..[ \t]+...[ \t]+(.+)",
};

ArsonShnlenProcess::ArsonShnlenProcess (const ArsonConfig &cfg, const char *infile)
	: ArsonUtilityProcess(cfg, Stdout, i18n("shnlen"), ArsonConfig::PROGRAM_SHNLEN),
	m_seconds(0), m_bytes(0)
{
	setOutput(Stdout, events, ARRSIZE(events));
	
	deferArg(quoteFile(infile));
}

void ArsonShnlenProcess::onStdout (int event, KRegExp *pre)
{
	if (event == EVENT_SHNLEN)
	{
		m_seconds = QString(pre->group(2)).toInt() +
			(QString(pre->group(1)).toInt() * 60);

		if (QString(pre->group(3)).toInt() >= 50)
			++m_seconds;

		m_bytes = QString(pre->group(4)).toInt();
	}
}

/*========================================================*/
/*	SOX process
 *========================================================*/

ArsonSoxProcess::ArsonSoxProcess (ArsonProcessMgr *pMgr, const char *infile)
	: ArsonProcess(i18n("sox"), pMgr, ArsonConfig::PROGRAM_SOX),
	m_infile(infile)
{
	QFileInfo fi (infile);
	m_outfile = fi.dir().filePath(QString("_") + fi.fileName());

	for (int index = 0; QFileInfo(m_outfile).exists(); ++index)
		m_outfile = QString::number(index) + fi.fileName();
	
	THIS << quoteFile(infile)
		 << "-r" << "44100"
		 << "-c" << "2";

	deferArg(quoteFile(m_outfile));
}

/*========================================================*/

bool ArsonSoxProcess::execute (void)
{
	if (ArsonProcess::execute())
	{
		if (ArsonProcessUI *pUI = ui())
		{
			pUI->setText(
				i18n("Fixing track %1").arg(m_infile));
			
			pUI->setMaxProgress(-1);
		}

		return true;
	}

	return false;
}

/*========================================================*/

void ArsonSoxProcess::slotExited (KProcess *ptr)
{
	ArsonProcess *p = (ArsonProcess *) ptr;

	if (p->successful())
	{
		QFile::remove(m_infile);
		rename(m_outfile, m_infile);
	}

	ArsonProcess::slotExited(ptr);
}

/*========================================================*/
