/* Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// net_udp.c

// Modified for Armagetron by Manuel Moos (manuel@moosnet.de)

//#include "quakedef.h"
#include "nNet.h"

#include <string>
#include <stdio.h>

#include <sys/types.h>

#ifndef WIN32
#include <arpa/inet.h> 
#include <netinet/in.h>
#include <netdb.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#else
#define ioctl(a,b,c) ioctlsocket((a),(b),(u_long *)(c))

#undef EINVAL
#undef EINTR
#define EWOULDBLOCK WSAEWOULDBLOCK
#define ECONNREFUSED WSAECONNREFUSED
#define ENOTSOCK WSAENOTSOCK
#define ECONNRESET WSAECONNRESET
#define ECONNREFUSED WSAECONNREFUSED
#define ENETDOWN WSAENETDOWN
#define EINTR WSAEINTR
#define EINVAL WSAEINVAL
#define EISCONN WSAEISCONN
#define ENETRESET WSAENETRESET
#define EOPNOTSUPP WSAEOPNOTSUPP
#define EAFNOSUPPORT WSAEAFNOSUPPORT
#define EADDRNOTAVAIL WSAEADDRNOTAVAIL
#define ESHUTDOWN WSAESHUTDOWN
#define EMSGSIZE WSAEMSGSIZE
#define ETIMEDOUT WSAETIMEDOUT
#define close closesocket
#endif


#include <errno.h>
#include <stdlib.h>


#ifdef __sun__
#include <sys/filio.h>
#endif

#ifdef NeXT
#include <libc.h>
#endif

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 256
#endif

#ifndef WIN32
#include <unistd.h>
#endif

//extern int gethostname (char *, int);
//extern int close (int);

unsigned int net_hostport=4534;
tString my_tcpip_address;
bool tcpipAvailable=false;
tString hostname="UNNAMED";

int net_acceptsocket = -1;  // socket for fielding new connections
static int net_controlsocket;
static int net_broadcastsocket = 0;
static struct sockaddr broadcastaddr;

static unsigned long myAddr;

#include "nNet.h"


#ifdef WIN32
typedef int NET_SIZE; 
#else
#ifdef MACOSX
typedef int NET_SIZE;
#else
typedef size_t NET_SIZE;
#endif
#include <unistd.h>
#endif

enum nSocketError
{
	nSocketError_Ignore,				// nothing special happened
	nSocketError_Reset					// socket needs to be reset
};

nSocketError ANET_Error()
{
#ifdef WIN32
	int error = WSAGetLastError();
#else
	int error = errno;
#endif

	switch ( error )
	{
		case ENOTSOCK:
			return nSocketError_Reset;
			break;
		case ECONNRESET:
			return nSocketError_Reset;
			break;
		case ECONNREFUSED:
		case EWOULDBLOCK:
			return nSocketError_Ignore;
			break;
//		case NOTINITIALISED:
//			break;
		case ENETDOWN:
			break;
		case EFAULT:
			break;
		case EINTR:
			break;
		case EINVAL:
			break;
		case EISCONN:
			break;
		case ENETRESET:
			break;
		case EOPNOTSUPP:
		case EAFNOSUPPORT:
		case EADDRNOTAVAIL:
			return nSocketError_Reset;
			break;
		case ESHUTDOWN:
			break;
#ifndef WIN32
		case EMSGSIZE:
			break;
#endif
		case ETIMEDOUT:
			break;
		default:
			st_Breakpoint();
			break;
	}

//	st_Breakpoint();
	return nSocketError_Ignore;
};

//=============================================================================

int ANET_Init (void)
{
  struct hostent *local;
  char	buff[MAXHOSTNAMELEN];
  struct sockaddr addr;
  char *colon;
  
#ifdef WIN32
	WSADATA		winsockdata;
	WSAStartup( MAKEWORD(1, 1), &winsockdata );
#endif

  /* not for armagetron
     if (COM_CheckParm ("-noudp"))
     return -1;
  */
  
  // determine my name & address
	gethostname(buff, MAXHOSTNAMELEN);
  local = gethostbyname(buff);
  myAddr = *reinterpret_cast<int *>(local->h_addr_list[0]);
  
  // if the quake hostname isn't set, set it to the machine name
  if (strcmp(hostname, "UNNAMED") == 0)
    {
      buff[15] = 0;
      hostname=buff;
    }
  
  if ((net_controlsocket = ANET_OpenSocket (0)) == -1)
    Sys_Error("ANET_Init: Unable to open control socket\n");
  
  reinterpret_cast<struct sockaddr_in *>(&broadcastaddr)->sin_family = AF_INET;
  reinterpret_cast<struct sockaddr_in *>(&broadcastaddr)->sin_addr.s_addr = INADDR_BROADCAST;
  reinterpret_cast<struct sockaddr_in *>(&broadcastaddr)->sin_port = htons(net_hostport);
  
  ANET_GetSocketAddr (net_controlsocket, &addr);
  my_tcpip_address=ANET_AddrToString (&addr);
  colon = strrchr (my_tcpip_address, ':');
  if (colon)
    *colon = 0;
  
  //  Con_Printf("UDP Initialized\n");
  tcpipAvailable = true;
  
  return net_controlsocket;
}

//=============================================================================

void ANET_Shutdown (void)
{
  ANET_Listen (false);
  ANET_CloseSocket (net_controlsocket);
#ifdef WIN32
  WSACleanup();
#endif
}

//=============================================================================
static bool sn_listen = false;

void ANET_Listen (bool state)
{
  sn_listen = state;

  // enable listening
  if (state)
    {
      if (net_acceptsocket != -1)
	return;
      if ((net_acceptsocket = ANET_OpenSocket (net_hostport)) == -1)
	Con_Printf ("ANET_Listen: Unable to open accept socket on desired port, Trying next port...\n");
      return;
    }

  // disable listening
  if (net_acceptsocket == -1)
    return;
  ANET_CloseSocket (net_acceptsocket);
  net_acceptsocket = -1;
}

//=============================================================================

int ANET_OpenSocket (int port)
{
  int newsocket;
  struct sockaddr_in address;
  bool _true = true;

  if ((newsocket = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
    return -1;

  if (ioctl (newsocket, FIONBIO, reinterpret_cast<char *>(&_true)) == -1)
    goto ErrorReturn;

  address.sin_family = AF_INET;
  address.sin_addr.s_addr = INADDR_ANY;
  address.sin_port = htons(port);
  if( bind (newsocket, reinterpret_cast<const sockaddr *>(&address), sizeof(address)) == -1)
    goto ErrorReturn;

  return newsocket;

 ErrorReturn:
  close (newsocket);
  return -1;
}

//=============================================================================

int ANET_CloseSocket (int sock)
{
  if (sock == net_broadcastsocket)
    net_broadcastsocket = 0;
  return close (sock);
}


//=============================================================================
/*
  ============
  PartialIPAddress

  this lets you type only as much of the net address as required, using
  the local network components to fill in the rest
  ============
*/
static int PartialIPAddress (const char *in, struct sockaddr *hostaddr)
{
  char buff[256];
  char *b;
  int addr;
  int num;
  int mask;
  int run;
  int port;
	
  buff[0] = '.';
  b = buff;
  strcpy(buff+1, in);
  if (buff[1] == '.')
    b++;

  addr = 0;
  mask=-1;
  while (*b == '.')
    {
      b++;
      num = 0;
      run = 0;
      while (!( *b < '0' || *b > '9'))
	{
	  num = num*10 + *b++ - '0';
	  if (++run > 3)
	    return -1;
	}
      if ((*b < '0' || *b > '9') && *b != '.' && *b != ':' && *b != 0)
	return -1;
      if (num < 0 || num > 255)
	return -1;
      mask<<=8;
      addr = (addr<<8) + num;
    }
	
  if (*b++ == ':')
    port = atoi(b);
  else
    port = net_hostport;

  hostaddr->sa_family = AF_INET;
  ((struct sockaddr_in *)hostaddr)->sin_port = htons((short)port);	
  ((struct sockaddr_in *)hostaddr)->sin_addr.s_addr = (myAddr & htonl(mask)) | htonl(addr);
	
  return 0;
}
//=============================================================================

int ANET_Connect (int sock, struct sockaddr *addr)
{
  return 0;
}

//=============================================================================

void ANET_ResetSocket( int& sock )
{
	if ( sn_listen && sock == net_acceptsocket )
	{
		ANET_Listen( false );
		ANET_Listen( true );
		sock = net_acceptsocket;
	}
	
	else if ( sock == net_controlsocket )
	{
		bool listen = sn_listen;
		ANET_Shutdown();
		ANET_Init();
		ANET_Listen( listen );
		sock = net_controlsocket;
	}
	else
	{
		ANET_CloseSocket( sock );
		sock = ANET_OpenSocket( 0 );
	}
}

//=============================================================================

int ANET_CheckNewConnections (void)
{
  unsigned long	available;

  if (net_acceptsocket == -1)
    return -1;

  if (ioctl (net_acceptsocket, FIONREAD, &available) == -1)
	{
		switch ( ANET_Error() )
		{
			case nSocketError_Reset:
				ANET_Shutdown();
				ANET_Init();
				ANET_Listen( true );
				break;
			case nSocketError_Ignore:
				break;
			default:
			    Sys_Error ("UDP: ioctlsocket (FIONREAD) failed\n");
				break;			
		}

		return -1;
	}

  if (available)
    return net_acceptsocket;
  return -1;
}

//=============================================================================

int ANET_Read (int& sock, int8 *buf, int len, struct sockaddr *addr)
{
  NET_SIZE addrlen = sizeof (struct sockaddr);

  int ret;
  
  ret = recvfrom (sock, buf, len, 0, (struct sockaddr *)addr, &addrlen);

	if ( ret < 0 )
	{
		switch ( ANET_Error() )
		{
			case nSocketError_Reset:
				{
					ANET_ResetSocket( sock );
					return 0;
				}
				break;
			case nSocketError_Ignore:
				return 0;
				break;
		}
	}

  return ret;
}

//=============================================================================

int ANET_MakeSocketBroadcastCapable (int sock)
{
  int				i = 1;

  // make this socket broadcast capable
  if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (char *)&i, sizeof(i)) < 0)
    return -1;
  net_broadcastsocket = sock;

  return 0;
}

//=============================================================================

int ANET_Broadcast (int& sock, const int8 *buf, int len, unsigned int port)
{
  int ret;

  if (sock != net_broadcastsocket)
    {
      if (net_broadcastsocket != 0)
	Sys_Error("Attempted to use multiple broadcasts sockets\n");
      ret = ANET_MakeSocketBroadcastCapable (sock);
      if (ret == -1)
	{
	  Con_Printf("Unable to make socket broadcast capable\n");
	  return ret;
	}
    }

  reinterpret_cast<struct sockaddr_in *>(&broadcastaddr)->sin_port = htons(port);

  return ANET_Write (sock, buf, len, &broadcastaddr);
}

//=============================================================================

int ANET_Write (int& sock, const int8 *buf, int len, const struct sockaddr *addr)
{
  int ret;

  ret = sendto (sock, buf, len, 0, (struct sockaddr *)addr, sizeof(struct sockaddr));

	if ( ret < 0 )
	{
		switch ( ANET_Error() )
		{
			case nSocketError_Reset:
				{
					ANET_ResetSocket( sock );
					return 0;
				}
				break;
			case nSocketError_Ignore:
				return 0;
				break;
		}
	}

  return ret;
}

//=============================================================================

char *ANET_AddrToString (const struct sockaddr *addr)
{
  static char buffer[22];
  int haddr;
  
  haddr = ntohl(((struct sockaddr_in *)addr)->sin_addr.s_addr);
  sprintf(buffer, "%d.%d.%d.%d:%d", (haddr >> 24) & 0xff, (haddr >> 16) & 0xff, (haddr >> 8) & 0xff, haddr & 0xff, ntohs(((struct sockaddr_in *)addr)->sin_port));
  return buffer;
}

//=============================================================================

int ANET_StringToAddr (const char *string, struct sockaddr *addr)
{
  int ha1, ha2, ha3, ha4, hp;
  int ipaddr;

  sscanf(string, "%d.%d.%d.%d:%d", &ha1, &ha2, &ha3, &ha4, &hp);
  ipaddr = (ha1 << 24) | (ha2 << 16) | (ha3 << 8) | ha4;

  addr->sa_family = AF_INET;
  ((struct sockaddr_in *)addr)->sin_addr.s_addr = htonl(ipaddr);
  ((struct sockaddr_in *)addr)->sin_port = htons(hp);
  return 0;
}

//=============================================================================

int ANET_GetSocketAddr (int sock, struct sockaddr *addr)
{
  NET_SIZE addrlen = sizeof(struct sockaddr);
  unsigned int a;
  
  memset(addr, 0, sizeof(struct sockaddr));
  getsockname(sock, (struct sockaddr *)addr, &addrlen);
  a = ((struct sockaddr_in *)addr)->sin_addr.s_addr;
  if (a == 0 || a == inet_addr("127.0.0.1"))
    ((struct sockaddr_in *)addr)->sin_addr.s_addr = myAddr;
  
  return 0;
}

//=============================================================================

int ANET_GetNameFromAddr (const struct sockaddr *addr, char *name)
{
  struct hostent *hostentry;
  
  hostentry = gethostbyaddr ((char *)&((struct sockaddr_in *)addr)->sin_addr, sizeof(struct in_addr), AF_INET);
  if (hostentry)
    {
      strncpy (name, (char *)hostentry->h_name, NET_NAMELEN - 1);
      return 0;
    }
  
  strcpy (name, ANET_AddrToString (addr));
  return 0;
}

//=============================================================================

int ANET_GetAddrFromName(const char *name, struct sockaddr *addr)
{
  struct hostent *hostentry;

  if (name[0] >= '0' && name[0] <= '9')
    return PartialIPAddress (name, addr);
	
  hostentry = gethostbyname (name);
  if (!hostentry)
    return -1;

  addr->sa_family = AF_INET;
  ((struct sockaddr_in *)addr)->sin_port = htons(net_hostport);	
  ((struct sockaddr_in *)addr)->sin_addr.s_addr = *(int *)hostentry->h_addr_list[0];

  return 0;
}

//=============================================================================

int ANET_AddrCompare (struct sockaddr *addr1, struct sockaddr *addr2)
{
  if (addr1->sa_family != addr2->sa_family)
    return -1;

  if (((struct sockaddr_in *)addr1)->sin_addr.s_addr != ((struct sockaddr_in *)addr2)->sin_addr.s_addr)
    return -1;

  if (((struct sockaddr_in *)addr1)->sin_port != ((struct sockaddr_in *)addr2)->sin_port)
    return 1;

  return 0;
}

//=============================================================================

int ANET_GetSocketPort (struct sockaddr *addr)
{
  return ntohs(((struct sockaddr_in *)addr)->sin_port);
}


int ANET_SetSocketPort (struct sockaddr *addr, int port)
{
  ((struct sockaddr_in *)addr)->sin_port = htons(port);
  return 0;
}

//=============================================================================




