// vs_text_layout.h                     -*-c++-*-
//
// Copyright 2004 Daniel Burrows

#ifndef VS_TEXT_LAYOUT_H
#define VS_TEXT_LAYOUT_H

#include "vscreen_widget.h"
#include "fragment_contents.h"

class fragment;

/** Code to display formatted text.
 *
 *  The text to display is composed of a tree of "fragments".  A
 *  fragment stores some amount of text; at any time, it can be
 *  formatted to a particular width.  The main layout mechanisms
 *  provided are flowboxes and clipboxes: flowboxes word-wrap their
 *  contents to a particular width, while clipboxes clip their
 *  contents to a particular width.  These boxes can be nested, if the
 *  user feels like it for some reason or other, although some
 *  nestings are non-sensical (for instance, placing a flowbox inside
 *  a smaller flowbox is likely to lead to really ugly text).
 *
 *  This provides some primitive layout mechanisms; higher-level
 *  layouts can be expressed in terms of these.
 */
class vs_text_layout:public vscreen_widget
{
public:
  /** Create an empty vs_text_layout. */
  vs_text_layout();

  /** Create a vs_text_layout with the given root fragment.
   *
   *  All fragments are implicitly placed within a clipbox of width
   *  equal to the width of this widget.
   */
  vs_text_layout(fragment *f);

  /** Handle the given keypress.  Returns \b true if the keystroke
   *  was "consumed" by this widget.
   */
  bool handle_char(chtype ch);

  /** Change the fragment being displayed in this layout widget. */
  void set_fragment(fragment *f);

  /** Return the requested size of this widget. */
  size size_request();

  /** Return \b true iff the cursor is visible in this widget. */
  bool get_cursorvisible();

  /** Return the location of the cursor in this widget. */
  point get_cursorloc();

  /** Return \b true iff this widget should be given focus. */
  bool focus_me();

  /** Paint this widget. */
  void paint();

  /** Move the view one line down. */
  void line_down();

  /** Move the view one line up. */
  void line_up();

  /** Move the view to the top of the widget. */
  void move_to_top();

  /** Move the view to the bottom of the widget. */
  void move_to_bottom();

  /** Move a page forward. */
  void page_down();

  /** Move a page back. */
  void page_up();

  /** Page based on a scrollbar signal.
   *
   *  \param dir the direction to page: if \b true, call page_up();
   *       else call page_down().
   */
  void scroll(bool dir);

  /** Calculate the number of lines needed to format this at the given width.
   *
   *  \param w the desired width of the widget
   *
   *  \return the number of lines of text which will be generated at
   *  the given width.
   */
  unsigned int height_for_width(unsigned int w);


  /** Delete the root fragment. */
  ~vs_text_layout();

  /** A signal that is called whenever the "location" of the view
   *  within the text changes.
   */
  SigC::Signal2<void, int, int> location_changed;

  static keybindings *bindings;

  static void init_bindings();
private:
  /** Move the starting location of the widget. */
  void set_start(unsigned int new_start);

  /** Update the cached contents of the widget, if necessary. */
  void freshen_contents();

  /** Called when this needs layout. */
  void layout_me();

  /** Emits the above signal based on the present location of the display. */
  void do_signal();

  /** The line which is currently at the top of the widget. */
  size_t start;

  /** The root fragment of this layout.  This is always a clipbox. */
  fragment *f;

  /** Cache the current contents of the widget. */
  fragment_contents contents;

  /** If \b true, the current cached contents need to be updated. */
  bool stale;

  /** The width of the widget the last time we updated the cached contents. */
  int lastw;
};

#endif
