// vs_pager.cc
//
//  Copyright 2000 Daniel Burrows

#include "vs_pager.h"

#include "vscreen.h"
#include "vs_minibuf_win.h"
#include "vs_editline.h"
#include "config/keybindings.h"

#include <unistd.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/fcntl.h>

#include <sigc++/object_slot.h>

using namespace std;

keybindings *vs_pager::bindings=NULL;

vs_pager::vs_pager(const char *text, int len)
  :vscreen_widget(), first_line(0), first_column(0), text_width(0)
{
  set_text(text, len);

  do_layout.connect(slot(*this, &vs_pager::layout_me));
}

vs_pager::vs_pager(const string &s)
  :vscreen_widget(), first_line(0), first_column(0), text_width(0)
{
  set_text(s);

  do_layout.connect(slot(*this, &vs_pager::layout_me));
}

vs_pager::~vs_pager() {}

void vs_pager::set_text(const string &s)
{
  set_text(s.c_str(), s.size());
}

void vs_pager::set_text(const char *text, string::size_type len)
{
  string::size_type loc=0;

  text_width=0;

  lines.clear();

  while(loc<len)
    {
      string curline;
      col_count cur_width=0;

      while(loc<len && text[loc]!='\n')
	{
	  if(text[loc]=='\t')
	    cur_width+=8;
	  else
	    ++cur_width;

	  curline+=text[loc];

	  ++loc;
	}

      if(loc<len)
	++loc;

      text_width=max(cur_width, text_width);

      lines.push_back(curline);
    }

  // Bouncing to the start is easiest.
  first_line=0;
  first_column=0;

  do_line_signal();
  vscreen_queuelayout();
  vscreen_redraw();
}

void vs_pager::do_line_signal()
{
  int realmax=max<int>(lines.size()-getmaxy(), 0);
  line_changed(first_line, realmax);
}

void vs_pager::do_column_signal()
{
  int realmax=max<int>(text_width-getmaxx(), 0);
  column_changed(first_column, realmax);
}

void vs_pager::scroll_up(line_count nlines)
{
  if(first_line<nlines)
    first_line=0;
  else
    first_line-=nlines;

  do_line_signal();
  vscreen_update();
}

void vs_pager::scroll_down(line_count nlines)
{
  first_line=min(first_line+nlines, lines.size()-getmaxy());

  do_line_signal();
  vscreen_update();
}

void vs_pager::scroll_left(col_count ncols)
{
  if(first_column<ncols)
    first_column=0;
  else
    first_column-=ncols;

  do_column_signal();
  vscreen_update();
}

void vs_pager::scroll_right(col_count ncols)
{
  first_column=min(first_column+ncols, text_width-getmaxx());

  do_column_signal();
  vscreen_update();
}

void vs_pager::scroll_top()
{
  first_line=0;

  do_line_signal();
  vscreen_update();
}

void vs_pager::scroll_bottom()
{
  first_line=lines.size()-getmaxy();

  do_line_signal();
  vscreen_update();
}

void vs_pager::scroll_page(bool dir)
{
  if(dir)
    scroll_up(getmaxy());
  else
    scroll_down(getmaxy());
}

void vs_pager::search_for(string s)
{
  if(s!="")
    last_search=s;
  else if(last_search=="")
    {
      beep();
      return;
    }

  line_count i=first_line+1;

  for( ; i<lines.size(); ++i)
    {
      col_count loc=lines[i].find(last_search);

      if(loc!=string::npos)
	{
	  first_line=i;
	  do_line_signal();

	  if(loc<first_column)
	    {
	      first_column=loc;
	      do_column_signal();
	    }
	  else if(loc+last_search.size()>=first_column+getmaxx())
	    {
	      if(last_search.size()>(col_count) getmaxx())
		first_column=loc;
	      else
		first_column=loc+last_search.size()-getmaxx();

	      do_column_signal();
	    }

	  vscreen_update();
	  return;
	}
    }
  beep();
}

bool vs_pager::handle_char(chtype ch)
{
  if(bindings->key_matches(ch, "Up"))
    scroll_up(1);
  else if(bindings->key_matches(ch, "Down"))
    scroll_down(1);
  else if(bindings->key_matches(ch, "Left"))
    scroll_left(1);
  else if(bindings->key_matches(ch, "Right"))
    scroll_right(1);
  else if(bindings->key_matches(ch, "PrevPage"))
    scroll_up(getmaxy());
  else if(bindings->key_matches(ch, "NextPage"))
    scroll_down(getmaxy());
  else if(bindings->key_matches(ch, "Begin"))
    scroll_top();
  else if(bindings->key_matches(ch, "End"))
    scroll_bottom();
  else
    return vscreen_widget::handle_char(ch);

  return true;
}

// Could find out what dimensions changed and only update along those?
void vs_pager::layout_me()
{
  do_line_signal();
  do_column_signal();
}

void vs_pager::paint()
{
  int width,height;
  getmaxyx(height, width);

  for(int y=0; y<height && first_line+y<lines.size(); ++y)
    {
      const string &s=lines[first_line+y];
      col_count x=0, curr=0;

      while(curr<s.size() && x<first_column+width)
	{
	  if(s[curr]=='\t')
	    x+=8;
	  else if(x>=first_column)
	    {
	      mvaddch(y, x-first_column, (unsigned char) s[curr]);
	      ++x;
	    }
	  else
	    ++x;

	  ++curr;
	}
    }
}

size vs_pager::size_request()
{
  return size(text_width, lines.size());
}

void vs_pager::init_bindings()
{
  bindings=new keybindings(&global_bindings);
}

vs_file_pager::vs_file_pager():vs_pager("")
{
}

vs_file_pager::vs_file_pager(string filename):vs_pager("")
{
  load_file(filename);
}

vs_file_pager::vs_file_pager(const char *text, int size)
  :vs_pager(text, size)
{
}

void vs_file_pager::load_file(string filename)
{
  int fd=open(filename.c_str(), O_RDONLY, 0644);

  if(fd==-1)
    set_text("open: "+filename+": "+strerror(errno));
  else
    {
      struct stat buf;
      if(fstat(fd, &buf)<0)
	{
	  close(fd);
	  fd=-1;
	  set_text("fstat: "+filename+": "+strerror(errno));
	}
      else
	{
	  const char *contents=(const char *) mmap(NULL,
						   buf.st_size,
						   PROT_READ,
						   MAP_SHARED,
						   fd,
						   0);

	  if(contents==MAP_FAILED)
	    {
	      close(fd);
	      fd=-1;
	      set_text("mmap: "+filename+": "+strerror(errno));
	      // FIXME: just display something in the widget itself?
	    }
	  else
	    vs_pager::set_text(contents, buf.st_size);


	  if(fd!=-1)
	    {
	      munmap((void *) contents, buf.st_size);
	      close(fd);
	    }
	}
    }
}
