// ---------------------------------------------------------------------------
// - Real.cpp                                                                -
// - standard object library - real class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Real.hpp"
#include "Vector.hpp"
#include "Recycle.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "cmth.hpp"
#include "ccnv.hpp"

namespace aleph {
  /// the real supported quarks
  static const long QUARK_OPP      = String::intern ("++");
  static const long QUARK_OMM      = String::intern ("--");
  static const long QUARK_ADD      = String::intern ("+");
  static const long QUARK_SUB      = String::intern ("-");
  static const long QUARK_MUL      = String::intern ("*");
  static const long QUARK_DIV      = String::intern ("/");
  static const long QUARK_EQL      = String::intern ("==");
  static const long QUARK_NEQ      = String::intern ("!=");
  static const long QUARK_LTH      = String::intern ("<");
  static const long QUARK_LEQ      = String::intern ("<=");
  static const long QUARK_GTH      = String::intern (">");
  static const long QUARK_GEQ      = String::intern (">=");
  static const long QUARK_AEQ      = String::intern ("+=");
  static const long QUARK_SEQ      = String::intern ("-=");
  static const long QUARK_MEQ      = String::intern ("*=");
  static const long QUARK_DEQ      = String::intern ("/=");
  static const long QUARK_QEQ      = String::intern ("?=");
  static const long QUARK_ABS      = String::intern ("abs");
  static const long QUARK_LOG      = String::intern ("log");
  static const long QUARK_EXP      = String::intern ("exp");
  static const long QUARK_SIN      = String::intern ("sin");
  static const long QUARK_COS      = String::intern ("cos");
  static const long QUARK_TAN      = String::intern ("tan");
  static const long QUARK_SQRT     = String::intern ("sqrt");
  static const long QUARK_NANP     = String::intern ("nan-p");
  static const long QUARK_ASIN     = String::intern ("asin");
  static const long QUARK_ACOS     = String::intern ("acos");
  static const long QUARK_ATAN     = String::intern ("atan");
  static const long QUARK_SINH     = String::intern ("sinh");
  static const long QUARK_COSH     = String::intern ("cosh");
  static const long QUARK_TANH     = String::intern ("tanh");
  static const long QUARK_ASINH    = String::intern ("asinh");
  static const long QUARK_ACOSH    = String::intern ("acosh");
  static const long QUARK_ATANH    = String::intern ("atanh");
  static const long QUARK_FLOOR    = String::intern ("floor");
  static const long QUARK_ZEROP    = String::intern ("zero-p");
  static const long QUARK_FORMAT   = String::intern ("format");
  static const long QUARK_CEILING  = String::intern ("ceiling");

  // the real recycler
  static Recycle recycler;

  // allocate a new real
  void* Real::operator new (const t_size size) {
    return recycler.pop (size);
  }

  // delete a real
  void Real::operator delete (void* handle) {
    recycler.push (handle);
  }

  // the default precision
  t_real Real::d_precision = 0.00001;

  // create a new default real

  Real::Real (void) {
    d_value = 0.0;
  }

  // create a new real from a native real

  Real::Real (const t_real value) {
    d_value = value;
  }

  // create a new real from an Integer

  Real::Real (const Integer& value) {
    d_value = value.d_value;
  }

  // create a new real from a string

  Real::Real (const String& value) {
    bool status = false;
    const char* data = value.tochar ();
    d_value = c_atod (data,status);
    delete [] data;
    if (status == false) 
      throw Exception ("literal-error", "illegal string real number",value);
  }
  
  // copy constructor for this real

  Real::Real (const Real& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Real::repr (void) const {
    return "Real";
  }

  // return a literal representation of this real

  String Real::toliteral (void) const {
    return tostring ();
  }

  // return a string representation of this real

  String Real::tostring (void) const {
    char* buffer = c_dtoa (d_value);
    String result (buffer);
    delete [] buffer;
    return result;
  }

  // return a clone of this real

  Object* Real::clone (void) const {
    return new Real (*this);
  }

  // return the real serial code

  t_byte Real::serialid (void) const {
    return SERIAL_REAL_ID;
  }

  // serialize this real

  void Real::wrstream (Output& os) const {
    rdlock ();
    String sval = tostring ();
    sval.wrstream (os);
    unlock ();
  }

  // deserialize this real

  void Real::rdstream (Input& is) {
    wrlock ();
    String sval;
    sval.rdstream (is);
    *this = sval;
    unlock ();
  }

  // return a format string of this real

  String Real::format (const long precision) const {
    if (precision < 0) 
      throw Exception ("precision-error",
		       "invalid negative precision with real format");
    char* buffer = c_dtoap (d_value, precision);
    String result (buffer);
    delete [] buffer;
    return result;
  }

  // return the double representation

  t_real Real::toreal (void) const {
    return d_value;
  }

  // get an integer from this real
  
  t_long Real::tointeger (void) const {
    return (t_long) d_value;
  }
  
  // set an real with a value

  Real& Real::operator = (const double value) {
    d_value = value;
    return *this;
  }

  // set an real with a value

  Real& Real::operator = (const Real& value) {
    d_value = value.d_value;
    return *this;
  }

  // compare an real with a native value
  
  bool Real::operator == (const long value) const {
    return (d_value == value);
  }
  
  // compare an real with a native value
  
  bool Real::operator != (const long value) const {
    return (d_value != value);
  }

  // compare an real with a double value
  
  bool Real::operator == (const t_real value) const {
    return (d_value == value);
  }
  
  // compare an real with a native value
  
  bool Real::operator != (const t_real value) const {
    return (d_value != value);
  }

  // compare two reals

  bool Real::operator == (const Real& value) const {
    return (d_value == value.d_value);
  }

  bool Real::operator != (const Real& value) const {
    return (d_value != value.d_value);
  }

  bool Real::operator < (const Real& value) const {
    return (d_value < value.d_value);
  }

  bool Real::operator <= (const Real& value) const {
    return (d_value <= value.d_value);
  }

  bool Real::operator > (const Real& value) const {
    return (d_value > value.d_value);
  }

  bool Real::operator >= (const Real& value) const {
    return (d_value >= value.d_value);
  }

  // add two double reals together from two double reals

  Real operator + (const Real& x, const Real& y) {
    return Real (x.d_value + y.d_value);
  }

  // add a real to this one

  Real& Real::operator += (const Real& x) {
    d_value += x.d_value;
    return *this;
  }

  // substract a real to this one

  Real& Real::operator -= (const Real& x) {
    d_value -= x.d_value;
    return *this;
  }

  // substract two double reals together from two double reals

  Real operator - (const Real& x, const Real& y) {
    return Real (x.d_value - y.d_value);
  }

  // compute the opposite of a real

  Real operator - (const Real& x) {
    return Real (-x.d_value);
  }

  // multiply two double reals together from two double reals

  Real operator * (const Real& x, const Real& y) {
    return Real (x.d_value * y.d_value);
  }

  // multiply a real with this one

  Real& Real::operator *= (const Real& x) {
    d_value *= x.d_value;
    return *this;
  }

  // divide two double reals together from two double reals

  Real operator / (const Real& x, const Real& y) {
    return Real (x.d_value / y.d_value);
  }

  // return true if the number is nan

  bool Real::isnan (void) const {
    return c_isnan (d_value);
  }

  // return the ceiling of this number

  Real Real::ceiling (void) const {
    double result = c_ceiling (d_value);
    return Real (result);
  }

  // return the floor of this number

  Real Real::floor (void) const {
    double result = c_floor (d_value);
    return Real (result);
  }

  // return the absolute value of this number
  
  Real Real::abs (void) const {
    double result = c_abs (d_value);
    return Real (result);
  }
  
  // return the remainder of this number with its argument
  
  Real Real::mod (const Real& x) const {
    double result = c_mod (d_value,x.d_value);
    return Real (result);
  }
  
  // return the square root of this real
  
  Real Real::sqrt (void) const {
    bool   status = false;
    double result = c_sqrt (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with sqrt call");
    return Real (result);
  }
  
  // return the natural logarithm of this real
  
  Real Real::log (void) const {
    bool   status = false;
    double result = c_log (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with log call");
    return Real (result);
  }
  
  // return the exponential of this number
  
  Real Real::exp (void) const {
    double result = c_exp (d_value);
    return Real (result);
  }
  
  // return the power of this number with the argument
  
  Real Real::pow (const Real& x) const {
    double result = c_pow (d_value,x.d_value);
    return Real (result);
  }
  
  // return the sine of this number
  
  Real Real::sin (void) const {
    double result = c_sin (d_value);
    return Real (result);
  }
  
  // return the cosine of this number
  
  Real Real::cos (void) const {
    double result = c_cos (d_value);
    return Real (result);
  }
  
  // return the tangent of this number
  
  Real Real::tan (void) const {
    double result = c_tan (d_value);
    return Real (result);
  }
  
  // return the arc sine of this number
  
  Real Real::asin (void) const {
    bool   status = false;
    double result = c_asin (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with asin call");
    return Real (result);
  }
  
  // return the arc cosine of this number
  
  Real Real::acos (void) const {
    bool   status = false;
    double result = c_acos (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with acos call");
    return Real (result);
  }
  
  // return the arc tangent of this number
  
  Real Real::atan (void) const {
    bool   status = false;
    double result = c_atan (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with atan call");
    return Real (result);
  }
  
  // return the hyperbolic sine of this number
  
  Real Real::sinh (void) const {
    bool   status = false;
    double result = c_sinh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with sinh call");
    return Real (result);
  }
  
  // return the hyperbolic cosine of this number
  
  Real Real::cosh (void) const {
    bool   status = false;
    double result = c_cosh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with cosh call");
    return Real (result);
  }

  // return the hyperbolic tangent of this number

  Real Real::tanh (void) const {
    bool   status = false;
    double result = c_tanh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with tanh call");
    return Real (result);
  }
  
  // return the hyperbolic arc sine of this number

  Real Real::asinh (void) const {
    bool   status = false;
    double result = c_asinh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with asinh call");
    return Real (result);
  }
  
  // return the hyperbolic arc cosine of this number
  
  Real Real::acosh (void) const {
    bool   status = false;
    double result = c_acosh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with acosh call");
    return Real (result);
  }

  // return the hyperbolic arc tangent of this number
  
  Real Real::atanh (void) const {
    bool   status = false;
    double result = c_atanh (d_value,status);
    if (status == false) throw Exception ("math-error",
					  "math error with atanh call");
    return Real (result);
  }

  // evaluate an object to a real value

  double Real::evalto (Runnable* robj, Nameset* nset, Object* object) {
    Object* obj = (object == nilp) ? nilp : object->eval (robj, nset);
    Real*   val = dynamic_cast <Real*> (obj);
    if (val == nilp) throw Exception ("type-error", "nil object to evaluate");
    return val->toreal ();
  }

  // create a new real in a generic way

  Object* Real::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Real;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with real constructor");
    // try to map the real argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Real;

    // try an integer object
    Integer* ival = dynamic_cast <Integer*> (obj);
    if (ival != nilp) return new Real (ival->tointeger ());

    // try a real object
    Real* rval = dynamic_cast <Real*> (obj);
    if (rval != nilp) return new Real (*rval);

    // try a character object
    Character* cval = dynamic_cast <Character*> (obj);
    if (cval != nilp) return new Real (cval->tochar ());

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Real (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with real constructor",
		     obj->repr ());
  }

  // operate this real with another object

  Object* Real::oper (Runnable* robj, t_oper type, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    Real*    dobj = dynamic_cast <Real*>    (object);
    switch (type) {
    case Object::ADD:
      if (iobj != nilp) return new Real (d_value + iobj->d_value);
      if (dobj != nilp) return new Real (d_value + dobj->d_value);
      break;
    case Object::SUB:
      if (iobj != nilp) return new Real (d_value - iobj->d_value);
      if (dobj != nilp) return new Real (d_value - dobj->d_value);
      break;
    case Object::MUL:
      if (iobj != nilp) return new Real (d_value * iobj->d_value);
      if (dobj != nilp) return new Real (d_value * dobj->d_value);
      break;
    case Object::DIV:
      if (iobj != nilp) return new Real (*this / *iobj);
      if (dobj != nilp) return new Real (*this / *dobj);
      break;
    case Object::EQL:
      if (iobj != nilp) return new Boolean (d_value == iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value == dobj->d_value);
      break;
    case Object::NEQ:
      if (iobj != nilp) return new Boolean (d_value != iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value != dobj->d_value);
      break;
    case Object::GEQ:
      if (iobj != nilp) return new Boolean (d_value >= iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value >= dobj->d_value);
      break;
    case Object::GTH:
      if (iobj != nilp) return new Boolean (d_value > iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value > dobj->d_value);
      break;
    case Object::LEQ:
      if (iobj != nilp) return new Boolean (d_value <= iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value <= dobj->d_value);
      break;
    case Object::LTH:
      if (iobj != nilp) return new Boolean (d_value < iobj->d_value);
      if (dobj != nilp) return new Boolean (d_value < dobj->d_value);
      break;
    case Object::MINUS:
      return new Real (-d_value);
      break;
    }
    throw Exception ("type-error", "invalid operand with real",
		     Object::repr (object));
  }

  // set an object to this real

  Object* Real::vdef (Runnable* robj, Nameset* nset, Object* object) {
    Integer* iobj = dynamic_cast <Integer*> (object);
    if (iobj != nilp) {
      d_value = iobj->d_value;
      return this;
    }
    Real* fobj = dynamic_cast <Real*> (object);
    if (fobj != nilp) {
      d_value = fobj->d_value;
      return this;
    }
    throw Exception ("type-error", "invalid object with real vdef",
		     Object::repr (object));
  }

  // apply this real with a set of arguments and a quark

  Object* Real::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_ABS)      return new Real   (abs     ());
      if (quark == QUARK_LOG)      return new Real   (log     ());
      if (quark == QUARK_EXP)      return new Real   (exp     ());
      if (quark == QUARK_SIN)      return new Real   (sin     ());
      if (quark == QUARK_COS)      return new Real   (cos     ());
      if (quark == QUARK_TAN)      return new Real   (tan     ());
      if (quark == QUARK_SINH)     return new Real   (sinh    ());
      if (quark == QUARK_COSH)     return new Real   (cosh    ());
      if (quark == QUARK_TANH)     return new Real   (tanh    ());
      if (quark == QUARK_ASIN)     return new Real   (asin    ());
      if (quark == QUARK_ACOS)     return new Real   (acos    ());
      if (quark == QUARK_ATAN)     return new Real   (atan    ());
      if (quark == QUARK_SQRT)     return new Real   (sqrt    ());
      if (quark == QUARK_NANP)     return new Real   (isnan   ());
      if (quark == QUARK_ASINH)    return new Real   (asinh   ());
      if (quark == QUARK_ACOSH)    return new Real   (acosh   ());
      if (quark == QUARK_ATANH)    return new Real   (atanh   ());
      if (quark == QUARK_FLOOR)    return new Real   (floor   ());
      if (quark == QUARK_CEILING)  return new Real   (ceiling ());

      if (quark == QUARK_OPP) {
	d_value++;
	return this;
      }
      if (quark == QUARK_OMM) {
	d_value--;
	return this;
      }
      if (quark == QUARK_ZEROP) {
	return new Boolean (d_value == 0.0);
      }
    }

    if (argc == 1) {
      if (quark == QUARK_ADD) return oper (robj, Object::ADD, argv->get (0));
      if (quark == QUARK_SUB) return oper (robj, Object::SUB, argv->get (0)); 
      if (quark == QUARK_MUL) return oper (robj, Object::MUL, argv->get (0)); 
      if (quark == QUARK_DIV) return oper (robj, Object::DIV, argv->get (0)); 
      if (quark == QUARK_EQL) return oper (robj, Object::EQL, argv->get (0)); 
      if (quark == QUARK_NEQ) return oper (robj, Object::NEQ, argv->get (0)); 
      if (quark == QUARK_LTH) return oper (robj, Object::LTH, argv->get (0)); 
      if (quark == QUARK_LEQ) return oper (robj, Object::LEQ, argv->get (0)); 
      if (quark == QUARK_GTH) return oper (robj, Object::GTH, argv->get (0)); 
      if (quark == QUARK_GEQ) return oper (robj, Object::GEQ, argv->get (0)); 

      if (quark == QUARK_AEQ) {
	t_real val = argv->getireal (0);
	d_value += val;
	return this;
      }
      if (quark == QUARK_SEQ) {
	t_real val = argv->getireal (0);
	d_value -= val;
	return this;
      }
      if (quark == QUARK_MEQ) {
	t_real val = argv->getireal (0);
	d_value *= val;
	return this;
      }
      if (quark == QUARK_DEQ) {
	t_real val = argv->getireal (0);
	if (val == 0) throw Exception ("divide-error", "division by zero");
	d_value = d_value / val;
	return this;
      }
      if (quark == QUARK_QEQ) {
	t_real val = argv->getreal (0);
	t_real pos = (d_value >= val) ? d_value - val : val - d_value;
	return new Boolean (pos <= Real::d_precision);
      }
      if (quark == QUARK_FORMAT) {
	long precision = argv->getint (0);
	return new String (format (precision));
      }
    }

    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
