// ---------------------------------------------------------------------------
// - HtmlPage.cpp                                                            -
// - aleph:www library - html page class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Runnable.hpp"
#include "HtmlPage.hpp"
#include "Exception.hpp"

namespace aleph {

  // the html page supported quarks
  static const long QUARK_ADDHTTP   = String::intern ("add-http");
  static const long QUARK_ADDHEAD   = String::intern ("add-head");
  static const long QUARK_ADDBODY   = String::intern ("add-body");
  static const long QUARK_ADDMETA   = String::intern ("add-meta");
  static const long QUARK_ADDTITLE  = String::intern ("add-title");
  static const long QUARK_ADDSTYLE  = String::intern ("add-style");
  static const long QUARK_WRITECGI  = String::intern ("write-cgi");
  static const long QUARK_ADDCOOKIE = String::intern ("add-cookie");
  static const long QUARK_ADDAUTHOR = String::intern ("add-author");
  static const long QUARK_GETBUFFER = String::intern ("get-buffer");
  static const long QUARK_WRITEHTTP = String::intern ("write-http");
  static const long QUARK_WRITEHEAD = String::intern ("write-head");
  static const long QUARK_WRITEBODY = String::intern ("write-body");
  static const long QUARK_WRITEPAGE = String::intern ("write-page");

  // the base http header
  static const char* HTTPTYP = "Content-type: text/html\n";
  // the page document type
  static const char* DOCTYPE = 
  "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">";
  static const char* CONTENT = "<meta http-equiv=\"Content-Type\""
                               "content=\"text/html; charset=iso-8859-1\">\n";

  // create a new html page

  HtmlPage::HtmlPage (void) {
    // add the page type to the http header
    addhttp (HTTPTYP);
    // add the content type to the head
    addhead (CONTENT);
  }

  // create a new html page with an initial header

  HtmlPage::HtmlPage (const String& header) {
    // add the page type to the http header
    addhttp (HTTPTYP);
    // add the content type to the head
    addhead (header);
  }

  // return the class name

  String HtmlPage::repr (void) const {
    return "HtmlPage";
  }

  // add a title to this page

  void HtmlPage::addtitle (const String& title) {
    String line = "<title>";
    line = line + title + "</title>\n";
    addhead (line);
  }

  // add a author name

  void HtmlPage::addauthor (const String& author) {
    addmeta ("author", author);
  }

  // add a page style

  void HtmlPage::addstyle (const String& style) {
    String line = "<link href=";
    line = line + style + " rel=\"stylesheet\" type=\"text/css\">\n";
    addhead (line);
  }

  // add a meta markup by name and content

  void HtmlPage::addmeta (const String& name, const String& content) {
    String line = "<meta name=";
    line = line + '"' + name + '"';
    line = line + ' ' + "content=";
    line = line + '"' + content + "\">\n";
    addhead (line);
  }

  // add some content to the http header
  
  void HtmlPage::addhttp (const String& content) {
    wrlock ();
    d_http.add (content);
    unlock ();
  }

  // add some content to the head
  
  void HtmlPage::addhead (const String& content) {
    wrlock ();
    d_head.add (content);
    unlock ();
  }

  // add some content to the body
  
  void HtmlPage::addbody (const String& content) {
    wrlock ();
    d_body.add (content);
    unlock ();
  }

  void HtmlPage::addcookie (Cookie* cookie) {
    if (cookie == nilp) return;
    wrlock ();
    try {
      String data = cookie->tostring () + eolc;
      addhttp (data);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write the http header
  void HtmlPage::writehttp (Output& os) const {
    rdlock ();
    try {
      d_http.write (os);
      os.newline ();
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write the head content
  
  void HtmlPage::writehead (Output& os) const {
    rdlock ();
    try {
      os << "<head>" << eolc;
      d_head.write (os);
      os << "</head>" << eolc;
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write the body content
  
  void HtmlPage::writebody (Output& os) const {
    rdlock ();
    try {
      os << "<body>" << eolc;
      d_body.write (os);
      os << "</body>" << eolc;
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write the page content

  void HtmlPage::writepage (Output& os) const {
    rdlock ();
    try {
      // add the dtd and html markup
      os << DOCTYPE  << eolc;
      os << "<html>" << eolc;
      writehead (os);
      writebody (os);
      os << "</html>" << eolc;
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();    
  }

  // write the page as a cgi return

  void HtmlPage::writecgi (Output& os) const {
    rdlock ();
    try {
      writehttp (os);
      writepage (os);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // return a buffer with the html page

  Buffer* HtmlPage::getbuffer (void) const {
    rdlock ();
    Buffer* result = new Buffer;
    try {
      // add the dtd and html markup
      result->add (String (DOCTYPE));
      result->add (eolc);
      result->add (String ("<html>\n"));
      // add the head
      result->add (String ("<head>\n"));
      result->add (d_head);
      result->add (String ("</head>\n"));
      // add the body
      result->add (String ("<body>\n"));
      result->add (d_body);
      result->add (String ("</body>\n"));
      // close the html markup
      result->add (String ("</html>\n"));
    } catch (...) {
      delete result;
      unlock ();
      throw;
    }
    unlock ();
    return result;
  }

  // create a new html page in a generic way

  Object* HtmlPage::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new HtmlPage;
    // wrong arguments
    throw Exception ("argument-error", 
		     "too many arguments with html page object");
  }

  // apply this html page object with a set of arguments and a quark

  Object* HtmlPage::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch generic quarks
    if (quark == QUARK_ADDHTTP) {
      String result;
      for (long i = 0; i < argc; i++) {
	Literal* lobj = dynamic_cast <Literal*> (argv->get (i));
	if (lobj == nilp) 
	  throw Exception ("type-error", "invalid object for http header"); 
	result = result + lobj->tostring ();
      }
      addhttp (result);
      return nilp;
    }
    if (quark == QUARK_ADDHEAD) {
      String result;
      for (long i = 0; i < argc; i++) {
	Literal* lobj = dynamic_cast <Literal*> (argv->get (i));
	if (lobj == nilp) 
	  throw Exception ("type-error", "invalid object to add in body"); 
	result = result + lobj->tostring ();
      }
      addhead (result);
      return nilp;
    }
    if (quark == QUARK_ADDBODY) {
      String result;
      for (long i = 0; i < argc; i++) {
	Literal* lobj = dynamic_cast <Literal*> (argv->get (i));
	if (lobj == nilp) 
	  throw Exception ("type-error", "invalid object to add in body"); 
	result = result + lobj->tostring ();
      }
      addbody (result);
      return nilp;
    }

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETBUFFER) return getbuffer ();
      if (quark == QUARK_WRITEHTTP) {
	Output* os = (robj == nilp) ? nilp : robj->getos ();
	if (os == nilp) return nilp;
	writehttp (*os);
	return nilp;
      }
      if (quark == QUARK_WRITEHEAD) {
	Output* os = (robj == nilp) ? nilp : robj->getos ();
	if (os == nilp) return nilp;
	writehead (*os);
	return nilp;
      }
      if (quark == QUARK_WRITEBODY) {
	Output* os = (robj == nilp) ? nilp : robj->getos ();
	if (os == nilp) return nilp;
	writebody (*os);
	return nilp;
      }
      if (quark == QUARK_WRITEPAGE) {
	Output* os = (robj == nilp) ? nilp : robj->getos ();
	if (os == nilp) return nilp;
	writepage (*os);
	return nilp;
      }
      if (quark == QUARK_WRITECGI) {
	Output* os = (robj == nilp) ? nilp : robj->getos ();
	if (os == nilp) return nilp;
	writecgi (*os);
	return nilp;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADDTITLE) {
	String title = argv->getstring (0);
	addtitle (title);
	return nilp;
      }
      if (quark == QUARK_ADDAUTHOR) {
	String author = argv->getstring (0);
	addauthor (author);
	return nilp;
      }
      if (quark == QUARK_ADDSTYLE) {
	String style = argv->getstring (0);
	addstyle (style);
	return nilp;
      }
      if (quark == QUARK_ADDCOOKIE) {
	Cookie* cookie = dynamic_cast <Cookie*> (argv->get (0));
	if (cookie == nilp) 
	  throw Exception ("type-error", "cookie expected with add-cookie");
	addcookie (cookie);
	return nilp;
      }
      if (quark == QUARK_WRITEHTTP) {
	Output* os = dynamic_cast <Output*> (argv->get (0));
	if (os == nilp) 
	  throw Exception ("type-error", "output stream expected with write");
	writehttp (*os);
	return nilp;
      }
      if (quark == QUARK_WRITEHEAD) {
	Output* os = dynamic_cast <Output*> (argv->get (0));
	if (os == nilp) 
	  throw Exception ("type-error", "output stream expected with write");
	writehead (*os);
	return nilp;
      }
      if (quark == QUARK_WRITEBODY) {
	Output* os = dynamic_cast <Output*> (argv->get (0));
	if (os == nilp) 
	  throw Exception ("type-error", "output stream expected with write");
	writebody (*os);
	return nilp;
      }
      if (quark == QUARK_WRITEPAGE) {
	Output* os = dynamic_cast <Output*> (argv->get (0));
	if (os == nilp) 
	  throw Exception ("type-error", "output stream expected with write");
	writepage (*os);
	return nilp;
      }
      if (quark == QUARK_WRITECGI) {
	Output* os = dynamic_cast <Output*> (argv->get (0));
	if (os == nilp) 
	  throw Exception ("type-error", "output stream expected with write");
	writecgi (*os);
	return nilp;
      }
    }

    // dispatch 2 argument
    if (argc == 2) {
      if (quark == QUARK_ADDMETA) {
	String name    = argv->getstring (0);
	String content = argv->getstring (1);
	addmeta (name, content);
	return nilp;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
