/**********************************************************************
 ** Flags class: manages a flag object
 **
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef FLAGS_C
#define FLAGS_C

#ifndef CONVERTER
#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "flags.h"
#include "global.h"
#include "errlog.h"
#include "lexer.h"
#include "builder.h"
#include "newfuncts.h"
#include "player.h"
#else
#include "../../include/strings.h"
#include "../../include/sysdep.h"
#include "../../include/flags.h"
#include "convconfig.h"
#endif

/***********************************************************************
 ** Flags (constructor) - Constructs the flags object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Flags::Flags()
{
   int i;
   flag_long_int *tmp_flags = NULL;
   num_flags = DEFAULT_NUM_FLAGS;
   flag_list = NULL;

   /* initialize them to zero */ 
   for (i=0; i<num_flags; i++)
   {
      if (flag_list == NULL)
      {
         flag_list = new flag_long_int();
         tmp_flags = flag_list;
      }
      else
      {
         tmp_flags->next_flag = new flag_long_int();
         tmp_flags = tmp_flags->next_flag;
      }
      tmp_flags->next_flag = NULL;
      BZERO(( char * )tmp_flags, sizeof(flag_long_int));
   }
}

/***********************************************************************
 ** Flags (constructor) - Constructs the flag object, taking as a parameter
 **                       the number of long ints to have
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Flags::Flags(int num_long_int)
{
   int i;
   flag_long_int *tmp_flags = NULL;
   num_flags = num_long_int;
   flag_list = NULL;

   /* initialize them to zero */ 
   for (i=0; i<num_flags; i++)
   {
      if (flag_list == NULL)
      {
         flag_list = new flag_long_int();
         tmp_flags = flag_list;
      }
      else
      {
         tmp_flags->next_flag = new flag_long_int();
         tmp_flags = tmp_flags->next_flag;
      }
      tmp_flags->next_flag = NULL;
      BZERO(( char * )tmp_flags, sizeof(flag_long_int));
   }
}


/***********************************************************************
 ** ~Flags (destructor) - cleans up and destroys the flags object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Flags::~Flags()
{ 
   flag_long_int *tmp_flags; 

   tmp_flags = flag_list;
   while (tmp_flags != NULL)
   {
      flag_list = tmp_flags->next_flag;
      delete tmp_flags;
      tmp_flags = flag_list;
   }

}


/***********************************************************************
 ** operator = - copies one flags object to the other
 **
 ** Parameters: old_flags - the old flags to copy to this one
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Flags::operator = (Flags *old_flags)
{
   return copy_flags(old_flags->get_flagstruct());   
}


/***********************************************************************
 ** copy_flags - copies one flags object to the other
 **
 ** Parameters: old_flags - the old flags to copy to this one
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Flags::copy_flags(Flags *old_flags)
{
   return copy_flags(old_flags->get_flagstruct());   
}


/***********************************************************************
 ** copy_flags - copies one flag structure to another
 **
 ** Parameters: old_flags - the old flags to copy to this one
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Flags::copy_flags(flag_long_int *old_flags)
{
   flag_long_int *tmp_oldflag;
   flag_long_int *tmp_newflag;
   int count = 0;

   if (old_flags == NULL)
      return -1;

   tmp_oldflag = old_flags;
   while (tmp_oldflag != NULL)
   {
      count++;
      tmp_oldflag = tmp_oldflag->next_flag;
   }

   if (count != num_flags)
      return -1;

   tmp_oldflag = old_flags;
   tmp_newflag = flag_list;
   while (tmp_oldflag != NULL)
   {
      memcpy(&(tmp_newflag->flags), &(tmp_oldflag->flags), sizeof(long));   
      tmp_newflag = tmp_newflag->next_flag;
      tmp_oldflag = tmp_oldflag->next_flag;
   }
   return 1;
}


/***********************************************************************
 ** operator = - copies one flags object to the other
 **
 ** Parameters: old_flags - the old flags to copy to this one
 **
 ** Returns:  1 for success, -1 for failure
 **
 ***********************************************************************/

int Flags::operator = (flag_long_int *old_flags)
{
   return copy_flags(old_flags);
}


/***********************************************************************
 ** set_flag - sets a certain flag in the list
 **
 ** Parameters: flag_num - the flag number to set
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::set_flag(int flag_num)
{
   int flag_offset;
   flag_long_int *tmp_flags;
   int i;
   int which_flag;

   flag_offset = flag_num / 32;
   if (flag_offset > num_flags)
      return -1;

   tmp_flags = flag_list;

   for (i=0; i<flag_offset; i++)
      tmp_flags = tmp_flags->next_flag;

   which_flag = flag_num - (flag_offset * 32);

   tmp_flags->flags |= (1 << which_flag);

   return 1;
}

/***********************************************************************
 ** get_flag - sets a certain flag in the list
 **
 ** Parameters: flag_num - the flag number to check
 **
 ** Returns: 1 if set, 0 if not set
 **
 ***********************************************************************/

int Flags::get_flag(int flag_num)
{
   int flag_offset;
   flag_long_int *tmp_flags;
   int i;
   int which_flag;

   flag_offset = flag_num / 32;
   if (flag_offset > num_flags)
      return -1;

   tmp_flags = flag_list;

   for (i=0; i<flag_offset; i++)
      tmp_flags = tmp_flags->next_flag;

   which_flag = flag_num - (flag_offset * 32);

   if ((tmp_flags->flags & (1 << which_flag)) != 0)
      return 1;
   else
      return 0; 
}


/***********************************************************************
 ** clr_flag - sets a certain flag in the list
 **
 ** Parameters: flag_num - the flag number to check
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::clr_flag(int flag_num)
{
   int flag_offset;
   flag_long_int *tmp_flags;
   int i;
   int which_flag;

   flag_offset = flag_num / 32;
   if (flag_offset > num_flags)
      return -1;

   tmp_flags = flag_list;

   for (i=0; i<flag_offset; i++)
      tmp_flags = tmp_flags->next_flag;

   which_flag = flag_num - (flag_offset * 32);

   tmp_flags->flags &= ~(1 << which_flag);

   return 1;
}

/***********************************************************************
 ** write_flag - writes a flag to a file
 **
 ** Parameters: flag_num - the flag number to check
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::write_flag(FILE *the_file)
{
   flag_long_int *tmp_flags;

   tmp_flags = flag_list;
   while (tmp_flags != NULL)
   {
      fprintf(the_file, "%ld\n", tmp_flags->flags);
      tmp_flags = tmp_flags->next_flag;
   }
   fprintf(the_file, "\n");
   return 1;
}

#ifndef CONVERTER
/***********************************************************************
 ** display_flags - displays the flags, which are set and which arent
 **
 ** Parameters: flag_num - the flag number to check
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::display_flags(Player *the_player, char *the_name, 
                        char *nametable[], char *flagname, int frozen[])
{
   int i = 0;
   Strings holder1;
   Strings holder2;
   int flag_frozen;

   the_player->send_plr("&+W%s for %s&*\n", flagname, the_name);
   the_player->send_plr("&+WFlagName\t &+g|  &+WStatus  &+g| "
                        "&+WFlagName\t\t &+g|  &+WStatus  &+g|\n");
   the_player->send_plr(
     "&+g-------------------------------------------------------------\n");
   while (nametable[i] != NULL)
   {  
      flag_frozen = 0;
      if (is_frozen(frozen, i))
         flag_frozen = 1;

      holder1.sprintf("&+c%s%s&+g | %s &+g|", nametable[i],
            ((strlen(nametable[i]) > 7) ? ((strlen(nametable[i]) > 15) ? 
					    "" : "\t") : "\t\t"),
	      (flag_frozen) ? " &+CFrozen&* " : 
                       ((get_flag(i) ? "&+G   On   &*" : "&+R   Off  ")));
      if (nametable[i+1] != NULL)
      {
         flag_frozen = 0;
         if (is_frozen(frozen, i+1))
            flag_frozen = 1;

         holder2.sprintf("&+c%s%s&+g | %s &+g|", nametable[i+1],
            ((strlen(nametable[i+1]) > 1) ?
            ((strlen(nametable[i+1]) > 9) ? ((strlen(nametable[i+1]) > 17) ?
					    "" : "\t") : "\t\t") : "\t\t\t"),
	      (flag_frozen) ? " &+CFrozen&* " : 
                     ((get_flag(i+1) ? "&+G   On   &*" : "&+R   Off  ")));

         the_player->send_plr("%s %s\n", holder1.str_show(), 
                                                     holder2.str_show()); 
      }
      else
      {
         the_player->send_plr("%s\n", holder1.str_show());
         i -= 1;
      }
      i += 2;
   }
   the_player->send_plr(
     "&+g-------------------------------------------------------------\n");
   the_player->send_plr("&+WNum of flags: &+G%d&*\n", i);

   return 1;
}


/***********************************************************************
 ** display_flags - displays the flags to builders, which are set and 
 **                 which arent
 **
 ** Parameters: flag_num - the flag number to check
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::display_flags(Builder *the_builder, char *the_name, 
                           char *nametable[], char *flagname, int frozen[])
{
   int i = 0;
   Strings holder1;
   Strings holder2;
   int flag_frozen;

   the_builder->send_bldr("&+W%s for %s&*\n", flagname, the_name);
   the_builder->send_bldr("&+WFlagName\t &+g|  &+WStatus  &+g| "
                           "&+WFlagName\t\t &+g|  &+WStatus  &+g|\n");
   the_builder->send_bldr(
     "&+g-------------------------------------------------------------\n");
   while (nametable[i] != NULL)
   {  
      flag_frozen = 0;
      if (is_frozen(frozen, i))
         flag_frozen = 1;

      holder1.sprintf("&+c%s%s&+g | %s &+g|", nametable[i],
            ((strlen(nametable[i]) > 7) ? ((strlen(nametable[i]) > 15) ? 
					    "" : "\t") : "\t\t"),
	      (flag_frozen) ? " &+CFrozen&* " : 
                       ((get_flag(i) ? "&+G   On   &*" : "&+R   Off  ")));

      if (nametable[i+1] != NULL)
      {
         flag_frozen = 0;
         if (is_frozen(frozen, i+1))
            flag_frozen = 1;
   
         holder2.sprintf("&+c%s%s&+g | %s &+g|", nametable[i+1],
            ((strlen(nametable[i+1]) > 1) ?
            ((strlen(nametable[i+1]) > 9) ? ((strlen(nametable[i+1]) > 17) ? 
					    "" : "\t") : "\t\t") : "\t\t\t"),
	      (flag_frozen) ? " &+CFrozen&* " : 
                     ((get_flag(i+1) ? "&+G   On   &*" : "&+R   Off  ")));

         the_builder->send_bldr("%s %s\n", holder1.str_show(), 
                                                     holder2.str_show()); 
      }
      else
      {
         the_builder->send_bldr("%s\n", holder1.str_show());
         i -= 1;
      }
      i += 2;
   }
   the_builder->send_bldr(
    "&+g-------------------------------------------------------------\n");
   the_builder->send_bldr("&+WNum of flags: &+G%d&*\n", i);

   return 1;
}
#endif

/***********************************************************************
 ** get_by_name - gets a flagnumber from a character name of the flag
 **
 ** Parameters: the_name - the flagname to look for
 **             flaglist - the list to look through for it
 **
 ** Returns: flagnumber if success, -1 if failure
 **
 ***********************************************************************/

int Flags::get_by_name(char *the_name, char *flaglist[])
{
   int i = 0;
   
   if ((the_name == NULL) || (flaglist == NULL))
      return -1;

   while (flaglist[i] != NULL)
   {
      if (!STRNCASECMP(the_name, flaglist[i], strlen(the_name)))
         return i;
      i++;
   }
   return -1;
}


/***********************************************************************
 ** get_by_name - gets a flagnumber from a character name of the flag
 **
 ** Parameters: the_name - the flagname to look for
 **             flaglist - the list to look through for it
 **
 ** Returns: flagnumber if success, -1 if failure
 **
 ***********************************************************************/

flag_long_int *Flags::get_flagstruct(void)
{
   return flag_list;
}

/***********************************************************************
 ** set_all - sets all the flags to on
 **
 ** Parameters: flagnames
 **
 ** Returns: num set if success, -1 if failure
 **
 ***********************************************************************/

int Flags::set_all(char *flagnames[])
{
   int i = 0;

   while (flagnames[i] != NULL)
   {
      set_flag(i);
      i++;
   }
   return i;
}

/***********************************************************************
 ** clr_all - clears all the flags to on
 **
 ** Parameters: flagnames
 **
 ** Returns: num set if success, -1 if failure
 **
 ***********************************************************************/

int Flags::clr_all(char *flagnames[])
{
   int i = 0;

   while (flagnames[i] != NULL)
   {
      clr_flag(i);
      i++;
   }
   return i;
}

/***********************************************************************
 ** set_by_table - sets all the flags according to a table that is passed in
                   The last in the table should be a -1
 **
 ** Parameters: flat_table
 **
 ** Returns: num set if success, -1 if failure
 **
 ***********************************************************************/

int Flags::set_by_table(int flag_table[])
{
   int i = 0;

   while (flag_table[i] != -1)
   {
      set_flag(flag_table[i]);
      i++;
   }
   return i;
}


/***********************************************************************
 ** copy_save_struct - copies a structure designed to be saved into the
 **                    users database into this flag struct
 **
 ** Parameters: flat_table
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::copy_from_save_struct(save_flags *the_flags)
{
   flag_long_int *tmp_flags;

   tmp_flags = flag_list;
   
   if ((the_flags == NULL) || (num_flags != 6))
      return -1;

   memcpy(&(tmp_flags->flags), &(the_flags->flag1), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(tmp_flags->flags), &(the_flags->flag2), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(tmp_flags->flags), &(the_flags->flag3), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(tmp_flags->flags), &(the_flags->flag4), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(tmp_flags->flags), &(the_flags->flag5), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(tmp_flags->flags), &(the_flags->flag6), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   return 1;
}


/***********************************************************************
 ** copy_to_save_struct - copies a structure designed to be saved into the
 **                    users database into this flag struct
 **
 ** Parameters: flat_table
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::copy_to_save_struct(save_flags *the_flags)
{
   flag_long_int *tmp_flags;

   tmp_flags = flag_list;
   
   if ((the_flags == NULL) || (num_flags != 6))
      return -1;

   BZERO(( char * )the_flags, sizeof(save_flags));

   memcpy(&(the_flags->flag1), &(tmp_flags->flags), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(the_flags->flag2), &(tmp_flags->flags), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(the_flags->flag3), &(tmp_flags->flags), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(the_flags->flag4), &(tmp_flags->flags), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(the_flags->flag5), &(tmp_flags->flags), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;
   memcpy(&(the_flags->flag6), &(tmp_flags->flags), sizeof(long));   
   tmp_flags = tmp_flags->next_flag;

   return 1;
}


/***********************************************************************
 ** read_flags - reads a flag from a file and loads it into here
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: 1 if success, -1 if failure
 **
 ***********************************************************************/

int Flags::read_flags(FILE *the_file, ErrLog *error_log)
{
   flag_long_int *tmp_flags;
   token_record  *the_token;
   long          long_flag = 0;
   Strings       token_str;

   tmp_flags = flag_list;
   while (tmp_flags != NULL)
   {  
      the_token = get_token(the_file, '\0');
      token_str = the_token->the_string;
      if (the_token->token_type == T_MINUS)
      {
         the_token = get_token(the_file, '\0');
         token_str.str_cat(the_token->the_string);
      }
#ifndef CONVERTER
      if (the_token->token_type != T_NUMERICAL)
      {
          if (error_log != NULL)
            error_log->log_err("Error in flags format", "read_flags");
         return -1;
      }
#endif

      long_flag = atol(token_str.str_show());

      BZERO(( char * )&(tmp_flags->flags), sizeof(long));
      memcpy(&(tmp_flags->flags), &long_flag, sizeof(long));
      tmp_flags = tmp_flags->next_flag;
   }
   return 1;
}


/***********************************************************************
 ** clr_frozen - clears a list of frozen flags that should not be set
 **
 ** Parameters: the_list - the list of flags that should be cleared
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Flags::clr_frozen(int the_list[])
{
   int i = 0;

   while (the_list[i] != -1)
   {
      clr_flag(the_list[i]);
      i++;
   }
}


/***********************************************************************
 ** is_frozen - detects if a flag is frozen or not
 **
 ** Parameters: the_list - the list of frozen flags
 **
 ** Returns: 1 for frozen, 0 for not
 **
 ***********************************************************************/

int Flags::is_frozen(int the_list[], int the_flag)
{
   int i = 0;

   while (the_list[i] != -1)
   {
      if (the_list[i] == the_flag)
         return 1;

      i++;
   }
   return 0;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Flags::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Flags::get_mem_size_dynamic()
{
   int size = 0;
   flag_long_int *tmp_flag;
 
   tmp_flag = flag_list;
   while (tmp_flag != NULL)
   {
      size += sizeof(tmp_flag);
      tmp_flag = tmp_flag->next_flag;
   }

   return size;
}

int Flags::check_null()
{
   if (flag_list == NULL)
      RAISE(11);

   return 1;
}

#endif









