(in-package "ACL2")

#|

This file deals with the RTL primitives, natp, bvecp, unknown, unknown2, reset,
and reset2.  It includes the ACL2 functions that are used in the formalization
of the RTL semantics.

|#

;; The book rtlarr.lisp contains the definitions of our "array" operators ag
;; and as, and has nice rules about their definition.
(include-book "rtlarr")

;; 1. bit-vector constants

;; This is not generated by translate-rtl when :output-for-proof is set to T.
(defmacro n! (i n)
  (declare (ignore n))
  i)


;; 2. equality comparison

(defun log= (x y)
  (if (equal x y) 1 0))

(defun log<> (x y)
  (if (equal x y) 0 1))


;; 3. unsigned inequalities

(defun log< (x y)
  (if (< x y) 1 0))

(defun log<= (x y)
  (if (<= x y) 1 0))

(defun log> (x y)
  (if (> x y) 1 0))

(defun log>= (x y)
  (if (>= x y) 1 0))


;; 4. signed inequalities

;; The following function is not generated by translate-rtl, it is only needed
;; for the definitions of comp2<, comp2<=, etc.
(defun comp2 (x n)
  (if (< x (expt 2 (1- n)))
      x
    (- (- (expt 2 n) x))))

(defun comp2< (x y n) 
  (log< (comp2 x n) (comp2 y n)))

(defun comp2<= (x y n)
  (log<= (comp2 x n) (comp2 y n)))

(defun comp2> (x y n)
  (log> (comp2 x n) (comp2 y n)))

(defun comp2>= (x y n)
  (log>= (comp2 x n) (comp2 y n)))


;; 5. unary logical operations

(defun logand1 (x n)
  (log= x (1- (expt 2 n))))

(defun logior1 (x)
  (if (equal x 0) 0 1))

(defun logxor1 (src)
  (if (oddp (logcount src)) 1 0))


;; 6. bit-vector shifting operations

;; The following function will not be seen in the output from translate-rtl, it
;; is only provided here to define shft.
(defun fl (x) (floor x 1))

;; The following function will not be seen in the output from translate-rtl, it
;; is only provided here to define lshft and rshft.
(defun shft (x s l)
  (mod (fl (* (expt 2 s) x)) (expt 2 (nfix l))))

(defmacro lshft (x s l) 
  `(shft ,x ,s ,l))

(defmacro rshft (x s l)
  `(shft ,x (- ,s) ,l))


;; 7. concatenation operations

(defun cat (x y n)
  (+ (* (expt 2 (nfix n)) (nfix x)) (nfix y)))

(defun mulcat (l n x)
  (if (and (integerp n) (> n 0))
      (cat (mulcat l (1- n) x)
	   x
	   l)
    0))


;; 8. bit-vector access and update

(defun bits (x i j)
  (if (or (not (integerp i))
          (not (integerp j)))
      0
    (fl (/ (mod x (expt 2 (1+ i))) (expt 2 j)))))

(defun bitn (x n)
  (bits x n n))

; The new outer call to bits only effects things if the indices i,j are bad.
; It forces setbits to return something of the advertised size w
(defun setbits (x w i j y)
  (bits (cat (bits x (1- w) (1+ i))
             (cat (bits y (- i j) 0)
                  (bits x (1- j) 0)
                  j)
             (1+ i))
        (1- w)
        0))

(defun setbitn (x w n y)
  (setbits x w n n y))


;; 9. bitwise operations

;; logand, logior, logxor are predefined ACL2 functions

(defun comp1 (x n)
  (if (natp n)
      (+ -1 (expt 2 n) (- (bits x (+ -1 n) 0)))
    0))


;; 10. array access and update

;; aref1, aref2, aset1, aset2 are predefined ACL2 functions

;; Actually, we now generate ag and as, which are defined in rtlarr.lisp
;; in the rtl library.


;; 11. arithmetic operations

(defmacro mod+ (x y n)
  `(bits (+ ,x ,y) (1- ,n) 0))

(defmacro mod* (x y n)
  `(bits (* ,x ,y) (1- ,n) 0))

;; the following function is not generated in the translate-rtl output. It is
;; only needed to define 'mod-
(defun comp2-inv (x n)
  (if (< x 0)
      (+ x (expt 2 n))
    x))

(defun mod- (x y n)
  (comp2-inv (- x y) n))

;; NOTE -- the following definition of decode is "flawed". We still need to add
;; assumptions to "allow" this definition to be used.

(defun decode (x n)
  (if (and (natp x) (< x n)) 
      (ash 1 x) 
    0))

(defun encode (x n)
  (if (zp n) 
      0
    (if (= x (ash 1 n))
        n
      (encode x (1- n)))))

;; floor, rem are predefined ACL2 functions


;; 12. evaluation control operators

(defmacro bind (v x y)
  `(let ((,v ,x)) ,y))

(defmacro if1 (x y z)
  `(if (eql ,x 0) ,z ,y))


;; 13. extra operators

(defun natp1 (x)
  (if (and (integerp x)
           (>= x 0))
      1
    0))

;;Two functions that occur in the translated RTL, representing bit vectors of
;;determined length but undetermined value:

(encapsulate 
 ((reset (key size) t))
 (local (defun reset (key size) (declare (ignore key size)) 0))
 (defthm bvecp-reset (bvecp (reset key size) size)
   :hints (("Goal" :in-theory (enable bvecp)))
   :rule-classes 
   (:rewrite 
    (:forward-chaining :trigger-terms ((reset key size)))	
    (:type-prescription :corollary
                        (and (integerp (reset key size)) 
                             (>= (reset key size) 0))
                        :hints
                        (("Goal" :in-theory '(implies bvecp)))))))

(encapsulate 
 ((unknown (key size n) t))
 (local (defun unknown (key size n) (declare (ignore key size n)) 0))
 (defthm bvecp-unknown (bvecp (unknown key size n) size)
   :hints (("Goal" :in-theory (enable bvecp)))
   :rule-classes 
   (:rewrite
    (:forward-chaining :trigger-terms ((unknown key size n)))
    (:type-prescription :corollary
                        (and (integerp (unknown key size n))
                             (>= (unknown key size n) 0))
                        :hints
                        (("Goal" :in-theory '(implies bvecp)))))))

(encapsulate 
 ((reset2 (key size) t))
 (local (defun reset2 (key size) (declare (ignore key size)) nil))
 
;do we need rule-classes on this thm?
 (defthm bv-arrp-reset2
   (bv-arrp (reset2 key size) size)
   :hints (("goal" :in-theory (enable bv-arrp)))))

(encapsulate 
 ((unknown2 (key size n) t))
 (local (defun unknown2 (key size n) (declare (ignore key size n)) nil))
 
;do we need rule-classes on this thm?
 (defthm bv-arrp-unknown2
   (bv-arrp (unknown2 key size n) size)
   :hints (("goal" :in-theory (enable bv-arrp)))))

(in-theory
 (disable
  log= log<>
  log< log<= log> log>=
  comp2 comp2< comp2<= comp2> comp2>=
  logand logior logxor
  logand1 logior1 logxor1
  fl shft
  cat mulcat
  bits bitn setbits setbitn
  aref1 aref2 aset1 aset2
  comp1 
  comp2-inv mod-
  decode encode ash
  floor mod
;  natp1 ;leave enabled
  ))
